
/*
 *  Diverse SLab audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * FILE:
 *	audioEngineALSA.c
 *
 * NOTES:
 *	This was taken from latency.c, and tweaked to be the device configurator
 *	for the SLab audio interface. This file will only contain the code used by
 *	the SLab audio daemon - configures the desired resolutions, fragments, etc.
 *
 *	The code for the GUI, which is only concerned with the sound source mixing
 *	interface, is separate, in audioGUIALSA.c
 *
 * CALLS:
 *	asoundlib routines.
 *
 * IS CALLED BY:
 *	If it remains as clean as I intend, it should only come from audioEngine.c
 *	if the ALSA_SUPPORT flags are configured on this device. Oops, there ended
 *	up with two exceptions, which I could remove. audioWrite()/audioRead() are
 *	called from libslabadiod. Yes, I should change this, but I do not want yet
 *	another level of indirection for the read/write calls.
 *
 * As of 14/12/99 this uses a single duplex channel. Going to change it to use
 * up to two channels: record channel if not WRONLY, playback channel if not
 * RDONLY device. This is a fair bit of work.
 *
 * 1/3/00 - incorporated alterations for ALSA Rev 0.5.X, with lots of #ifdef's.
 * 27/11/00 - ALSA still not operational. Going back to single duplex dev.
 * 1/1/1 - ALSA operational, but issues with IO Errors after recording period.
 */

#include "slabrevisions.h"
#include "slabcdefs.h"
#include "slabaudiodev.h"
#include "slabalsadev.h"

#include <sys/poll.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if (MACHINE == LINUX)
#include <sched.h>
#endif

#ifdef SUBFRAGMENT
#if (MACHINE == LINUX)
#include <malloc.h>
#endif
#endif

#define DEF_FRAGSIZE 4096
#define DEF_FRAGCNT 32

#include <time.h>

void setscheduler(audioDev)
duplexDev *audioDev;
{
#if (MACHINE == LINUX)
	struct sched_param sched_param;

	if (sched_getparam(0, &sched_param) < 0) {
		printf("Scheduler getparam failed...\n");
		return;
	}
	sched_param.sched_priority = sched_get_priority_max(SCHED_FIFO);
	if (!sched_setscheduler(0, SCHED_FIFO, &sched_param)) {
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Scheduler set to FIFO with priority %i...\n",
				sched_param.sched_priority);
		return;
	}
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("!!!Scheduler set to FIFO with priority %i FAILED!!!\n",
			sched_param.sched_priority);
#else
	printf("Cannot reschedule audio API, wrong machine type\n");
#endif
}

#ifdef ALSA_SUPPORT

/*
 * This assumes the API will manage the memory blocks allocated to the handles,
 * so will only reset them to NULL, not free them.
 */
int
alsaDevClose(duplexDev *audioDev)
{
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("	alsaDevClose(%08x): %08x, %08x\n",
			audioDev,
			&alsaDev[audioDev->devID].chandle,
			&alsaDev[audioDev->devID].phandle);

	if (alsaDev[audioDev->devID].chandle != (snd_pcm_t *) NULL)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("	closing alsa capture channel\n");
		snd_pcm_close((void *) alsaDev[audioDev->devID].chandle);

		alsaDev[audioDev->devID].chandle = (snd_pcm_t *) NULL;
	}

	if (alsaDev[audioDev->devID].phandle != (snd_pcm_t *) NULL)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("	closing alsa playback channel\n");
		snd_pcm_close((void *) alsaDev[audioDev->devID].phandle);

		alsaDev[audioDev->devID].phandle = (snd_pcm_t *) NULL;
	}

	audioDev->fd = audioDev->fd2 = -1;
}

#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5)

/*
 * These are for cases where we have no specific parameters given for the
 * fragment size and count
 */

#include <sys/asoundlib.h>

struct snd_pcm {
	int card;
	int device;
	int mode;
	int ver;
	int fd[2];
	int setup_is_valid[2];
	snd_pcm_channel_setup_t setup[2];
	snd_pcm_mmap_control_t *mmap_caddr[2];
	char *mmap_daddr[2];
	long mmap_size[2];
	snd_pcm_plugin_t *plugin_first[2];
	snd_pcm_plugin_t *plugin_last[2];
	void *plugin_alloc_ptr[4];
	long plugin_alloc_size[4];
	int plugin_alloc_lock[4];
	void *plugin_alloc_xptr[2];
	long plugin_alloc_xsize[2];
};

snd_pcm_mmap_control_t *p_control = NULL, *c_control = NULL;
char *pmap_data = NULL, *cmap_data = NULL;

static char *xitoa(int aaa)
{
	static char str[12];

	sprintf(str, "%i", aaa);
	return str;
}

int
setformat(void *chandle, void *phandle, duplexDev *audioDev, int flags)
{
	int err, silence = 0;

#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
	int again;
#else
	snd_pcm_format_t pformat, cformat;
#endif

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("setformat(%08x, %08x, %08x, %i)\n", chandle, phandle, audioDev,
			flags);

	/*
	 * If samplerate has not been explicitly configured, default it.
	 */
	if (audioDev->writeSampleRate == 0)
		audioDev->writeSampleRate = 44100;
	if (audioDev->readSampleRate == 0)
		audioDev->readSampleRate = 44100;
	/*
	 * If we did not get a channel configuration value, set this to stereo.
	 */
	if (audioDev->channels == 0)
		audioDev->channels = 2;

	/*
	 * The configuration method differences between 0.4.X and 0.5.X are quite
	 * extensive, the structures have all changed. Going to have one big ifdef
	 * over the whole lot.
	 */
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
	/*
	 * Not sure about synchro yet
	if (sync)
		syncro_id(&params.sync);
	 *
	 * Or queues.
	 *
	params.buf.stream.queue_size = *queue;
	 */

	/*
	 * Not sure if we need to do the info request, but since a large part of
	 * this code was ripped out of aplay, and this is what it does, then it can
	 * stay in here. Funnily enough, its also in everybody elses code, perhaps
	 * we are using the same source?
	 */
	if (flags & SND_PCM_OPEN_PLAYBACK)
	{
		int bps, size;
		snd_pcm_channel_info_t pinfo;
		snd_pcm_channel_params_t params;
		snd_pcm_channel_setup_t setup;

		bzero(&params, sizeof(params));
		bzero(&setup, sizeof(setup));
		bzero(&pinfo, sizeof(pinfo));

		params.mode = SND_PCM_MODE_BLOCK;
		params.channel = SND_PCM_CHANNEL_PLAYBACK;

		if (audioDev->cflags & SLAB_8_BIT_OUT)
			params.format.format = SND_PCM_SFMT_S8;
		else
			params.format.format = SND_PCM_SFMT_S16_LE;
		params.format.interleave = 1;
		params.format.voices = audioDev->channels;
		params.format.rate = audioDev->writeSampleRate;
		params.time = 1;

		params.start_mode = SND_PCM_START_GO;
		params.start_mode = SND_PCM_START_DATA;
		params.stop_mode = SND_PCM_STOP_STOP;

		/*
		 * This was taken from aplay.c. It is broken. It calculates an optimal
		 * fragment size, since (at least when using channels) this frag size
		 * is used to calculate the number of available frags. This is fine 
		 * unless the driver redefines the frag size, which it does depending on
		 * any sample rate interpolations it does (44100<->48000 spdif). It can
		 * end up with just a few small fragments, which results in overruns
		 * and/or underruns.
		 *
		 * This needs to be looked at, since we now get IO errors with the
		 * small frag size. MARK.
		 */
		bps = params.format.rate * params.format.voices;
		if (params.format.format == SND_PCM_SFMT_S16_LE)
			bps <<= 1;
		if ((bps >>= 2) < 16)
			bps = 16;

		size = 1;
		while ((size << 1) < bps)
			size <<= 1;

		/*
		 * This should come from the controlBuffer somewhere.
		 */
		if (audioDev->fragSize == 0)
			params.buf.block.frag_size = DEF_FRAGSIZE;
		else
			params.buf.block.frag_size = audioDev->fragSize;

		/*
		 * This looks ugly, but its not so bad. The number of buffers we need
		 * to maintain should be related to the preload factor used by the
		 * audio daemon. This is a function itself of the output segment size
		 * which is in stereo interleaved 32 bit numbers, and the desired frag
		 * size.

		params.buf.block.frags_max =
			audioDev->preLoad * audioDev->OSegmentSize /
			params.buf.block.frag_size / 8;

		 * This should actually come from the preLoad parameter, although we
		 * should consider ensuring that the output segment can provide this
		 * many fragments?
		 */
		params.buf.block.frags_max = audioDev->preLoad;
		params.buf.block.frags_min = 1;

	   	pinfo.channel = SND_PCM_CHANNEL_PLAYBACK;
		if ((err = snd_pcm_channel_info(phandle, &pinfo)) < 0) {
			printf("Playback info error (%i): %s\n", err, snd_strerror(err));
			exit(0);
		}

		/*
		 * Put in known state
		 */
		snd_pcm_channel_flush(phandle, params.channel);

		if ((err = snd_pcm_channel_params(phandle, &params)) < 0) {
			printf("Playback setup error: %s\n", snd_strerror(err));
			exit(0);
		}

		if (audioDev->cflags & SLAB_AUDIODBG)
			printPcmParams(&params);

		if (snd_pcm_channel_prepare(phandle, pinfo.channel) < 0) {
			fprintf(stderr, "unable to prepare channel\n");
			exit(0);
		}
		setup.mode = params.mode;
		setup.channel = params.channel;
		if (snd_pcm_channel_setup(phandle, &setup) < 0) {
			fprintf(stderr, "unable to obtain setup\n");
			exit(1);
		}
		audioDev->writeSampleRate = params.format.rate;
		audioDev->readSampleRate = params.format.rate;

		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("actual playback fragsize is %i\n",
				setup.buf.block.frag_size);

		audioDev->fragSize = setup.buf.block.frag_size;
	}

	if (flags & SND_PCM_OPEN_CAPTURE)
	{
		int bps, size;
		snd_pcm_channel_info_t cinfo;
		snd_pcm_channel_params_t params;
		snd_pcm_channel_setup_t setup;

		bzero(&params, sizeof(params));
		bzero(&setup, sizeof(setup));
		bzero(&cinfo, sizeof(cinfo));

		params.mode = SND_PCM_MODE_BLOCK;
		params.channel = SND_PCM_CHANNEL_CAPTURE;
		if (audioDev->cflags & SLAB_8_BIT_IN)
			params.format.format = SND_PCM_SFMT_S8;
		else
			params.format.format = SND_PCM_SFMT_S16_LE;
		params.format.interleave = 1;
		params.format.voices = audioDev->channels;
		params.format.rate = audioDev->readSampleRate;
		params.start_mode = SND_PCM_START_GO;
		params.start_mode = SND_PCM_START_DATA;
		params.stop_mode = SND_PCM_STOP_STOP;
		params.time = 1;

		params.buf.block.frags_min = 1;
		if (audioDev->cflags & SLAB_8_BIT_IN)
			params.format.format = SND_PCM_SFMT_S8;
		else
			params.format.format = SND_PCM_SFMT_S16_LE;

		cinfo.channel = SND_PCM_CHANNEL_CAPTURE;

		/*
		 * This was taken from aplay.c. It is broken. It calculates an optimal
		 * fragment size, since (at leaast when using channels) this frag size
		 * is used to calculate the number of available frags. This is fine 
		 * unless the driver redefines the frag size, which it does depending on
		 * any sample rate interpolations it does (44100<->48000 spdif). It can
		 * end up with just a few small fragments, which results in overruns
		 * and/or underruns.
		 */
		bps = params.format.rate * params.format.voices;
		if (params.format.format == SND_PCM_SFMT_S16_LE)
			bps <<= 1;
		if ((bps >>= 2) < 16)
			bps = 16;

		size = 1;
		while ((size << 1) < bps)
			size <<= 1;

		if (audioDev->fragSize == 0)
			params.buf.block.frag_size = DEF_FRAGSIZE;
		else
			params.buf.block.frag_size = audioDev->fragSize;

		/*
		 * This looks ugly, but its not so bad. The number of buffers we need
		 * to maintain should be related to the preload factor used by the
		 * audio daemon. This is a function itself of the output segment size
		 * which is in stereo interleaved 32 bit numbers, and the desired frag
		 * size.
		 */
		params.buf.block.frags_max =
			audioDev->preLoad * audioDev->OSegmentSize /
			params.buf.block.frag_size / 8;
		params.buf.block.frags_max = audioDev->preLoad;

		if ((err = snd_pcm_channel_info(chandle, &cinfo)) < 0) {
			printf("Capture params error (%i): %s\n", err, snd_strerror(err));
			exit(0);
		}

		/*
		 * Put in known state
		 */
		snd_pcm_channel_flush(chandle, params.channel);

		if ((err = snd_pcm_channel_params(chandle, &params)) < 0) {
			printf("Capture setup error: %s\n", snd_strerror(err));
			exit(0);
		}

		if (audioDev->cflags & SLAB_AUDIODBG)
			printPcmParams(&params);

		if (snd_pcm_channel_prepare(chandle, cinfo.channel) < 0) {
			fprintf(stderr, "unable to prepare channel\n");
			exit(0);
		}
		setup.mode = params.mode;
		setup.channel = params.channel;
		if (snd_pcm_channel_setup(chandle, &setup) < 0) {
			fprintf(stderr, "unable to obtain setup\n");
			exit(1);
		}
		audioDev->writeSampleRate = params.format.rate;
		audioDev->readSampleRate = params.format.rate;

		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("actual capture fragsize is %i\n",
				setup.buf.block.frag_size);

		audioDev->fragSize = setup.buf.block.frag_size;
	}

	/*
	 * Hm, going to just comment this out for now.
	 *
	if (psetup.buf.stream.queue_size > *queue) {
		*queue = psetup.buf.stream.queue_size;
		again++;
	}
	if (csetup.buf.stream.queue_size > *queue) {
		*queue = csetup.buf.stream.queue_size;
		again++;
	}
	if ((err = snd_pcm_playback_prepare(phandle)) < 0) {
		printf("Playback prepare error: %s\n", snd_strerror(err));
		exit(0);
	}
	if ((err = snd_pcm_capture_prepare(chandle)) < 0) {
		printf("Capture prepare error: %s\n", snd_strerror(err));
		exit(0);
	}	
	 */

#else /* Rest is for ALSA REV 0.4.X */

	bzero(&cformat, sizeof(cformat));

	if (audioDev->cflags & (SLAB_8_BIT_OUT|SLAB_8_BIT_IN))
		cformat.format = SND_PCM_SFMT_U8;
	else
		cformat.format = SND_PCM_SFMT_S16_LE;

	cformat.channels = audioDev->channels;
	/*
	 * And assume the same resolution in each direction.
	 */
	cformat.rate = audioDev->writeSampleRate;

	/*
	 * We have the assumption that each channel has the same set of basic 
	 * parameters. This may not be true if we are using SB-AWE devices, but I
	 * no longer intend to support them in unbalanced IO resolutions.
	 */
	bcopy(&cformat, &pformat, sizeof(cformat));

	if (flags != SND_PCM_OPEN_CAPTURE)
	{
		if ((err = snd_pcm_playback_format(phandle, &pformat)) < 0) {
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("Playback format error: %s\n", snd_strerror(err));
			return -1;
		}
		if ((err = snd_pcm_playback_time(phandle, 1)) < 0) {
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("Playback time error: %s\n", snd_strerror(err));
			return -1;
		}
	}
	if (flags != SND_PCM_OPEN_PLAYBACK)
	{
		if ((err = snd_pcm_capture_format(chandle, &cformat)) < 0) {
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("Capture format error: %s\n", snd_strerror(err));
			return -1;
		}
		if ((err = snd_pcm_capture_time(chandle, 1)) < 0) {
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("Capture time error: %s\n", snd_strerror(err));
			return -1;
		}
	}
	audioDev->writeSampleRate = pformat.rate;
	audioDev->readSampleRate = cformat.rate;
#endif

	return 0;
}

int
setparams(snd_pcm_t *chandle, snd_pcm_t *phandle, int *fragmentsize, int flags, duplexDev *audioDev)
{
	int err, step = 4;
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
	snd_pcm_channel_info_t pinfo;
	snd_pcm_channel_info_t cinfo;
	snd_pcm_channel_params_t pparams;
	snd_pcm_channel_params_t cparams;
#else
	snd_pcm_playback_info_t pinfo;
	snd_pcm_capture_info_t cinfo;
	snd_pcm_playback_params_t pparams;
	snd_pcm_capture_params_t cparams;
#endif /* ALSA_REV_0_4_X */

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("setparams(%08x, %08x, %i, %i)\n",
			chandle, phandle, *fragmentsize, flags);

	if (*fragmentsize == 0)
		*fragmentsize = 4096;

	if (flags != SND_PCM_OPEN_CAPTURE)
	{
		pinfo.channel = SND_PCM_CHANNEL_PLAYBACK;
/*
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
		if ((err = snd_pcm_channel_info(phandle, &pinfo)) < 0)
#else
		if ((err = snd_pcm_playback_info(phandle, &pinfo)) < 0)
#endif
		{
			printf("Playback info error: %s\n", snd_strerror(err));
			return 0;
		}
		if (audioDev->cflags & SLAB_AUDIODBG)
			printPcmInfo(pinfo);
*/
	}
	if (flags != SND_PCM_OPEN_PLAYBACK)
	{
		cinfo.channel = SND_PCM_CHANNEL_CAPTURE;
/*
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
		if ((err = snd_pcm_channel_info(chandle, &cinfo)) < 0)
#else
		if ((err = snd_pcm_capture_info(chandle, &cinfo)) < 0)
#endif
		{
			printf("Capture info error: %s\n", snd_strerror(err));
			return 0;
		}
		if (audioDev->cflags & SLAB_AUDIODBG)
			printPcmInfo(cinfo);
*/
	}

	/*
	 * Going to paraphrase this bit, note that if fragsize does not fit, and
	 * we are not subfragmented we should fail here.
	if (*fragmentsize > pinfo.max_fragment_size)
		*fragmentsize = pinfo.max_fragment_size;
	if (*fragmentsize < cinfo.max_fragment_size)
		*fragmentsize = cinfo.max_fragment_size;
	 */

	/*
	 * This needs to change, since we will end up supporting fragment sizes 
	 * lower than this.
	 */
	if (*fragmentsize < 512)
		*fragmentsize = 4096;

#ifdef DEBUG
	printf("fragmentsize set to %i\n", *fragmentsize);
#endif

	if (flags != SND_PCM_OPEN_CAPTURE)
	{
		bzero(&pparams, sizeof(pparams));
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
		pparams.channel = SND_PCM_CHANNEL_PLAYBACK;
		pparams.mode = SND_PCM_MODE_BLOCK;
		pparams.format.interleave = 1;
		/*
		 * These are wrong, it should come from the controlBuffer.
		 */
		pparams.format.format = SND_PCM_SFMT_S16_LE;
		pparams.format.rate = audioDev->writeSampleRate;
		pparams.format.voices = 1;
		pparams.buf.block.frag_size = *fragmentsize;
/*
		if (audioDev->cflags & SLAB_AUDIODBG)
			printPcmParams(&pparams);
		if ((err = snd_pcm_channel_params(phandle, &pparams)) < 0) {
			printf("snd_pcm_channel_params() playback (%i): %s\n", err,
				snd_strerror(err));
			return -1;
		}
*/
#else
		pparams.fragment_size = *fragmentsize;
		pparams.fragments_max = -1;	/* maximum */
		pparams.fragments_room = 1;
		if ((err = snd_pcm_playback_params(phandle, &pparams)) < 0) {
			printf("snd_pcm_playback_params() failure: %s\n",
				snd_strerror(err));
			return -1;
		}
#endif
		/*
		 * This bit is now a requirement for continuous IO management.
		 */
		audioDev->fd = phandle->fd[pparams.channel];
	}
	if (flags != SND_PCM_OPEN_PLAYBACK)
	{
		bzero(&cparams, sizeof(cparams));

#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
		cparams.channel = SND_PCM_CHANNEL_CAPTURE;
		cparams.mode = SND_PCM_MODE_BLOCK;
		cparams.start_mode = SND_PCM_START_DATA;
		cparams.stop_mode = SND_PCM_STOP_ROLLOVER;
		cparams.format.interleave = 1;
		/*
		 * These are wrong, it should come from the controlBuffer.
		 */
		cparams.format.format = SND_PCM_SFMT_S16_LE;
		cparams.format.rate = audioDev->readSampleRate;
		cparams.format.voices = 2;
		cparams.buf.block.frag_size = *fragmentsize;
		cparams.buf.block.frags_min = 1;
		cparams.buf.block.frags_max = 8;
/*
		if (audioDev->cflags & SLAB_AUDIODBG)
			printPcmParams(&cparams);

printf("	%x, %x, fd %i, %i\n", chandle, &cparams,
chandle->fd[cparams.channel],
cparams.channel);

		if ((err = snd_pcm_channel_params(chandle, &cparams)) < 0) {
			printf("snd_pcm_channel_params() capture (%i): %s\n", err,
				snd_strerror(err));
			return -1;
		}
*/
#else
		cparams.fragment_size = *fragmentsize;
		/*
		 * This should be configured to preload fragments plus one.
		 */
		cparams.fragments_min = 1;	/* wakeup if at least one fragment ready */
		if ((err = snd_pcm_capture_params(chandle, &cparams)) < 0) {
			printf("snd_pcm_capture_params() failure\n");
			return -1;
		}
#endif
		/*
		 * This bit is now a requirement for continuous IO management.
		 */
		audioDev->fd2 = chandle->fd[cparams.channel];
	}

	if (*fragmentsize <= 512) {
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Trying fragment size %i...\n", *fragmentsize);
		return 0;
	}
	return 0;
}

printPcmSetup(setup)
snd_pcm_channel_setup_t *setup;
{
	printf("S	%i, %i, f %i, f %i, f %i, f %i, b %i, b %i, b %i\n",
		setup->channel,
		setup->mode,
		setup->format.interleave,
		setup->format.format,
		setup->format.rate,
		setup->format.voices,
		setup->buf.block.frag_size,
		setup->buf.block.frags_min,
		setup->buf.block.frags_max);
}

printPcmParams(params)
snd_pcm_channel_params_t *params;
{
	printf("P	%i, %i, f %i, f %i, f %i, f %i, %i, %i, b %i, b %i, b %i\n",
		params->channel,
		params->mode,
		params->format.interleave,
		params->format.format,
		params->format.rate,
		params->format.voices,
		params->start_mode,
		params->stop_mode,
		params->buf.block.frag_size,
		params->buf.block.frags_min,
		params->buf.block.frags_max);
}

printPcmInfo(pinfo)
snd_pcm_channel_info_t pinfo;
{
	printf("I	%i, %s, %i, %i, %x, %x, %x, %i, %i, %i, %i\n\
I	%i, %i, %i, %i, %i, %i, %i, %i\n",
		pinfo.subdevice,			/* subdevice number */
		pinfo.subname,		/* subdevice name */
		pinfo.channel,			/* channel information */
		pinfo.mode,			/* transfer mode */
		pinfo.flags,		/* see to SND_PCM_CHNINFO_XXXX */
		pinfo.formats,		/* supported formats */
		pinfo.rates,		/* hardware rates */
		pinfo.min_rate,			/* min rate (in Hz) */
		pinfo.max_rate,			/* max rate (in Hz) */
		pinfo.min_voices,			/* min voices */
		pinfo.max_voices,			/* max voices */
		pinfo.buffer_size,		/* max buffer size in bytes */
		pinfo.min_fragment_size,		/* min fragment size in bytes */
		pinfo.max_fragment_size,		/* max fragment size in bytes */
		pinfo.fragment_align,		/* align fragment value */
		pinfo.fifo_size,			/* stream FIFO size in bytes */
		pinfo.transfer_block_size,	/* bus transfer block size in bytes */
		pinfo.dig_mask,	/* AES/EBU/IEC958 bits, zero = no AES/EBU/IEC958 */
		pinfo.mmap_size,			/* mmap data size */
		pinfo.mixer_device,		/* mixer device */
		pinfo.mixer_eid);	/* mixer element identification */
}

int
playbackunderrun(void *phandle)
{
	int err;
	snd_pcm_channel_status_t pstatus;

	if ((err = snd_pcm_channel_status(phandle, &pstatus)) < 0) {
		printf("Playback status error: %s\n", snd_strerror(err));
		return 0;
	}
	return pstatus.underrun;
}

int
capturefragment(void *rhandle, char *buffer, int index, int fragmentsize)
{
	int err;

	buffer += index * fragmentsize;
	if ((err = snd_pcm_read(rhandle, buffer, fragmentsize)) != fragmentsize) {
		printf("Read error: %s\n", err < 0 ? snd_strerror(err) : xitoa(err));
		return 0;
	}
	return fragmentsize;
}

int
playfragment(void *phandle, char *buffer, int index, int fragmentsize)
{
	int err;

	buffer += index * fragmentsize;
	if ((err = snd_pcm_write(phandle, buffer, fragmentsize)) != fragmentsize) {
		printf("Write error: %s\n", err < 0 ? snd_strerror(err) : xitoa(err));
		return 0;
	}
	return fragmentsize;
}

/*
 * Main will be replaced by the new configure calls. devID is the index for
 * debugging messages, fragSize is desired value if ocnfigured, may be zero.
 */
int
alsaDevOpen(duplexDev *audioDev, int devID, int flags, int fragSize)
{
	/*
	 * These should come from the audioDevName?
	int pcard = 0, pdevice = 0;
	int rcard = 0, rdevice = 0;
	 */
	int err;
	int ridx, pidx, size, ok;
	snd_pcm_channel_status_t pstatus;
	snd_pcm_channel_status_t rstatus;

	if (flags == SLAB_OWRONLY)
		flags = SND_PCM_OPEN_PLAYBACK;
	else if (flags == SLAB_ORDONLY)
		flags = SND_PCM_OPEN_CAPTURE;
	else if (flags == SLAB_ORDWR)
		flags = SND_PCM_OPEN_DUPLEX;
	else {
		printf("	WHAT WERE THOSE FLAGS: %x\n", flags);
		/*
		 * We may as well take some default value if we are unsure about the
		 * request.
		 */
		flags = SND_PCM_OPEN_DUPLEX;
	}

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("alsaDevOpen(%08x, %i, %i, %i)\n",
			audioDev, devID, flags, fragSize);

	/*
	 * There are issues with how we detect if the soundcard was already open.
	 * We will use the same code as that in the OSS library files.
	if ((audioDev->fd != -1) || (audioDev->fd2 != -1))
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Dev already open?\n");
		return(-1);
	}
	 */

	setscheduler(audioDev);

	audioDev->fragSize = fragSize;

	if (audioDev->fragBuf != (char *) NULL)
		free(audioDev->fragBuf);

	/*
	 * If this is the max device, assume we are working for slabPlayer, and 
	 * set some defaults.
	 */
	if (devID == MAX_DEVICES)
	{
		audioDev->fragSize = fragSize = 8192;
		audioDev->OSegmentSize = 32768;
	}

	audioDev->fragBuf = (char *) NULL;
	/*
	 * If we are actually a subfragmented device then we need to add
	 * the fragSize and fragBuf.
	 */
#ifdef SUBFRAGMENT
	if ((audioDev->cflags & SLAB_SUBFRAGMENT) == 0)
		audioDev->fragSize = audioDev->OSegmentSize;
#endif

	/* 
	 * This needs to be changed, to support SLab player. The player should use
	 * the "ninth" audio device, but we do not have that yet.
	 */
	bzero(&alsaDev[devID], sizeof(aDev));

	if (flags != SND_PCM_OPEN_CAPTURE)
	{
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
		int lflags = SND_PCM_OPEN_PLAYBACK;
#else
		int lflags = O_WRONLY;
#endif
		/*
		 * If we are not a readonly device, open the phandle in write only.
		 */
		if ((err = snd_pcm_open(&alsaDev[devID].phandle,
			atoi(audioDev->devName), atoi(audioDev->mixerName),	lflags))
			< 0)
		{
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("Duplex (write) open error: %s\n", snd_strerror(err));
			return -1;
		}
	}

	if (flags != SND_PCM_OPEN_PLAYBACK)
	{
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
		int lflags = SND_PCM_OPEN_CAPTURE;
#else
		int lflags = O_RDONLY;
#endif
		/*
		 * If we are not a writeonly device, open the chandle in read only.
		 */
		if ((err = snd_pcm_open(&alsaDev[devID].chandle,
			atoi(audioDev->devName), atoi(audioDev->mixerName),	lflags))
			< 0)
		{
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("Duplex (read) open error: %s\n", snd_strerror(err));
			return -1;
		}
	}

	if (setformat(alsaDev[devID].chandle, alsaDev[devID].phandle,
		audioDev, flags) < 0)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("setformat() failure\n");
		return -1;
	}

	if (setparams(alsaDev[devID].chandle, alsaDev[devID].phandle,
		&audioDev->fragSize, flags, audioDev)
		< 0)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("setparams() failure\n");
		return -1;
	}
	if (audioDev->cflags & SLAB_AUDIODBG)
	{
		printSndPcm(alsaDev[devID].phandle);
		printSndPcm(alsaDev[devID].chandle);
	}

	/*
	 * Stop the device already, and configure non block mode. The audio start
	 * will put it back in block mode. Binned the block mode request, turning
	 * it into block enable (which is admittedly default anyway).
	snd_pcm_block_mode((void *) alsaDev[devID].phandle, 1);
	snd_pcm_block_mode((void *) alsaDev[devID].chandle, 1);
	snd_pcm_playback_pause((void *) alsaDev[devID].phandle, 1);

	snd_pcm_capture_flush((void *) alsaDev[devID].chandle);
	snd_pcm_playback_flush((void *) alsaDev[devID].phandle);
	 */

	/*
	 * Mallocing a fragment already?
	 */
	if (audioDev->fragSize == 0)
	{
		printf("FRAGSIZE ISSUE\n");
		audioDev->fragSize = DEF_FRAGSIZE;
	}
	if (audioDev->fragBuf)
		free(audioDev->fragBuf);
	audioDev->fragBuf = (char *) malloc(audioDev->fragSize);
	bzero(audioDev->fragBuf, audioDev->fragSize);

	return 0;
}

int
alsaDevAudioStart(duplexDev *audioDev)
{
	char buffer[2];

#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
	/*
	 * We should really check on our IO config, but the API does a check on 
	 * the legality of the snd_pcm_t struct for us anyway.
	 */
	snd_pcm_channel_go(alsaDev[audioDev->devID].phandle,
		SND_PCM_CHANNEL_PLAYBACK);
	snd_pcm_channel_go(alsaDev[audioDev->devID].chandle,
		SND_PCM_CHANNEL_CAPTURE);
/*	slabgettimeofday(&audioDev->startTime); */
#else /* This is 0.4.X */
/*
 	snd_pcm_block_mode((void *) &alsaDev[audioDev->devID].phandle, 1);
 	snd_pcm_block_mode((void *) &alsaDev[audioDev->devID].chandle, 1);
*/
	snd_pcm_playback_pause(alsaDev[audioDev->devID].phandle, 0);
	/*
	 * To force the capture channels to open we should consider putting in here:
	snd_pcm_read(alsaDev[audioDev->devID].chandle, buffer, 0);
	snd_pcm_read(alsaDev[audioDev->devID].chandle, buffer, 0);
	 */
#endif
}

getAlsaInputStatus(duplexDev *audioDev)
{
	snd_pcm_channel_status_t *cstatus;

	if (audioDev->flags == SLAB_WRONLY)
		return;

	cstatus = &alsaDev[audioDev->devID].cstatus;
	bzero(cstatus, sizeof(snd_pcm_channel_status));

	cstatus->mode = SND_PCM_MODE_BLOCK;
	cstatus->channel = SND_PCM_CHANNEL_CAPTURE;

	snd_pcm_channel_status(alsaDev[audioDev->devID].chandle, cstatus);

	return;
}

getAlsaOutputStatus(duplexDev *audioDev)
{
	snd_pcm_channel_status_t *pstatus;

	if (audioDev->flags == SLAB_RDONLY)
		return;

	pstatus = &alsaDev[audioDev->devID].pstatus;
	bzero(pstatus, sizeof(snd_pcm_channel_status));

	pstatus->mode = SND_PCM_MODE_BLOCK;
	pstatus->channel = SND_PCM_CHANNEL_CAPTURE;

	snd_pcm_channel_status(alsaDev[audioDev->devID].phandle, pstatus);

	return;
}

/*
 * channel status - for sync of input to output.
 *
 * This will be used, amongst other things, to force sync. It returns the number
 * of samples that should be inserted into the input channel to sync things up.
 *
 * This needs to be extended for multihead support - if the device is not the
 * primary then we should calculate the number of samples to be inserted into
 * each channel to sync the auxiliary heads to the primary. This will probably
 * be made into a separate routine called "alsaHeadStatus(?)" and it will 
 * assume the channels are in sync, returning a single value to be slipped into
 * each direction.
 */
alsaChannelStatus(duplexDev *audioDev)
{
	int deltaTime, deltaSamples;

	getAlsaInputStatus(audioDev);
	getAlsaOutputStatus(audioDev);

	if (audioDev->cflags & SLAB_AUDIODBG)
	{
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 5)
		printf("Irate%i %i, frags %i, scount %i, count %i, free %i, over %i\n",
			audioDev->devID,
			alsaDev[audioDev->devID].cstatus.status,
			alsaDev[audioDev->devID].cstatus.frag,
			alsaDev[audioDev->devID].cstatus.scount,
			alsaDev[audioDev->devID].cstatus.count,
			alsaDev[audioDev->devID].cstatus.free,
			alsaDev[audioDev->devID].cstatus.overrun);
		printf("  secs %i, usecs %i\n",
			alsaDev[audioDev->devID].cstatus.stime.tv_sec,
			alsaDev[audioDev->devID].cstatus.stime.tv_usec);
		printf("Orate%i %i, frags %i, scount %i, count %i, free %i, under %i\n",
			audioDev->devID,
			alsaDev[audioDev->devID].pstatus.status,
			alsaDev[audioDev->devID].pstatus.frag,
			alsaDev[audioDev->devID].pstatus.scount,
			alsaDev[audioDev->devID].pstatus.count,
			alsaDev[audioDev->devID].pstatus.free,
			alsaDev[audioDev->devID].pstatus.underrun);
		printf("  secs %i, usecs %i\n",
			alsaDev[audioDev->devID].pstatus.stime.tv_sec,
			alsaDev[audioDev->devID].pstatus.stime.tv_usec);
#else
		printf("Irate%i %i, frags %i, fSize %i, count %i, free %i, over %i\n",
			audioDev->devID,
			alsaDev[audioDev->devID].cstatus.rate,
			alsaDev[audioDev->devID].cstatus.fragments,
			alsaDev[audioDev->devID].cstatus.fragment_size,
			alsaDev[audioDev->devID].cstatus.count,
			alsaDev[audioDev->devID].cstatus.free,
			alsaDev[audioDev->devID].cstatus.overrun);
		printf("  time %i, stime %i, scount %i, orange %i\n",
			alsaDev[audioDev->devID].cstatus.time,
			alsaDev[audioDev->devID].cstatus.stime,
			alsaDev[audioDev->devID].cstatus.scount,
			alsaDev[audioDev->devID].cstatus.overrange);
		printf("Orate%i %i, frags %i, fSize %i, count %i, Q %i, under %i\n",
			audioDev->devID,
			alsaDev[audioDev->devID].pstatus.rate,
			alsaDev[audioDev->devID].pstatus.fragments,
			alsaDev[audioDev->devID].pstatus.fragment_size,
			alsaDev[audioDev->devID].pstatus.count,
			alsaDev[audioDev->devID].pstatus.queue,
			alsaDev[audioDev->devID].pstatus.underrun);
		printf("  time %i, stime %i, scount %i\n",
			alsaDev[audioDev->devID].pstatus.time,
			alsaDev[audioDev->devID].pstatus.stime,
			alsaDev[audioDev->devID].pstatus.scount);
#endif
	}

	/* 
	 * To find sync we need to find out what the start delta time was, and how
	 * many samples this delta represents. We can then find the actual delta
	 * from the [pc]status structures. From these two we calculate the value by
	 * which we need to adjust.
	 *
	 * This is only required for duplex devs, but that is not enforced yet.
	 *
	 * We are going to use a reasonable assumption that playback starts before
	 * capture. This is based on the premise that the playback channel is
	 * preloaded.
	 */
	if (audioDev->flags != SLAB_FULL_DUPLEX)
	{
		/*
		 * If we are not full duplex there is no channel sync to evaluate.
		 */
		return(0);
	} else {
		if (alsaDev[audioDev->devID].cstatus.stime.tv_sec ==
			alsaDev[audioDev->devID].pstatus.stime.tv_sec)
		{
			deltaTime = alsaDev[audioDev->devID].cstatus.stime.tv_usec -
				alsaDev[audioDev->devID].pstatus.stime.tv_usec;
		} else {
			deltaTime = (1000000 -
				alsaDev[audioDev->devID].pstatus.stime.tv_usec) +
				alsaDev[audioDev->devID].cstatus.stime.tv_usec;
		}
	}

	deltaSamples = audioDev->writeSampleRate * deltaTime / 1000000;

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("Delta count: %i (%i us)\n",
			deltaSamples, deltaTime);

	return(deltaSamples);
}

/*
 * Head status is for sync between multiple audio devices.
 */
alsaHeadStatus(duplexDev *audioDev)
{
	int deltaTime;

	/*
	 * We are going to compare when each device started, and return a number of
	 * samples to slip to sync multiple heads. We will assume the main device
	 * started before this one.
	 */
	if (audioDev->flags == SLAB_RDONLY)
	{
		if (alsaDev[audioDev->devID].cstatus.stime.tv_sec ==
			alsaDev[0].pstatus.stime.tv_sec)
		{
			deltaTime = alsaDev[audioDev->devID].cstatus.stime.tv_usec -
				alsaDev[0].pstatus.stime.tv_usec;
		} else {
			deltaTime = (1000000 - alsaDev[0].pstatus.stime.tv_usec) +
				alsaDev[audioDev->devID].cstatus.stime.tv_usec;
		}
	} else {
		if (alsaDev[audioDev->devID].pstatus.stime.tv_sec ==
			alsaDev[0].pstatus.stime.tv_sec)
		{
			deltaTime = alsaDev[audioDev->devID].pstatus.stime.tv_usec -
				alsaDev[0].pstatus.stime.tv_usec;
		} else {
			deltaTime = (1000000 - alsaDev[0].pstatus.stime.tv_usec) +
				alsaDev[audioDev->devID].pstatus.stime.tv_usec;
		}
	}
	return(audioDev->writeSampleRate * deltaTime / 1000000);
}

printSndPcm(snd_pcm_t *handle)
{
	if (handle == (snd_pcm_t *) NULL)
		return;

	printf("PCM SETUP: fd0 %i, fd1 %i, v0 %i, v1 %i\n",
		handle->fd[0], handle->fd[1],
		handle->setup_is_valid[0], handle->setup_is_valid[1]);
	printPcmSetup(&handle->setup[0]);
	printPcmSetup(&handle->setup[1]);
}

slabAlsaSyncData(duplexDev *audioDev)
{
	snd_pcm_channel_status_t *cstatus;
	float et;

	cstatus = &alsaDev[audioDev->devID].cstatus;
	/*
	 * Get current time point.
	slabgettimeofday(&audioDev->timeLastRead);
	 */
	if (audioDev->timeLastRead.tv_sec == audioDev->startTime.tv_sec)
		et = (audioDev->timeLastRead.tv_usec - audioDev->startTime.tv_usec);
	else
		et = (((float) (audioDev->timeLastRead.tv_sec -
			audioDev->startTime.tv_sec - 1)) * 1000000)
			+ (audioDev->timeLastRead.tv_usec + 1000000 -
			audioDev->startTime.tv_usec);
	/*
	 * We now have a number of seconds, as a float, since start. Evaluate 
	 * sample rate. I should remove the number of preloaded samples.
	 */
printf("Sample rate estimate is %f at time %f, bytes %i\n",
((float) cstatus->scount) * 250000.0 / et, et, cstatus->scount);
}

#else /* (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5) */

#define SND_PCM_OPEN_PLAYBACK SND_PCM_STREAM_PLAYBACK
#define SND_PCM_OPEN_CAPTURE SND_PCM_STREAM_CAPTURE

snd_output_t *output = NULL;

alsaChannelConfigure(duplexDev *audioDev, snd_pcm_t **handle,
	snd_pcm_hw_params_t **h_params,
	snd_pcm_sw_params_t **s_params,
	char *dir)
{
	char *devicename;
	int err, stream;
	int nfds;
	struct pollfd *pfds;

	snd_pcm_hw_params_alloca(h_params);
	snd_pcm_sw_params_alloca(s_params);

	if (audioDev->fragSize == 0)
		audioDev->fragSize = DEF_FRAGSIZE;

	devicename = strdup(audioDev->devName);

	if (strcmp(dir, "capture") == 0)
		stream = SND_PCM_STREAM_CAPTURE;
	else
		stream = SND_PCM_STREAM_PLAYBACK;

	/*
	 * Open the device.
	 */
	if (snd_pcm_open(handle, devicename, stream, 0) < 0)
	{
		fprintf(stderr, "Error opening PCM device %s\n", devicename);
		return(-1);
	}

	if (snd_pcm_hw_params_any(*handle, *h_params) < 0)
	{
		printf("Cound not get %s any params\n", dir);
		return(-1);
	}
	if (snd_pcm_hw_params_set_access(*handle,
		*h_params, SND_PCM_ACCESS_RW_INTERLEAVED)
			< 0)
	{
		printf("Could not set %s access methods\n", dir);
		return(-1);
	}
	if (snd_pcm_hw_params_set_format(*handle,
		*h_params, SND_PCM_FORMAT_S16_LE) < 0)
	{
		printf("Could not set %s format\n", dir);
		return(-1);
	}
	if (snd_pcm_hw_params_set_channels(*handle,
		*h_params, audioDev->channels) < 0)
	{
		printf("Could not set channels\n");
		return(-1);
	}
	if (snd_pcm_hw_params_set_rate_near(*handle,
		*h_params, audioDev->writeSampleRate, 0) <0)
	{
		printf("Could not set %s rate\n", dir);
		return(-1);
	}
	if (snd_pcm_hw_params_set_period_size(*handle,
		*h_params, audioDev->fragSize / 4, 0) < 0)
	{
		printf("Could not configure %s period size\n", dir);
printf("period size is %i, %i %i\n",
audioDev->fragSize,
audioDev->writeSampleRate,
audioDev->channels);
		return(-1);
	}
	if (snd_pcm_hw_params_set_buffer_size_near(
		*handle,
		*h_params, audioDev->fragSize * audioDev->preLoad) < 0)
	{
		printf("Could not configure %s buffer size\n", dir);
		return(-1);
	}
	if (snd_pcm_hw_params(*handle,
		*h_params) < 0)
	{
		printf("Could not set %s hardware parameters\n", dir);
		return(-1);
	}

	if (snd_pcm_sw_params_current(*handle,
		*s_params) < 0)
	{
		printf("Could not get %s current configuration\n", dir);
		return(-1);
	}
	if (snd_pcm_sw_params_set_start_threshold(
		*handle,
		*s_params, audioDev->fragSize * 1024) < 0)
	{
		printf("Could not set %s start threshold\n", dir);
		return(-1);
	}
	if (snd_pcm_sw_params_set_xfer_align(*handle,
		*s_params, 1) < 0)
	{
		printf("Could not set %s transfer alignment\n", dir);
		return(-1);
	}
	if (snd_pcm_sw_params(*handle,
		*s_params) < 0)
	{
		printf("Could not configure %s software parameters\n", dir);
		return(-1);
	}

	snd_pcm_prepare(*handle);

	if (audioDev->cflags & SLAB_AUDIODBG)
		snd_pcm_dump(*handle, output);

	nfds = snd_pcm_poll_descriptors_count(*handle);
	pfds = (struct pollfd *) malloc(sizeof (struct pollfd) * nfds);
	snd_pcm_poll_descriptors(*handle, pfds, nfds);

	if (strcmp(dir, "capture") == 0)
		audioDev->fd2 = pfds[0].fd;
	else
		audioDev->fd = pfds[0].fd;

	free(pfds);

	return(0);
}

/*
 * The rest of this code is for ALSA 0.9, and hopefully greater. Each new
 * minor revision introduces some serious redefinition of the interfaces.
 */
alsaDevOpen(duplexDev *audioDev, int devID, int flags, int fragSize)
{
	char *devicename;
	int err;

	setscheduler(audioDev);

if (audioDev->cflags & SLAB_AUDIODBG)
printf("opening device %s, flags %08x\n", audioDev->devName, audioDev->flags);

	if (audioDev->channels == 0)
		audioDev->channels = 2;

	if (audioDev->writeSampleRate == 0)
		audioDev->writeSampleRate = 44100;
	if (audioDev->readSampleRate == 0)
		audioDev->readSampleRate = 44100;

	if (flags == SLAB_OWRONLY)
		flags = SLAB_OWRONLY;
	else if (flags == SLAB_ORDONLY)
		flags = SLAB_RDONLY;
	else if (flags == SLAB_ORDWR)
		flags = SLAB_OWRONLY|SLAB_RDONLY;

	err = snd_output_stdio_attach(&output, stdout, 0);

	/*
	 * Open and configure the playback channel.
	 */
	if (flags & SLAB_OWRONLY)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("open playback on %s, pre %i\n",
				audioDev->devName, audioDev->preLoad);

		if (alsaChannelConfigure(audioDev,
				&alsaDev[audioDev->devID].phandle,
				&alsaDev[audioDev->devID].p_params,
				&alsaDev[audioDev->devID].p_swparams,
				"playback") < 0)
			return(-1);
	}

	/*
	 * And do the same for the capture channel
	 */
	if (flags & SLAB_RDONLY)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("open capture on %s: pre %i\n",
				audioDev->devName, audioDev->preLoad);

		if (alsaChannelConfigure(audioDev,
				&alsaDev[audioDev->devID].chandle,
				&alsaDev[audioDev->devID].c_params,
				&alsaDev[audioDev->devID].c_swparams,
				"capture") < 0)
			return(-1);
	}

	if (audioDev->fragBuf)
		free(audioDev->fragBuf);
	audioDev->fragBuf = (char *) malloc(audioDev->fragSize);

	return(0);
}

int
alsaChannelStatus(duplexDev *audioDev)
{
}

int
alsaHeadStatus(duplexDev *audioDev)
{
}

int
alsaDevAudioStart(duplexDev *audioDev)
{
	int err;

	if ((audioDev->flags == ADIOD_OUTPUT)
		|| (audioDev->flags == ADIOD_DUPLEX))
	{
		if ((err = snd_pcm_start(alsaDev[audioDev->devID].phandle)) < 0) {
			printf("Go error: %s\n", snd_strerror(err));
			return(-1);
		}
	}
	if ((audioDev->flags == ADIOD_INPUT)
		|| (audioDev->flags == ADIOD_DUPLEX))
	{
		if ((err = snd_pcm_start(alsaDev[audioDev->devID].chandle)) < 0) {
			printf("Go error: %s\n", snd_strerror(err));
			return(-1);
		}
	}
}

void showstat(snd_pcm_t *handle, size_t frames)
{
	int err;
	snd_pcm_status_t *status;

	snd_pcm_status_alloca(&status);
	if ((err = snd_pcm_status(handle, status)) < 0) {
		printf("Stream status error: %s\n", snd_strerror(err));
		exit(0);
	}
	printf("*** frames = %li ***\n", (long)frames);
	snd_pcm_status_dump(status, output);
}

#endif /* (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5) */
#endif /* ALSA_SUPPORT */

int
audioWrite(audioDev, buffer, size)
duplexDev *audioDev;
char *buffer;
int size;
{
#ifdef ALSA_SUPPORT
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5)
	if (audioDev->cflags & SLAB_AUDIODBG)
	{
		printf("audioWrite(%08x, %08x, %i): %08x\n", audioDev, buffer, size,
			&alsaDev[audioDev->devID].phandle);
		if (audioDev->siflags & AUDIO_ALSA)
		{
			alsaChannelStatus(audioDev);
		}
	}

	if (audioDev->siflags & AUDIO_ALSA)
	{
		int res;

		res = snd_pcm_write(alsaDev[audioDev->devID].phandle, buffer, size);

		if (res != size)
		{
			getAlsaOutputStatus(audioDev);

if (audioDev->cflags & SLAB_AUDIODBG)
printf("issue: wrote %i, status %i at %i\n", res,
alsaDev[audioDev->devID].pstatus.status,
alsaDev[audioDev->devID].pstatus.count);

if (alsaDev[audioDev->devID].pstatus.status == 0)
return(size);
			if (alsaDev[audioDev->devID].pstatus.status
				== SND_PCM_STATUS_UNDERRUN)
			{
				if (audioDev->cflags & SLAB_AUDIODBG)
					printf("Underrun (%i) at %i\n",
						alsaDev[audioDev->devID].pstatus.status,
						alsaDev[audioDev->devID].pstatus.count);

				/*
				 * If we can prepare the channel again, do so and continue.
				 */
				if (snd_pcm_plugin_prepare(alsaDev[audioDev->devID].phandle,
						SND_PCM_CHANNEL_PLAYBACK) <0)
				{
					printf("underrun: playback channel prepare error\n");
					return(res);
				}

				return(audioWrite(audioDev, buffer, size));
			}
			return(res);
		}

		return(res);
	}
#else /* (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5) */
	/*
	 * This if for ALSA 0.9
	 */
	if (audioDev->siflags & AUDIO_ALSA)
	{
		long r;

		/* 
		 * Slab historically used byte buffer sizes. ALSA 0.9 uses frames, so
		 * we are going to assume (hm) for now that a frame is stereo 16 bits.
		 */
		while((r = 
			snd_pcm_writei(alsaDev[audioDev->devID].phandle, buffer, size >> 2))
			== EAGAIN)
			printf("Do again\n");

		if (r < 0)
		{
			printf("	Write Error: %s %i\n", snd_strerror(r));
			return(r);
		}
		return(size);
	}
#endif /* (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5) */
#endif /* ALSA_SUPPORT */

	/*
	 * Otherwise just to a normal OSS fd write operation.
	 */
	return(write(audioDev->fd, buffer, size));
}

int
audioRead(audioDev, buffer, size)
duplexDev *audioDev;
char *buffer;
int size;
{
#ifdef ALSA_SUPPORT
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5)
	if (audioDev->siflags & AUDIO_ALSA)
	{
		int res;

		res = snd_pcm_read(alsaDev[audioDev->devID].chandle, buffer, size);

		if (res < 0)
		{
			getAlsaInputStatus(audioDev);

			/*
			 * If we are prepared, continue - or flush input?
			 */
			if (alsaDev[audioDev->devID].cstatus.status
				== SND_PCM_STATUS_PREPARED)
				return(size);
			/*
			 * If we are still running, continue.
			 */
			if (alsaDev[audioDev->devID].cstatus.status
				== SND_PCM_STATUS_RUNNING)
				return(audioRead(audioDev, buffer, size));

/*
			if (alsaDev[audioDev->devID].cstatus.status
				== SND_PCM_STATUS_OVERRUN)
*/
			{
				if (audioDev->cflags & SLAB_AUDIODBG)
					printf("Overrun at %i\n",
						alsaDev[audioDev->devID].cstatus.count);

				/*
				 * If we can prepare the channel again, do so and continue.
				 */
				if (snd_pcm_plugin_prepare(alsaDev[audioDev->devID].chandle,
						SND_PCM_CHANNEL_CAPTURE) <0)
				{
					printf("overrun: capture channel prepare error\n");
					return(res);
				}

				return(audioRead(audioDev, buffer, size));
			}
		}

		return(res);
	}
#else /* (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5) */
	/*
	 * This if for ALSA 0.9
	 */
	if (audioDev->siflags & AUDIO_ALSA)
	{
		return(4 *
			snd_pcm_readi(alsaDev[audioDev->devID].chandle, buffer, size >> 2));
	}
#endif /* (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5) */
#endif

	/*
	 * If this is not an ALSA dev, do a normal OSS fd read operation.
	 */
	return(read(audioDev->fd2, buffer, size));
}

