/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "ActionGenerator.h"
#include <ItkFilter.hxx>
#include <Classification.hxx>
using namespace coreschema;

// Includes from Qt
#include <QDate>
#include <QTime>
#include <QPair>
#include <QMapIterator>
#include <QTextStream>

// includes from std
#include <iostream>

bool ActionGenerator::generateActionFiles(QString xmlFileName, QString devDirectoryName, QString * elementClassName) {
    try {
        ActionGenerator * generator = new ActionGenerator(xmlFileName, devDirectoryName);
        generator->generateActionFiles();

        if (elementClassName != NULL)
            (*elementClassName) = generator->className;


    }
    catch (QString msg) {
        std::cout << msg.toStdString() << std::endl;
        std::cout << "Could not generate component files, sorry..." << std::endl;
        return false;
    }


    return true;

}


ActionGenerator::ActionGenerator(QString xmlFilename, QString devDirectoryName)  {

    className = QString("");
    hasParameters = false;
    setXmlFileName(xmlFilename);
    setDevDirectoryName(devDirectoryName);

    createElement();
}

void ActionGenerator::setXmlFileName(QString xmlFileName) throw (QString) {
    QFileInfo xmlFile(xmlFileName);

    if ((! xmlFile.exists()) || (! xmlFile.isFile())) {
        QString msg = "Exception from Action generation: \n     The file " + xmlFileName +  " does not exist or is not a file...\n";
        throw (msg);
    }

    this->xmlFileName = xmlFile;
}

void ActionGenerator::setDevDirectoryName(QString devDirectoryName) throw (QString) {
    QFileInfo devDir(devDirectoryName);

    if ( ! devDir.isDir()) {
        QString msg = "Exception from action generation: \n     The path " + devDirectoryName + " is not a directory\n";
        throw (msg);
    }

    this->devDirectoryName.cd(devDirectoryName);
}


void ActionGenerator::createElement() throw (QString) {
    std::string xmlFileStr = this->xmlFileName.canonicalFilePath().toStdString();
    // TODO manage exception (despite Qt...).
    // TODO find how to read an xml file with an absolute path !!!
    this->theAction = action(xmlFileStr, xml_schema::flags::dont_validate);

    this->className = QString(this->theAction->classNames().actionClass().get().c_str());

    this->hasParameters = theAction->parameters().present();

    if (hasParameters) {
        Parameters::parameter_sequence theParams = this->theAction->parameters()->parameter();

        for (Parameters::parameter_const_iterator it = theParams.begin(); it < theParams.end(); it++) {
            QString parameterName = QString((*it).name().c_str());
            QString parameterType = QString((*it).type().c_str());
            QString parameterValue ="";

            if ((*it).defaultValue().present()) {
                parameterValue = QString((*it).defaultValue()->c_str());
            }
            else {
                if (parameterType == "int")
                    parameterValue =    "0";
                else
                    if (parameterType == "bool")
                        parameterValue = "false";
                    else
                        if (parameterType == "double")
                            parameterValue = "0.0";
                        else
                            if (parameterType == "QString")
                                parameterValue = "\"Hello World\"";
                            else
                                if (parameterType == "QDate") {
                                    QDate today = QDate::currentDate();
                                    QString defValue;
                                    QTextStream in(&defValue);
                                    in << "QDate(" << today.year() << ", " << today.month() << ", " << today.day() << ")";
                                    parameterValue = defValue;
                                }
                                else
                                    if (parameterType == "QTime") {
                                        QTime now = QTime::currentTime();
                                        QString defValue;
                                        QTextStream in(&defValue);
                                        in << "QTime(" << now.hour() << ", " << now.minute() << ", " << now.second() << ")";
                                        parameterValue = defValue;
                                    }
                                    else
                                        if (parameterType == "QColor")
                                            parameterValue = "QColor(255, 255, 255, 255)";
                                        else
                                            if (parameterType == "QPoint")
                                                parameterValue = "QPoint(0, 0)";
                                            else
                                                if (parameterType == "QPointF")
                                                    parameterValue = "QPointF(0.0, 0.0)";
                                                else
                                                    if (parameterType == "QVector3D")
                                                        parameterValue = "QVector3D(0.0, 0.0, 0.0)";
            }

            // Create a table with name and default value
            parameters.insert(parameterName, QPair<QString, QString>(parameterType, parameterValue));

            // Select the additional includes
            if ((parameterType != "int") && (parameterType != "bool") && (parameterType != "double")) {
                this->additionalIncludes.insert(parameterType);
            }


        }
    }

    this->isItkFilter = theAction->classification().itkFilter().present();

}

void ActionGenerator::generateActionFiles() throw (QString) {
    writeHFile();
    writeCFile();

    if (isItkFilter) {
        writeImplementationFile();
    }
}


void ActionGenerator::writeHFile() throw (QString) {
    QString componentName(theAction->classNames().componentClass().c_str());

    QFile initFile(":/resources/Action.h.in");
    initFile.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream in(&initFile);


    QFileInfo extFilePath;
    extFilePath.setFile(this->devDirectoryName, className + ".h");
    QFile extFile(extFilePath.absoluteFilePath());

    if (! extFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFilePath.fileName() + "\n";
        throw (msg);
    }

    QTextStream out(&extFile);

    QString text;

    do {
        text = in.readLine();
        text.replace(QRegExp("@HEADDEF@"), className.toUpper());
        text.replace(QRegExp("@COMPONENTNAME@"), componentName);
        text.replace(QRegExp("@ACTIONCLASSNAME@"), className);

        if (text.contains(QRegExp("@IF_NOCOMP@"))) {
            if (componentName != "") {
                do {
                    text = in.readLine();
                }
                while ( (! text.contains(QRegExp("@ELSEIF_NOCOMP@"))) &&
                        (! text.contains(QRegExp("@END_NOCOMP@"))));
            }
        }
        else
            if (text.contains(QRegExp("@ELSEIF_NOCOMP@"))) {
                if (componentName == "") {
                    do {
                        text = in.readLine();
                    }
                    while (! text.contains(QRegExp("@END_NOCOMP@")));
                }
            }
            else
                if (text.contains(QRegExp("@END_NOCOMP@"))) {
                    // Go to next line...
                }

                else
                    if (text.contains(QRegExp("@IF_DEFCOMPONENT@"))) {
                        if ((componentName == "Component") || (componentName == "")) {
                            do {
                                text = in.readLine();
                            }
                            while ( (! text.contains(QRegExp("@ENDIF_DEFCOMPONENT@"))) &&
                                    (! text.contains(QRegExp("@ELSEIF_DEFCOMPONENT@"))));
                        }
                    }
                    else
                        if ((text.contains(QRegExp("@ELSEIF_DEFCOMPONENT@")))) {
                            if ((componentName != "Component") && (componentName != "")) {
                                do {
                                    text = in.readLine();
                                }
                                while (! text.contains(QRegExp("@ENDIF_DEFCOMPONENT@")));
                            }
                        }
                        else
                            if (text.contains(QRegExp("@ENDIF_DEFCOMPONENT@"))) {
                                // Go to next line...
                            }
                            else
                                if (text.contains(QRegExp("@IF_IMAGECOMPONENT@"))) {
                                    if ((componentName != "ImageComponent") || (isItkFilter)) {
                                        do {
                                            text = in.readLine();
                                        }
                                        while (! text.contains(QRegExp("@ENDIF_IMAGECOMPONENT@")));
                                    }
                                }
                                else
                                    if (text.contains(QRegExp("@ENDIF_IMAGECOMPONENT@"))) {
                                        // Go to the next line...
                                    }
                                    else
                                        if (text.contains(QRegExp("@IF_ITKFILTER@"))) {
                                            if (! isItkFilter) {
                                                do {
                                                    text = in.readLine();
                                                }
                                                while (! text.contains(QRegExp("@ENDIF_ITKFILTER@")));
                                            }
                                        }
                                        else
                                            if (text.contains(QRegExp("@ENDIF_ITKFILTER@"))) {
                                                // Go to the next line...
                                            }
                                            else {
                                                out << text << endl;
                                            }
    }
    while (! text.isNull());

    extFile.close();
    initFile.close();

}

void ActionGenerator::writeCFile() throw (QString) {

    //Getting the parameters
    QString actionName(this->theAction->name().c_str());
    QString actionDescription(this->theAction->description().c_str());
    QString componentName(theAction->classNames().componentClass().c_str());
    QString family(theAction->classification().family().c_str());

    // Opening init file
    QFile initFile(":/resources/Action.cpp.in");
    initFile.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream in(&initFile);
    // Opening destination file
    QFileInfo extFilePath;
    extFilePath.setFile(this->devDirectoryName, className + ".cpp");
    QFile extFile(extFilePath.absoluteFilePath());

    if (! extFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFilePath.fileName() + "\n";
        throw (msg);
    }

    QTextStream out(&extFile);

    // Parsing init file to fill destinaiton file.
    QString text;

    do {
        text = in.readLine();
        text.replace(QRegExp("@COMPONENTNAME@"), componentName);
        text.replace(QRegExp("@ACTIONCLASSNAME@"), className);
        text.replace(QRegExp("@ACTION_NAME@"), actionName);
        text.replace(QRegExp("@ACTION_DESCRIPTION@"), actionDescription);
        text.replace(QRegExp("@FAMILY@"), family);

        if (text.contains(QRegExp("@ADDITIONAL_INCLUDES@"))) {
            std::set<QString>::const_iterator it;

            for (it = additionalIncludes.begin(); it != additionalIncludes.end(); it++) {
                out << QString("#include <") << (*it) << ">" << endl;
            }
        }
        else
            if (text.contains(QRegExp("@IF_NOCOMP@"))) {
                if (componentName != "") {
                    do {
                        text = in.readLine();
                    }
                    while ( (! text.contains(QRegExp("@ELSEIF_NOCOMP@"))) &&
                            (! text.contains(QRegExp("@END_NOCOMP@"))));
                }
            }
            else
                if (text.contains(QRegExp("@ELSEIF_NOCOMP@"))) {
                    if (componentName == "") {
                        do {
                            text = in.readLine();
                        }
                        while (! text.contains(QRegExp("@END_NOCOMP@")));
                    }
                }
                else
                    if (text.contains(QRegExp("@END_NOCOMP@"))) {
                        // Go to next line...
                    }
                    else
                        if (text.contains(QRegExp("@IF_DEFCOMPONENT@"))) {
                            if ((componentName == "Component") || (componentName == "")) {
                                do {
                                    text = in.readLine();
                                }
                                while ( (! text.contains(QRegExp("@ENDIF_DEFCOMPONENT@"))) &&
                                        (! text.contains(QRegExp("@ELSEIF_DEFCOMPONENT@"))));
                            }
                        }
                        else
                            if ((text.contains(QRegExp("@ELSEIF_DEFCOMPONENT@")))) {
                                if (componentName != "Component") {
                                    do {
                                        text = in.readLine();
                                    }
                                    while (! text.contains(QRegExp("@ENDIF_DEFCOMPONENT@")));
                                }
                            }
                            else
                                if (text.contains(QRegExp("@ENDIF_DEFCOMPONENT@"))) {
                                    // Go to next line...
                                }
                                else
                                    if (text.contains(QRegExp("@IF_IMAGECOMPONENT@"))) {
                                        if ((componentName != "ImageComponent") || (isItkFilter)) {
                                            do {
                                                text = in.readLine();
                                            }
                                            while (! text.contains(QRegExp("@ENDIF_IMAGECOMPONENT@")));
                                        }
                                    }
                                    else
                                        if (text.contains(QRegExp("@ENDIF_IMAGECOMPONENT@"))) {
                                            // Go to the next line...
                                        }
                                        else
                                            if (text.contains(QRegExp("@IF_ITKFILTER@"))) {
                                                if (! isItkFilter) {
                                                    do {
                                                        text = in.readLine();
                                                    }
                                                    while (! text.contains(QRegExp("@ENDIF_ITKFILTER@")));
                                                }
                                            }
                                            else
                                                if (text.contains(QRegExp("@ENDIF_ITKFILTER@"))) {
                                                    // Go to the next line...
                                                }
                                                else
                                                    if (text.contains(QRegExp("@BEGIN_ADDTAGS@"))) {
                                                        text = in.readLine();
                                                        int nbTags = theAction->classification().tag().size();
                                                        int indexTag = 0;

                                                        if (nbTags > 0) {
                                                            QString * addOneTag = new QString[nbTags];

                                                            for (indexTag = 0; indexTag < nbTags; indexTag++)
                                                                addOneTag[indexTag] = "";

                                                            while (! text.contains(QRegExp("@END_ADDTAG@")) && ! text.contains(QRegExp("@ELSE_ADDTAG@")) ) {
                                                                Action::classification_type::tag_const_iterator it;
                                                                indexTag = 0;

                                                                for (it = theAction->classification().tag().begin();
                                                                        it < theAction->classification().tag().end(); it++) {
                                                                    QString textTmp = text;
                                                                    textTmp.replace(QRegExp("@TAG@"), QString((*it).c_str()));
                                                                    addOneTag[indexTag].append(textTmp).append("\n");
                                                                    indexTag++;
                                                                }

                                                                text = in.readLine();
                                                            }

                                                            if (text.contains("@ELSE_ADDTAG@")) {
                                                                while (! text.contains("@END_ADDTAG@")) {
                                                                    text = in.readLine();
                                                                }
                                                            }

                                                            for (indexTag = 0; indexTag < nbTags; indexTag++) {
                                                                out << addOneTag[indexTag];
                                                            }
                                                        }
                                                        else {
                                                            while ((! text.contains("@ELSE_ADDTAG@")) && (! text.contains("@END_ADDTAG@")))
                                                                text = in.readLine();

                                                            if (text.contains("@ELSE_ADDTAG@")) {
                                                                text = in.readLine();

                                                                while (! text.contains("@END_ADDTAG@")) {
                                                                    out << text << endl;
                                                                    text = in.readLine();
                                                                }
                                                            }

                                                            text = in.readLine();
                                                        }
                                                    }
                                                    else
                                                        if (text.contains(QRegExp("@IF_ADDPROPERTIES@"))) {
                                                            text = in.readLine();

                                                            while (! text.contains(QRegExp("@ENDIF_ADDPROPERTIES@"))) {
                                                                if ( ! parameters.isEmpty()) {
                                                                    out << text << endl;
                                                                }

                                                                text = in.readLine();
                                                            }
                                                        }
                                                        else
                                                            if (text.contains(QRegExp("@BEGIN_ADDPROPERTIES@"))) {
                                                                text = in.readLine();
                                                                int nbProps = parameters.size();
                                                                int indexProp = 0;

                                                                if (nbProps > 0) {
                                                                    QString * addOneProp = new QString[nbProps];

                                                                    for (indexProp = 0; indexProp < nbProps; indexProp++)
                                                                        addOneProp[indexProp] = "";

                                                                    while (! text.contains(QRegExp("@END_ADDPROPERTIES@")) &&
                                                                            ! text.contains(QRegExp("@ELSE_ADDPROPERTIES@")) ) {
                                                                        QMapIterator<QString, QPair<QString, QString> > it(parameters);
                                                                        indexProp = 0;

                                                                        while (it.hasNext()) {
                                                                            it.next();
                                                                            QString name = it.key();
                                                                            // create a name that can be used as C++ variable name (remove all white space)
                                                                            QString cppName = name.simplified(); // transform all to whitespace
                                                                            QStringList cppNameList = cppName.split(" "); // split words
                                                                            // lower case for the first letter of the first word
                                                                            QString firstWord = cppNameList.takeFirst();

                                                                            // only if there is more than one word... (otherwise it means the user has just enter a longNameWithPossiblySomeUpperCaseLetters)
                                                                            if (cppNameList.size()>0)
                                                                                cppName = firstWord.toLower();

                                                                            // lowercase the first letter of each word and concatenate
                                                                            foreach(QString s, cppNameList) {
                                                                                cppName += s.left(1).toUpper() + s.mid(1).toLower();
                                                                            }

                                                                            QString type = it.value().first;
                                                                            QString val  = it.value().second;
                                                                            QString toString = cppName;
                                                                            QString toType;

                                                                            if ((type == "int") || (type == "bool") || (type == "float") || (type == "double")) {
                                                                                // using toType() method
                                                                                toType = "to" + type.left(1).toUpper() + type.mid(1) + "()";
                                                                            }
                                                                            else {
                                                                                toType = QString("value<").append(type).append(">()");

                                                                                if (type == "QPoint" || type == "QPointF") {
                                                                                    toString = "\"(\" + QString::number(" + cppName + ".x()) + \",\" + QString::number(" + cppName + ".y()) + \")\"";
                                                                                }
                                                                                else
                                                                                    if (type == "QVector3D") {
                                                                                        toString = "\"(\" + QString::number(" + cppName + ".x()) + \",\" + QString::number(" + cppName + ".y()) + \",\" + QString::number(" + cppName + ".z()) + \")\"";
                                                                                    }
                                                                                    else
                                                                                        if (type == "QColor") {
                                                                                            toString.append(".name()");
                                                                                        }
                                                                                        else
                                                                                            if (type != "QString") {
                                                                                                toString.append(".toString()");
                                                                                            }
                                                                            }

                                                                            QString textTmp = text;
                                                                            textTmp.replace(QRegExp("@PROPERTY_TYPE@"), type);
                                                                            textTmp.replace(QRegExp("@PROPERTY_NAME@"), name);
                                                                            textTmp.replace(QRegExp("@PROPERTY_CPP_NAME@"), cppName);
                                                                            textTmp.replace(QRegExp("@PROPERTY_NAME_STRING@"), toString);

                                                                            // QVector3D and QColor are not in QtCore
                                                                            // but in QtGui, QVariant class could not provide
                                                                            // a conversion functions or copy constructor
                                                                            // => the operator QVariant() of these classes is needed
                                                                            if (type=="QVector3D" || type=="QColor") {
                                                                                textTmp.replace(QRegExp("@PROPERTY_QVARIANT@"), type);
                                                                            }
                                                                            else {
                                                                                textTmp.replace(QRegExp("@PROPERTY_QVARIANT@"), "QVariant");
                                                                            }
                                                                            
                                                                            textTmp.replace(QRegExp("@PROPERTY_VALUE@"), val);
                                                                            textTmp.replace(QRegExp("@PROPERTY_TOTYPE@"), toType);
                                                                            addOneProp[indexProp].append(textTmp).append("\n");
                                                                            indexProp++;
                                                                        }

                                                                        text = in.readLine();
                                                                    }

                                                                    if (text.contains("@ELSE_ADDPROPERTIES@")) {
                                                                        while (! text.contains("@END_ADDPROPERTIES@")) {
                                                                            text = in.readLine();
                                                                        }
                                                                    }

                                                                    for (indexProp = 0; indexProp < nbProps; indexProp++) {
                                                                        out << addOneProp[indexProp];
                                                                    }
                                                                }
                                                                else {
                                                                    while ((! text.contains("@ELSE_ADDPROPERTIES@")) && (! text.contains("@END_ADDPROPERTIES@")))
                                                                        text = in.readLine();

                                                                    if (text.contains("@ELSE_ADDPROPERTIES@")) {
                                                                        text = in.readLine();

                                                                        while (! text.contains("@END_ADDPROPERTIES@")) {
                                                                            out << text << endl;
                                                                            text = in.readLine();
                                                                        }
                                                                    }
                                                                }
                                                            }
                                                            else {
                                                                out << text << endl;
                                                            }
    }
    while (! text.isNull());

    extFile.close();
    initFile.close();

}

void ActionGenerator::writeImplementationFile() throw (QString) {
    Classification::itkFilter_type schemaFilter = theAction->classification().itkFilter().get();
    QString outputType = schemaFilter.outputType().get().c_str();

    QString initFileName = "";

    if (outputType == "Same as Input") {
        initFileName = QString(":/resources/ActionImplementation.sai.cpp.in");
    }
    else {
        initFileName = QString(":/resources/ActionImplementation.cpp.in");
    }

    QFile initFile(initFileName);
    initFile.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream in(&initFile);

    QFileInfo extFilePath;
    extFilePath.setFile(this->devDirectoryName, className + ".impl");
    QFile extFile(extFilePath.absoluteFilePath());

    if (! extFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = "Exception from extension generation: \n    Cannot write on file " + extFilePath.fileName() + "\n";
        throw (msg);
    }

    QTextStream out(&extFile);

    QString text;

    do {
        text = in.readLine();
        text.replace(QRegExp("@ACTIONCLASSNAME@"), className);
        text.replace(QRegExp("@OUTPUTTYPE@"), outputType);

        out << text << endl;

    }
    while (! text.isNull());

    extFile.close();
    initFile.close();
}
