"""Helper code for multi-threaded Python programs

Quoting from
http://code.activestate.com/recipes/496735-workaround-for-missed-sigint-in-multithreaded-prog/

  Multithreaded Python programs often ignore the SIGINT generated by a
  Keyboard Interrupt, especially if the thread that gets the signal is
  waiting or sleeping. This module provides a workaround by forking a
  child process that executes the rest of the program while the parent
  process waits for signals and kills the child process.


How to use:

  from chemfp import Watcher

  def main():
      ...
      Watcher.Watcher()
      ... start multi-threaded code ...

  if __name__ == "__main__":
      main()

Created by Allen Downey and distributed under the PSF license for Python.

"""

import threading, time, os, signal, sys

class Watcher(object):
    """this class solves two problems with multithreaded
    programs in Python, (1) a signal might be delivered
    to any thread (which is just a malfeature) and (2) if
    the thread that gets the signal is waiting, the signal
    is ignored (which is a bug).

    The watcher is a concurrent process (not thread) that
    waits for a signal and the process that contains the
    threads.  See Appendix A of The Little Book of Semaphores.
    http://greenteapress.com/semaphores/

    I have only tested this on Linux.  I would expect it to
    work on the Macintosh and not work on Windows.
    """
    
    def __init__(self):
        """ Creates a child thread, which returns.  The parent
            thread waits for a KeyboardInterrupt and then kills
            the child thread.
        """
        self.child = os.fork()
        if self.child == 0:
            return
        else:
            self.watch()

    def watch(self):
        try:
            os.wait()
        except KeyboardInterrupt:
            # I put the capital B in KeyBoardInterrupt so I can
            # tell when the Watcher gets the SIGINT
            print 'KeyBoardInterrupt'
            self.kill()
        sys.exit()

    def kill(self):
        try:
            os.kill(self.child, signal.SIGKILL)
        except OSError: pass
