# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: Storage.py 347 2008-01-24 00:43:09Z camrdale-guest $

"""Low-level writing of files.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type MAXREADSIZE: C{long}
@var MAXREADSIZE: the maximum number of bytes that can be read at a time
@type MAXLOCKSIZE: C{long}
@var MAXLOCKSIZE: the maximum size to lock at a time (windows only)
@type MAXLOCKRANGE: C{long}
@var MAXLOCKRANGE: the maximum range to lock in a file (windows only)
@type _pool: L{DebTorrent.piecebuffer.BufferPool}
@var _pool: the buffer for temporary storage of pieces
@type PieceBuffer: C{method} L{DebTorrent.piecebuffer.BufferPool.new}
@var PieceBuffer: returns a new or existing L{DebTorrent.piecebuffer.SingleBuffer}

"""

from DebTorrent.piecebuffer import BufferPool
from threading import Lock
from time import time, strftime, localtime
import os
from os.path import exists, getsize, getmtime, basename, split
from os import makedirs
import logging
from os import fsync
from bisect import bisect
    
logger = logging.getLogger('DebTorrent.BT1.Storage')

MAXREADSIZE = 32768
MAXLOCKSIZE = 1000000000L
MAXLOCKRANGE = 3999999999L   # only lock first 4 gig of file

_pool = BufferPool()
PieceBuffer = _pool.new

def dummy_status(fractionDone = None, activity = None):
    """Dummy function to print nothing."""
    pass

class Storage:
    """Low-level writing of files.
    
    Control the low-level management of files in the download. Contains
    functions to open and close, read and write, enable and disable, 
    flush and delete, all the files in the download. Also serves as an 
    abstraction layer, as the reading and writing is called with no 
    knowledge of file boundaries.
    
    @type files: C{list} of C{tuple} of (C{string}, C{long})
    @ivar files: the files list from the info of the metainfo dictionary
    @type piece_lengths: C{list} of C{long}
    @ivar piece_lengths: the list of piece lengths
    @type doneflag: C{threading.Event}
    @ivar doneflag: the flag that indicates when the program is to be shutdown
    @type disabled: C{list} of C{boolean}
    @ivar disabled: list of true for the files that are disabled
    @type file_ranges: C{list} of (C{long}, C{long}, C{long}, C{string})
    @ivar file_ranges: for each file, the start offset within the download, 
        end offset, offset within the file, and file name
    @type file_pieces: C{list} of (C{int}, C{int})
    @ivar file_pieces: for each file, the starting and ending piece of the file
    @type piece_files: C{dictionary}
    @ivar piece_files: for each piece, the starting and ending offset of the 
        piece in the file, the length of the file, and the file name
    @type disabled_ranges: C{list} of C{tuple}
    @ivar disabled_ranges: for each file, a tuple containing the working range, 
        shared pieces, and disabled range (see L{_get_disabled_ranges} for their
        meaning)
    @type working_ranges: C{list} of C{list} of (C{long}, C{long}, C{long}, C{string})
    @ivar working_ranges: For each file, the list of files to be written when
        writing to that file (may not be the actual file, i.e if it is 
        disabled). Ranges are temporarily stored here, before eventually being
        written to self.ranges by L{_reset_ranges} to be used.
    @type handles: C{dictionary} of {C{string}, C{file handle}}
    @ivar handles: the file handles that are open, keys are file names and
        values are the file handles
    @type whandles: C{dictionary} of {C{string}, C{int}}
    @ivar whandles: the files that are open for writing, keys are the file
        names and values are all 1
    @type tops: C{dictionary} of {C{string}, C{long}}
    @ivar tops: the current length of each file (by name)
    @type sizes: C{dictionary} of {C{string}, C{long}}
    @ivar sizes: the desired length of each file (by name)
    @type mtimes: C{dictionary} of {C{string}, C{long}}
    @ivar mtimes: the last modified time of each file (by name)
    @type lock_file: C{method}
    @ivar lock_file: locks a file (if file locking is enabled, otherwise does
        nothing)
    @type unlock_file: C{method}
    @ivar unlock_file: unlocks a file (if file locking is enabled, otherwise 
        does nothing)
    @type lock_while_reading: C{boolean}
    @ivar lock_while_reading: whether to lock files while reading them
    @type lock: C{lock}
    @ivar lock: a threading lock object for synchorizing threads (semaphore)
    @type total_length: C{long}
    @ivar total_length: the total length in bytes of the download
    @type max_files_open: C{int}
    @ivar max_files_open: the maximum number of files to have open at a time
        (0 means no maximum)
    @type handlebuffer: C{list}
    @ivar handlebuffer: the list of open files, in the order of most recently
        accessed, with the most recently accessed at the end of the list
        (only if there is a limit on the number of open files, otherwise None)
    @type ranges: C{list} of C{list} of (C{long}, C{long}, C{long}, C{string})
    @ivar ranges: for each file, the list of files to be written when
        writing to that file (may not be the actual file, i.e if it is 
        disabled)
    @type begins: C{list} of C{long}
    @ivar begins: the offset within the download that each non-disabled file
        begins at
    @type bufferdir: C{string}
    @ivar bufferdir: the buffer directory
    @type reset_file_status: C{method}
    @ivar reset_file_status: a shortcut to the _reset_ranges method

    """
    
    def __init__(self, files, piece_lengths, doneflag, config,
                 enabled_files = None):
        """Initializes the Storage.
        
        Initializes some variables, and calculates defaults for others,
        such as which pieces are contained by which file.
        
        @type files: C{list} of C{tuple} of (C{string}, C{long})
        @param files: the files list from the info of the metainfo dictionary
        @type piece_lengths: C{list} of C{long}
        @param piece_lengths: the list of piece lengths
        @type doneflag: C{threading.Event}
        @param doneflag: the flag that indicates when the program is to be shutdown
        @type config: C{dictionary}
        @param config: the configuration information
        @type enabled_files: C{list} of C{boolean}
        @param enabled_files: list of true for the files that are enabled
            (optional, default is all files disabled)
        
        """
        
        self.files = files
        self.piece_lengths = piece_lengths
        self.doneflag = doneflag
        self.disabled = [True] * len(files)
        self.file_ranges = []
        self.file_pieces = []
        self.piece_files = {}
        self.disabled_ranges = []
        self.working_ranges = []
        numfiles = 0
        total = 0l
        so_far = 0l
        cur_piece = 0
        piece_total = 0l
        self.handles = {}
        self.whandles = {}
        self.tops = {}
        self.sizes = {}
        self.mtimes = {}
        if config.get('lock_files', True):
            self.lock_file, self.unlock_file = self._lock_file, self._unlock_file
        else:
            self.lock_file, self.unlock_file = lambda x1,x2: None, lambda x1,x2: None
        self.lock_while_reading = config.get('lock_while_reading', False)
        self.lock = Lock()

        if not enabled_files:
            enabled_files = [False] * len(files)

        for i in xrange(len(files)):
            file, length = files[i]
            if doneflag.isSet():    # bail out if doneflag is set
                return
            self.disabled_ranges.append(None)
            if length == 0:
                self.file_ranges.append(None)
                self.working_ranges.append([])
                
                # Assign any zero-length pieces to the empty file
                start_piece = cur_piece
                for cur_piece in xrange(start_piece,len(self.piece_lengths)):
                    if self.piece_lengths[cur_piece] > 0:
                        break
                    self.piece_files[cur_piece] = (0L, 0L, 0L, '')
                end_piece = cur_piece-1
                if self.piece_lengths[cur_piece] == 0:
                    # Special case if the last file in a torrent is empty
                    end_piece = cur_piece
                self.file_pieces.append((start_piece, end_piece))
            else:
                range = (total, total + length, 0, file)
                self.file_ranges.append(range)
                self.working_ranges.append([])
                numfiles += 1
                total += length
                start_piece = cur_piece
                cur_piece_offset = 0l
                for cur_piece in xrange(start_piece,len(self.piece_lengths)+1):
                    if piece_total >= total:
                        break
                    self.piece_files[cur_piece] = (cur_piece_offset, 
                           cur_piece_offset + self.piece_lengths[cur_piece], 
                           length, file)
                    piece_total += self.piece_lengths[cur_piece]
                    cur_piece_offset += self.piece_lengths[cur_piece]
                end_piece = cur_piece-1
                if piece_total > total:
                    cur_piece -= 1
                    piece_total -= self.piece_lengths[cur_piece]
                self.file_pieces.append((start_piece, end_piece))
                if enabled_files[i]:
                    if exists(file):
                        l = getsize(file)
                        if l > length:
                            h = open(file, 'rb+')
                            h.truncate(length)
                            h.flush()
                            h.close()
                            l = length
                    else:
                        l = 0
                        self.make_directories(file)
                        h = open(file, 'wb+')
                        h.flush()
                        h.close()
                    self.mtimes[file] = getmtime(file)
                    self.tops[file] = l
                else:
                    l = 0
                self.sizes[file] = length
                so_far += l

        self.total_length = total
        self._reset_ranges()

        self.max_files_open = config['max_files_open']
        if self.max_files_open > 0 and numfiles > self.max_files_open:
            self.handlebuffer = []
        else:
            self.handlebuffer = None


    if os.name == 'nt':
        def _lock_file(self, name, f):
            """Lock a file on Windows.
            
            @type name: C{string}
            @param name: The file name to lock (only used to get the file size)
            @type f: C{file}
            @param f: a file handle for the file to lock

            """
            
            import msvcrt
            for p in range(0, min(self.sizes[name],MAXLOCKRANGE), MAXLOCKSIZE):
                f.seek(p)
                msvcrt.locking(f.fileno(), msvcrt.LK_LOCK,
                               min(MAXLOCKSIZE,self.sizes[name]-p))

        def _unlock_file(self, name, f):
            """Unlock a file on Windows.
            
            @type name: C{string}
            @param name: The file name to unlock (only used to get the file size)
            @type f: C{file}
            @param f: a file handle for the file to unlock

            """
            
            import msvcrt
            for p in range(0, min(self.sizes[name],MAXLOCKRANGE), MAXLOCKSIZE):
                f.seek(p)
                msvcrt.locking(f.fileno(), msvcrt.LK_UNLCK,
                               min(MAXLOCKSIZE,self.sizes[name]-p))

    elif os.name == 'posix':
        def _lock_file(self, name, f):
            """Lock a file on Linux.
            
            @type name: C{string}
            @param name: The file name to lock (only used to get the file size)
            @type f: C{file}
            @param f: a file handle for the file to lock

            """
            
            import fcntl
            fcntl.flock(f.fileno(), fcntl.LOCK_EX)

        def _unlock_file(self, name, f):
            """Unlock a file on Linux.
            
            @type name: C{string}
            @param name: The file name to unlock (only used to get the file size)
            @type f: C{file}
            @param f: a file handle for the file to unlock

            """
            
            import fcntl
            fcntl.flock(f.fileno(), fcntl.LOCK_UN)

    else:
        def _lock_file(self, name, f):
            """Dummy function to not Lock a file on other systems."""
            pass

        def _unlock_file(self, name, f):
            """Dummy function to not unlock a file on other systems."""
            pass


    def was_preallocated(self, pos, length):
        """Check if a download location was pre-allocated.
        
        @type pos: C{long}
        @param pos: the location to start checking
        @type length: C{long}
        @param length: the amount of the download to check
        @rtype: C{boolean}
        @return: whether the download location was pre-allocated
        
        """
        
        for file, begin, end in self._intervals(pos, length):
            if self.tops.get(file, 0) < end:
                return False
        return True


    def _sync(self, file):
        """Synchronize a file to disk.
        
        Closes the open file so that it is synchronize to disk, the next time
        it is referenced it will be reopened.
        
        @type file: C{string}
        @param file: the name of the file to sync
        
        """
        
        self._close(file)
        if self.handlebuffer:
            self.handlebuffer.remove(file)

    def sync(self):
        """Synchronize all read/write files to disk."""
        for file in self.whandles.keys():
            self._sync(file)


    def set_readonly(self, f=None):
        """Set a file (or all files) to be read-only.
        
        Synchronizes the file (or all read/write files if none is given) to
        disk. The next time the file is accessed it will be reopened.
        
        @type f: C{int}
        @param f: the number of the file to set read-only 
            (optional, default is to set all read/write files read-only)
        
        """
        
        if f is None:
            self.sync()
            return
        file = self.files[f][0]
        if self.whandles.has_key(file):
            self._sync(file)
            

    def get_total_length(self):
        """Get the total length of the download.
        
        @rtype: C{long}
        @return: the total length of the download
        
        """
        
        return self.total_length


    def _open(self, file, mode):
        """Open a file with the given mode.
        
        @type file: C{string}
        @param file: the file name to open
        @type mode: C{string}
        @param mode: the mode to open the file in 
            (r = read, w = write, a = append, add b for binary)
        @rtype: C{file handle}
        @return: the file handle to access the file with
        @raise IOError: if the file has been modified since it was last read
        
        """
        
        if self.mtimes.has_key(file):
            try:
              if self.handlebuffer is not None:
                assert getsize(file) == self.tops[file]
                newmtime = getmtime(file)
                oldmtime = self.mtimes[file]
                assert newmtime <= oldmtime+1
                assert newmtime >= oldmtime-1
            except:
                logger.exception(file+' modified: '
                            +strftime('(%x %X)',localtime(self.mtimes[file]))
                            +strftime(' != (%x %X) ?',localtime(getmtime(file))))
                raise IOError('modified during download')
        try:
            self.make_directories(file)
            return open(file, mode)
        except:
            logger.exception('Error opening the file: '+file)
            raise


    def _close(self, file):
        """Close the file
        
        @type file: C{string}
        @param file: the name of the file to close

        """
        
        f = self.handles[file]
        del self.handles[file]
        if self.whandles.has_key(file):
            del self.whandles[file]
            f.flush()
            self.unlock_file(file, f)
            f.close()
            self.tops[file] = getsize(file)
            self.mtimes[file] = getmtime(file)
        else:
            if self.lock_while_reading:
                self.unlock_file(file, f)
            f.close()


    def _close_file(self, file):
        """Close the file and release the handle.
        
        @type file: C{string}
        @param file: the name of the file to close

        """
        
        if not self.handles.has_key(file):
            return
        self._close(file)
        if self.handlebuffer:
            self.handlebuffer.remove(file)
        

    def _get_file_handle(self, file, for_write):
        """Get a new or existing flie handle for the file.
        
        @type file: C{string}
        @param file: the name of the file to get a handle for
        @type for_write: C{boolean}
        @param for_write: whether to open the file for writing
        @rtype: C{file handle}
        @return: the file handle that can be used for the file
        @raise IOError: if the file can not be opened

        """

        if self.handles.has_key(file):
            if for_write and not self.whandles.has_key(file):
                self._close(file)
                try:
                    f = self._open(file, 'rb+')
                    self.handles[file] = f
                    self.whandles[file] = 1
                    self.lock_file(file, f)
                except (IOError, OSError), e:
                    logger.exception('unable to reopen: '+file)
                    raise IOError('unable to reopen '+file+': '+str(e))

            if self.handlebuffer:
                if self.handlebuffer[-1] != file:
                    self.handlebuffer.remove(file)
                    self.handlebuffer.append(file)
            elif self.handlebuffer is not None:
                self.handlebuffer.append(file)
        else:
            try:
                if for_write:
                    f = self._open(file, 'rb+')
                    self.handles[file] = f
                    self.whandles[file] = 1
                    self.lock_file(file, f)
                else:
                    f = self._open(file, 'rb')
                    self.handles[file] = f
                    if self.lock_while_reading:
                        self.lock_file(file, f)
            except (IOError, OSError), e:
                logger.exception('unable to open: '+file)
                raise IOError('unable to open '+file+': '+str(e))
            
            if self.handlebuffer is not None:
                self.handlebuffer.append(file)
                if len(self.handlebuffer) > self.max_files_open:
                    self._close(self.handlebuffer.pop(0))

        return self.handles[file]


    def _reset_ranges(self):
        """Re-initialize the ranges from the working copies."""
        self.ranges = []
        for l in self.working_ranges:
            self.ranges.extend(l)
        self.begins = [i[0] for i in self.ranges]

    def get_file_range(self, index):
        """Get the file name and range that corresponds to this piece.
        
        @type index: C{int}
        @param index: the piece index to get a file range for
        @rtype: (C{long}, C{long}, C{long}, C{string})
        @return: the start and end offsets of the piece in the file, the length
            of the file, and the name of the file
        
        """
        
        return self.piece_files[index]

    def _intervals(self, pos, amount):
        """Get the files that are within the range.
        
        Finds all the files that occur within a given range in the download,
        and return a list of them. Includes the range of the file that is
        inside the range, which will be the start (0) and end (length) of the 
        file unless it goes past the beginning or end of the range.
        
        @type pos: C{long}
        @param pos: the start of the range within the download
        @type amount: C{long}
        @param amount: the length of the range
        @rtype: C{list} of C{tuple} of (C{string}, C{long}, C{long})
        @return: the list of files and their start and end offsets in the range
        
        """
        
        r = []
        stop = pos + amount
        p = max(bisect(self.begins, pos) - 1,0)
        while p < len(self.ranges):
            begin, end, offset, file = self.ranges[p]
            if begin >= stop:
                break
            if end > pos:
                r.append(( file,
                           offset + max(pos, begin) - begin,
                           offset + min(end, stop) - begin   ))
            p += 1
        return r


    def read(self, pos, amount, flush_first = False):
        """Read data from the download.
        
        @type pos: C{long}
        @param pos: the offset in the download to start reading from
        @type amount: C{long}
        @param amount: the length of the data to read
        @type flush_first: C{boolean}
        @param flush_first: whether to flush the files before reading the data
            (optional, default is not to flush)
        @rtype: L{DebTorrent.piecebuffer.SingleBuffer}
        @return: the data that was read
        @raise IOError: if the read data is not complete
        
        """
        
        r = PieceBuffer()
        for file, pos, end in self._intervals(pos, amount):
            logger.debug('reading '+file+' from '+str(pos)+' to '+str(end))
            self.lock.acquire()
            h = self._get_file_handle(file, False)
            if flush_first and self.whandles.has_key(file):
                h.flush()
                fsync(h)
            h.seek(pos)
            while pos < end:
                length = min(end-pos, MAXREADSIZE)
                data = h.read(length)
                if len(data) != length:
                    raise IOError('error reading data from '+file)
                r.append(data)
                pos += length
            self.lock.release()
        return r

    def write(self, pos, s):
        """Write data to the download.
        
        @type pos: C{long}
        @param pos: the offset in the download to start writing at
        @type s: C{string}
        @param s: data to write
        
        """

        # might raise an IOError
        total = 0
        for file, begin, end in self._intervals(pos, len(s)):
            logger.debug('writing '+file+' from '+str(pos)+' to '+str(end))
            self.lock.acquire()
            h = self._get_file_handle(file, True)
            h.seek(begin)
            h.write(s[total: total + end - begin])
            self.lock.release()
            total += end - begin

    def make_directories(self, file):
        """Create missing parent directories for a file.
        
        @type file: C{string}
        @param file: the file name to create directories for
        
        """
        
        file = split(file)[0]
        if file != '' and not exists(file):
            logger.debug('creating directories: '+file)
            makedirs(file)

    def top_off(self):
        """Extend all files to their appropriate length."""
        for begin, end, offset, file in self.ranges:
            l = offset + end - begin
            if l > self.tops.get(file, 0):
                self.lock.acquire()
                h = self._get_file_handle(file, True)
                h.seek(l-1)
                h.write(chr(0xFF))
                self.lock.release()

    def flush(self):
        """Flush all files to disk."""
        # may raise IOError or OSError
        for file in self.whandles.keys():
            self.lock.acquire()
            self.handles[file].flush()
            self.lock.release()

    def close(self):
        """Close all open files."""
        for file, f in self.handles.items():
            try:
                self.unlock_file(file, f)
            except:
                pass
            try:
                f.close()
            except:
                pass
        self.handles = {}
        self.whandles = {}
        self.handlebuffer = None


    def _get_disabled_ranges(self, f):
        """Calculate the file ranges for the disabled file.
        
        Calculates, based on the file lengths and piece lengths, the ranges
        to write for the file. There are three lists calculated.
        
        The working range is the list of files and file offsets to write if 
        the file is enabled.
        
        The shared pieces is a list of piece numbers that the file shares 
        with other files.
        
        The disabled range is the list of files and file offsets to write if
        the file is disabled.
        
        @type f: C{int}
        @param f: the index of the file
        @rtype: C{tuple}
        @return: a tuple containing the working range, shared pieces, and 
            disabled range
        
        """
        
        if not self.file_ranges[f]:
            return ((),(),())
        r = self.disabled_ranges[f]
        if r:
            return r
        start, end, offset, file = self.file_ranges[f]
        start_piece, end_piece = self.file_pieces[f]
        logger.debug('calculating disabled range for '+self.files[f][0])
        logger.debug('bytes: '+str(start)+'-'+str(end))
        logger.debug('file spans pieces '+str(start_piece)+'-'+str(end_piece))
        pieces = range(start_piece, end_piece+1)
        offset = 0
        disabled_files = []
        # Dramatically simplified since files always start and end on piece boundaries
        working_range = [(start, end, offset, file)]
        update_pieces = []

        logger.debug('working range: '+str(working_range))
        logger.debug('update pieces: '+str(update_pieces))
        r = (tuple(working_range), tuple(update_pieces), tuple(disabled_files))
        self.disabled_ranges[f] = r
        return r
        

    def set_bufferdir(self, dir):
        """Sets the buffer directory.
        
        @type dir: C{string}
        @param dir: the new buffer directory
        
        """
        
        self.bufferdir = dir

    def enable_file(self, f):
        """Enable a file for writing.
        
        @type f: C{int}
        @param f: the index of the file to enable
        
        """
        
        if not self.disabled[f]:
            return
        logger.info('enabling file '+self.files[f][0])
        self.disabled[f] = False
        r = self.file_ranges[f]
        if not r:
            return
        file = r[3]
        if not exists(file):
            self.make_directories(file)
            h = open(file, 'wb+')
            h.flush()
            h.close()
        if not self.tops.has_key(file):
            self.tops[file] = getsize(file)
        if not self.mtimes.has_key(file):
            self.mtimes[file] = getmtime(file)
        self.working_ranges[f] = [r]

    def disable_file(self, f):
        """Disable a file from writing.
        
        @type f: C{int}
        @param f: the index of the file to disable
        
        """
        if self.disabled[f]:
            return
        logger.info('disabling file '+self.files[f][0])
        self.disabled[f] = True
        r = self._get_disabled_ranges(f)
        if not r:
            return
        for begin, end, offset, file in r[2]:
            if not os.path.isdir(self.bufferdir):
                os.makedirs(self.bufferdir)
            if not exists(file):
                h = open(file, 'wb+')
                h.flush()
                h.close()
            if not self.tops.has_key(file):
                self.tops[file] = getsize(file)
            if not self.mtimes.has_key(file):
                self.mtimes[file] = getmtime(file)
        self.working_ranges[f] = r[2]

    reset_file_status = _reset_ranges


    def get_piece_update_list(self, f):
        """Get the list of pieces the file shares with other files.
        
        @type f: C{int}
        @param f: the index of the file to disable
        @rtype: C{list} of C{int}
        @return: the list of piece indexes
        
        """
        
        return self._get_disabled_ranges(f)[1]


    def delete_file(self, f):
        """Delete the file.
        
        @type f: C{int}
        @param f: the index of the file to delete
        
        """
        try:
            os.remove(self.files[f][0])
        except:
            pass


    def pickle(self):
        """Create a dictionary representing the current state of the download.
        
        Pickled data format::
    
            d['files'] = {file name: [size, mtime], file name: [size, mtime], ...}
                        file name in torrent, and the size and last modification
                        time for those files.  Missing files are either empty
                        or disabled.
        
        @rtype: C{dictionary}
        @return: the pickled current status of the download
        
        """
        
        files = {}
        for i in xrange(len(self.files)):
            if not self.files[i][1]:    # length == 0
                continue
            if self.disabled[i]:
                continue
            file = self.files[i][0]
            files[file] = [getsize(file), int(getmtime(file))]
        return {'files': files}


    def unpickle(self, data):
        """Extract the current status of the download from a pickled dictionary.
        
        Assumes all previously-disabled files have already been disabled.
        
        @type data: C{dictionary}
        @param data: the pickled current status of the download
        @rtype: C{list} of C{int}
        @return: a list of the currently enabled pieces
        
        """

        try:
            files = data['files']
            logger.debug('saved file characteristics: %r', files)
            
            valid_pieces = {}
            for i in xrange(len(self.files)):
                if self.disabled[i]:
                    continue
                r = self.file_ranges[i]
                if not r:
                    continue
                start, end, offset, file =r
                start_piece, end_piece = self.file_pieces[i]
                logger.debug('adding '+file)
                for p in xrange(start_piece, end_piece+1):
                    valid_pieces[p] = 1

            logger.info('Saved list of valid pieces: '+str(valid_pieces.keys()))
            
            def test(old, size, mtime):
                """Test that the file has not changed since the status save.                
                
                @type old: C{list} of [C{long}, C{long}]
                @param old: the previous size and modification time of the file
                @type size: C{long}
                @param size: the current size of the file
                @type mtime: C{long}
                @param mtime: the current modification time of the file
                @rtype: C{boolean}
                @return: whether the file has been changed

                """
                
                oldsize, oldmtime = old
                if size != oldsize:
                    logger.debug('Size of file has changed: %d to %d', oldsize, size)
                    return False
                if mtime > oldmtime+1:
                    logger.debug('Time of file has changed: %d to %d', oldmtime, mtime)
                    return False
                if mtime < oldmtime-1:
                    logger.debug('Time of file has changed: %d to %d', oldmtime, mtime)
                    return False
                return True

            for i in xrange(len(self.files)):
                if self.disabled[i]:
                    continue
                file, size = self.files[i]
                if not size:
                    continue
                if ( not files.has_key(file)
                     or not test(files[file],getsize(file),getmtime(file)) ):
                    start, end, offset, file = self.file_ranges[i]
                    start_piece, end_piece = self.file_pieces[i]
                    logger.debug('removing '+file)
                    for p in xrange(start_piece, end_piece+1):
                        if valid_pieces.has_key(p):
                            del valid_pieces[p]
        except:
            if 'files' in data:
                logger.exception('Error unpickling data cache')
            return []

        logger.info('Final list of valid pieces: '+str(valid_pieces.keys()))
        return valid_pieces.keys()

