C Copyright 1981-2007 ECMWF
C 
C Licensed under the GNU Lesser General Public License which
C incorporates the terms and conditions of version 3 of the GNU
C General Public License.
C See LICENSE and gpl-3.0.txt for details.
C

      INTEGER FUNCTION IAOGDEF (PWEST, PEAST, PNORTH, PSOUTH,
     1   KGAUSS, KAREA, OCHNG, OIWEGLOBE, OINPOLE, OISPOLE,
     2   KIAREA, KPR, KERR)
C
C---->
C**** *IAOGDEF*
C
C     PURPOSE
C     _______
C
C     Set the output field definition variables.
C
C     INTERFACE
C     _________
C
C     IERR = IAOGDEF (PWEST, PEAST, PNORTH, PSOUTH, KGAUSS, KAREA,
C    1   OCHNG, OIWEGLOBE, OINPOLE, OISPOLE, KIAREA, KPR, KERR)
C
C     Input parameters
C     ________________
C
C     PWEST      - The Western limit of the output field area.
C
C     PEAST      - The Eastern limit of the output field area.
C
C     PNORTH     - The Northern limit of the output field area.
C
C     PSOUTH     - The Southern limit of the output field area.
C
C     KGAUSS     - The Gaussian truncation for the output field.
C
C     KAREA      - The previous output field area used for checking
C                  for definition changes.
C
C     OIWEGLOBE  - Flag indicating whether the INPUT field spans the
C                  globe in the West-East direction.
C
C     OINPOLE    - Flag indicating whether the INPUT field includes
C                  the North polar line of latitude.
C
C     OISPOLE    - Flag indicating whether the INPUT field includes
C                  the South polar line of latitude.
C
C     KIAREA     - The input field area which will be used for an
C                  uninitialised output field area when the input
C                  field is not global.
C
C     KPR        - The debug print switch.
C                  0  , No debugging output.
C                  1  , Produce debugging output.
C
C     KERR       - The error control flag.
C                  -ve, No error message. Return error code.
C                  0  , Hard failure with error message.
C                  +ve, Print error message. Return error code.
C
C     Output parameters
C     ________________
C
C     KAREA      - The output field area if the definition has changed.
C
C     OCHNG      - A flag variable indicating whether the definition
C                  has changed.
C
C     Return value
C     ____________
C
C     The error indicator (INTEGER).
C
C     Error and Warning Return Values
C     _______________________________
C
C     21601 An illegal Gaussian truncation was requested.
C     21602 The default West-East output limits for a non-global
C           input field did not match the increment.
C     21603 Default North-South limits were specified for a
C           non-global input field. This is not allowed for a
C           Gaussian output field.
C
C     Common block usage
C     __________________
C
C     nofld.common - This file contains all the output field
C                    definition variables.
C
C     NOGAUSS      - Output field Gaussian truncation is set.
C     NOREPR       - Output field representation is set.
C     NOSCNM       - Output field scanning mode is set.
C
C     EXTERNALS
C     _________
C
C     INTLOG     - Logs messages.
C
C     METHOD
C     ______
C
C     Check the output definition against the previous definition and
C     update appropriate definition variables.
C
C     REFERENCE
C     _________
C
C     None
C
C     COMMENTS
C     ________
C
C     Program contains sections 0 to 3 and 9
C
C     AUTHOR
C     ______
C
C     K. Fielding      *ECMWF*      Apr 1994
C
C     MODIFICATIONS
C     _____________
C
C     None
C
C----<
C     _______________________________________________________
C
C
C*    Section 0. Definition of variables.
C     _______________________________________________________
C
C*    Prefix conventions for variable names
C
C     Logical      L (but not LP), global or common.
C                  O, dummy argument
C                  G, local variable
C                  LP, parameter.
C     Character    C, global or common.
C                  H, dummy argument
C                  Y (but not YP), local variable
C                  YP, parameter.
C     Integer      M and N, global or common.
C                  K, dummy argument
C                  I, local variable
C                  J (but not JP), loop control
C                  JP, parameter.
C     REAL         A to F and Q to X, global or common.
C                  P (but not PP), dummy argument
C                  Z, local variable
C                  PP, parameter.
C
C     Implicit statement to force declarations
C
      IMPLICIT NONE
C
#include "parim.h"
#include "nofld.common"
C
C     Subroutine arguments
C
      LOGICAL OCHNG, OIWEGLOBE, OINPOLE, OISPOLE
      INTEGER KGAUSS, KPR, KERR
      INTEGER KAREA (4), KIAREA (4)
      REAL PWEST, PEAST, PNORTH, PSOUTH
C
C     Local variables
      LOGICAL GLOBE, GNS
      INTEGER INORTH, ISOUTH, IWEST, IEAST, IWEINC, IWETMP, ITEMP
      INTEGER IERR
      INTEGER JPROUTINE
      PARAMETER (JPROUTINE = 21600)
C
C     External functions
C
C     Transform definition variables that must be preserved
C
C
C     _______________________________________________________
C
C*    Section 1. Initialisation
C     _______________________________________________________
C
  100 CONTINUE
C
      IF (KPR.GE.1) CALL INTLOG(JP_DEBUG,'IAOGDEF: Section 1.',JPQUIET)
C
      IAOGDEF = 0
C
      IERR = 0
C
      OCHNG = .FALSE.
C
      GNS = OINPOLE .AND. OISPOLE
C
      GLOBE = OIWEGLOBE .AND. GNS
C
      IF (KPR .GE. 1) THEN
         CALL INTLOG(JP_DEBUG,'IAOGDEF: Input parameters',JPQUIET)
         CALL INTLOG(JP_DEBUG,'IAOGDEF: Gaussian truncation = ',KGAUSS)
         CALL INTLOGR(JP_DEBUG,'IAOGDEF: Area North = ',PNORTH)
         CALL INTLOGR(JP_DEBUG,'IAOGDEF: Area West = ',PWEST)
         CALL INTLOGR(JP_DEBUG,'IAOGDEF: Area South = ',PSOUTH)
         CALL INTLOGR(JP_DEBUG,'IAOGDEF: Area East = ',PEAST)
      ENDIF
C
C     _______________________________________________________
C
C
C*    Section 2. Convert parameters to integer
C     _______________________________________________________
C
  200 CONTINUE
C
      IF (KPR.GE.1) CALL INTLOG(JP_DEBUG,'IAOGDEF: Section 2.',JPQUIET)
C
C     Representation
C
      IF (NOREPR .NE. JPGAUSSIAN) THEN
         NOREPR = JPGAUSSIAN
         OCHNG = .TRUE.
      ENDIF
C
      NOSCNM = 0
C
C     North, South, West and East limits
C
      INORTH = NINT (PNORTH * PPMULT)
      ISOUTH = NINT (PSOUTH * PPMULT)
      IWEST = NINT (PWEST * PPMULT)
      IEAST = NINT (PEAST * PPMULT)
C
C     Direction increments for regular grids
C
      IWEINC = JP90 / KGAUSS
C
      IF (IWEINC * KGAUSS .NE. JP90) THEN
        IAOGDEF = JPROUTINE + 1
        IF (KERR .GE. 0) THEN
          CALL INTLOG(JP_ERROR,
     X      'IAOGDEF: Illegal gaussian truncation = ',KGAUSS)
          IF (KERR .EQ. 0) CALL INTLOG(JP_FATAL,
     X      'IAOGDEF: Interpolation failing.',JPQUIET)
        ENDIF
        GO TO 900
      ENDIF
C
C
C     _______________________________________________________
C
C
C*    Section 3. Adjust grid stride and area
C     _______________________________________________________
C
  300 CONTINUE
C
      IF (KPR.GE.1) CALL INTLOG(JP_DEBUG,'IAOGDEF: Section 3.',JPQUIET)
C
C     Global field if all area limits are 0
C
      IF (INORTH .EQ. 0 .AND. ISOUTH .EQ. 0 .AND.
     X  IWEST .EQ. 0 .AND. IEAST .EQ. 0) THEN
C
        INORTH = JP90
        ISOUTH = - JP90
C
        IF (GLOBE) THEN
C
          IWEST = 0
          IEAST = JP360 - IWEINC
C
        ELSE
          IF (GNS) THEN
C
            IWEST = KIAREA (JPWEST)
            IEAST = KIAREA (JPEAST)
C
            IF (MOD (IEAST - IWEST, IWEINC) .NE. 0) THEN
              IAOGDEF = JPROUTINE + 2
              IF (KERR .GE. 0) THEN
                CALL INTLOG(JP_ERROR,
     X            'IAOGDEF: West-East increment ', IWEINC)
                CALL INTLOG(JP_ERROR,
     X            'IAOGDEF: does not match the range from ',IWEST)
                CALL INTLOG(JP_ERROR,
     X            'IAOGDEF: to ',IEAST)
                IF (KERR .EQ. 0) CALL INTLOG(JP_FATAL,
     X            'IAOGDEF: Interpolation failing.',JPQUIET)
              ENDIF
              GO TO 900
            ENDIF
          ELSE
            IAOGDEF = JPROUTINE + 3
              IF (KERR .GE. 0) THEN
                CALL INTLOG(JP_ERROR,
     X            'IAOGDEF: Must specify a range if Gaussian',JPQUIET)
                CALL INTLOG(JP_ERROR,
     X            'IAOGDEF: output and non-global input.',JPQUIET)
                IF (KERR .EQ. 0) CALL INTLOG(JP_FATAL,
     X            'IAOGDEF: Interpolation failing.',JPQUIET)
              ENDIF
              GO TO 900
          ENDIF
        ENDIF
C
      ELSE
C
C     Try to produce fields representable in GRIB for 0.5625 degree
C     strides
C
        IF (IEAST .LT. IWEST) THEN
          IF (IEAST .LE. 0) THEN
            IEAST = IEAST + JP360
          ELSE
            IWEST = IWEST - JP360
          ENDIF
        ENDIF
C
        GLOBE = IEAST - IWEST + IWEINC .EQ. JP360 .OR.
     X      IEAST - IWEST + IWEINC .EQ. 0
C
        IF (MOD (IWEINC, 10) .EQ. 5 .AND. .NOT. GLOBE) THEN
          IWETMP = IWEINC * 2
        ELSE
          IWETMP = IWEINC
        ENDIF
C
        IF (MOD (IEAST - IWEST, IWETMP) .NE. 0) THEN
C
          IF (MOD (IEAST, IWETMP) .NE. 0) THEN
C
            ITEMP = ABS (IEAST) / IWETMP
C
            IF (IEAST .GT. 0) THEN
              IEAST = (ITEMP + 1) * IWETMP
              IF (IEAST .GT. JP360) IEAST = IEAST - IWETMP
            ELSE
              IEAST = - ITEMP * IWETMP
            ENDIF
          ENDIF
C
          IF (MOD (IWEST, IWETMP) .NE. 0) THEN
C
            ITEMP = ABS (IWEST) / IWETMP
C
            IF (IWEST .LT. 0) THEN
              IWEST = - (ITEMP + 1) * IWETMP
              IF (IWEST .LT. - JP360) IWEST = IWEST + IWETMP
            ELSE
              IWEST = ITEMP * IWETMP
            ENDIF
          ENDIF
        ENDIF
      ENDIF
C
      IF (NOGAUSS .NE. KGAUSS) THEN
        NOGAUSS = KGAUSS
        OCHNG = .TRUE.
      ENDIF
C
      IF (KAREA (JPNORTH) .NE. INORTH) THEN
        KAREA (JPNORTH) = INORTH
        OCHNG = .TRUE.
      ENDIF
C
      IF (KAREA (JPSOUTH) .NE. ISOUTH) THEN
        KAREA (JPSOUTH) = ISOUTH
        OCHNG = .TRUE.
      ENDIF
C
      IF (IWEST .NE. KAREA (JPWEST) ) THEN
        KAREA (JPWEST) = IWEST
        OCHNG = .TRUE.
      ENDIF
C
      IF (IEAST .NE. KAREA (JPEAST) ) THEN
        KAREA (JPEAST) = IEAST
        OCHNG = .TRUE.
      ENDIF
C
      IF (KPR .GE. 1) THEN
        CALL INTLOG(JP_DEBUG,
     X    'IAOGDEF: Output field definition parameters.',JPQUIET)
        CALL INTLOG(JP_DEBUG,
     X    'IAOGDEF: Representation is ',NOREPR)
        CALL INTLOG(JP_DEBUG,
     X    'IAOGDEF: Gaussian truncation is ',NOGAUSS)
        CALL INTLOG(JP_DEBUG,
     X    'IAOGDEF: Area North is ',KAREA(1))
        CALL INTLOG(JP_DEBUG,
     X    'IAOGDEF: Area West  is ',KAREA(2))
        CALL INTLOG(JP_DEBUG,
     X    'IAOGDEF: Area South is ',KAREA(3))
        CALL INTLOG(JP_DEBUG,
     X    'IAOGDEF: Area East  is ',KAREA(4))
        IF( OCHNG ) THEN
          CALL INTLOG(JP_DEBUG,
     X      'IAOGDEF: Output field change is TRUE.',JPQUIET)
        ELSE
          CALL INTLOG(JP_DEBUG,
     X      'IAOGDEF: Output field change is TRUE.',JPQUIET)
        ENDIF
C
      ENDIF
C
C     _______________________________________________________
C
C*    Section 9. Return to calling routine. Format statements
C     _______________________________________________________
C
  900 CONTINUE
C
      IF (KPR.GE.1) CALL INTLOG(JP_DEBUG,'IAOGDEF: Section 9.',JPQUIET)
C
      RETURN
      END
