/*
 * Derived from MPlayer (libao2 - ao_oss).
 *
 * Copyright (c) FAUMachine Team.
 * Copyright (c) MPlayer Team.
 *
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * OSS audio output driver for FAUmachine
 */

#include <stdio.h>
#include <stdlib.h>

#include <sys/ioctl.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include "config.h"
#include "glue-audio_internal.h"

static ao_info_t info = {
	"OSS/ioctl audio output",
	"oss",
	"A'rpi",
	""
};

/* Support for >2 output channels added 2001-11-25 - Steve Davies <steve@daviesfam.org> */

LIBAO_EXTERN(oss);

static int oss_outburst = OUTBURST;
static int oss_buffersize = 0;
static int oss_bps = 0;

static audio_buf_info zz;
static int audio_fd=-1;
static int audio_delay_method=2;

/* open & setup audio device
 * return: 1=success 0=fail
 */
static int
init(void)
{
#ifdef LINUX
	audio_fd=open(PATH_DEV_DSP, O_WRONLY | O_NONBLOCK);
#else
	audio_fd=open(PATH_DEV_DSP, O_WRONLY);
#endif
	if(audio_fd<0){
		return 0;
	}

#ifdef LINUX
	/* Remove the non-blocking flag */
	if(fcntl(audio_fd, F_SETFL, 0) < 0) {
		return 0;
	}  
#endif

#if defined(FD_CLOEXEC) && defined(F_SETFD)
	fcntl(audio_fd, F_SETFD, FD_CLOEXEC);
#endif

	int format=GLUE_AUDIO_FORMAT;
	if(ioctl(audio_fd, SNDCTL_DSP_SETFMT, &format)<0 ||
			format != GLUE_AUDIO_FORMAT) {
		return 0;
	}

	int channels = GLUE_AUDIO_CHANNELS - 1;
	if (ioctl (audio_fd, SNDCTL_DSP_STEREO, &channels) == -1) {
		return 0;
	}

	// set rate
	int samplerate = GLUE_AUDIO_RATE;
	ioctl (audio_fd, SNDCTL_DSP_SPEED, &samplerate);

	if(ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &zz)==-1){
		int r=0;
		if(ioctl(audio_fd, SNDCTL_DSP_GETBLKSIZE, &r)!=-1){
			oss_outburst=r;
		}
	} else {
		if(oss_buffersize==-1) oss_buffersize=zz.bytes;
		oss_outburst=zz.fragsize;
	}

	if(oss_buffersize==-1){
		// Measuring buffer size:
		oss_buffersize=0;
#ifdef HAVE_AUDIO_SELECT
		void* data;
		data=malloc(oss_outburst); memset(data,0,oss_outburst);
		while(oss_buffersize<0x40000){
			fd_set rfds;
			struct timeval tv;
			FD_ZERO(&rfds); FD_SET(audio_fd,&rfds);
			tv.tv_sec=0; tv.tv_usec = 0;
			if(!select(audio_fd+1, NULL, &rfds, NULL, &tv)) break;
			write(audio_fd,data,oss_outburst);
			oss_buffersize+=oss_outburst;
		}
		free(data);
		if(oss_buffersize==0){
			return 0;
		}
#endif
	}

	oss_bps=channels * GLUE_AUDIO_FORMAT_BYTES;
	oss_outburst-=oss_outburst % oss_bps; // round down
	oss_bps*=samplerate;

	return 1;
}

/* close audio device */
static void
uninit(int immed)
{
	if(audio_fd == -1) return;
#ifdef SNDCTL_DSP_SYNC
	// to get the buffer played
	if (!immed)
		ioctl(audio_fd, SNDCTL_DSP_SYNC, NULL);
#endif
#ifdef SNDCTL_DSP_RESET
	if (immed)
		ioctl(audio_fd, SNDCTL_DSP_RESET, NULL);
#endif
	close(audio_fd);
	audio_fd = -1;
}

/* return: how many bytes can be played without blocking */
	static int
get_space()
{
	int playsize=oss_outburst;

#ifdef SNDCTL_DSP_GETOSPACE
	if(ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &zz)!=-1){
		// calculate exact buffer space:
		playsize = zz.fragments*zz.fragsize;
		if (playsize > MAX_OUTBURST)
			playsize = (MAX_OUTBURST / zz.fragsize) * zz.fragsize;
		return playsize;
	}
#endif

	// check buffer
#ifdef HAVE_AUDIO_SELECT
	{  fd_set rfds;
		struct timeval tv;
		FD_ZERO(&rfds);
		FD_SET(audio_fd, &rfds);
		tv.tv_sec = 0;
		tv.tv_usec = 0;
		if(!select(audio_fd+1, NULL, &rfds, NULL, &tv)) return 0; // not block!
	}
#endif

	return oss_outburst;
}

/* plays 'len' bytes of 'data'
 * it should round it down to outburst*n
 * return: number of bytes played
 */
static int
play(void* data,int len)
{
	len/=oss_outburst;
	len=write(audio_fd,data,len*oss_outburst);
	return len;
}


/* return: delay in seconds between first and last sample in buffer */
static float
get_delay()
{
	/* Calculate how many bytes/second is sent out */
	if(audio_delay_method==2){
#ifdef SNDCTL_DSP_GETODELAY
		int r=0;
		if(ioctl(audio_fd, SNDCTL_DSP_GETODELAY, &r)!=-1)
			return ((float)r)/(float)oss_bps;
#endif
		audio_delay_method=1; // fallback if not supported
	}
	if(audio_delay_method==1){
		// SNDCTL_DSP_GETOSPACE
		if(ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &zz)!=-1)
			return ((float)(oss_buffersize-zz.bytes))/(float)oss_bps;
		audio_delay_method=0; // fallback if not supported
	}
	return ((float)oss_buffersize)/(float)oss_bps;
}
