/*
 * Copyright (C) 2004-2012 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

/* ===================== RUNTIME || INIT ======================== */
#if defined(RUNTIME_RM) || defined(INIT_RM)

#define FD_CHANGELINE_SUPPORT	1

struct diskette_param_table_t {
	/* FIXME VOSSI */
	uint8_t x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10;
};

extern const struct diskette_param_table_t diskette_param_table;

#endif /* RUNTIME_RM || INIT_RM */
/* ==================== RUNTIME_RM ==================== */
#ifdef RUNTIME_RM

CODE16;

#include "assert.h"
#include "fixme.h"
#include "stdio.h"
#include "string.h"
#include "in.h"

#include "io.h"
#include "const.h"
#include "cmos.h"
#include "var.h"
#include "el_torito.h"
#include "ptrace.h"
#include "floppy.h"

/*
 * Since no provisions are made for multiple drive types, most
 * values in this table are ignored.  I set parameters for 1.44M
 * floppy here
 */
/* Should be located at 0xefc7 in ROM. FIXME VOSSI */

const struct diskette_param_table_t diskette_param_table = {
	0xAF,
	0x02, /* Head load time 0000001, DMA used. */
	0x25,
	0x02,
	18,
	0x1B,
	0xFF,
	0x6C,
	0xF6,
	0x0F,
	0x08
};

static void
floppy_poweron(uint16_t drive)
{
	var_put(fmot_tmout, 0x5a); /* About 5sec at 18Hz. */

	if (var_get(fmot_flag) & (1 << drive)) {
		/* Floppy already spinning... */
		return;
	}

	var_put(fmot_flag, var_get(fmot_flag) | (1 << drive));
	outb(0x0c | ((var_get(fmot_flag) & 0x0f) << 4) | drive, 0x3f2);

	/* Wait for floppy spinning up... */
	/* FIXME VOSSI */
}

static ALWAYS_INLINE void
floppy_poweroff(void)
{
	outb(0x0c, 0x3f2);
	var_put(fmot_flag, 0x00);
}

void
floppy_tick(void)
{
	uint8_t timeout;

	timeout = var_get(fmot_tmout);

	if (timeout == 0) {
		/* Nothing to do... */
		return;
	}

	timeout--;
	var_put(fmot_tmout, timeout);

	if (timeout != 0) {
		/* Nothing to do... */
		return;
	}

	/* Power-off motor. */
	floppy_poweroff();
}

/*static*/ uint8_t
floppy_err(uint8_t st0, uint8_t st1, uint8_t st2)
{
	uint8_t ret;

	if (((st0 >> 6) & 3) == 0) { /* Interrupt code. */
		ret = 0; /* No error. */

	} else {
		/* Error. */
		if ((st1 >> 0) & 1) { /* Missing address mark. */
			ret = 0x02; /* Missing address mark. */
		} else if ((st1 >> 1) & 1) { /* Not writable. */
			ret = 0x03; /* Write protected. */
		} else if ((st1 >> 2) & 1) { /* No data. */
			ret = 0x04; /* Sector not found / read error. */
		} else if ((st1 >> 4) & 1) { /* Overrun/underrun. */
			ret = 0x08; /* DMA overrun/underrun. */
		} else if ((st1 >> 5) & 1) { /* Data error. */
			ret = 0x10; /* CRC checksum error. */
		} else {
			ret = 0x80; /* Unknown error. */ /* FIXME VOSSI */
		}
	}

#if 0
	dprintf("floppy_err: ret=0x%02x\n", ret);
#endif

	return ret;
}

uint8_t
floppy_read_chs(
	uint8_t drive,
	uint8_t num,
	uint8_t cylinder,
	uint8_t head,
	uint8_t sector,
	uint16_t buffer_seg,
	uint16_t buffer_off
)
{
	uint8_t status;
	uint32_t size;
	uint32_t address;

	if (1 < drive) {
		return 0x01;	/* invalid parameter */
	}

	if (1 < head) {
		return 0x01;	/* invalid parameter */
	}

	size = num * 512 - 1;
	address = ((uint32_t) buffer_seg << 4) + buffer_off;
	if((address & 0xffff) + size > 0xffff) {
		return 0x09;	/* dma boundary */
	}

	/*
	 * Start motor
	 */
	floppy_poweron(drive);
	
	/*
	 * Set up DMA
	 */
	outb(0x06, 0x0a); /* Enable DMA 2. */
	
	outb(0x00, 0x0c); /* Clear byte pointer flip/flop. */
	outb((address >> 0) & 0xff, 0x04); /* Address low byte. */
	outb((address >> 8) & 0xff, 0x04); /* Address high byte. */
	
	outb(0x00, 0x0c); /* Clear byte pointer flip/flop. */
	outb((size >> 0) & 0xff, 0x05); /* Count low byte. */
	outb((size >> 8) & 0xff, 0x05); /* Count high byte. */
	
	/* Mode register: single block, inc addr, no autoinit, read, chan 2 */
	outb((1 << 6) | (0 << 5) | (0 << 4) | (1 << 2) | 2, 0x0b);

	outb((address >> 16) & 0xff, 0x81); /* Page register */

	/*
	 * Wait until DR is ready
	 */
	do {
		status = inb(0x3f4);
	} while ((status & 0xc0) != 0x80);

	/*
	 * Send read command.
	 */
	outb(0xc6, 0x3f5); /* READ with multitrack */
	outb((head << 2) | drive, 0x3f5);
	outb(cylinder, 0x3f5); /* cylinder */
	outb(head, 0x3f5); /* head */
	outb(sector, 0x3f5); /* sector */
	outb(0x02, 0x3f5); /* sectorsize = 0x02 >> 7 */
	outb(18, 0x3f5); /* sector per track/side */
	outb(0x1b, 0x3f5); /* length of GAP 3 */
	outb(0xff, 0x3f5); /* data length (0xff = unused) */
	
	/*
	 * Wait until command is has been finished
	 */
	var_put(f_recal, var_get(f_recal) & ~(1 << 7));
	while (! (var_get(f_recal) & (1 << 7))) {
		asm volatile (
			"sti\n\t"
			"hlt\n\t"
			"cli\n\t"
		);
	}

	/*
	 * Read result
	 */
	var_put(f_stat[0], inb(0x3f5));
	var_put(f_stat[1], inb(0x3f5));
	var_put(f_stat[2], inb(0x3f5));
	var_put(f_stat[3], inb(0x3f5));
	var_put(f_stat[4], inb(0x3f5));
	var_put(f_stat[5], inb(0x3f5));
	var_put(f_stat[6], inb(0x3f5));

	/*
	 * DMA
	 */
	outb(0x02, 0x0a); /* Disable DMA 2. */

	return floppy_err(var_get(f_stat[0]),
			var_get(f_stat[1]), var_get(f_stat[2]));
}

uint8_t
floppy_write_chs(
	uint8_t drive,
	uint8_t num,
	uint8_t cylinder,
	uint8_t head,
	uint8_t sector,
	uint16_t buffer_seg,
	uint16_t buffer_off
)
{
	uint8_t status;
	uint32_t size;
	uint32_t address;

	if (1 < drive) {
		return 0x01;	/* invalid parameter */
	}

	if (1 < head) {
		return 0x01;	/* invalid parameter */
	}

	size = num * 512 - 1;
	address = ((uint32_t) buffer_seg << 4) + buffer_off;
	if((address & 0xffff) + size > 0xffff) {
		return 0x09;	/* dma boundary */
	}

	/*
	 * Start motor
	 */
	floppy_poweron(drive);
	
	/*
	 * Set up DMA
	 */
	outb(0x06, 0x0a); /* Enable DMA 2. */
	
	outb(0x00, 0x0c); /* Clear byte pointer flip/flop. */
	outb((address >> 0) & 0xff, 0x04); /* Address low byte. */
	outb((address >> 8) & 0xff, 0x04); /* Address high byte. */
	
	outb(0x00, 0x0c); /* Clear byte pointer flip/flop. */
	outb((size >> 0) & 0xff, 0x05); /* Count low byte. */
	outb((size >> 8) & 0xff, 0x05); /* Count high byte. */
	
	/* Mode register: single block, inc addr, no autoinit, write, chan 2 */
	outb((1 << 6) | (0 << 5) | (0 << 4) | (2 << 2) | 2, 0x0b);

	outb((address >> 16) & 0xff, 0x81); /* Page register */

	/*
	 * Wait until DR is ready
	 */
	do {
		status = inb(0x3f4);
	} while ((status & 0xc0) != 0x80);

	/*
	 * Send write command.
	 */
	outb(0xc5, 0x3f5); /* WRITE */
	outb((head << 2) | drive, 0x3f5);
	outb(cylinder, 0x3f5); /* cylinder */
	outb(head, 0x3f5); /* head */
	outb(sector, 0x3f5); /* sector */
	outb(0x02, 0x3f5); /* sectorsize = 0x02 >> 7 */
	outb(18, 0x3f5); /* sectors per track/side */
	outb(0x1b, 0x3f5); /* length of GAP 3 */
	outb(0xff, 0x3f5); /* data length (0xff = unused) */
	
	/*
	 * Wait until command is has been finished
	 */
	var_put(f_recal, var_get(f_recal) & ~(1 << 7));
	while (! (var_get(f_recal) & (1 << 7))) {
		asm volatile (
			"sti\n\t"
			"hlt\n\t"
			"cli\n\t"
		);
	}

	/*
	 * Read result
	 */
	var_put(f_stat[0], inb(0x3f5));
	var_put(f_stat[1], inb(0x3f5));
	var_put(f_stat[2], inb(0x3f5));
	var_put(f_stat[3], inb(0x3f5));
	var_put(f_stat[4], inb(0x3f5));
	var_put(f_stat[5], inb(0x3f5));
	var_put(f_stat[6], inb(0x3f5));
	
	/*
	 * DMA
	 */
	outb(0x02, 0x0a); /* Disable DMA 2. */

	return floppy_err(var_get(f_stat[0]),
			var_get(f_stat[1]), var_get(f_stat[2]));
}

uint8_t
floppy_verify_chs(
	uint8_t drive,
	uint8_t num,
	uint8_t cylinder,
	uint8_t head,
	uint8_t sector,
	uint16_t buffer_seg,
	uint16_t buffer_off)
{
	return 0x01;
}

#if FD_CHANGELINE_SUPPORT
uint8_t
floppy_check_changeline(uint8_t drive)
{
	uint8_t status;
	uint8_t result;

	if (2 <= drive) {
		return 0x80;	/* drive not ready or present */
	}

	/* FIXME sand:
	 * not sure, whether this method to figure out
	 * disk change is correct...
	 */

	floppy_poweron(drive);

	if (inb(0x3f7) & 0x80) {
		/*
		 * Wait until DR is ready
		 */
		do {
			status = inb(0x3f4);
		} while ((status & 0xc0) != 0x80);

		/*
		 * Send recalibrate command.
		 */
		outb(0x07, 0x3f5); /* RECALIBRATE */
		outb(drive, 0x3f5);

		/*
		 * Wait until command has been finished
		 */
		var_put(f_recal, var_get(f_recal) & ~(1 << 7));
		while (! (var_get(f_recal) & (1 << 7))) {
			asm volatile (
					"sti\n\t"
					"hlt\n\t"
					"cli\n\t"
				     );
		}

		/*
		 * Wait until DR is ready
		 */
		do {
			status = inb(0x3f4);
		} while ((status & 0xc0) != 0x80);

		/*
		 * Send sense interrupt status command.
		 */
		outb(0x08, 0x3f5); /* SENSE INTERRUPT STATUS */
		outb(drive, 0x3f5);

		/*
		 * Wait until DR is ready
		 */
		do {
			status = inb(0x3f4);
		} while ((status & 0xd0) != 0xd0);

		var_put(f_stat[0], inb(0x3f5));
		var_put(f_stat[1], inb(0x3f5));

		if (var_get(f_stat[1]) == 1
		 && (var_get(f_stat[0]) & 0xf0) == 0x20) {
			if (inb(0x37f) & 0x80) {
				result = 0x80;	/* drive not ready or present */
			} else {
				result = 0x06; /* change line active */
			}
		} else {
			result = 0x80; /* drive not ready or present */
		}
	} else {
		result = 0; /* disk not changed */
	}

	floppy_poweroff();

	return result;
}
#endif /* FD_CHANGELINE_SUPPORT */

#endif /* RUNTIME_RM */
/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

#include "assert.h"
#include "stdio.h"
#include "string.h"
#include "in.h"
#include "io.h"
#include "cmos.h"
#include "var.h"
#include "el_torito.h"
#include "ptrace.h"
#include "floppy.h"
#include "const.h"
#include "video.h"

const char *
get_identify_floppy(uint16_t drive)
{
	static const char * const floppy_type[] = {
		"None",
		"360K, 5.25 in.",
		"1.2M, 5.25 in.",
		"720K, 3.5 in.",
		"1.44M, 3.5 in.",
		"2.88M, 3.5 in.",
		"160K, 5.25 in.",
		"180K, 5.25 in.",
		"320K, 5.25 in.",
	};
	uint8_t drive_type;

	drive_type = cmos_get(fd_config);

	if (drive == 0) {
		drive_type >>= 4;
	} else {
		drive_type &= 0xf;
	}
	
	return floppy_type[drive_type];
}

void
floppy_init(void)
{
	uint8_t drive_type;
	uint16_t equipment;

	/* Reset floppy controller. */
	outb(0x08, 0x3f2);
	outb(0x0c, 0x3f2);

	/* Call sense interrupt status. */
	outb(0x08, 0x3f5);

	/* Get result bytes. */
	(void) inb(0x3f5);
	(void) inb(0x3f5);

	/* Call configure to enable implied seeks. */
	outb(0x13, 0x3f5); /* Configure command */
	outb(0x00, 0x3f5); /* Dummy */
	outb(0 << 7 /* ? */
	   | 1 << 6 /* Enable implied seek */
	   | 1 << 5 /* Enable FIFO */
	   | 1 << 4 /* Disable polling */
	   | 0 << 0, /* FIFO threshold = 1 */
		0x3f5);
	outb(0x00, 0x3f5); /* Precompensation beginning at track 0 */

	/* Get result bytes. */
	/* None. */

	/* int $0x1e vector is diskette_param_table. */
	put_word(0x0000, 0x1e * 4 + 0, PTR_OFF(&diskette_param_table));
	put_word(0x0000, 0x1e * 4 + 2, PTR_SEG(&diskette_param_table));

	/* adjust sys_conf settings for floppy drives */
	/* bit   0: drives are present */
	/* bit 7-6: number of drives - 1 */
	drive_type = cmos_get(fd_config);
	equipment = var_get(sys_conf) & ~0xc1;
	if ((drive_type >> 4) & 0xf) {
		equipment |= (1 << 0);
	}
	if ((drive_type >> 0) & 0xf) {
		equipment |= (1 << 6);
	}
	var_put(sys_conf, equipment);
}

#endif /* INIT_RM */
