# FIAIF is an Intelligent firewall, version: $Revision: 1.10 $
#
# description: Automates a packet filtering firewall with iptables.
#
# Script Author:	Anders Fugmann <afu at fugmann dot net>
# 
# FIAIF is an Intelligent firewall
# Copyright (C) 2002-2003 Anders Peter Fugmann
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

BEGIN {
    rules_deleted    = 0;
    FS               = "NO SPLITTING";
    rule_no          = 0;

    if ( TABLE == "" ) 
	TABLE="filter";
}

function scan_rule(string,
                   target, chain, spec_start, specification)
{
    start = match(string, "-A[ ]");
    if (start == 1) {
	string = substr(string, RLENGTH+1, length(string) - RLENGTH);
	if (match(string, "[a-zA-Z0-9_+]+") != 1) {
	    printf("Error scanning rule: '%s'\n", string);
	    return;
	}
        chain = substr(string, RSTART, RLENGTH);
	spec_start = RLENGTH+2;

        if ( match(string, "-j[ ][a-zA-Z0-9_+]+.*") == 0) {
	    printf("Rule with no target: -A %s %s\n", chain, specification);
	    return;
	}
	target = substr(string, RSTART+3, RLENGTH-4);
	specification = substr(string, spec_start, length(string) - spec_start - (RLENGTH));

	if ( specification == "" && target == "" ) {
	    printf("Rule error: %s", string);
	}
        rule_num++;
	RULES[rule_num, "CHAIN"]         = chain;
	RULES[rule_num, "SPECIFICATION"] = specification;
        RULES[rule_num, "TARGET"]        = target;
	REFERENCES[target]++;
	CHAIN_RULES[chain]++;
	#printf("Scanned rule: -t '%s' -A '%s' '%s' -j '%s'\n", TABLE, chain, specification, target);
	#printf("REFERENCES: '%s' %d\n", target, REFERENCES[target]);

    } else {
	# Store all user chains.
 	if ( match(string, ":[a-zA-Z0-9_+]+[ ]-") == 1 ) {
	    match(string,":[a-zA-Z0-9_+]+");
	    USER_CHAIN[substr(string, RSTART+1, RLENGTH-1)] = 1;
	}
    }
}

## Delete references to empty chains.
## Return number of deletions.
function delete_chain_references(chain,
				 rule, ret, rule_index)
{
    ret = 0
    # Find all rules with reference to this chain.
    for (rule = 1; rule <= rule_num; rule++)
    {
	if ( RULES[rule, "TARGET"] == chain ) {
	    rule_index = rule_index=get_rule_index(RULES[rule, "CHAIN"], rule);
	    printf("iptables -t %s -D %s %d\n", 
		   TABLE, RULES[rule, "CHAIN"],  rule_index);
	    # Decrement all counters.
	    REFERENCES[RULES[rule, "TARGET"]]--;
	    CHAIN_RULES[RULES[rule, "CHAIN"]]--;
	    delete RULES[rule, "CHAIN"];
	    delete RULES[rule, "TARGET"];
	    delete RULES[rule, "SPECIFICATION"];
	    ret++;
	}
    }
    return ret
}

function delete_chain(chain,
		      rule)
{
    printf("iptables -t %s -F %s\n", TABLE, chain);
    printf("iptables -t %s -X %s\n", TABLE, chain);
    CHAIN_RULES[chain]=0;
    USER_CHAIN[chain]=0
    # Delete all rules
    for (rule = 1; rule <= rule_num; rule++) {
	if (RULES[rule, "CHAIN"] == chain) {
	    REFERENCES[RULES[rule, "TARGET"]]--;
	    delete RULES[rule, "CHAIN"];
	    delete RULES[rule, "TARGET"];
	    delete RULES[rule, "SPECIFICATION"];
	}
    }
}

# Merge two rules.
function merge_rule ( old_rule, new_rule,
		      ret, has_interface, has_source, has_destination)
{
    # Merge two rules.

    ret = "";

    if (old_rule == "")
        ret = new_rule;
    else if (new_rule == "")
	ret = old_rule;
    else if (old_rule == new_rule)
        ret = new_rule;	
    else {
	match(old_rule,"([-][isd][ ][a-zA-Z0-9_./]+[ ]*)*");
	if ((RSTART == 1) && (RLENGTH == length(old_rule))) {
	    # the old rule has only source/interface.
	    has_interface = match(old_rule,"[-]i[ ][a-zA-Z0-9_+]+");
	    has_source = match(old_rule,"[-]s[ ][a-zA-Z0-9_./]+");
	    has_destinaion = match(old_rule,"[-]d[ ][a-zA-Z0-9_./]+");
	    
	    # Only replace if no dublets.
	    if ( (! has_interface   || ! match(new_rule,"[-]i[ ][a-zA-Z0-9_./]+")) &&
		 (! has_source      || ! match(new_rule,"[-]s[ ][a-zA-Z0-9_./]+")) &&
		 (! has_destination || ! match(new_rule,"[-]d[ ][a-zA-Z0-9_./]+")) ) {

		ret = sprintf("%s %s", old_rule, new_rule);
	    }
	    else
		printf("# Old_Rule: '%s' New_rule: '%s'\n", old_rule, new_rule);
	} 
    }
    #printf("Merge: %s + %s => %s\n", old_rule, new_rule, ret);
    return ret;
}

## Return the index of the rule in the specified chain.
## The index starts with 1.
## Returns 0 if the rule is not found.
function get_rule_index(chain, rule_id, 
			 ret)
{
    ret = 0;
    for (rule = 1; rule <= rule_num; rule++) {
	if ( RULES[rule, "CHAIN"] == chain ) {
	    ret++;
	    if (rule_id == rule)
	        break;
	}
    }
    return ret;
}

# Merge one rule from chain with all rules that references the chain.
# Retunr the number of merged rules.
function merge_chain (chain,
		      rule, specification, target, new_spec, rule_index, merged, 
		      new_target)
{
    merged = 0;    
    for (rule = 1; rule <= rule_num; rule++) {
	if ( RULES[rule, "CHAIN"] == chain ) {
	    specification = RULES[rule,"SPECIFICATION"];
	    target = RULES[rule,"TARGET"];
	    #printf("Rule: '%s' '%s' '%s'\n", chain, specification, target);
	    break;
	}
    }

    for (rule = 1; rule <= rule_num; rule++) {
	if ( RULES[rule, "TARGET"] == chain ) {
	    new_spec = merge_rule(RULES[rule, "SPECIFICATION"], specification);

	    if ( new_spec != "" ) {
		rule_index=get_rule_index(RULES[rule, "CHAIN"], rule);

		# Delete both rules if it was a return statement.
		if ( target == "RETURN" ) {
		    printf("iptables -t %s -D %s %d \n", 
			   TABLE, RULES[rule, "CHAIN"], rule_index);
		    # Update structures on the deleted chain.
		    REFERENCES[RULES[rule, "TARGET"]]--;
		    CHAIN_RULES[RULES[rule, "CHAIN"]]--;
		    delete RULES[rule, "CHAIN"];
		    delete RULES[rule, "TARGET"];
		    delete RULES[rule, "SPECIFICATION"];
		} else {
		    if (target == "")
			printf("# Target zero\n");

		    printf("iptables -t %s -R %s %d %s -j %s\n", 
			   TABLE, RULES[rule, "CHAIN"], rule_index, new_spec, target);
		    # Update structures.
		    REFERENCES[RULES[rule, "TARGET"]]--;
		    RULES[rule, "TARGET"] = target;
		    REFERENCES[target]++;
		    RULES[rule, "SPECIFICATION"] = new_spec;

		}
		merged++;
	    }
	}
    }
    return merged;
}

# If a catch all rule exists in a chain, then delete all following rules.
function truncate_chain(chain,
			catch_all, ret, rule, rule_index)
{
    ret = 0;
    catch_all = 0;
    for ( rule = 1; rule <= rule_num; rule++ ) {
	if ( RULES[rule, "CHAIN"] == chain ) {
	    if ( catch_all == 0 ) {
                # See if the rule is a catch_all rule.
		if ( RULES[rule, "SPECIFICATION"] == "" && 
		     ( RULES[rule, "TARGET"] == "DROP" || RULES[rule, "TARGET"] == "ACCEPT" ||
		       RULES[rule, "TARGET"] == "RETURN" || RULES[rule, "TARGET"] == "REJECT" ) ) {
		    catch_all = 1;
		    if (RULES[rule, "TARGET"] == "RETURN") {
			# Do not leave empry returns.
			rule_index = get_rule_index(chain, rule);
			printf("iptables -t %s -D %s %d\n", TABLE, chain, rule_index);
			REFERENCES[RULES[rule, "TARGET"]]--;
			CHAIN_RULES[RULES[rule, "CHAIN"]]--;
			delete RULES[rule, "CHAIN"];
			delete RULES[rule, "TARGET"];
			delete RULES[rule, "SPECIFICATION"];   		
			ret++;
		    }
		}
	    } else {
		# Delete all following rules.
		rule_index = get_rule_index(chain, rule);
		printf("iptables -t %s -D %s %d\n", TABLE, chain, rule_index);
		
		REFERENCES[RULES[rule, "TARGET"]]--;
		CHAIN_RULES[RULES[rule, "CHAIN"]]--;
		delete RULES[rule, "CHAIN"];
		delete RULES[rule, "TARGET"];
		delete RULES[rule, "SPECIFICATION"];   		
		ret++;
	    }
	}
    }
    return ret;
}


## Main function 
{
    scan_rule($1);
}


END {   
    iterations = 0;
    has_work = 1;
    while(has_work > 0) {
	iterations++;
	has_work = 0;
	for (chain in USER_CHAIN) {
	    #printf("Testing chain: '%s'. #rules %d, #references %d\n", chain, CHAIN_RULES[chain], REFERENCES[chain]);
	}

	for (chain in USER_CHAIN) {
	    if (USER_CHAIN[chain] != 1) 
		continue;

	    if (REFERENCES[chain] == 0) {
		delete_chain(chain);
		has_work++;
	    }
	    if (CHAIN_RULES[chain] > 0) {
		has_work += truncate_chain(chain);
	    }

	    if (CHAIN_RULES[chain] == 1) {
		has_work += merge_chain(chain);
	    }

	    if (CHAIN_RULES[chain] == 0) {
	        has_work += delete_chain_references(chain);
	    }	    
	}
    }
    if (VERBOSE == 1) printf("# Iterations: %d\n", iterations);
}
