/* arcellipse.c  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* contains functions assoc. with arc sections of ellipse and circle composition */
/*
 * $Log: arcellipse.c,v $
 * Revision 1.2  2000/12/06 20:56:01  moz
 * GPL stuff.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:30  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:29  moz
 * CVS Import
 *
 * Revision 1.10  2000/02/18 16:59:35  moz
 * Compile fixes.
 *
 * Revision 1.9  1999/11/15 02:02:50  moz
 * Use rounded trig.
 *
 * Revision 1.8  1999/05/22 15:30:58  moz
 * Draw arrows last to fill white arrows.
 *
 * Revision 1.7  1999/05/19 17:06:04  moz
 * 1.0 Checkin.
 *
 * Revision 1.6  1999/05/04 14:40:50  moz
 * Arrows for arcellipse.
 *
 * Revision 1.5  1999/04/22 23:27:14  moz
 * Draw lines when moving object.
 *
 * Revision 1.4  1999/04/22 16:17:03  moz
 * Fix for arc circles, not ellipses.
 *
 * Revision 1.3  1999/04/20 03:14:13  moz
 * Unwanted debugging info removed.
 *
 * Revision 1.2  1999/04/20 03:07:44  moz
 * Correct angle before drawing arcellipse.
 *
 * Revision 1.1  1999/03/30 00:04:17  moz
 * Initial revision
 *
 */    

/* see mouse_button.c for general behaviour */  
 
#include <math.h>  
#include "include/figurine.h"  
#include "include/extern.h"
 
/* NOTE: everywhere you see arcellipse, think
	arccircle. xfig can't handle arcellipses ! */   
	 
static long cx=0;
static long cy=0;
static long xr=0;
static long yr=0;
static double sa=0; /* in DEGREES * 64 */ 
static double ea=0;
static double *ca=NULL; /* this points to sa, ea, or NULL - NULL means sizing  */  
static Boolean enda; /* TRUE when editing enda, FALSE otherwise  */  
static Boolean just_started=FALSE; 

static Boolean held=FALSE;

void 
commit_arcellipse(View *view, long x)
{
	Object *ob; 
	 
	/* calculate end angle finally  */   
	ea = min(P2D(100,view),max(P2D(-100,view),XP2D(x, view) - cx));   
	ea = ea/P2D(100,view);
	ea *= 360*64;
	 
	/* constrain */
 	if (state.control_down) 
 		constrain_ellipse(&xr,&yr);

	ob = (Object *)malloc(sizeof(Object));
	if (ob==NULL)
		return;
	
	ob->type = ARCELLIPSE;
	ob->ticket = ob_ticket++;
	ob->derries = NULL; 
	ob->depth = view->doc->ob_depth--;  
	ob->ls = view->linestyle;
	ob->lw = view->linewidth; 
	ob->es = view->endstyle; 
	ob->colour = view->colour; 
	ob->fillcolour = view->fillcolour; 
	if (view->fillon)
		ob->fs = view->fillstyle;
	else
		ob->fs = NONE;

	ob->dash = 5;
	ob->gap = 4;
	ob->bbox.x1 = cx - xr;
	ob->bbox.y1 = cy - yr;
	ob->bbox.x2 = cx + xr;
	ob->bbox.y2 = cy + yr;
	ob->ob.arcellipse.open = view->openarc; 
	ob->ob.arcellipse.centre.x = cx - ob->bbox.x1;
	ob->ob.arcellipse.centre.y = cy - ob->bbox.y1;
	ob->ob.arcellipse.xradius = xr;
	ob->ob.arcellipse.yradius = yr;
	ob->ob.arcellipse.start = sa;
	ob->ob.arcellipse.end = ea;
	if (view->barrow_on && view->openarc)
		ob->barrow = make_arrow(view->barrow);
	else
		ob->barrow = NULL;
	if (view->farrow_on && view->openarc)
		ob->farrow = make_arrow(view->farrow);
	else
		ob->farrow = NULL;

	
 	normalise_rectangle(&ob->bbox.x1, &ob->bbox.y1, &ob->bbox.x2, &ob->bbox.y2); 
	ob->bbox.x2 = cx + xr +1;
	ob->bbox.y2 = cy + yr +1;
	 
	view->doc->o = add_object(view->doc->o, &view->doc->lo, ob);
	 
	register_undo(UNDO_PASTE,ob,view->doc); 
	send_redraw_object(view,ob); 
}

void 
arcellipse_button(BEvent *bev, View *view)
{
	 
	if (((held && bev->button==Button2) || bev->button==Button3) && state.busy_drawing )
		{
		/* cancel the current ellipse */
		XUngrabPointer(display,CurrentTime);
		
		toggle_arcellipse(view, bev->x, bev->y);
			 
		state.busy_drawing = FALSE;
		held = FALSE;
		sa = 0;
		ea = 0;
		ca = NULL;
		enda = FALSE; 
		}
	else if (bev->button==Button1)
		{
		if (!P_IN_DOC(bev->x, bev->y, view) && state.busy_drawing)
			{
			/* ok, cancel the ellipse  we're drawing  */
			state.busy_drawing = FALSE;
			 
			if (!held)
				XUngrabPointer(display,CurrentTime);
			else 
				held = FALSE;
			
			toggle_arcellipse(view, bev->x, bev->y);
			sa = 0;
			ea = 0;
			ca = NULL;
			enda = FALSE; 
			}
		else
			{
			switch (bev->type)
				{
				case BUTTON_HELD:
					
					if (!state.busy_drawing)
						{
						if (ca==NULL) /* ie sizing  */  
							{ 
							/* start drawing */
							/* move into the circle sizing mode */  
							state.busy_drawing = TRUE;
							held = TRUE;
							if (view->gridon)
								{
								cx = GXP2D(bev->x,view);
								cy = GYP2D(bev->y,view);
								}
							else
								{
								cx = XP2D(bev->x,view);
								cy = YP2D(bev->y,view);
								};	
							
							xr = 1;
							yr = 1; 
							just_started = TRUE; 
								
							toggle_arcellipse(view, bev->x, bev->y);  
							}; 
						};

					break; /* BUTTON_HELD  */ 

				case BUTTON_CLICKED:
					if (held)
					   {
						XGrabPointer(display, view->draw_window->win, False, ButtonPressMask | ButtonReleaseMask
										 | PointerMotionMask | ButtonMotionMask | Button1MotionMask, 
										 GrabModeAsync, GrabModeAsync, None, None, CurrentTime);
				      } 
					else
						{ 
 						toggle_arcellipse(view, bev->x, bev->y);
						XUngrabPointer(display,CurrentTime);
						};

						 
					if (ca==NULL && !just_started)
						{
						/* move into angling */
						if (held) 
							toggle_arcellipse(view,bev->x,bev->y); 
						ca = &sa; 
						enda = FALSE; 
						if (held) 
							toggle_arcellipse(view,bev->x,bev->y); 
						}
					else if (!enda && !just_started)
						{
						/* angle start of arc  */ 

						toggle_arcellipse(view,bev->x,bev->y); 
						sa = min(P2D(100,view),max(P2D(-100,view),XP2D(bev->x, view) - cx));   
						sa = sa/P2D(100,view);
						sa *= 360*64;
						ca = &ea; 
						enda = TRUE; 
						toggle_arcellipse(view,bev->x,bev->y); 
						}
					else if (!just_started)
						{
						XUngrabPointer(display,CurrentTime);
						/* angle end of arc  */ 
						commit_arcellipse(view, bev->x);
						state.busy_drawing=FALSE;
						ca = NULL;
						sa = 0;
						ea = 0;
						enda = FALSE;
						just_started = FALSE;
						};
					
					if (just_started)
						just_started = FALSE;
						
					break;
				
				case BUTTON_RELEASED:
				 	toggle_arcellipse(view, bev->x, bev->y);
						 
					if (ca==NULL)
						{
						/* move into angling */ 
						ca = &sa; 
						toggle_arcellipse(view,bev->x,bev->y); 
						enda = FALSE; 
						just_started = FALSE;
						}
					else if (!enda)
						{
						/* angle start of arc  */ 
						sa = min(P2D(100,view),max(P2D(-100,view),XP2D(bev->x, view) - cx));   
						sa = sa/P2D(100,view);
						sa *= 360*64;
						ca = &ea; 
						enda = TRUE; 
						}
					else
						{
						if (!held)
							XUngrabPointer(display,CurrentTime);
						/* angle end of arc  */ 
						commit_arcellipse(view, bev->x);
						state.busy_drawing=FALSE;
						ca = NULL;
						sa = 0;
						ea = 0;
						enda = FALSE;
						just_started = FALSE;
						};
					break;
				};
			}; 
		}; 
}

void 
toggle_arcellipse(View *view, int x, int y)
{
	if (ca==NULL)
		{
		if (view->gridon)
			{
			xr = abs(GXP2D(x, view) - cx);
			yr = abs(GYP2D(y, view) - cy);
			}
		else
			{
			xr = abs(XP2D(x, view) - cx);
			yr = abs(YP2D(y, view) - cy);
			};
			 
		/* always constrain, fig2dev can't handle arcellipses, only circles ! */
		constrain_ellipse(&xr,&yr); 

		XDrawArc(display, view->draw_window->win, blackxorgc, XD2P(cx-xr,view), YD2P(cy-yr,view),
					(uint)D2P(2*xr,view), (uint)D2P(2*yr,view), 0, 360*64);
		}
	else if (ca==&sa)
		{
		double f;

		/* calculate the angle based on mouse position  */ 
		f = min(P2D(100,view),max(P2D(-100,view),XP2D(x, view) - cx));   
		f = f/P2D(100,view);
		f *= 360*64;
		 
		XDrawArc(display,view->draw_window->win, blackxorgc, XD2P(cx-xr,view), YD2P(cy-yr,view),
					(uint)D2P(2*xr,view), (uint)D2P(2*yr,view), (int)f, 128);
		}
	else
		{
		double f;

		/* calculate the angle based on mouse position  */ 
		f = min(P2D(100,view),max(P2D(-100,view),XP2D(x, view) - cx));   
		f = f/P2D(100,view);
		f *= 360*64;
		 
		XDrawArc(display,view->draw_window->win, blackxorgc, XD2P(cx-xr,view), YD2P(cy-yr,view),
					(uint)D2P(2*xr,view), (uint)D2P(2*yr,view), (int)sa, (int)f-(int)sa);
		};
}

void 
draw_arcellipse(Object *ob, View *view, GC gc, long x,long y, double rx, double ry)
{
	GC tgc;
	int r; 
	double theta; 
	double theta2;
	double a,b; 
	Boolean clipped = FALSE; 
	VLine line; VLine *pl; 
			 

	if (gc==NULL)
		tgc = ugc;
	else 
		tgc = gc;
	
	if (state.tied_corner != NOTSCALING)
		{
		/* we are scaling, move object correctly */  
		corner_magic(ob, &x, &y, rx, ry);
		r = ellipse_box(XD2P(x+ob->ob.arcellipse.centre.x,view),YD2P(y+ob->ob.arcellipse.centre.y,view),
							 D2P(ob->ob.arcellipse.xradius,view), D2P(ob->ob.arcellipse.yradius,view),
							 0, 0, (signed long)view->window.w, (signed long)view->window.h);

		if (r!=NO_INTERSECT && r!=CONTAINS)
			{
			double diff = ob->ob.arcellipse.end - ob->ob.arcellipse.start;
			if (diff>360.0*64.0)
			   diff -= 360.0*64.0;
			theta = ob->ob.arcellipse.start; 
			if (theta>(180.0*64.0))
				theta = theta - (360.0*64.0);
			XDrawArc(display, view->draw_window->win, tgc, 
						XD2P(x,view),
						YD2P(y,view),
						(uint)D2P(abs(R(2*ob->ob.arcellipse.xradius,rx)),view), 
						(uint)D2P(abs(R(2*ob->ob.arcellipse.yradius,ry)),view),
						(int)theta, (int)diff);
			};
		}
	else
		{
		/* not scaling */  
		r = ellipse_box(XD2P(x+ob->ob.arcellipse.centre.x,view),YD2P(y+ob->ob.arcellipse.centre.y,view),
							 D2P(ob->ob.arcellipse.xradius,view), D2P(ob->ob.arcellipse.yradius,view),
							 0, 0, (signed long)view->draw_window->w, (signed long)view->draw_window->h);
	
		if (!ob->ob.arcellipse.open)
			{
			if (r==CONTAINED && ob->fs!=NONE && gc!=blackxorgc)
				{
				/* closed, fill the arc if necessary  */  
			   double diff = ob->ob.arcellipse.end - ob->ob.arcellipse.start;
			   if (diff>360.0*64.0)
			      diff -= 360.0*64.0;
			   theta = ob->ob.arcellipse.start; 
			   if (theta>(180.0*64.0))
				   theta = theta - (360.0*64.0);
				XFillArc(display, view->draw_window->win, fillgc,
							XD2P(x,view),
							YD2P(y,view),
							(uint)D2P(2*ob->ob.arcellipse.xradius,view), 
							(uint)D2P(2*ob->ob.arcellipse.yradius,view),
							(int)theta, (int)diff);
				};
			 
			if (r==CONTAINS && ob->fs!=NONE && gc!=blackxorgc)
				XFillRectangle(display,view->draw_window->win,fillgc,0,0,view->draw_window->w, view->draw_window->h);
			};

		if (r!=NO_INTERSECT && r!=CONTAINS && (gc==blackxorgc || ob->lw!=0))
			{
		   double diff = ob->ob.arcellipse.end - ob->ob.arcellipse.start;
		   if (diff>360.0*64.0)
		      diff -= 360.0*64.0;
			theta = ob->ob.arcellipse.start; 
			if (theta>(180.0*64.0))
			   theta = theta - (360.0*64.0);
			XDrawArc(display, view->draw_window->win, tgc, 
						XD2P(x,view),
						YD2P(y,view),
						(uint)D2P(2*ob->ob.arcellipse.xradius,view), 
						(uint)D2P(2*ob->ob.arcellipse.yradius,view), 
						(int)theta, (int)diff);
			};
			 
		if (!ob->ob.arcellipse.open || ob->farrow!=NULL || ob->barrow!=NULL)
			{
			/* draw the two lines of a closed arcellipse and/or the two arrows  */  
			a = ob->ob.arcellipse.xradius; 
			b = ob->ob.arcellipse.yradius;
			theta = (((double)(ob->ob.arcellipse.start))/64.0)*(PI/180.0); 
			if (theta<(-1.0*PI))
				theta = 2.0*PI + theta;
			if (theta>(1.0*PI))
				theta = theta - 2.0*PI;
			theta2 = theta+0.1; 
			line.x1 = XD2P(x + ob->ob.arcellipse.centre.x,view);
			line.y1 = YD2P(y + ob->ob.arcellipse.centre.y,view);
 			line.x2 = a * cosround(theta);
			line.y2 = - b * sinround(theta);
			line.x2 += x + ob->ob.arcellipse.centre.x;
			line.y2 += y + ob->ob.arcellipse.centre.y;
			line.x2 = XD2P(line.x2,view);
			line.y2 = YD2P(line.y2,view);
			 
			pl = clip_line(0,0,(signed long)view->draw_window->w,(signed long)view->draw_window->h, &line, &clipped, 2); 
			if (!ob->ob.arcellipse.open && pl!=NULL) 
				XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
			
			if (ob->barrow!=NULL && pl!=NULL)
				{
				/* calculate a point a little less, then pass to draw_arrow  */  
				if (ob->ob.arcellipse.start>ob->ob.arcellipse.end)
					theta2 = theta-0.1; 
				else 
					theta2 = theta+0.1; 
				line.x1 = a * cosround(theta2);
				line.y1 = -b * sinround(theta2);
				line.x1 += x + ob->ob.arcellipse.centre.x;
				line.y1 += y + ob->ob.arcellipse.centre.y;
				line.x1 = XD2P(line.x1,view);
				line.y1 = YD2P(line.y1,view);
				draw_arrow(view,gc,ob->barrow, line.x1, line.y1, pl->x2, pl->y2);
				gc_lw(view,gc,ob); 
				};

			theta = (((double)(ob->ob.arcellipse.end))/64.0)*(PI/180.0); 
			if (theta<(-1.0*PI))
				theta = 2.0*PI + theta;
			if (theta>(1.0*PI))
				theta = theta - 2.0*PI;
			line.x1 = XD2P(x + ob->ob.arcellipse.centre.x,view);
			line.y1 = YD2P(y + ob->ob.arcellipse.centre.y,view);
 			line.x2 = a * cosround(theta);
			line.y2 = - b * sinround(theta);
			line.x2 += x + ob->ob.arcellipse.centre.x;
			line.y2 += y + ob->ob.arcellipse.centre.y;
			line.x2 = XD2P(line.x2,view);
			line.y2 = YD2P(line.y2,view);
			 
			pl = clip_line(0,0,(signed long)view->draw_window->w,(signed long)view->draw_window->h, &line, &clipped, 2); 
			if (!ob->ob.arcellipse.open && pl!=NULL) 
				XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
			
			if (ob->farrow!=NULL && pl!=NULL)
				{
				/* calculate a point a little less, then pass to draw_arrow  */  
				if (ob->ob.arcellipse.start<ob->ob.arcellipse.end)
					theta2 = theta-0.1; 
				else 
					theta2 = theta+0.1; 
				line.x1 = a * cosround(theta2);
				line.y1 = -b * sinround(theta2);
				line.x1 += x + ob->ob.arcellipse.centre.x;
				line.y1 += y + ob->ob.arcellipse.centre.y;
				line.x1 = XD2P(line.x1,view);
				line.y1 = YD2P(line.y1,view);
				draw_arrow(view,gc,ob->farrow, line.x1, line.y1, pl->x2, pl->y2);
				gc_lw(view,gc,ob); 
				};
			};

		};
}
