/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  File-Roller
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <fnmatch.h>
#include <gnome.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-directory-filter.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "bookmarks.h"
#include "dlg-add.h"
#include "dlg-batch-add.h"
#include "dlg-extract.h"
#include "dlg-viewer-or-app.h"
#include "fr-archive.h"
#include "file-utils.h"
#include "window.h"
#include "main.h"
#include "menu-callbacks.h"
#include "menu.h"
#include "toolbar.h"
#include "typedefs.h"

#include "pixmaps/up.xpm"
#include "pixmaps/down.xpm"
#include "pixmaps/file.xpm"
#include "pixmaps/folder.xpm"

#define CLIST_ROW_PAD 5
#define ACTIVITY_DELAY 100
#define DEFAULT_COLUMN_WIDTH 100


enum {
	TARGET_STRING,
	TARGET_URL
};

static GtkTargetEntry target_table[] = {
	{ "STRING",        0, TARGET_STRING },
	{ "text/plain",    0, TARGET_STRING },
	{ "text/uri-list", 0, TARGET_URL }
};

static guint n_targets = sizeof (target_table) / sizeof (target_table[0]);


/* -- window_update_file_list -- */


static void _window_update_recent_list (FRWindow *window);


static GList *
_window_get_current_dir (FRWindow *window)
{
	GList *scan;
	GList *dir_list = NULL;

	scan = window->archive->command->file_list;
	for (; scan; scan = scan->next) {
		FileData *fdata = scan->data;

		if (strncmp (fdata->full_path, 
			     window->current_dir, 
			     strlen (window->current_dir)) == 0) 
			dir_list = g_list_prepend (dir_list, fdata);
	}

	return g_list_reverse (dir_list);
}


static gint
sort_by_name (gconstpointer  ptr1,
              gconstpointer  ptr2)
{
	const FileData *fdata1 = ptr1, *fdata2 = ptr2;
	return strcmp (fdata1->list_name, fdata2->list_name);
}


static gint
sort_by_size (gconstpointer  ptr1,
              gconstpointer  ptr2)
{
	const FileData *fdata1 = ptr1, *fdata2 = ptr2;

	if (fdata1->is_dir != fdata2->is_dir) {
		if (fdata1->is_dir)
			return -1;
		else
			return 1;
	} else if (fdata1->is_dir && fdata2->is_dir) 
		return sort_by_name (ptr1, ptr2);

	if (fdata1->size == fdata2->size)
		return sort_by_name (ptr1, ptr2);
	else if (fdata1->size > fdata2->size)
		return 1;
	else
		return -1;
}


static gint
sort_by_type (gconstpointer  ptr1,
              gconstpointer  ptr2)
{
	gint result;
	const FileData *fdata1 = ptr1, *fdata2 = ptr2;

	if (fdata1->is_dir != fdata2->is_dir) {
		if (fdata1->is_dir)
			return -1;
		else
			return 1;
	} else if (fdata1->is_dir && fdata2->is_dir) 
		return sort_by_name (ptr1, ptr2);

	result = strcasecmp (fdata1->type, fdata2->type);
	if (result == 0)
		return sort_by_name (ptr1, ptr2);
	else
		return result;
}


static gint
sort_by_time (gconstpointer  ptr1,
              gconstpointer  ptr2)
{
	const FileData *fdata1 = ptr1, *fdata2 = ptr2;

	if (fdata1->is_dir != fdata2->is_dir) {
		if (fdata1->is_dir)
			return -1;
		else
			return 1;
	} else if (fdata1->is_dir && fdata2->is_dir) 
		return sort_by_name (ptr1, ptr2);


	if (fdata1->modified == fdata2->modified)
		return sort_by_name (ptr1, ptr2);
	else if (fdata1->modified > fdata2->modified)
		return 1;
	else
		return -1;
}


static gint
sort_by_path (gconstpointer  ptr1,
              gconstpointer  ptr2)
{
	gint result;
	const FileData *fdata1 = ptr1, *fdata2 = ptr2;

	if (fdata1->is_dir != fdata2->is_dir) {
		if (fdata1->is_dir)
			return -1;
		else
			return 1;
	} else if (fdata1->is_dir && fdata2->is_dir) 
		return sort_by_name (ptr1, ptr2);

	result = strcasecmp (fdata1->path, fdata2->path);
	if (result == 0)
		return sort_by_name (ptr1, ptr2);
	else
		return result;
}


static GCompareFunc
get_compare_func_from_idx (int column_index)
{
	static GCompareFunc compare_funcs[5] = {
		sort_by_name,
		sort_by_type,
		sort_by_size,
		sort_by_time,
		sort_by_path
	};

	return compare_funcs [column_index];
}


static void
compute_file_list_name (FRWindow *window, 
			FileData *fdata)
{
	gchar *scan, *end;

	fdata->is_dir = FALSE;
	if (fdata->list_name != NULL)
		g_free (fdata->list_name);

	if (window->list_mode == WINDOW_LIST_MODE_FLAT) {
		fdata->list_name = g_strdup (fdata->name);
		return;
	}

	scan = fdata->full_path + strlen (window->current_dir);
	end = strchr (scan, '/');
	if (end == NULL)
		fdata->list_name = g_strdup (scan);
	else {
		fdata->is_dir = TRUE;
		fdata->list_name = g_strndup (scan, end - scan);
	}
}


static void
set_check_menu_item_state (FRWindow *window,
                           GtkWidget *mitem, 
                           gboolean active)
{
        gtk_signal_handler_block_by_data (GTK_OBJECT (mitem), window);
        gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), active);
        gtk_signal_handler_unblock_by_data (GTK_OBJECT (mitem), window);
}


static void
_window_sort_file_list (FRWindow *window)
{
	GList *scan;

	scan = window->archive->command->file_list;
	for (; scan; scan = scan->next) {
		FileData *fdata = scan->data;
		compute_file_list_name (window, fdata);
	}

	window->archive->command->file_list = g_list_sort (window->archive->command->file_list, get_compare_func_from_idx (window->sort_method));

	if (window->sort_type == GTK_SORT_DESCENDING)
		window->archive->command->file_list = g_list_reverse (window->archive->command->file_list);
}


static void
_go_up_one_level (FRWindow *window)
{
	char *new_dir;

	g_return_if_fail (window != NULL);

	if (window->list_mode != WINDOW_LIST_MODE_AS_DIR)
		return;
	if (window->current_dir == NULL)
		return;
	if (strcmp (window->current_dir, "/") == 0)
		return;

	window->current_dir[strlen (window->current_dir) - 1] = 0;
	new_dir = remove_level_from_path (window->current_dir);
	g_free (window->current_dir);
	if (new_dir[strlen (new_dir) - 1] == '/')
		window->current_dir = new_dir;
	else {
		window->current_dir = g_strconcat (new_dir, "/", NULL);
		g_free (new_dir);
	}	
}


static void _window_update_statusbar_list_info (FRWindow *window);


void
window_update_file_list (FRWindow *window)
{
	GList *scan;
	GList *dir_list = NULL;
	GHashTable *names_hash = NULL;
	gint i;
	GtkWidget *folder_pixmap;
	GtkWidget *file_pixmap;
	gboolean list_empty;

	gtk_clist_freeze (GTK_CLIST (window->clist));
	gtk_clist_clear (GTK_CLIST (window->clist));

	if (! window->archive_present || window->archive_new) {
		/* Resize columns. */

		for (i = 0; i < 5; i++) 
			gtk_clist_set_column_width (GTK_CLIST (window->clist),
						    i, 
						    DEFAULT_COLUMN_WIDTH);

		gtk_clist_thaw (GTK_CLIST (window->clist));
		return;
	}

	file_pixmap = gnome_pixmap_new_from_xpm_d (file_xpm);	
	folder_pixmap = gnome_pixmap_new_from_xpm_d (folder_xpm);

	if (window->list_mode == WINDOW_LIST_MODE_AS_DIR) 
		names_hash = g_hash_table_new (g_str_hash, g_str_equal);
	
	_window_sort_file_list (window);
	if (window->list_mode == WINDOW_LIST_MODE_FLAT)
		scan = window->archive->command->file_list;
	else {
		dir_list = _window_get_current_dir (window);

		while ((dir_list == NULL) 
		       && (strcmp (window->current_dir, "/") != 0)) {
			_go_up_one_level (window);
			dir_list = _window_get_current_dir (window);
		}

		scan = dir_list;
	}
	
	list_empty = scan == NULL;
	
	for (; scan; scan = scan->next) {
		FileData *   fdata = scan->data;
		char *       buf[6];
		char *       s_size;
		char         s_time[32];
		struct tm *  tm;
		int          row;
		GdkPixmap *  pixmap;
		GdkBitmap *  mask;
		
		s_size = gnome_vfs_format_file_size_for_display (fdata->size);
		tm = localtime (& fdata->modified);
		strftime (s_time, 31, "%d %b %Y, %H:%M", tm);

		if ((window->list_mode == WINDOW_LIST_MODE_AS_DIR)
		    && fdata->is_dir) {
			if (g_hash_table_lookup (names_hash, 
						 fdata->list_name) != NULL) {
				g_free (s_size);
				continue;
			}
			
			g_hash_table_insert (names_hash, 
					     fdata->list_name, 
					     GINT_TO_POINTER (1));
		}

		buf[0] = fdata->list_name;
		if (fdata->is_dir) {
			buf[1] = "";
			buf[2] = "";
			buf[3] = "";
			buf[4] = "";
		} else {
			buf[1] = fdata->type;
			buf[2] = s_size;
			buf[3] = s_time;
			buf[4] = fdata->path;
		}
		buf[5] = NULL;

		row = gtk_clist_append (GTK_CLIST (window->clist), buf);
		gtk_clist_set_row_data (GTK_CLIST (window->clist), row, fdata);
			
		g_free (s_size);
		
		/* Set the icon. */
		
		if (fdata->is_dir) {
			pixmap = GNOME_PIXMAP (folder_pixmap)->pixmap;
			mask = GNOME_PIXMAP (folder_pixmap)->mask;
		} else {
			pixmap = GNOME_PIXMAP (file_pixmap)->pixmap;
			mask = GNOME_PIXMAP (file_pixmap)->mask;
		}
			
		gtk_clist_set_pixtext (GTK_CLIST (window->clist),
				       row,
				       0,
				       fdata->list_name,
				       5,
				       pixmap,
				       mask);
	}		

	if (dir_list != NULL)
		g_list_free (dir_list);
	if (window->list_mode == WINDOW_LIST_MODE_AS_DIR) 
		g_hash_table_destroy (names_hash);

	gtk_widget_destroy (file_pixmap);
	gtk_widget_destroy (folder_pixmap);

	/**/

	for (i = 0; i < 5; i++) {
		gtk_widget_hide (window->up_arrows[i]);
		gtk_widget_hide (window->down_arrows[i]);
	}
	if (window->sort_type == GTK_SORT_ASCENDING)
		gtk_widget_show (window->down_arrows[window->sort_method]);
	else
		gtk_widget_show (window->up_arrows[window->sort_method]);

	/**/

	set_check_menu_item_state (window, 
				   window->mitem_sort[window->sort_method], 
				   TRUE);
	set_check_menu_item_state (window, 
				   window->mitem_sort_reversed,
				   window->sort_type != GTK_SORT_ASCENDING);

	/* Resize columns. */

	for (i = 0; i < 5; i++) {
		gint w;
		w = list_empty ? DEFAULT_COLUMN_WIDTH : gtk_clist_optimal_column_width (GTK_CLIST (window->clist), i);
		gtk_clist_set_column_width (GTK_CLIST (window->clist), i, w + 5);
	}

	gtk_clist_thaw (GTK_CLIST (window->clist));

	_window_update_statusbar_list_info (window);
}


static void
_window_update_title (FRWindow *window)
{
	if (! window->archive_present)
		gtk_window_set_title (GTK_WINDOW (window->app), 
				      _("File Roller"));
	else {
		gchar *title;
		title = g_strdup_printf ("%s - %s %s",
					 _("File Roller"),
					 file_name_from_path (window->archive_filename),
					 (window->archive->read_only) ? _("[read only]") : "");

		gtk_window_set_title (GTK_WINDOW (window->app), title);
		g_free (title);
	}
}


static GList *
_get_selection_as_fd (FRWindow *window)
{
	GList *list, *scan;

	g_return_val_if_fail (window != NULL, NULL);

	list = NULL;
        scan = GTK_CLIST (window->clist)->selection;
        for (; scan; scan = scan->next) {
                FileData *fd = gtk_clist_get_row_data (GTK_CLIST (window->clist), GPOINTER_TO_INT (scan->data));

		if (!fd)
			continue;

		if (fd->is_dir)
			continue;

		list = g_list_prepend (list, fd);
        }

        return g_list_reverse (list);
}


static void
_window_update_statusbar_list_info (FRWindow *window)
{
	gchar *info, *archive_info, *selected_info, *size_txt, *sel_size_txt;
	gint tot_n, sel_n;
	GnomeVFSFileSize tot_size, sel_size;
	GList *scan;
	GList *selection;

	if (window == NULL)
		return;

	if ((window->archive == NULL)
	    || (window->archive->command == NULL)) {
		gnome_appbar_set_default (GNOME_APPBAR (window->statusbar),
					  "");
		return;
	}

	tot_n = 0;
	tot_size = 0;

	if (window->archive_present) {
		scan = window->archive->command->file_list;
		for (; scan; scan = scan->next) {
			FileData *fd = scan->data;
			tot_n++;
			tot_size += fd->size;
		}
	}

	sel_n = 0;
	sel_size = 0;

	if (window->archive_present) {
		selection = _get_selection_as_fd (window);
		for (scan = selection; scan; scan = scan->next) {
			FileData *fd = scan->data;
			sel_n++;
			sel_size += fd->size;
		}
		g_list_free (selection);
	}

	size_txt = gnome_vfs_format_file_size_for_display (tot_size);
	sel_size_txt = gnome_vfs_format_file_size_for_display (sel_size);

	if (tot_n == 0)
		archive_info = g_strdup ("");
	else if (tot_n == 1)
		archive_info = g_strdup_printf (_("1 file (%s)"),
						size_txt);
	else
		archive_info = g_strdup_printf (_("%d files (%s)"),
						tot_n,
						size_txt);


	if (sel_n == 0)
		selected_info = g_strdup ("");
	else if (sel_n == 1)
		selected_info = g_strdup_printf (_("1 file selected (%s)"), 
						 sel_size_txt);
	else
		selected_info = g_strdup_printf (_("%d files selected (%s)"), 
						 sel_n, 
						 sel_size_txt);

	info = g_strconcat (archive_info,
			    ((sel_n == 0) ? NULL : ", "), 
			    selected_info, 
			    NULL);

	gnome_appbar_set_default (GNOME_APPBAR (window->statusbar), info);

	g_free (size_txt);
	g_free (sel_size_txt);
	g_free (archive_info);
	g_free (selected_info);
	g_free (info);
}


static gboolean
one_file_is_selected (GtkWidget *clist)
{
	FileData *fdata;
	gint row;

        if (GTK_CLIST (clist)->selection == NULL)
                return FALSE;

	if (GTK_CLIST (clist)->selection->next != NULL)
		return FALSE;

	row = GPOINTER_TO_INT (GTK_CLIST (clist)->selection->data);
	fdata = gtk_clist_get_row_data (GTK_CLIST (clist), row);

        return ! fdata->is_dir;
}


static void
_window_update_sensitivity (FRWindow *window)
{
	gboolean no_archive;
	gboolean ro;
	gboolean file_op;
	gboolean running;
	gboolean compr_file;
	gboolean one_file_selected;

	running      = window->activity_ref > 0;
	no_archive   = ! window->archive_present;
	ro           = window->archive->read_only;
	file_op      = (window->archive_present 
			&& ! window->archive_new 
			&& ! running);
	compr_file   = window->archive->is_compressed_file;
	one_file_selected = one_file_is_selected (window->clist);

	gtk_widget_set_sensitive (window->mitem_new_archive, ! running);
	gtk_widget_set_sensitive (window->mitem_open_archive, ! running);
	gtk_widget_set_sensitive (window->mitem_close, ! no_archive);

	gtk_widget_set_sensitive (window->mitem_archive_prop, file_op);

	gtk_widget_set_sensitive (window->mitem_move_archive, file_op && ! ro);
	gtk_widget_set_sensitive (window->mitem_copy_archive, file_op);
	gtk_widget_set_sensitive (window->mitem_rename_archive, file_op && ! ro && ! compr_file);
	gtk_widget_set_sensitive (window->mitem_delete_archive, file_op && ! ro);

	gtk_widget_set_sensitive (window->mitem_add, ! no_archive && ! ro && ! running && ! compr_file);
	gtk_widget_set_sensitive (window->mitem_delete, ! no_archive && ! ro && ! window->archive_new && ! running && ! compr_file);
	gtk_widget_set_sensitive (window->mitem_extract, file_op);
	gtk_widget_set_sensitive (window->mitem_open, file_op);
	gtk_widget_set_sensitive (window->mitem_view, file_op && one_file_selected);
	gtk_widget_set_sensitive (window->mitem_stop, running);

	/* toolbar */

	gtk_widget_set_sensitive (window->toolbar_new, ! running);
	gtk_widget_set_sensitive (window->toolbar_open, ! running);
	gtk_widget_set_sensitive (window->toolbar_add, ! no_archive && ! ro && ! running && ! compr_file);
	gtk_widget_set_sensitive (window->toolbar_extract, file_op);
	gtk_widget_set_sensitive (window->toolbar_view, file_op && one_file_selected);
	gtk_widget_set_sensitive (window->toolbar_stop, running);

	/* popup menu */

	gtk_widget_set_sensitive (window->popupmenu_file[0], ! no_archive && ! ro && ! running && ! compr_file);
	gtk_widget_set_sensitive (window->popupmenu_file[1], ! no_archive && ! ro && ! window->archive_new && ! running && ! compr_file);
	gtk_widget_set_sensitive (window->popupmenu_file[2], file_op);
	gtk_widget_set_sensitive (window->popupmenu_file[4], file_op);
	gtk_widget_set_sensitive (window->popupmenu_file[5], file_op && one_file_selected);
}


static void
location_opt_changed_cb (GtkWidget * caller,
			 FRWindow *window)
{
	GtkWidget *item;
        GtkWidget *menu;
	char *path;
	char *location;

	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (window->location_opt));
        item = gtk_menu_get_active (GTK_MENU (menu));
	path = gtk_object_get_data (GTK_OBJECT (item), "path");
	location = g_strconcat (path,
				(strcmp (path, "/") == 0) ? NULL : "/",
				NULL);

	window_go_to_location (window, location);
	g_free (location);
}


static void
_window_update_current_location (FRWindow *window)
{
	GtkWidget *menu;
	GtkWidget *item;
	gchar *current_dir;

	if (window->list_mode == WINDOW_LIST_MODE_FLAT) {
		gtk_widget_hide (window->location_bar);
		return;
	} else
		gtk_widget_show (window->location_bar);

	if (! window->archive_present) {
		menu = gtk_menu_new ();
		item = gtk_menu_item_new_with_label (" ");
		gtk_menu_append (GTK_MENU (menu), item);
		gtk_widget_show_all (menu);
		gtk_option_menu_set_menu (GTK_OPTION_MENU (window->location_opt),
					  menu);
		
		gtk_widget_set_sensitive (window->up_button, FALSE);
		gtk_widget_set_sensitive (window->location_bar, FALSE);
		return;
	} else {
		gtk_widget_set_sensitive (window->location_bar, TRUE);
		gtk_widget_set_sensitive (window->up_button, strcmp (window->current_dir, "/") != 0);
	}

	menu = gtk_menu_new ();

	current_dir = window->current_dir;
	do {
		char *sub_dir;

		sub_dir = remove_level_from_path (current_dir);
		item = gtk_menu_item_new_with_label (sub_dir);

		gtk_object_set_data_full (GTK_OBJECT (item), 
					  "path", 
					  sub_dir,
					  g_free);
		gtk_signal_connect (GTK_OBJECT (item), 
                            "activate",
                            (GtkSignalFunc) location_opt_changed_cb, 
                            window);

		gtk_menu_append (GTK_MENU (menu), item);
		current_dir = sub_dir;
	} while (strcmp (current_dir, "/") != 0);

	gtk_widget_show_all (menu);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (window->location_opt),
				  menu);

	if (! GTK_WIDGET_VISIBLE (window->location_bar))
		gtk_widget_show (window->location_bar);
}


static void
_action_started (FRArchive *archive,
		 FRAction action, 
		 gpointer data)
{
	FRWindow *window = data;

	window_start_activity_mode (window);
	_window_update_sensitivity (window);

#ifdef DEBUG
	switch (action) {
	case FR_ACTION_LIST:
		g_print ("List");
		break;

	case FR_ACTION_DELETE:
		g_print ("Delete");
		break;

	case FR_ACTION_ADD:
		g_print ("Add");
		break;
	case FR_ACTION_EXTRACT:
		g_print ("Extract");
		break;

	default:
	}
	g_print (" [START]\n");
#endif
}


static void
_window_add_to_recent (FRWindow *window, char *filename, gboolean add)
{
	Bookmarks *recent;

	recent = bookmarks_new (RC_RECENT_FILE);
	bookmarks_load_from_disk (recent);
	if (add)
		bookmarks_add (recent, filename);
	else
		bookmarks_remove (recent, filename);
	bookmarks_set_max_lines (recent, preferences.max_history_len);
	bookmarks_write_to_disk (recent);
	bookmarks_free (recent);
	
	_window_update_recent_list (window);
}


static void drag_drop_add_file_list (FRWindow *window);
static void _window_batch_start_current_action (FRWindow *window);


static void
_action_performed (FRArchive *archive,
		   FRAction action, 
		   FRProcError *error, 
		   gpointer data)
{
	FRWindow *window = data;

	window_stop_activity_mode (window);

	if (error->type == FR_PROC_ERROR_STOPPED) {
		/* FIXME
		GtkWidget *dialog;
		dialog = gnome_warning_dialog_parented (_("Operation stopped"), GTK_WINDOW (window->app));
		gtk_widget_show (dialog);
		*/
	} else if (error->type != FR_PROC_ERROR_NONE) {
		char *msg = NULL;
		char *details = NULL;
		char *full_message;
		GtkWidget *dialog;

		if (action == FR_ACTION_LIST) 
			window_archive_close (window);

		switch (action) {
		case FR_ACTION_EXTRACT:
			msg = _("An error occurred while extracting files.");
			break;

		case FR_ACTION_LIST:
			msg = _("An error occurred while loading the archive.");
			break;
			
		case FR_ACTION_DELETE:
			msg = _("An error occurred while deleting files from the archive.");
			break;
			
		case FR_ACTION_ADD:
			msg = _("An error occurred while adding files to the archive.");
			break;
		}

		switch (error->type) {
		case FR_PROC_ERROR_COMMAND_NOT_FOUND:
			details = _("Command not found.");
			break;
		case FR_PROC_ERROR_EXITED_ABNORMALLY:
			details = _("Command exited abnormally.");
			break;
		case FR_PROC_ERROR_PIPE:
			details = _("System error: pipe.");
			break;
		case FR_PROC_ERROR_FORK:
			details = _("System error: fork.");
			break;
		case FR_PROC_ERROR_CANNOT_CHDIR:
			details = _("Cannot change to specified directory.");
			break;
		default:
		case FR_PROC_ERROR_GENERIC:
			details = _("No further information on error.");
			break;
		}

		full_message = g_strdup_printf ("%s\n%s", msg, details);
		dialog = gnome_error_dialog_parented (full_message, 
						      GTK_WINDOW(window->app));
		gtk_widget_show (dialog);
		g_free (full_message);
	}

	switch (action) {
	case FR_ACTION_LIST:
		if (error->type != FR_PROC_ERROR_NONE) {
			window_archive_close (window);
			break;
		}

		if (! window->archive_present) {
			window->archive_present = TRUE;
			
			if (window->current_dir != NULL)
				g_free (window->current_dir);
			window->current_dir = g_strdup ("/");
			
			if (window->open_default_dir != NULL) 
				g_free (window->open_default_dir);
			window->open_default_dir = remove_level_from_path (window->archive_filename);

			window->archive_new = FALSE;
		}

		window_update_file_list (window);
		_window_update_title (window);
		_window_update_current_location (window);
		break;

	case FR_ACTION_DELETE:
		window_archive_reload (window);
		return;

	case FR_ACTION_ADD:
		if (error->type != FR_PROC_ERROR_NONE) {
			window_archive_reload (window);
			break;
		}

		if (window->archive_new) {
			window->archive_new = FALSE;
			/* the archive file is created only when you add some
			 * file to it. */
			_window_add_to_recent (window, 
					       window->archive_filename,
					       TRUE);
		}

		if (window->adding_dropped_files) 
			/* continue adding dropped files. */
			drag_drop_add_file_list (window);
		else
			window_archive_reload (window);
		return;

	default:
	}

	_window_update_sensitivity (window);
	_window_update_statusbar_list_info (window);

	if (error->type != FR_PROC_ERROR_NONE) 
		window_batch_mode_stop (window);
	else {
		if (! window->batch_mode) 
			return;

		window->batch_action = g_list_next (window->batch_action);
		_window_batch_start_current_action (window);
	}
}


static void
button_title_clicked (GtkWidget * caller,
		      FRWindow * window)
{
	int i;

	/* Get the clicked column. */

	for (i = 0; i < 5; i++)
		if (GTK_CLIST (window->clist)->column[i].button == caller)
			break;

	if (window->sort_method == i) {
		if (window->sort_type == GTK_SORT_ASCENDING)
			window->sort_type = GTK_SORT_DESCENDING;
		else
			window->sort_type = GTK_SORT_ASCENDING;
	} else {
		window->sort_method = i;
		window->sort_type = GTK_SORT_ASCENDING;
	}

	window_update_file_list (window);
}


static gboolean 
clist_utils_row_is_selected (GtkCList *clist, 
                             gint row)
{
        GList *scan;

        for (scan = clist->selection; scan; scan = scan->next)
                if (GPOINTER_TO_INT (scan->data) == row) 
                        return TRUE;

        return FALSE;
}


static int
file_button_press_cb (GtkWidget *widget, 
		      GdkEventButton *event,
		      gpointer data)
{
        FRWindow *window = data;
	GtkCList *clist = GTK_CLIST (window->clist);
	FileData *fdata;
	gint row, col;
	
	if ((event->type == GDK_BUTTON_PRESS) 
	    && ((event->button == 2) || (event->button == 3))) {
		if (!gtk_clist_get_selection_info (clist,
						   event->x, 
						   event->y, 
						   &row, 
						   &col))
			return FALSE;

		if (! clist_utils_row_is_selected (clist, row)) {
			gtk_clist_unselect_all (clist);
			gtk_clist_select_row (clist, row, col);
		}
	}
	
	if ((event->type == GDK_BUTTON_PRESS) && (event->button == 3)) {
		gtk_menu_popup (GTK_MENU (window->file_popup_menu),
				NULL, NULL, NULL, 
				window, 
				event->button,
				event->time);
		return FALSE;
	}

	if ((event->state & GDK_SHIFT_MASK)
            || (event->state & GDK_CONTROL_MASK))
                return FALSE;

	if ((event->type == GDK_2BUTTON_PRESS) && (event->button == 1)) {

		if (!gtk_clist_get_selection_info (clist, 
						   event->x, 
						   event->y, 
						   &row, 
						   &col))
			return FALSE;

		fdata = gtk_clist_get_row_data (clist, row);
		if (fdata->is_dir) {
			char *new_dir;
			
			new_dir = g_strconcat (window->current_dir, 
					       fdata->list_name,
					       "/",
					       NULL);
			
			g_free (window->current_dir);
			window->current_dir = new_dir;
			
			window_update_file_list (window);
			_window_update_current_location (window);
		} else 
			window_view_or_open_file (window, 
						  fdata->original_path);

		return FALSE;
	}

	return FALSE;
}


/* -- drag and drop -- */


static GList *
get_file_list_from_url_list (gchar *url_list)
{
	GList *list = NULL;
	gint i;
	gchar *url_start, *url_end;

	i = 0;
	url_start = url_list;
	while (url_list[i] != '\0')	{
		while ((url_list[i] != '\0')
		       && (url_list[i] != '\r')
		       && (url_list[i] != '\n')) i++;

		url_end = url_list + i;
		if (strncmp (url_start, "file:", 5) == 0) {
			url_start += 5;
			if ((url_start[0] == '/') 
			    && (url_start[1] == '/')) url_start += 2;
		}
		list = g_list_prepend (list, g_strndup (url_start, url_end - url_start));

		while ((url_list[i] != '\0')
		       && ((url_list[i] == '\r')
			   || (url_list[i] == '\n'))) i++;
		url_start = url_list + i;
	}
	
	return g_list_reverse (list);
}


static void
drag_drop_add_file_list (FRWindow *window)
{
	GList *     list = window->dropped_file_list;
	FRArchive * archive = window->archive;
	GList *     scan;
	gchar *     first_basedir;
	gboolean    same_dir;

	if (window->activity_ref > 0) 
		return;

	if (window->archive->read_only) {
		GtkWidget *dialog;
		dialog = gnome_error_dialog_parented (_("You don't have permissions to add files to this archive."), GTK_WINDOW (window->app));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		window->adding_dropped_files = FALSE;
		return;
	}

	if (list == NULL) {
		window->adding_dropped_files = FALSE;
		window_archive_reload (window);
		return;
	}

	for (scan = list; scan; scan = scan->next) {
		if (strcmp (scan->data, window->archive_filename) == 0) {
			GtkWidget *dialog;

			window->adding_dropped_files = FALSE;
			dialog = gnome_error_dialog_parented (_("You can't add an archive to itself."), GTK_WINDOW (window->app));
			gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
			return;
		}
	}

	/* add directories. */

	for (scan = list; scan; scan = scan->next) {
		gchar *path = scan->data;
		gchar *base_dir;

		if (! path_is_dir (path)) 
			continue;

		window->dropped_file_list = g_list_remove_link (list, scan);
		window->adding_dropped_files = TRUE;

		base_dir = remove_level_from_path (path);
		fr_archive_add_directory (archive, 
					  file_name_from_path (path),
					  base_dir,
					  window->update_dropped_files);
		g_free (base_dir);
		g_free (path);
		return;
	}

	window->adding_dropped_files = FALSE;

	/* if all files are in the same directory call fr_archive_add once. */

	same_dir = TRUE;
	first_basedir = remove_level_from_path (list->data);

	if (first_basedir == NULL) 
		return;

	for (scan = list->next; scan; scan = scan->next) {
		gchar *basedir;

		basedir = remove_level_from_path (scan->data);
		if (basedir == NULL) {
			same_dir = FALSE;
			break;
		}

		if (strcmp (first_basedir, basedir) != 0) {
			same_dir = FALSE;
			g_free (basedir);
			break;
		}
		g_free (basedir);
	}

	if (same_dir) {
		GList *only_names_list = NULL;

		for (scan = list; scan; scan = scan->next)
			only_names_list = g_list_prepend (only_names_list, file_name_from_path (scan->data));

		fr_archive_add (archive,
				only_names_list,
				first_basedir,
				window->update_dropped_files);

		g_list_free (only_names_list);
		g_free (first_basedir);

		return;
	}
	g_free (first_basedir);
	
	/* ...else call fr_command_add for each file.  This is needed to add
	 * files without path info. */

	fr_process_clear (archive->process);
	fr_command_uncompress (archive->command);
	for (scan = list; scan; scan = scan->next) {
		gchar *fullpath = scan->data;
		gchar *basedir;
		GList *singleton;
		
		basedir = remove_level_from_path (fullpath);
		singleton = g_list_prepend (NULL, 
					    shell_escape (file_name_from_path (fullpath)));
		fr_command_add (archive->command,
				singleton,
				basedir,
				window->update_dropped_files);
		path_list_free (singleton);
		g_free (basedir);
	}
	fr_command_recompress (archive->command);
	fr_process_start (archive->process, FALSE);
}


void  
window_drag_data_received  (GtkWidget          *widget,
			    GdkDragContext     *context,
			    gint                x,
			    gint                y,
			    GtkSelectionData   *data,
			    guint               info,
			    guint               time,
			    gpointer            extra_data)
{
	FRWindow *window = extra_data;
	GList *list;
	gboolean one_file;
	gboolean is_an_archive;

#ifdef DEBUG
	g_print ("::DragDataReceived -->\n");
#endif

	if (! ((data->length >= 0) && (data->format == 8))) {
		gtk_drag_finish (context, FALSE, FALSE, time);
		return;
	}

	if (window->activity_ref > 0) {
		gtk_drag_finish (context, FALSE, FALSE, time);
		return;
	}

	gtk_drag_finish (context, TRUE, FALSE, time);

	list = get_file_list_from_url_list ((gchar *)data->data);
	if (list == NULL)
		return;

	if (window->dropped_file_list != NULL)
		path_list_free (window->dropped_file_list);
	window->dropped_file_list = list;
	window->update_dropped_files = FALSE;

	one_file = (list->next == NULL);
	if (one_file)
		is_an_archive = (fr_archive_utils_get_file_name_ext (list->data) != NULL);
	else
		is_an_archive = FALSE;

	if (window->archive_present && ! window->archive->is_compressed_file) {
		if (one_file && is_an_archive) {
			GtkWidget *dialog;
			GtkWidget *label;
			gint r;
			
			dialog = gnome_dialog_new (_("File Roller"),
						   GNOME_STOCK_PIXMAP_ADD,
						   GNOME_STOCK_PIXMAP_OPEN,
						   GNOME_STOCK_BUTTON_CANCEL,
						   NULL);
			label = gtk_label_new (_("Do you want to add this file to the current archive or open it as a new archive ?"));
			gtk_widget_show (label);
			gtk_container_add (GTK_CONTAINER ((GNOME_DIALOG (dialog)->vbox)), label);
			gnome_dialog_set_default (GNOME_DIALOG (dialog), 2);
			gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (window->app));
			r = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
			if (r == 0) { /* Add */
				/* this way we do not free the list saved in
				 * window->dropped_file_list. */
				list = NULL;

				drag_drop_add_file_list (window);
			} else if (r == 1) { /* Open */
				/* if this window already has an archive 
				 * create a new window. */
				if (window->archive_present) {
					FRWindow *new_window;
					new_window = window_new ();
					gtk_widget_show (new_window->app);
					window_archive_open (new_window, list->data);
				} else
					window_archive_open (window, list->data);
			}
 		} else {
			/* this way we do not free the list saved in
			 * window->dropped_file_list. */
			list = NULL;

			drag_drop_add_file_list (window);
		}
	} else {
		if (one_file && is_an_archive) 
			window_archive_open (window, list->data);
		else {
			GtkWidget *dialog;
			GtkWidget *label;
			gint r;

			dialog = gnome_dialog_new (_("File Roller"),
						   GNOME_STOCK_BUTTON_YES,
						   GNOME_STOCK_BUTTON_NO,
						   NULL);
			label = gtk_label_new (_("Do you want to create a new archive with these files ?"));
			gtk_widget_show (label);
			gtk_container_add (GTK_CONTAINER ((GNOME_DIALOG (dialog)->vbox)), label);
			gnome_dialog_set_default (GNOME_DIALOG (dialog), 2);
			gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (window->app));
			r = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
			if (r == 0) {
				window->add_dropped_files = TRUE;

				/* this way we do not free the list saved in
				 * window->dropped_file_list. */
				list = NULL; 

				new_archive_cb (NULL, window);
			}
		}
	}

	if (list != NULL) {
		path_list_free (list);
		window->dropped_file_list = NULL;
	}

#ifdef DEBUG
	g_print ("::DragDataReceived <--\n");
#endif
}


static gchar *
make_url_list (GList *list, 
	       gboolean plain_text)
{
	gchar *url_list;
	gint   url_list_length;
	gchar *prefix;
	gint   prefix_length;
	gchar *url_sep = "\r\n";
	gint   url_sep_length;
	GList *scan;

	if (list == NULL)
		return NULL;
	
	prefix = (plain_text) ? g_strdup ("") : g_strdup ("file://");
	prefix_length = strlen (prefix);
	url_sep_length = strlen (url_sep);

	url_list_length = 0;
	for (scan = list; scan; scan = scan->next)
		url_list_length += (prefix_length 
				    + strlen (scan->data)
				    + url_sep_length);

	url_list = g_malloc (url_list_length + 1);
	*url_list = 0;

	for (scan = list; scan; scan = scan->next) {
		url_list = strncat (url_list, prefix, prefix_length);
		url_list = strncat (url_list, scan->data, strlen (scan->data));
		url_list = strncat (url_list, url_sep, url_sep_length);
	}

	g_free (prefix);

	return url_list;
}


static gchar *
_get_temp_dir_name ()
{
	static int count = 0;
	return g_strdup_printf ("%s%s.%d.%d",
				g_get_tmp_dir (),
				"/file-roller",
				getpid (),
				count++);
}


static GList *
_get_selection_as_names (FRWindow *window)
{
	GList *list, *scan;

	g_return_val_if_fail (window != NULL, NULL);

	list = NULL;
        scan = GTK_CLIST (window->clist)->selection;
        for (; scan; scan = scan->next) {
                FileData *fd = gtk_clist_get_row_data (GTK_CLIST (window->clist), GPOINTER_TO_INT (scan->data));

		if (!fd)
			continue;

		list = g_list_prepend (list, g_strdup (fd->list_name));
        }

        return g_list_reverse (list);
}


static void  
file_list_drag_begin (GtkWidget          *widget,
		      GdkDragContext     *context,
		      gpointer            data)
{
	FRWindow *window = data;

#ifdef DEBUG
	g_print ("::DragBegin -->\n");
#endif

	if (window->activity_ref > 0) 
		return;

	window->drag_file_list = window_get_file_list_selection (window, TRUE, & (window->dragging_dirs));

	if (window->drag_file_list_names != NULL) {
		path_list_free (window->drag_file_list_names);
		window->drag_file_list_names = NULL;
	}

	if (window->dragging_dirs)
		window->drag_file_list_names = _get_selection_as_names (window);

	if (window->drag_file_list != NULL) {
		window->drag_temp_dir = _get_temp_dir_name ();
		window->drag_temp_dirs = g_list_prepend (window->drag_temp_dirs, window->drag_temp_dir);

		ensure_dir_exists (window->drag_temp_dir);
		fr_archive_extract (window->archive,
				    window->drag_file_list,
				    window->drag_temp_dir,
				    FALSE,
				    TRUE,
				    FALSE);

		while (window->activity_ref > 0)
			while (gtk_events_pending())
				gtk_main_iteration();
	}

#ifdef DEBUG
	g_print ("::DragBegin <--\n");
#endif
}


static void  
file_list_drag_end (GtkWidget * widget,
		    GdkDragContext * context,
		    gpointer data)
{
	FRWindow *window = data;

#ifdef DEBUG
	g_print ("::DragEnd -->\n");
#endif

	if (window->activity_ref > 0) 
		return;

	if (window->drag_file_list != NULL) {
		path_list_free (window->drag_file_list);
		window->drag_file_list = NULL;
	}

#ifdef DEBUG
	g_print ("::DragEnd <--\n");
#endif
}


static void  
file_list_drag_data_get  (GtkWidget          *widget,
			  GdkDragContext     *context,
			  GtkSelectionData   *selection_data,
			  guint               info,
			  guint               time,
			  gpointer            data)
{
	FRWindow *window = data;
	GList *list, *scan;
	gchar *url_list;

#ifdef DEBUG
	g_print ("::DragDataGet -->\n"); 
#endif

	if (window->activity_ref > 0) 
		return;

	if (window->drag_file_list == NULL) 
		return;
	
	list = NULL;
	if (! window->dragging_dirs) 
		for (scan = window->drag_file_list; scan; scan = scan->next) {
			gchar *url;
			url = g_strconcat (window->drag_temp_dir, 
					   "/", 
					   scan->data, 
					   NULL);
			list = g_list_prepend (list, url);
		}
	else
		for (scan = window->drag_file_list_names; scan; scan = scan->next) {
			gchar *url;
			url = g_strconcat (window->drag_temp_dir, 
					   window->current_dir,
					   scan->data, 
					   NULL);
			list = g_list_prepend (list, url);
		}


	url_list = make_url_list (list, (info == TARGET_STRING));
	path_list_free (list);

	if (url_list == NULL) 
		return;

	gtk_selection_data_set (selection_data, 
				selection_data->target,
				8, 
				url_list, 
				strlen (url_list));

	g_free (url_list);

#ifdef DEBUG
	g_print ("::DragDataGet <--\n");
#endif
}


/* -- window_new -- */


static GtkWidget *
create_icon (char *name)
{
	GtkWidget *icon;
	GdkPixbuf *pixbuf;
	GdkPixmap *pixmap;
        GdkBitmap *mask;
	char *file;
	
	file = g_strconcat (ICONDIR, "/", name, NULL);
	pixbuf = gdk_pixbuf_new_from_file (file);
	g_free (file);

	if (pixbuf == NULL)
		return NULL;

	gdk_pixbuf_render_pixmap_and_mask (pixbuf, &pixmap, &mask, 112);
        gdk_pixbuf_unref (pixbuf);
	icon = gtk_pixmap_new (pixmap, mask);
	gdk_pixmap_unref (pixmap);
	gdk_bitmap_unref (mask);

	return icon;
}


static gint
key_press_cb (GtkWidget *widget, 
              GdkEventKey *event,
              gpointer data)
{
        FRWindow *window = data;
	
	switch (event->keyval) {
	case GDK_Escape:
		stop_cb (NULL, window);
		return TRUE;

	case GDK_Delete:
		dlg_delete (NULL, window);
		return TRUE;

	default:
		break;
	}

	return FALSE;
}


static int
selection_changed_cb (GtkWidget *widget, 
		      gint row, 
		      gint col, 
		      GdkEvent *event,
		      gpointer data)
{
	FRWindow *window = data;
	_window_update_statusbar_list_info (window);
	_window_update_sensitivity (window);
	return FALSE;
}


static void
window_delete_event_cb (GtkWidget *caller, 
			GdkEvent *event, 
			FRWindow *window)
{
#ifdef DEBUG
	g_print ("DELETE\n");
#endif
	window_close (window);
}



FRWindow *
window_new ()
{
	static gchar * titles[] = {N_("Name"), 
				   N_("Type"), 
				   N_("Size"), 
				   N_("Modified"), 
				   N_("Path")};
	FRWindow *window;
	GtkWidget *toolbar;
	GtkWidget *scrolled_window;
	GtkWidget *vbox;
	GtkWidget *location_label;
	GtkWidget *location_box;
	int i, height;

	window = g_new (FRWindow, 1);

	/* Create the application. */

        window->app = gnome_app_new ("main", _("File Roller"));
        gnome_window_icon_set_from_default (GTK_WINDOW (window->app));

	gtk_signal_connect (GTK_OBJECT (window->app), "delete_event",
			    GTK_SIGNAL_FUNC (window_delete_event_cb),
			    window);

	gtk_window_set_default_size (GTK_WINDOW (window->app), 600, 480);

	gtk_drag_dest_set (window->app,
			   GTK_DEST_DEFAULT_ALL,
			   target_table, n_targets,
			   GDK_ACTION_COPY);

	gtk_signal_connect (GTK_OBJECT (window->app), 
			    "drag_data_received",
			    GTK_SIGNAL_FUNC (window_drag_data_received), 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->app), 
                            "key_press_event",
                            (GtkSignalFunc) key_press_cb, 
                            window);

	/* Create the widgets. */

	/* * File list. */

	window->clist = gtk_clist_new (5);
	height = (GTK_WIDGET (window->clist)->style->font->ascent 
		  + GTK_WIDGET (window->clist)->style->font->descent 
		  + CLIST_ROW_PAD);
	gtk_clist_set_row_height (GTK_CLIST (window->clist), height);
	gtk_clist_set_selection_mode (GTK_CLIST (window->clist),
				      GTK_SELECTION_EXTENDED);

        gtk_drag_source_set (window->clist, 
			     GDK_BUTTON2_MASK,
			     target_table, n_targets, 
			     GDK_ACTION_COPY);
	gtk_signal_connect_after (GTK_OBJECT (window->clist), 
			    "select_row",
			    (GtkSignalFunc) selection_changed_cb,
			    window);
	gtk_signal_connect_after (GTK_OBJECT (window->clist), 
			    "unselect_row",
			    (GtkSignalFunc) selection_changed_cb,
			    window);
	gtk_signal_connect_after (GTK_OBJECT (window->clist), 
                            "button_press_event",
                            (GtkSignalFunc) file_button_press_cb, 
                            window);

	gtk_signal_connect (GTK_OBJECT (window->clist), 
			    "drag_data_get",
			    GTK_SIGNAL_FUNC (file_list_drag_data_get), 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->clist), 
			    "drag_begin",
			    GTK_SIGNAL_FUNC (file_list_drag_begin), 
			    window);
	gtk_signal_connect (GTK_OBJECT (window->clist), 
			    "drag_end",
			    GTK_SIGNAL_FUNC (file_list_drag_end), 
			    window);

	for (i = 0; i < 5; i++) {
		GtkWidget *button_title;
		GtkWidget *hbox;
		GtkWidget *label = NULL;
		GtkWidget *pixmap_up;
		GtkWidget *pixmap_down;

		hbox = gtk_hbox_new (FALSE, 0);

		label = gtk_label_new (_(titles[i]));

		pixmap_up = gnome_pixmap_new_from_xpm_d (up_xpm);
		pixmap_down = gnome_pixmap_new_from_xpm_d (down_xpm);

		gtk_box_pack_start (GTK_BOX (hbox), label, 
				    FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (hbox), pixmap_up, 
				    FALSE, FALSE, 2);
		gtk_box_pack_start (GTK_BOX (hbox), pixmap_down,
				    FALSE, FALSE, 2);

		window->up_arrows[i] = pixmap_up;
		window->down_arrows[i] = pixmap_down;

		gtk_widget_show (label);
		gtk_widget_show (hbox);
		gtk_clist_set_column_widget (GTK_CLIST (window->clist),
					     i, 
					     hbox);

		button_title = GTK_CLIST (window->clist)->column[i].button;
		gtk_signal_connect (GTK_OBJECT (button_title), "clicked",
				    (GtkSignalFunc) button_title_clicked,
				    window);

		gtk_clist_set_column_resizeable (GTK_CLIST (window->clist), 
						 i,
						 FALSE);
	}
	gtk_clist_column_titles_show (GTK_CLIST (window->clist));

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_add (GTK_CONTAINER (scrolled_window), window->clist);

	/* * Location bar. */

	window->location_bar = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (window->location_bar),
				   gnome_preferences_get_toolbar_relief () ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);

	location_box = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (location_box), 2);
	gtk_container_add (GTK_CONTAINER (window->location_bar), location_box);

	window->up_button = gtk_button_new ();
	GTK_WIDGET_UNSET_FLAGS (window->up_button, GTK_CAN_FOCUS);
	gtk_button_set_relief (GTK_BUTTON (window->up_button),
			       gnome_preferences_get_toolbar_relief_btn () ? GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);
	gtk_container_add (GTK_CONTAINER (window->up_button), 
			   gnome_stock_pixmap_widget (window->app,
						      GNOME_STOCK_PIXMAP_UP));
	gtk_box_pack_start (GTK_BOX (location_box), 
			    window->up_button, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (window->up_button), "clicked",
			    go_up_one_level_cb,
			    window);

	gtk_box_pack_start (GTK_BOX (location_box), 
			    gtk_vseparator_new (),
			    FALSE, FALSE, 0);

	location_label = gtk_label_new (_("Current Location:"));
	gtk_box_pack_start (GTK_BOX (location_box), 
			    location_label, FALSE, FALSE, 0);

	window->location_opt = gtk_option_menu_new ();
	gtk_box_pack_start (GTK_BOX (location_box), 
			    window->location_opt, FALSE, FALSE, 0);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), window->location_bar, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);

	gnome_app_set_contents (GNOME_APP (window->app), vbox);
	gtk_widget_show_all (vbox);

	/* Create the main menu. */

	gnome_app_create_menus_with_data (GNOME_APP (window->app), main_menu, 
					  window);

	/* Create the toolbar. */

        toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
                                   GTK_TOOLBAR_BOTH);
        gnome_app_fill_toolbar_with_data (GTK_TOOLBAR (toolbar), 
                                          toolbar_data, 
                                          (GtkAccelGroup*) NULL,
                                          window);
	window->toolbar_add = 
		gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
					    GTK_TOOLBAR_CHILD_BUTTON,
					    NULL, 
					    _("Add"), 
					    _("Add files"),
					    NULL, 
					    create_icon ("add.xpm"),
					    add_cb, 
					    window,
					    TOOLBAR_SEP1 + 1);
	window->toolbar_extract = 
		gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
					    GTK_TOOLBAR_CHILD_BUTTON,
					    NULL, 
					    _("Extract"), 
					    _("Extract"),
					    NULL, 
					    create_icon ("extract.xpm"),
					    dlg_extract, 
					    window,
					    TOOLBAR_SEP1 + 2);
	window->toolbar_view = 
		gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
					    GTK_TOOLBAR_CHILD_BUTTON,
					    NULL, 
					    _("View"), 
					    _("View File"),
					    NULL, 
					    create_icon ("view.xpm"),
					    view_or_open_cb, 
					    window,
					    TOOLBAR_SEP1 + 3);
	gnome_app_set_toolbar (GNOME_APP (window->app), GTK_TOOLBAR (toolbar));

	/* Create popup menus. */

	window->file_popup_menu = gtk_menu_new ();
	gnome_app_fill_menu_with_data (GTK_MENU_SHELL (window->file_popup_menu),
				       file_popup_menu_data,
				       (GtkAccelGroup*) NULL,
				       FALSE,
				       0,
				       window);

	/* Create a statusbar. */

        window->statusbar = gnome_appbar_new (FALSE, TRUE, 
                                              GNOME_PREFERENCES_USER);
        gnome_app_set_statusbar (GNOME_APP (window->app), window->statusbar);
        gnome_app_install_appbar_menu_hints (GNOME_APPBAR (window->statusbar),
                                             main_menu);

	/* * Progress bar. */

        window->progress = gtk_progress_bar_new ();
        gtk_box_pack_start (GTK_BOX (window->statusbar), window->progress, 
                            FALSE, TRUE, 0);
	gtk_widget_show (window->progress);

	/* Data. */

	window->archive = fr_archive_new (NULL);
	gtk_signal_connect (GTK_OBJECT (window->archive),
			    "start",
			    GTK_SIGNAL_FUNC (_action_started),
			    window);
	gtk_signal_connect (GTK_OBJECT (window->archive),
			    "done",
			    GTK_SIGNAL_FUNC (_action_performed),
			    window);

	window->sort_method = preferences.sort_method;
	window->sort_type = preferences.sort_type;

	window->list_mode = preferences.list_mode;
	window->current_dir = g_strdup ("/");

	window->mitem_new_archive = file_menu[FILE_MENU_NEW_ARCHIVE].widget;
	window->mitem_open_archive = file_menu[FILE_MENU_OPEN_ARCHIVE].widget;
	window->mitem_close = file_menu[FILE_MENU_CLOSE_ARCHIVE].widget;

	window->mitem_archive_prop = file_menu[FILE_MENU_ARCHIVE_PROP].widget;
	window->mitem_move_archive = file_menu[FILE_MENU_MOVE_ARCHIVE].widget;
	window->mitem_copy_archive = file_menu[FILE_MENU_COPY_ARCHIVE].widget;
	window->mitem_rename_archive = file_menu[FILE_MENU_RANAME_ARCHIVE].widget;
	window->mitem_delete_archive = file_menu[FILE_MENU_DELETE_ARCHIVE].widget;

	window->mitem_bookmarks = GTK_MENU_ITEM (main_menu[0].widget)->submenu;

	window->mitem_add = actions_menu[ACTIONS_MENU_ADD].widget;
	window->mitem_delete = actions_menu[ACTIONS_MENU_DELETE].widget;
	window->mitem_extract = actions_menu[ACTIONS_MENU_EXTRACT].widget;
	window->mitem_open = actions_menu[ACTIONS_MENU_OPEN].widget;
	window->mitem_view = actions_menu[ACTIONS_MENU_VIEW].widget;
	window->mitem_stop = actions_menu[ACTIONS_MENU_STOP].widget;

	window->mitem_view_flat   = view_list[VIEW_LIST_VIEW_ALL].widget;
	window->mitem_view_as_dir = view_list[VIEW_LIST_AS_DIR].widget;
	set_check_menu_item_state (window, window->mitem_view_as_dir, TRUE);
	for (i = 0; i < 5; i++)
		window->mitem_sort[i] = sort_by_radio_list[i].widget;
	window->mitem_sort_reversed = sort_menu[2].widget;

	for (i = 0; i < 8; i++) 
		window->popupmenu_file[i] = file_popup_menu_data[i].widget;

	window->toolbar_new = toolbar_data[TOOLBAR_NEW].widget;
	window->toolbar_open = toolbar_data[TOOLBAR_OPEN].widget;
	window->toolbar_stop = toolbar_data[TOOLBAR_STOP].widget;

	window->bookmarks_tooltips = NULL;

	window->open_default_dir = g_strdup (g_get_home_dir ());
	window->add_default_dir = g_strdup (g_get_home_dir ());
	window->extract_default_dir = g_strdup (g_get_home_dir ());

	window->activity_ref = 0;
        window->activity_progress = 0.0;
	window->activity_timeout_handle = 0;

	window->archive_present = FALSE;
	window->archive_new = FALSE;
	window->archive_filename = NULL;

	window->drag_temp_dir = NULL;
	window->drag_temp_dirs = NULL;
	window->drag_file_list = NULL;
	window->drag_file_list_names = NULL;

	window->dropped_file_list = NULL;
	window->add_dropped_files = FALSE;
	window->adding_dropped_files = FALSE;

	window->batch_mode = FALSE;
	window->batch_action_list = NULL;
	window->batch_action = NULL;

	/* update menu items */

	if (window->list_mode == WINDOW_LIST_MODE_FLAT)
		set_check_menu_item_state (window, window->mitem_view_flat, TRUE);
	else
		set_check_menu_item_state (window, window->mitem_view_as_dir, TRUE);
	set_check_menu_item_state (window, window->mitem_sort[window->sort_method], TRUE);
	if (window->sort_type == GTK_SORT_DESCENDING)
		set_check_menu_item_state (window, window->mitem_sort_reversed, TRUE);

	_window_update_title (window);
	_window_update_sensitivity (window);
	window_update_file_list (window);
	_window_update_current_location (window);
	_window_update_recent_list (window);
	window_update_columns_visibility (window);

	window_list = g_list_prepend (window_list, window);
	
	return window;
}


/* -- window_close -- */


static void
_window_remove_drag_temp_dirs (FRWindow *window)
{
	GList *scan;

	for (scan = window->drag_temp_dirs; scan; scan = scan->next) 
		if (path_is_dir (scan->data)) {
			char *command;
			
			command = g_strconcat ("rm -rf ",
					       scan->data,
					       NULL);
			gnome_execute_shell (g_get_tmp_dir (), command); 
			g_free (command);
		}

	if (window->drag_temp_dirs != NULL) {
		path_list_free (window->drag_temp_dirs);
		window->drag_temp_dirs = NULL;
	}
}


static void
_window_free_batch_data (FRWindow *window)
{
	GList *scan;

	for (scan = window->batch_action_list; scan; scan = scan->next) {
		FRBatchActionDescription *adata = scan->data;
		if ((adata->data != NULL) && (adata->free_func != NULL))
			(*adata->free_func) (adata->data);
		g_free (adata);
	}

	g_list_free (window->batch_action_list);
	window->batch_action_list = NULL;
	window->batch_action = NULL;
}


void
window_close (FRWindow *window)
{
	g_return_if_fail (window != NULL);

	if (window->current_dir != NULL)
		g_free (window->current_dir);
	if (window->open_default_dir != NULL)
		g_free (window->open_default_dir);
	if (window->add_default_dir != NULL)
		g_free (window->add_default_dir);
	if (window->extract_default_dir != NULL)
		g_free (window->extract_default_dir);
	if (window->archive_filename != NULL)
		g_free (window->archive_filename);

	gtk_widget_destroy (window->app);
	gtk_object_unref (GTK_OBJECT (window->archive));

	if (window->drag_file_list != NULL)
		path_list_free (window->drag_file_list);
	_window_remove_drag_temp_dirs (window);

	if (window->dropped_file_list != NULL) {
		path_list_free (window->dropped_file_list);
		window->dropped_file_list = NULL;
	}

	if (window->drag_file_list_names != NULL) {
		path_list_free (window->drag_file_list_names);
		window->drag_file_list_names = NULL;
	}

	_window_free_batch_data (window);

	/* save preferences. */

	preferences.sort_method = window->sort_method;
	preferences.sort_type = window->sort_type;
	preferences.list_mode = window->list_mode;

	window_list = g_list_remove (window_list, window);
	g_free (window);

	if (window_list == NULL)
                gtk_main_quit ();
}


void
window_archive_new (FRWindow *window, 
		    char *filename)
{
	g_return_if_fail (window != NULL);

	if (fr_archive_utils_get_file_name_ext (filename) == NULL) {
		GtkWidget *dialog;

		dialog = gnome_error_dialog_parented (_("File type not supported."), GTK_WINDOW (window->app));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		return;
	}

	if (window->current_dir != NULL)
		g_free (window->current_dir);
	window->current_dir = g_strdup ("/");

	if (window->archive_filename != NULL)
		g_free (window->archive_filename);
	window->archive_filename = g_strdup (filename);

	window->archive_present = TRUE;
	window->archive_new = TRUE;

	fr_archive_new_file (window->archive, filename);

	window_update_file_list (window);
	_window_update_title (window);
	_window_update_sensitivity (window);
	_window_update_current_location (window);

	if (window->add_dropped_files) {
		drag_drop_add_file_list (window);
		window->add_dropped_files = FALSE;
	}
}


/* -- _window_update_recent_list -- */


static void 
bookmark_selected_cb (GtkWidget *widget, 
		      gpointer data)
{
	FRWindow *window = data;
	gchar *path;

	path = gtk_object_get_data (GTK_OBJECT (widget), "full_path");
	window_archive_open (window, path);
}


static void
_window_update_recent_list (FRWindow *window)
{
	GList *scan, *l;
	Bookmarks *bookmarks;
	gint i, offset;
	GtkWidget *mitem;

	bookmarks = bookmarks_new (RC_RECENT_FILE);
	bookmarks_load_from_disk (bookmarks);

	if (window->bookmarks_tooltips != NULL) 
                gtk_object_destroy (GTK_OBJECT (window->bookmarks_tooltips));
	window->bookmarks_tooltips = gtk_tooltips_new ();

	l = gtk_container_children (GTK_CONTAINER (window->mitem_bookmarks));

	if (gnome_preferences_get_menus_have_tearoff ()) 
		offset = 2;
	else 
		offset = 1;

	for (i = 0; i < FILE_MENU_EXIT + offset; i++) 
		l = l->next;

	for (scan = l; scan; scan = scan->next)
		if (GTK_IS_WIDGET (scan->data))
			gtk_widget_destroy (GTK_WIDGET (scan->data));

	if (l) g_list_free (l);

	if (bookmarks->list == NULL) {
		bookmarks_free (bookmarks);
		return;
	}

	mitem =  gtk_menu_item_new ();
	gtk_widget_show (mitem);
	gtk_widget_set_sensitive (mitem, FALSE);
	gtk_menu_append (GTK_MENU (window->mitem_bookmarks), mitem);

	/* Update bookmarks menu. */

	for (i = 1, scan = bookmarks->list; scan; scan = scan->next) {
		char * label;

		label = g_strdup_printf ("%d. %s", i++, file_name_from_path (scan->data));
		mitem =  gtk_menu_item_new_with_label (label);
		g_free (label);

		gtk_tooltips_set_tip (window->bookmarks_tooltips, 
				      GTK_WIDGET (mitem),
				      scan->data,
				      NULL);	
		gtk_object_set_data (GTK_OBJECT (mitem), "tooltips", 
				     window->bookmarks_tooltips);
		gtk_widget_show (mitem);
		
		gtk_object_set_data_full (GTK_OBJECT (mitem), "full_path", 
					  g_strdup (scan->data),
					  g_free);
		gtk_menu_append (GTK_MENU (window->mitem_bookmarks), mitem);

		gtk_signal_connect (GTK_OBJECT (mitem), 
				    "activate", 
				    GTK_SIGNAL_FUNC (bookmark_selected_cb), 
				    window);
	}

	bookmarks_free (bookmarks);
}


void
window_archive_open (FRWindow *window, char *filename)
{
	gboolean success;
	gchar *path;

	g_return_if_fail (window != NULL);

	window_archive_close (window);

	if (! g_path_is_absolute (filename)) {
		char *current_dir;
		current_dir = g_get_current_dir ();
		path = g_strconcat (current_dir, 
				    "/", 
				    filename, 
				    NULL);
		g_free (current_dir);
	} else
		path = g_strdup (filename);

	window->archive_present = FALSE;	

	if (window->archive_filename != NULL)
		g_free (window->archive_filename);
	window->archive_filename = g_strdup (path);

	success = fr_archive_load (window->archive, path);
	if (! success) 
		gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog_parented (_("Cannot load archive."), GTK_WINDOW (window->app))));
	
	_window_add_to_recent (window, path, success);
}


void
window_archive_reload (FRWindow *window)
{
	g_return_if_fail (window != NULL);

	if (window->archive_new)
		return;

	fr_archive_reload (window->archive);
}


void
window_archive_rename (FRWindow *window, 
		       char *filename)
{
	g_return_if_fail (window != NULL);

	if (window->archive_new) 
		window_archive_new (window, filename);
	else {
		fr_archive_rename (window->archive, filename);

		if (window->archive_filename != NULL)
			g_free (window->archive_filename);
		window->archive_filename = g_strdup (filename);

		_window_update_title (window);
		_window_add_to_recent (window, window->archive_filename, TRUE);
	}
}


/* -- window_archive_extract -- */


typedef struct {
	FRWindow *   window;
	gchar *      extract_to_dir;
	gboolean     do_not_extract;
} WindowArchiveExtractData;


static void
create_dir_cb (gint reply, 
	       gpointer callback_data)
{
	WindowArchiveExtractData *data = callback_data;
	
	if (reply != GNOME_YES) {
		data->do_not_extract = TRUE;
		return;
	}
	
	if (! ensure_dir_exists (data->extract_to_dir)) {
		GtkWidget *dialog;
		dialog = gnome_error_dialog_parented (_("Could not create the destination directory."), GTK_WINDOW (data->window->app));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		data->do_not_extract = TRUE;
	}
}


void
window_archive_extract (FRWindow *window,
			GList * file_list,
			char * extract_to_dir,
			gboolean skip_older,
			gboolean overwrite,
			gboolean junk_paths)
{
	WindowArchiveExtractData *data;

	data = g_new (WindowArchiveExtractData, 1);
	data->window = window;
	data->extract_to_dir = extract_to_dir;
	data->do_not_extract = FALSE;

	if ( ! path_is_dir (extract_to_dir)) {
		GtkWidget *dialog;
		dialog = gnome_question_dialog_parented (_("Directory does not exist.  Do you want to create it ?"), create_dir_cb, data, GTK_WINDOW (window->app));
		gnome_dialog_run (GNOME_DIALOG (dialog));
	}
	
	if (data->do_not_extract) {
		GtkWidget *dialog;
		dialog = gnome_error_dialog_parented (_("Extraction not performed."), GTK_WINDOW (window->app));
		gnome_dialog_run (GNOME_DIALOG (dialog));
		g_free (data);
		return;
	}

	fr_archive_extract (window->archive,
			    file_list,
			    extract_to_dir,
			    skip_older,
			    overwrite,
			    junk_paths);

	g_free (data);
}


void
window_archive_close (FRWindow *window)
{
	g_return_if_fail (window != NULL);

	_window_remove_drag_temp_dirs (window);

	window->archive_new = FALSE;
	window->archive_present = FALSE;

	_window_update_title (window);
	_window_update_sensitivity (window);
	window_update_file_list (window);
	_window_update_current_location (window);
	_window_update_statusbar_list_info (window);
}


void
window_go_to_location (FRWindow *window, char *path)
{
	g_return_if_fail (window != NULL);

	if (window->current_dir)
		g_free (window->current_dir);
	window->current_dir = g_strdup (path);

	window_update_file_list (window);
	_window_update_current_location (window);
}


void
window_go_up_one_level (FRWindow *window)
{
	g_return_if_fail (window != NULL);

	_go_up_one_level (window);
	window_update_file_list (window);
	_window_update_current_location (window);
}


void
window_set_list_mode (FRWindow *window, 
		      WindowListMode list_mode)
{
	g_return_if_fail (window != NULL);

	window->list_mode = list_mode;
	window_update_file_list (window);
	_window_update_current_location (window);
}


/* -- window_get_file_list_selection -- */


static GList *
get_dir_list (FRWindow *window,
	      FileData *fdata)
{
	GList *list;
	GList *scan;
	char *dirname;
	int dirname_l;

	dirname = g_strconcat (window->current_dir,
			       fdata->list_name,
			       "/",
			       NULL);
	dirname_l = strlen (dirname);

	list = NULL;
	scan = window->archive->command->file_list;
	for (; scan; scan = scan->next) {
		FileData *fd = scan->data;

		if (strncmp (fd->full_path, dirname, dirname_l) == 0)
			list = g_list_prepend (list, 
					       g_strdup (fd->original_path));
	}
	g_free (dirname);

	return g_list_reverse (list);
}


GList *
window_get_file_list_selection (FRWindow *window,
				gboolean recursive,
				gboolean *has_dirs)
{
	GList *list, *scan;

	g_return_val_if_fail (window != NULL, NULL);

	if (has_dirs != NULL)
		*has_dirs = FALSE;

	list = NULL;
        scan = GTK_CLIST (window->clist)->selection;
        for (; scan; scan = scan->next) {
                FileData *fd = gtk_clist_get_row_data (GTK_CLIST (window->clist), GPOINTER_TO_INT (scan->data));

		if (!fd)
			continue;

		if (fd->is_dir) {
			if (has_dirs != NULL)
				*has_dirs = TRUE;

			if (recursive)
				list = g_list_concat (list, get_dir_list (window, fd));
		} else
			list = g_list_prepend (list, g_strdup (fd->original_path));
        }

        return g_list_reverse (list);
}


/* -- window_get_file_list_pattern -- */


#define CASE_INSENSITIVE (1 << 4)


static gboolean
match_patterns (gchar **patterns, gchar *string)
{
	gint i;
	gint result;

	if (patterns[0] == NULL)
		return TRUE;

	if (string == NULL)
		return FALSE;

	result = FNM_NOMATCH;
	i = 0;
	while ((result != 0) && (patterns[i] != NULL)) {
		result = fnmatch (patterns[i], string, CASE_INSENSITIVE);
		i++;
	}

	return (result == 0);
}


static gchar **
search_util_get_patterns (gchar *pattern_string)
{
	gchar **patterns;
	gint i;

	patterns = g_strsplit (pattern_string, ";", 10);
	for (i = 0; patterns[i] != NULL; i++) {
		patterns[i] = g_strstrip (patterns[i]);
		
		if (strchr (patterns[i], '*') == NULL) {
			gchar *temp;
			temp = patterns[i];
			patterns[i] = g_strdup_printf ("*%s*", temp);
			g_free (temp);
		}
	}

	return patterns;
}


GList *
window_get_file_list_pattern (FRWindow *window,
			      gchar *pattern)
{
	GList *list, *scan;
	gchar **patterns;

	g_return_val_if_fail (window != NULL, NULL);

	patterns = search_util_get_patterns (pattern);

	list = NULL;
        scan = window->archive->command->file_list;
        for (; scan; scan = scan->next) {
                FileData *fd = scan->data;

		if (!fd)
			continue;

		if (match_patterns (patterns, fd->name))
			list = g_list_prepend (list, 
					       g_strdup (fd->original_path));
        }

	if (patterns != NULL) 
		g_strfreev (patterns);

        return g_list_reverse (list);
}


/* -- window_start/stop_activity_mode -- */


static void
window_progress (gfloat percent, 
                 gpointer data)
{
        FRWindow *window = data;
        gtk_progress_bar_update (GTK_PROGRESS_BAR (window->progress), percent);
}


static gint
activity_cb (gpointer data)
{
        FRWindow *window = data;

        window->activity_progress += 0.1;
        if (window->activity_progress > 1.0)
                window->activity_progress = 0.0;

        window_progress (window->activity_progress, data);

        return TRUE;
}


void
window_start_activity_mode (FRWindow *window)
{
        g_return_if_fail (window != NULL);

        if (window->activity_ref++ > 0)
                return;

        window->activity_progress = 0.0;
        gtk_progress_set_activity_mode (GTK_PROGRESS (window->progress), TRUE);
        window->activity_timeout_handle = gtk_timeout_add (ACTIVITY_DELAY,
							   activity_cb, 
							   window);
}


void
window_stop_activity_mode (FRWindow *window)
{
        g_return_if_fail (window != NULL);

        if (--window->activity_ref > 0)
                return;

        if (window->activity_timeout_handle == 0)
                return;

        gtk_timeout_remove (window->activity_timeout_handle);
        window->activity_timeout_handle = 0;
        gtk_progress_set_activity_mode (GTK_PROGRESS (window->progress), 
                                        FALSE);
        window_progress (0.0, window);
}


/* -- window_view_file -- */


typedef struct {
	FRWindow * window;
	gchar *    filename;
	gchar *    temp_dir;
} ViewData;


static void 
viewer_done (FRProcess *process,
	     FRProcError *error, 
	     gpointer callback_data)
{
	ViewData *vdata = callback_data;

	if ((vdata->temp_dir != NULL) 
	    && path_is_dir (vdata->temp_dir)) {
		char *command;
		command = g_strconcat ("rm -rf ",
				       vdata->temp_dir,
				       NULL);
		gnome_execute_shell (g_get_tmp_dir (), command); 
		g_free (command);
	}

	g_free (vdata->filename);
	g_free (vdata->temp_dir);
	g_free (vdata);

	if (process != NULL) {
		viewer_list = g_list_remove (viewer_list, process);
		gtk_object_destroy (GTK_OBJECT (process));		
	}
}


static void
view_file (FRArchive *archive,
	   FRAction action, 
	   FRProcError *error,
	   gpointer callback_data)
{
	FRProcess *proc;
	ViewData *vdata = callback_data;

	gtk_signal_disconnect_by_data (GTK_OBJECT (archive), vdata);

	if (error->type != FR_PROC_ERROR_NONE) {
		viewer_done (NULL, error, vdata);
		return;
	}

	proc = fr_process_new ();
	gtk_signal_connect (GTK_OBJECT (proc), "done",
			    GTK_SIGNAL_FUNC (viewer_done),
			    vdata);
	fr_process_use_standard_locale (proc, FALSE);
	fr_process_begin_command (proc, "fr-document-viewer");
	fr_process_add_arg (proc, vdata->filename);
	fr_process_end_command (proc);

	viewer_list = g_list_prepend (viewer_list, proc);
	fr_process_start (proc, FALSE);
}


void 
window_view_file (FRWindow *window, 
		  gchar *file)
{
	GList *file_list;
	ViewData *vdata;
	gchar *filename;

        g_return_if_fail (window != NULL);

	vdata = g_new (ViewData, 1);
	vdata->window = window;
	vdata->temp_dir = _get_temp_dir_name ();
	ensure_dir_exists (vdata->temp_dir);

	filename = g_strconcat (vdata->temp_dir,
				"/",
				file,
				NULL);
	vdata->filename = shell_escape (filename);
	g_free (filename);

	gtk_signal_connect (GTK_OBJECT (window->archive), "done",
			    GTK_SIGNAL_FUNC (view_file),
			    vdata);

	file_list = g_list_prepend (NULL, file);
	fr_archive_extract (window->archive,
			    file_list,
			    vdata->temp_dir,
			    FALSE,
			    TRUE,
			    FALSE);
	g_list_free (file_list);
}


/* -- window_open_files -- */


typedef struct {
	FRWindow * window;
	gchar *    command;
	GList *    file_list;
	gchar *    temp_dir;
} OpenData;


static void 
external_app_done (FRProcess *process,
		   FRProcError *error, 
		   gpointer callback_data)
{
	OpenData *odata = callback_data;

	if ((odata->temp_dir != NULL) 
	    && path_is_dir (odata->temp_dir)) {
		char *command;
		command = g_strconcat ("rm -rf ",
				       odata->temp_dir,
				       NULL);
		gnome_execute_shell (g_get_tmp_dir (), command); 
		g_free (command);
	}

	g_free (odata->command);
	path_list_free (odata->file_list);
	g_free (odata->temp_dir);
	g_free (odata);

	if (process != NULL) {
		viewer_list = g_list_remove (viewer_list, process);
		gtk_object_destroy (GTK_OBJECT (process));		
	}
}


static void
open_files (FRArchive *archive,
	    FRAction action, 
	    FRProcError *error,
	    gpointer callback_data)
{
	OpenData *  odata = callback_data;
	FRProcess * proc;
	GList *     scan;

	gtk_signal_disconnect_by_data (GTK_OBJECT (archive), odata);

	if (error->type != FR_PROC_ERROR_NONE) {
		external_app_done (NULL, error, odata);
		return;
	}

	proc = fr_process_new ();
	gtk_signal_connect (GTK_OBJECT (proc), "done",
			    GTK_SIGNAL_FUNC (external_app_done),
			    odata);
	fr_process_use_standard_locale (proc, FALSE);
	proc->term_on_stop = FALSE;

	fr_process_begin_command (proc, odata->command);
	for (scan = odata->file_list; scan; scan = scan->next) {
		gchar *filename = scan->data;
		fr_process_add_arg (proc, filename);
	}
	fr_process_end_command (proc);

	viewer_list = g_list_prepend (viewer_list, proc);
	fr_process_start (proc, FALSE);
}


void
window_open_files (FRWindow *window, 
		   gchar *command,
		   GList *file_list)
{
	OpenData *odata;
	GList * scan;

        g_return_if_fail (window != NULL);

	odata = g_new (OpenData, 1);
	odata->window = window;
	odata->command = g_strdup (command);
	odata->file_list = NULL;
	odata->temp_dir = _get_temp_dir_name ();
	ensure_dir_exists (odata->temp_dir);

	for (scan = file_list; scan; scan = scan->next) {
		gchar *file = scan->data;
		gchar *filename;

		filename = g_strconcat (odata->temp_dir,
					"/",
					file,
					NULL);
		odata->file_list = g_list_prepend (odata->file_list,
						   shell_escape (filename));
		g_free (filename);
	}

	gtk_signal_connect (GTK_OBJECT (window->archive), "done",
			    GTK_SIGNAL_FUNC (open_files),
			    odata);

	fr_archive_extract (window->archive,
			    file_list,
			    odata->temp_dir,
			    FALSE,
			    TRUE,
			    FALSE);
}


void
window_view_or_open_file (FRWindow *window, 
			  gchar *filename)
{
	GnomeVFSMimeAction * action;
	const char *         mime_type;
	char *               command;
	GList *              singleton;
	
	mime_type = gnome_vfs_mime_type_from_name_or_default (filename, NULL);
	if (mime_type == NULL) {
		window_view_file (window, filename);
		return;
	}
		
	action = gnome_vfs_mime_get_default_action (mime_type);
		
	if (action == NULL) {
		dlg_viewer_or_app (window, filename);
		return;
	}
		
	switch (action->action_type) {
	case GNOME_VFS_MIME_ACTION_TYPE_NONE:
	case GNOME_VFS_MIME_ACTION_TYPE_COMPONENT:
		window_view_file (window, filename);
		break;
		
	case GNOME_VFS_MIME_ACTION_TYPE_APPLICATION:
		command = application_get_command (action->action.application);
		singleton = g_list_append (NULL, filename);
		window_open_files (window, command, singleton);
		g_list_free (singleton);
		g_free (command);
		break;
	}
	gnome_vfs_mime_action_free (action);
}


void
window_set_default_dir (FRWindow *window,
			gchar *default_dir)
{
	g_return_if_fail (window != NULL);
	g_return_if_fail (default_dir != NULL);

	if (window->open_default_dir != NULL) 
		g_free (window->open_default_dir);
	window->open_default_dir = g_strdup (default_dir);

	if (window->add_default_dir != NULL) 
		g_free (window->add_default_dir);
	window->add_default_dir = g_strdup (default_dir);

	if (window->extract_default_dir != NULL) 
		g_free (window->extract_default_dir);
	window->extract_default_dir = g_strdup (default_dir);
}


void
window_update_columns_visibility (FRWindow *window)
{
	gtk_clist_set_column_visibility (GTK_CLIST (window->clist),
					 COLUMN_NAME,
					 preferences.show_name);
	gtk_clist_set_column_visibility (GTK_CLIST (window->clist),
					 COLUMN_TYPE,
					 preferences.show_type);
	gtk_clist_set_column_visibility (GTK_CLIST (window->clist),
					 COLUMN_SIZE,
					 preferences.show_size);
	gtk_clist_set_column_visibility (GTK_CLIST (window->clist),
					 COLUMN_TIME,
					 preferences.show_time);
	gtk_clist_set_column_visibility (GTK_CLIST (window->clist),
					 COLUMN_PATH,
					 preferences.show_path);
}




/* -- batch mode procedures -- */


void
window_batch_mode_clear (FRWindow *window)
{
	g_return_if_fail (window != NULL);
	_window_free_batch_data (window);
}


void
window_batch_mode_add_action (FRWindow *window,
			      FRBatchAction action,
			      void *data,
			      GFreeFunc free_func)
{
	FRBatchActionDescription *a_desc;

	g_return_if_fail (window != NULL);

	a_desc = g_new (FRBatchActionDescription, 1);
	a_desc->action    = action;
	a_desc->data      = data;
	a_desc->free_func = free_func;

	window->batch_action_list = g_list_append (window->batch_action_list,
						   a_desc);
}


typedef struct {
	gchar *archive_name;
	GList *file_list;
} OpenAndAddData;


void
open_and_add_data_free (OpenAndAddData *adata)
{
	if (adata == NULL)
		return;

	if (adata->archive_name != NULL)
		g_free (adata->archive_name);
	g_free (adata);
}


static void
_window_batch_start_current_action (FRWindow *window)
{
	FRBatchActionDescription *action;
	OpenAndAddData *adata;

	if (window->batch_action == NULL) {
		window->batch_mode = FALSE;
		return;
	}

	action = (FRBatchActionDescription *) window->batch_action->data;
	switch (action->action) {
	case FR_BATCH_ACTION_OPEN:
		window_archive_open (window, (gchar*) action->data);
		break;

	case FR_BATCH_ACTION_OPEN_AND_ADD:
		adata = (OpenAndAddData *) action->data;
		if (! path_is_file (adata->archive_name)) {
			window_batch_mode_add_action (window,
						      FR_BATCH_ACTION_QUIT,
						      NULL,
						      NULL);

			if (window->dropped_file_list != NULL)
				path_list_free (window->dropped_file_list);
			window->dropped_file_list = path_list_dup (adata->file_list);
			window->add_dropped_files = TRUE;
			window_archive_new (window, adata->archive_name);
		} else {
			window_batch_mode_add_action (window,
						      FR_BATCH_ACTION_ADD,
						      path_list_dup (adata->file_list),
						      (GFreeFunc) path_list_free);
			window_batch_mode_add_action (window,
						      FR_BATCH_ACTION_QUIT,
						      NULL,
						      NULL);
			window_archive_open (window, adata->archive_name);
		}
		break;

	case FR_BATCH_ACTION_ADD:
		if (window->dropped_file_list != NULL)
			path_list_free (window->dropped_file_list);
		window->dropped_file_list = path_list_dup ((GList*) action->data);

		drag_drop_add_file_list (window);
		break;

	case FR_BATCH_ACTION_ADD_INTERACT:
		dlg_batch_add_files (window, (GList*) action->data);
		break;

	case FR_BATCH_ACTION_EXTRACT:
		window_archive_extract (window,
					NULL,
					(gchar*) action->data,
					FALSE,
					TRUE,
					FALSE);
		break;

	case FR_BATCH_ACTION_EXTRACT_INTERACT:
		dlg_extract (NULL, window);
		break;

	case FR_BATCH_ACTION_QUIT:
		window_close (window);
		break;
	}
}


void
window_batch_mode_start (FRWindow *window)
{
	g_return_if_fail (window != NULL);

	if (window->batch_action_list == NULL)
		return;

	window->batch_mode = TRUE;
	window->batch_action = window->batch_action_list;
	_window_batch_start_current_action (window);
}


void
window_batch_mode_stop (FRWindow *window)
{
	window->batch_mode = FALSE;
}


void
window_archive__open_extract_close (FRWindow *window, 
				    char *filename,
				    char *dest_dir)
{
	window_batch_mode_clear (window);

	window_batch_mode_add_action (window,
				      FR_BATCH_ACTION_OPEN,
				      g_strdup (filename),
				      (GFreeFunc) g_free);
	if (dest_dir != NULL) 
		window_batch_mode_add_action (window,
					      FR_BATCH_ACTION_EXTRACT,
					      g_strdup (dest_dir),
					      (GFreeFunc) g_free);
	else
		window_batch_mode_add_action (window,
					      FR_BATCH_ACTION_EXTRACT_INTERACT,
					      NULL,
					      NULL);
	window_batch_mode_add_action (window,
				      FR_BATCH_ACTION_QUIT,
				      NULL,
				      NULL);

	window_batch_mode_start (window);	
}


void
window_archive__open_add_close (FRWindow *window, 
				char *archive,
				GList *file_list)
{
	window_batch_mode_clear (window);

	if (archive != NULL) {
		OpenAndAddData *adata;

		adata = g_new (OpenAndAddData, 1);
		adata->archive_name = g_strdup (archive);
		adata->file_list = file_list;
		window_batch_mode_add_action (window,
					      FR_BATCH_ACTION_OPEN_AND_ADD,
					      adata,
					      (GFreeFunc) open_and_add_data_free);
	} else 
		window_batch_mode_add_action (window,
					      FR_BATCH_ACTION_ADD_INTERACT,
					      file_list,
					      NULL);

	window_batch_mode_start (window);
}
