(* $Id: metacache.ml,v 1.8 2001/07/24 20:00:08 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)

type meta =
    { package_name : string;
      package_dir : string;
      meta_file : (string * (string list * string)) list;
    }
;;


module Metaentry =
  struct
    type t = meta
    type id_t = string
    let id m = m.package_name
  end
;;


module Metastore =
  Topo.Make(Metaentry)
;;


let ocamlpath = ref [];;
let ocamlstdlib = ref "";;

let store = Metastore.create();;


let init_cache path stdlib =
  ocamlpath := path;
  ocamlstdlib := stdlib
;;


let get_entry package package_dir meta_file =
  (* Parse the META file: *)
  let ch = open_in meta_file in
  try
    let mf = Metascanner.parse ch in
    let d =
      try
	Metascanner.lookup "directory" [] mf
      with
	  Not_found -> package_dir
    in
    let d' =
      if d = "" then
	package_dir
      else
	match d.[0] with
          | '^' 
	  | '+' -> Filename.concat
	      !ocamlstdlib
	      (String.sub d 1 (String.length d - 1))
	  | _ -> d
    in
    
    let e =
      { package_name = package;
	package_dir = d';
	meta_file = mf;
      } in
    close_in ch;
    e
  with
      Failure s ->
	close_in ch;
	failwith ("While parsing '" ^ meta_file ^ "': " ^ s)
    | Stream.Error s ->
	close_in ch;
	failwith ("While parsing '" ^ meta_file ^ "': " ^ s)
    | any ->
	close_in ch;
	raise any
;;


let query package =
  (* returns the 'meta' entry for 'package' or raises Not_found. *)

  let rec run_ocamlpath path =
    match path with
      [] -> raise Not_found
    | dir :: path' ->
	let package_dir = Filename.concat dir package in
	let meta_file_1 = Filename.concat package_dir "META" in
	let meta_file_2 = Filename.concat dir ("META." ^ package) in
	if Sys.file_exists meta_file_1 then begin
	  let entry = get_entry package package_dir meta_file_1 in
	  Metastore.add store entry;
	  entry
	end
	else
	  if Sys.file_exists meta_file_2 then begin
	    let entry = get_entry package package_dir meta_file_2 in
	    Metastore.add store entry;
	    entry
	  end
	  else
	    run_ocamlpath path'
  in

  try
    Metastore.find store package
  with
    Not_found ->
      run_ocamlpath !ocamlpath
;;


let package_definitions package =
  (* Return all META files defining this package
   *)
  let rec run_ocamlpath path =
    match path with
      [] -> []
    | dir :: path' ->
	let package_dir = Filename.concat dir package in
	let meta_file_1 = Filename.concat package_dir "META" in
	let meta_file_2 = Filename.concat dir ("META." ^ package) in
	if Sys.file_exists meta_file_1 then begin
	  meta_file_1 :: run_ocamlpath path'
	end
	else
	  if Sys.file_exists meta_file_2 then begin
	    meta_file_2 :: run_ocamlpath path'
	  end
	  else
	    run_ocamlpath path'
  in
  run_ocamlpath !ocamlpath
;;


let package_conflict_report() =
  Metastore.iter_up
    (fun pkg ->
       let c = package_definitions pkg.package_name in
       match c with
	   [] 
	 | [_] ->
	     ()
	 | _ ->
	     Printf.eprintf "WARNING: Package %s has multiple definitions in %s\n"
	       pkg.package_name
	       (String.concat ", " c)
    )
    store;
  flush stderr
;;


let requires plist package =
  (* returns names of package required by 'package'. It is checked that
   * the packages really exist.
   * 'plist': list of true predicates
   * - raises Not_found if there is no 'package'
   * - raises Failure if some of the ancestors do not exist
   *)
  let m = query package in
  let r =
    try Metascanner.lookup "requires" plist m.meta_file
	with Not_found -> ""
  in
  let ancestors = Split.in_words r in
  List.iter
    (fun p ->
      try
	let _ = query p in
	Metastore.let_le store p package
      with
	Not_found ->
	  failwith ("Findlib: package '" ^ p ^ "' not found (required by '" ^
		    package ^ "')")
      | Topo.Inconsistent_ordering ->
	  failwith ("Findlib: package '" ^ p ^ "' required by itself"))
    ancestors;
  ancestors
;;


let requires_deeply plist package_list =
  (* returns names of packages required by the packages in 'package_list',
   * either directly or indirectly.
   * It is checked that the packages really exist.
   * The list of names is sorted topologically; first comes the deepest
   * ancestor, last 'package' itself.
   * 'plist': list of true predicates
   * - raises Not_found if there is no 'package'
   * - raises Failure if some of the ancestors do not exist
   *)

  let done_pkgs = ref [] in

  let rec enter_packages pkglist =
    match pkglist with
      pkg :: pkglist' ->
	if not(List.mem pkg !done_pkgs) then begin
	  let pkg_ancestors = requires plist pkg in
	  done_pkgs := pkg :: !done_pkgs;
          enter_packages pkg_ancestors
	end;
	enter_packages pkglist'
    | [] ->
	()
  in

  enter_packages package_list;

  let l = ref [] in

  Metastore.iter_up_at
    (fun m ->
      l := m.package_name :: !l)
    store
    package_list;

  List.rev !l
;;

(* There are some more functions in Metacache_unix; however these are
 * not contained in findlib.cmx?a
 *)


(* ======================================================================
 * History:
 *
 * $Log: metacache.ml,v $
 * Revision 1.8  2001/07/24 20:00:08  gerd
 * 	Support for dynamically set stdlib
 *
 * Revision 1.7  2001/03/03 19:28:34  gerd
 * 	Added conflict reports.
 *
 * Revision 1.6  2001/02/24 20:21:45  gerd
 * 	New function get_entry; it was previously contained in
 * the function "query".
 * 	Function "users" moved to Metacache_unix.
 *
 * Revision 1.5  2000/02/28 20:20:38  gerd
 * 	Bugfix: The recursive collection of dependency had a strange
 * bug; it did not find all ascendents.
 *
 * Revision 1.4  1999/06/26 15:01:50  gerd
 * 	Added the -descendants option.
 *
 * Revision 1.3  1999/06/24 20:17:52  gerd
 * 	Further modifications (dont know which...)
 *
 * Revision 1.2  1999/06/20 22:23:18  gerd
 * 	Works now with the core libraries.
 *
 * Revision 1.1  1999/06/20 19:26:26  gerd
 * 	Major change: Added support for META files. In META files, knowlege
 * about compilation options, and dependencies on other packages can be stored.
 * The "ocamlfind query" subcommand has been extended in order to have a
 * direct interface for that. "ocamlfind ocamlc/ocamlopt/ocamlmktop/ocamlcp"
 * subcommands have been added to simplify the invocation of the compiler.
 *
 *
 *)
