# -*- Mode: Python; test-case-name: flumotion.test.test_pb -*-
# vi:si:et:sw=4:sts=4:ts=4
#
# Flumotion - a streaming media server
# Copyright (C) 2004,2005,2006,2007 Fluendo, S.L. (www.fluendo.com).
# All rights reserved.

# This file may be distributed and/or modified under the terms of
# the GNU General Public License version 2 as published by
# the Free Software Foundation.
# This file is distributed without any warranty; without even the implied
# warranty of merchantability or fitness for a particular purpose.
# See "LICENSE.GPL" in the source distribution for more information.

# Licensees having purchased or holding a valid Flumotion Advanced
# Streaming Server license may use this file in accordance with the
# Flumotion Advanced Streaming Server Commercial License Agreement.
# See "LICENSE.Flumotion" in the source distribution for more information.

# Headers in this file shall remain intact.

"""
Flumotion Perspective Broker using keycards

Inspired by L{twisted.spread.pb}
"""

import time

from twisted.cred import checkers, credentials, error
from twisted.cred.portal import IRealm, Portal
from twisted.internet import protocol, defer, reactor
from twisted.python import log, reflect, failure
from twisted.spread import pb, flavors
from twisted.spread.pb import PBClientFactory

from flumotion.configure import configure
from flumotion.common import keycards, interfaces, common, errors
from flumotion.common import log as flog
from flumotion.twisted import reflect as freflect
from flumotion.twisted import credentials as fcredentials
from flumotion.twisted.compat import implements
# TODO:
#   merge FMCF back into twisted

### Keycard-based FPB objects

# we made three changes to the standard PBClientFactory:
# 1) the root object has a getKeycardClasses() call that the server
#    uses to tell clients about the interfaces it supports
# 2) you can request a specific interface for the avatar to
#    implement, instead of only IPerspective
# 3) you send in a keycard, on which you can set a preference for an avatarId
# this way you can request a different avatarId than the user you authenticate
# with, or you can login without a username

class FPBClientFactory(pb.PBClientFactory, flog.Loggable):
    """
    I am an extended Perspective Broker client factory using generic
    keycards for login.


    @ivar keycard:              the keycard used last for logging in; set after
                                self.login has completed
    @type keycard:              L{keycards.Keycard}
    @ivar medium:               the client-side referenceable for the PB server
                                to call on, and for the client to call to the
                                PB server
    @type medium:               L{flumotion.common.medium.BaseMedium}
    @ivar perspectiveInterface: the interface we want to request a perspective
                                for
    @type perspectiveInterface: subclass of
                                L{flumotion.common.interfaces.IMedium}
    """
    logCategory = "FPBClientFactory"
    keycard = None
    medium = None
    perspectiveInterface = None # override in subclass

    def getKeycardClasses(self):
        """
        Ask the remote PB server for all the keycard interfaces it supports.

        @rtype: L{defer.Deferred} returning list of str
        """
        def getRootObjectCb(root):
            return root.callRemote('getKeycardClasses')

        d = self.getRootObject()
        d.addCallback(getRootObjectCb)
        return d
        
    def login(self, authenticator):
        """
        Login, respond to challenges, and eventually get perspective
        from remote PB server.

        Currently only credentials implementing IUsernamePassword are
        supported.

        @return: Deferred of RemoteReference to the perspective.
        """
        assert authenticator, "I really do need an authenticator"
        assert not isinstance(authenticator, keycards.Keycard)
        interfaces = []
        if self.perspectiveInterface:
            self.debug('perspectiveInterface is %r' % self.perspectiveInterface)
            interfaces.append(self.perspectiveInterface)
        else:
            self.warning('No perspectiveInterface set on %r' % self)
        if not pb.IPerspective in interfaces:
            interfaces.append(pb.IPerspective)
        interfaces = [reflect.qual(interface)
                          for interface in interfaces]
            
        def getKeycardClassesCb(keycardClasses):
            self.debug('supported keycard classes: %r' % keycardClasses)
            d = authenticator.issue(keycardClasses)
            return d

        def issueCb(keycard):
            self.keycard = keycard
            self.debug('using keycard: %r' % self.keycard)
            return self.keycard

        d = self.getKeycardClasses()
        d.addCallback(getKeycardClassesCb)
        d.addCallback(issueCb)
        d.addCallback(lambda r: self.getRootObject())
        d.addCallback(self._cbSendKeycard, authenticator, self.medium,
            interfaces)
        return d

    # we are a different kind of PB client, so warn
    def _cbSendUsername(self, root, username, password, avatarId, client, interfaces):
        self.warning("you really want to use cbSendKeycard")

        
    def _cbSendKeycard(self, root, authenticator, client, interfaces, count=0):
        self.debug("_cbSendKeycard(root=%r, authenticator=%r, client=%r, " \
            "interfaces=%r, count=%d" % (
            root, authenticator, client, interfaces, count))
        count = count + 1
        d = root.callRemote("login", self.keycard, client, *interfaces)
        return d.addCallback(self._cbLoginCallback, root, authenticator, client,
            interfaces, count)

    # we can get either a keycard, None (?) or a remote reference
    def _cbLoginCallback(self, result, root, authenticator, client, interfaces,
        count):
        if count > 5:
            # too many recursions, server is h0rked
            self.warning('Too many recursions, internal error.')
        self.debug("FPBClientFactory(): result %r" % result)

        if not result:
            self.warning('No result, raising.')
            raise error.UnauthorizedLogin()

        if isinstance(result, pb.RemoteReference):
            # everything done, return reference
            self.debug('Done, returning result %r' % result)
            return result

        # must be a keycard
        keycard = result
        if not keycard.state == keycards.AUTHENTICATED:
            self.debug("FPBClientFactory(): requester needs to resend %r" %
                keycard)
            d = authenticator.respond(keycard)
            def _loginAgainCb(keycard):
                d = root.callRemote("login", keycard, client, *interfaces)
                return d.addCallback(self._cbLoginCallback, root, authenticator,
                    client, interfaces, count)
            d.addCallback(_loginAgainCb)
            return d

        self.debug("FPBClientFactory(): authenticated %r" % keycard)
        return keycard

class ReconnectingPBClientFactory(pb.PBClientFactory, flog.Loggable,
                                  protocol.ReconnectingClientFactory):
    """
    Reconnecting client factory for normal PB brokers.

    Users of this factory call startLogin to start logging in, and should
    override getLoginDeferred to get the deferred returned from the PB server
    for each login attempt.
    """

    def __init__(self):
        pb.PBClientFactory.__init__(self)
        self._doingLogin = False

    def clientConnectionFailed(self, connector, reason):
        log.msg("connection failed, reason %r" % reason)
        pb.PBClientFactory.clientConnectionFailed(self, connector, reason)
        RCF = protocol.ReconnectingClientFactory
        RCF.clientConnectionFailed(self, connector, reason)

    def clientConnectionLost(self, connector, reason):
        log.msg("connection lost, reason %r" % reason)
        pb.PBClientFactory.clientConnectionLost(self, connector, reason,
                                             reconnecting=True)
        RCF = protocol.ReconnectingClientFactory
        RCF.clientConnectionLost(self, connector, reason)

    def clientConnectionMade(self, broker):
        log.msg("connection made")
        self.resetDelay()
        pb.PBClientFactory.clientConnectionMade(self, broker)
        if self._doingLogin:
            d = self.login(self._credentials, self._client)
            self.gotDeferredLogin(d)

    def startLogin(self, credentials, client=None):
        self._credentials = credentials
        self._client = client

        self._doingLogin = True
        
    # methods to override
    def gotDeferredLogin(self, deferred):
        """
        The deferred from login is now available.
        """
        raise NotImplementedError

class ReconnectingFPBClientFactory(FPBClientFactory,
                                   protocol.ReconnectingClientFactory):
    """
    Reconnecting client factory for FPB brokers (using keycards for login).

    Users of this factory call startLogin to start logging in.
    Override getLoginDeferred to get a handle to the deferred returned
    from the PB server.
    """

    def __init__(self):
        FPBClientFactory.__init__(self)
        self._doingLogin = False
        self._doingGetPerspective = False
        
    def clientConnectionFailed(self, connector, reason):
        log.msg("connection failed, reason %r" % reason)
        FPBClientFactory.clientConnectionFailed(self, connector, reason)
        RCF = protocol.ReconnectingClientFactory
        RCF.clientConnectionFailed(self, connector, reason)

    def clientConnectionLost(self, connector, reason):
        log.msg("connection lost, reason %r" % reason)
        FPBClientFactory.clientConnectionLost(self, connector, reason,
                                             reconnecting=True)
        RCF = protocol.ReconnectingClientFactory
        RCF.clientConnectionLost(self, connector, reason)

    def clientConnectionMade(self, broker):
        log.msg("connection made")
        self.resetDelay()
        FPBClientFactory.clientConnectionMade(self, broker)
        if self._doingLogin:
            d = self.login(self._authenticator)
            self.gotDeferredLogin(d)

    # TODO: This is a poorly named method; it just provides the appropriate
    # authentication information, and doesn't actually _start_ login at all.
    def startLogin(self, authenticator):
        assert not isinstance(authenticator, keycards.Keycard)
        self._authenticator = authenticator
        self._doingLogin = True
        
    # methods to override
    def gotDeferredLogin(self, deferred):
        """
        The deferred from login is now available.
        """
        raise NotImplementedError

### FIXME: this code is an adaptation of twisted/spread/pb.py
# it allows you to login to a FPB server requesting interfaces other than
# IPerspective.
# in other terms, you can request different "kinds" of avatars from the same
# PB server.
# this code needs to be sent upstream to Twisted
class _FPortalRoot:
    """
    Root object, used to login to bouncer.
    """

    implements(flavors.IPBRoot)
    
    def __init__(self, bouncerPortal):
        """
        @type bouncerPortal: L{flumotion.twisted.portal.BouncerPortal}
        """
        self.bouncerPortal = bouncerPortal

    def rootObject(self, broker):
        return _BouncerWrapper(self.bouncerPortal, broker)

class _BouncerWrapper(pb.Referenceable, flog.Loggable):

    logCategory = "_BouncerWrapper"

    def __init__(self, bouncerPortal, broker):
        self.bouncerPortal = bouncerPortal
        self.broker = broker

    def remote_getKeycardClasses(self):
        """
        @returns: the fully-qualified class names of supported keycard
                  interfaces
        @rtype:   L{defer.Deferred} firing list of str
        """
        return self.bouncerPortal.getKeycardClasses()

    def remote_login(self, keycard, mind, *interfaces):
        """
        Start of keycard login.

        @param interfaces: list of fully qualified names of interface objects

        @returns: one of
            - a L{flumotion.common.keycards.Keycard} when more steps
              need to be performed
            - a L{twisted.spread.pb.AsReferenceable} when authentication 
              has succeeded, which will turn into a
              L{twisted.spread.pb.RemoteReference} on the client side
            - a L{twisted.cred.error.UnauthorizedLogin} when authentication
              is denied
        """
        # corresponds with FPBClientFactory._cbSendKeycard
        self.log("remote_login(keycard=%s, *interfaces=%r" % (keycard, interfaces))
        interfaces = [freflect.namedAny(interface) for interface in interfaces]
        d = self.bouncerPortal.login(keycard, mind, *interfaces)
        d.addCallback(self._authenticateCallback, mind, *interfaces)
        return d

    def _authenticateCallback(self, result, mind, *interfaces):
        self.log("_authenticateCallback(result=%r, mind=%r, interfaces=%r" % (result, mind, interfaces))
        # FIXME: coverage indicates that "not result" does not happen,
        # presumably because a Failure is triggered before us
        if not result:
            return failure.Failure(error.UnauthorizedLogin())

        # if the result is a keycard, we're not yet ready
        if isinstance(result, keycards.Keycard):
            return result

        # authenticated, so the result is the tuple
        # FIXME: our keycard should be stored higher up since it was authd
        # then cleaned up sometime in the future
        # for that we probably need to pass it along
        return self._loggedIn(result)

    def _loggedIn(self, (interface, perspective, logout)):
        self.broker.notifyOnDisconnect(logout)
        return pb.AsReferenceable(perspective, "perspective")

class Authenticator(flog.Loggable, pb.Referenceable):
    """
    I am an object used by FPB clients to create keycards for me
    and respond to challenges.

    I encapsulate keycard-related data, plus secrets which are used locally
    and not put on the keycard.

    I can be serialized over PB connections to a RemoteReference and then
    adapted with RemoteAuthenticator to present the same interface.

    @cvar username: a username to log in with
    @type username: str
    @cvar password: a password to log in with
    @type password: str
    @cvar address:  an address to log in from
    @type address:  str
    @cvar avatarId: the avatarId we want to request from the PB server
    @type avatarId: str
    """
    logCategory = "authenticator"

    avatarId = None

    username = None
    password = None
    address = None
    # FIXME: we can add ssh keys and similar here later on

    def __init__(self, **kwargs):
        for key in kwargs:
            setattr(self, key, kwargs[key])

    def issue(self, keycardClasses):
        """
        Issue a keycard that implements one of the given interfaces.

        @param keycardClasses: list of fully qualified keycard classes
        @type  keycardClasses: list of str

        @rtype: L{defer.Deferred} firing L{keycards.Keycard}
        """
        # this method returns a deferred so we present the same interface
        # as the RemoteAuthenticator adapter
    
        # construct a list of keycard interfaces we can support right now
        supported = []
        # address is allowed to be None
        if self.username is not None and self.password is not None:
            # We only want to support challenge-based keycards, for
            # security. Maybe later we want this to be configurable
            # supported.append(keycards.KeycardUACPP)
            supported.append(keycards.KeycardUACPCC)
            supported.append(keycards.KeycardUASPCC)

        # expand to fully qualified names
        supported = [reflect.qual(k) for k in supported]

        for i in keycardClasses:
            if i in supported:
                self.debug('Keycard interface %s supported, looking up' % i)
                name = i.split(".")[-1]
                methodName = "issue_%s" % name
                method = getattr(self, methodName)
                keycard = method()
                self.debug('Issuing keycard %r of class %s' % (
                    keycard, name))
                keycard.avatarId = self.avatarId
                return defer.succeed(keycard)

        self.debug('Could not issue a keycard')
        return defer.succeed(None)

    # non-challenge types
    def issue_KeycardUACPP(self):
        return keycards.KeycardUACPP(self.username, self.password,
            self.address)

    # challenge types
    def issue_KeycardUACPCC(self):
        return keycards.KeycardUACPCC(self.username, self.address)

    def issue_KeycardUASPCC(self):
        return keycards.KeycardUASPCC(self.username, self.address)

    def respond(self, keycard):
        """
        Respond to a challenge on the given keycard, based on the secrets
        we have.

        @param keycard: the keycard with the challenge to respond to
        @type  keycard: L{keycards.Keycard}

        @rtype:   L{defer.Deferred} firing a {keycards.Keycard}
        @returns: a deferred firing the keycard with a response set
        """
        self.debug('responding to challenge on keycard %r' % keycard)
        methodName = "respond_%s" % keycard.__class__.__name__
        method = getattr(self, methodName)
        return defer.succeed(method(keycard))

    def respond_KeycardUACPCC(self, keycard):
        self.debug('setting password')
        keycard.setPassword(self.password)
        return keycard

    def respond_KeycardUASPCC(self, keycard):
        self.debug('setting password')
        keycard.setPassword(self.password)
        return keycard

    ### pb.Referenceable methods
    def remote_issue(self, interfaces):
        return self.issue(interfaces)

    def remote_respond(self, keycard):
        return self.respond(keycard)

class RemoteAuthenticator:
    """
    I am an adapter for a pb.RemoteReference to present the same interface
    as L{Authenticator}
    """

    avatarId = None # not serialized
    username = None # for convenience, will always be None
    password = None # for convenience, will always be None

    def __init__(self, remoteReference):
        self._remote = remoteReference

    def issue(self, interfaces):
        def issueCb(keycard):
            keycard.avatarId = self.avatarId
            return keycard

        d = self._remote.callRemote('issue', interfaces)
        d.addCallback(issueCb)
        return d

    def respond(self, keycard):
        return self._remote.callRemote('respond', keycard)


class Referenceable(pb.Referenceable, flog.Loggable):
    """
    @cvar remoteLogName: name to use to log the other side of the connection
    @type remoteLogName: str
    """
    logCategory = 'referenceable'
    remoteLogName = 'remote'


    # a referenceable that logs receiving remote messages
    def remoteMessageReceived(self, broker, message, args, kwargs):
        args = broker.unserialize(args)
        kwargs = broker.unserialize(kwargs)
        method = getattr(self, "remote_%s" % message, None)
        if method is None:
            raise pb.NoSuchMethod("No such method: remote_%s" % (message,))

        level = flog.DEBUG
        if message == 'ping': level = flog.LOG

        debugClass = self.logCategory.upper()
        # all this malarkey is to avoid actually interpolating variables
        # if it is not needed
        startArgs = [self.remoteLogName, debugClass, message]
        format, debugArgs = flog.getFormatArgs(
            '%s --> %s: remote_%s(', startArgs,
            ')', (), args, kwargs)
        # log going into the method
        logKwArgs = self.doLog(level, method, format, *debugArgs)

        # invoke the remote_ method
        try:
            state = method(*args, **kwargs)
        except TypeError:
            self.warning("%s didn't accept %s and %s" % (method, args, kwargs))
            raise

        # log coming out of the method
        if isinstance(state, defer.Deferred):
            # for a deferred, we currently can't log a better location than
            # the def line for the function/instance that we called above
            def callback(result):
                format, debugArgs = flog.getFormatArgs(
                    '%s <-- %s: remote_%s(', startArgs,
                    '): %r', (flog.ellipsize(result), ), args, kwargs)
                self.doLog(level, -1, format, *debugArgs, **logKwArgs)
                return result
            def errback(failure):
                format, debugArgs = flog.getFormatArgs(
                    '%s <-- %s: remote_%s(', startArgs,
                    '): failure %r', (failure, ), args, kwargs)
                self.doLog(level, -1, format, *debugArgs, **logKwArgs)
                return failure

            state.addCallback(callback)
            state.addErrback(errback)
        else:
            format, debugArgs = flog.getFormatArgs(
                '%s <-- %s: remote_%s(', startArgs,
                '): %r', (flog.ellipsize(state), ), args, kwargs)
            self.doLog(level, -1, format, *debugArgs, **logKwArgs)

        return broker.serialize(state, self.perspective)

class Avatar(pb.Avatar, flog.Loggable):
    """
    @cvar remoteLogName: name to use to log the other side of the connection
    @type remoteLogName: str
    """
    logCategory = 'avatar'
    remoteLogName = 'remote'

    def __init__(self, avatarId):
        self.avatarId = avatarId
        self.logName = avatarId
        self.mind = None
        self.debug("created new Avatar with id %s", avatarId)

    # a referenceable that logs receiving remote messages
    def perspectiveMessageReceived(self, broker, message, args, kwargs):
        args = broker.unserialize(args)
        kwargs = broker.unserialize(kwargs)
        method = getattr(self, "perspective_%s" % message, None)
        if method is None:
            raise pb.NoSuchMethod("No such method: perspective_%s" % (message,))

        level = flog.DEBUG
        if message == 'ping': level = flog.LOG
        debugClass = self.logCategory.upper()
        startArgs = [self.remoteLogName, debugClass, message]
        format, debugArgs = flog.getFormatArgs(
            '%s --> %s: perspective_%s(', startArgs,
            ')', (), args, kwargs)
        # log going into the method
        logKwArgs = self.doLog(level, method, format, *debugArgs)

        # invoke the perspective_ method
        try:
            state = method(*args, **kwargs)
        except TypeError:
            self.debug("%s didn't accept %s and %s" % (method, args, kwargs))
            raise
        except pb.Error, e:
            format, debugArgs = flog.getFormatArgs(
                '%s <-- %s: perspective_%s(', startArgs,
                '): pb.Error %r', (e, ), args, kwargs)
            self.doLog(level, -1, format, *debugArgs, **logKwArgs)
            raise e

        # log coming out of the method
        if isinstance(state, defer.Deferred):
            # for a deferred, we currently can't log a better location than
            # the def line for the function/instance that we called above
            def callback(result):
                format, debugArgs = flog.getFormatArgs(
                    '%s <-- %s: perspective_%s(', startArgs,
                    '): %r', (flog.ellipsize(result), ), args, kwargs)
                self.doLog(level, -1, format, *debugArgs, **logKwArgs)
                return result
            def errback(failure):
                format, debugArgs = flog.getFormatArgs(
                    '%s <-- %s: perspective_%s(', startArgs,
                    '): failure %r', (failure, ), args, kwargs)
                self.doLog(level, -1, format, *debugArgs, **logKwArgs)
                return failure

            state.addCallback(callback)
            state.addErrback(errback)
        else:
            format, debugArgs = flog.getFormatArgs(
                '%s <-- %s: perspective_%s(', startArgs,
                '): %r', (flog.ellipsize(state), ), args, kwargs)
            self.doLog(level, -1, format, *debugArgs, **logKwArgs)

        return broker.serialize(state, self, method, args, kwargs)

    def setMind(self, mind):
        """
        Tell the avatar that the given mind has been attached.
        This gives the avatar a way to call remotely to the client that
        requested this avatar.
        This is scheduled by the portal after the client has logged in.

        @type mind: L{twisted.spread.pb.RemoteReference}
        """
        self.mind = mind
        def nullMind(x):
            self.debug('%r: disconnected from %r' % (self, self.mind))
            self.mind = None
        self.mind.notifyOnDisconnect(nullMind)

        transport = self.mind.broker.transport
        tarzan = transport.getHost()
        jane = transport.getPeer()
        if tarzan and jane:
            self.debug("PB client connection seen by me is from me %s to %s" % (
                common.addressGetHost(tarzan),
                common.addressGetHost(jane)))
        self.log('Client attached is mind %s', mind)

    def mindCallRemoteLogging(self, level, stackDepth, name, *args,
                              **kwargs):
        """
        Call the given remote method, and log calling and returning nicely.

        @param level: the level we should log at (log.DEBUG, log.INFO, etc)
        @type  level: int
        @param stackDepth: the number of stack frames to go back to get
        file and line information, negative or zero.
        @type  stackDepth: non-positive int
        @param name: name of the remote method
        @type  name: str
        """
        if level is not None:
            debugClass = str(self.__class__).split(".")[-1].upper()
            startArgs = [self.remoteLogName, debugClass, name]
            format, debugArgs = flog.getFormatArgs(
                '%s --> %s: callRemote(%s, ', startArgs,
                ')', (), args, kwargs)
            logKwArgs = self.doLog(level, stackDepth - 1, format,
                                   *debugArgs)

        if not self.mind:
            self.warning('Tried to mindCallRemote(%s), but we are '
                         'disconnected', name)
            return defer.fail(errors.NotConnectedError())

        def callback(result):
            format, debugArgs = flog.getFormatArgs(
                '%s <-- %s: callRemote(%s, ', startArgs,
                '): %r', (flog.ellipsize(result), ), args, kwargs)
            self.doLog(level, -1, format, *debugArgs, **logKwArgs)
            return result

        def errback(failure):
            format, debugArgs = flog.getFormatArgs(
                '%s <-- %s: callRemote(%s', startArgs,
                '): %r', (failure, ), args, kwargs)
            self.doLog(level, -1, format, *debugArgs, **logKwArgs)
            return failure

        d = self.mind.callRemote(name, *args, **kwargs)
        if level is not None:
            d.addCallbacks(callback, errback)
        return d

    def mindCallRemote(self, name, *args, **kwargs):
        """
        Call the given remote method, and log calling and returning nicely.

        @param name: name of the remote method
        @type  name: str
        """
        return self.mindCallRemoteLogging(flog.DEBUG, -1, name, *args,
                                          **kwargs)

    def disconnect(self):
        """
        Disconnect the remote PB client. If we are already disconnected,
        do nothing.
        """
        if self.mind:
            return self.mind.broker.transport.loseConnection()

class PingableAvatar(Avatar):
    _pingCheckInterval = configure.heartbeatInterval * 2.5

    def perspective_ping(self):
        self._lastPing = time.time()
        return defer.succeed(True)

    def startPingChecking(self, disconnect):
        self._lastPing = time.time()
        self._pingCheckDisconnect = disconnect
        self._pingCheck()

    def _pingCheck(self):
        self._pingCheckDC = None
        if time.time() - self._lastPing > self._pingCheckInterval:
            self.info('no ping in %f seconds, closing connection',
                      self._pingCheckInterval)
            self._pingCheckDisconnect()
        else:
            self._pingCheckDC = reactor.callLater(self._pingCheckInterval,
                                                  self._pingCheck)

    def stopPingChecking(self):
        if self._pingCheckDC:
            self._pingCheckDC.cancel()
        self._pingCheckDC = None

    def setMind(self, mind):
        # chain up
        Avatar.setMind(self, mind)

        def stopPingCheckingCb(x):
            self.debug('stop pinging')
            self.stopPingChecking()
        self.mind.notifyOnDisconnect(stopPingCheckingCb)

        # Now we have a remote reference, so start checking pings.
        def _disconnect():
            if self.mind:
                self.mind.broker.transport.loseConnection()
        self.startPingChecking(_disconnect)

