package freenet.client;
import freenet.BadAddressException;
import freenet.transport.TCP;
import freenet.client.metadata.*;
import freenet.support.FileBucketFactory;
import freenet.support.Bucket;

import freenet.Core;
import freenet.support.Logger;
import freenet.client.listeners.EventLogger;


import java.util.Vector;


/**
 * A client that automatically does requests with a java interface. It's
 * really much smarter to use the RequestProcess classes themselves,
 * but I know you won't anyways, so I'll save my breath.
 *
 * @author oskar
 */

public class AutoRequester {

    private static final TCP tcp = new TCP(0);
    private ClientFactory clientFactory;

    private boolean doRedirect = true;
    private boolean doDateRedirect = false;
    private long drOffset = DateRedirect.DEFAULT_OFFSET; 
    private int drIncrement = DateRedirect.DEFAULT_INCREMENT;
    private String error;
    private Metadata metadata;
    private FreenetURI keyUri;

    private Vector listeners = new Vector();

    /**
     * Creates an AutoRequester that uses the default protocol (FCP) over
     * TCP. 
     * @param inetAddress The address of the node.
     */
    public AutoRequester(String inetAddress) throws BadAddressException {
        this(new FCPClient(tcp.getAddress(inetAddress)));
    }

    /**
     * Creates an AutoRequester that uses an arbitrary cleint backend.
     * @param  clientFactory  The client backend to use when making Freenet
     *                        requests.
     */
    public AutoRequester(ClientFactory clientFactory) {
        this.clientFactory = clientFactory;
    }

    /**
     * Sets whether to create and follow redirects when pertinent.
     * Default is true.
     * @param val  New setting.
     */
    public void doRedirect(boolean val) {
        this.doRedirect = val;
    }

    /**
     * Sets whether to make DateRedirects to data that is inserted.
     * Default is false.
     * @param val  New setting.
     */
    public void doDateRedirect(boolean val) {
        this.doDateRedirect = val;
    }

    /**
     * Sets the parameters to use if DateRedirects are used (default
     * are the default as per the metadat standard).
     * @param offset  The start time of the redirect calculation.
     * @param increment  The amount of time between updates
     */
    public void setDateRedirectOptions(long offset, int increment) {
        this.drOffset = offset;
        this.drIncrement = increment;
    }

    /**
     * Requests data for the given key from Freenet.
     * @param key     The key to request
     * @param data    The bucket in which to place the resulting data.
     * @param htl     Hops to live to give request.
     * @return  true if the request was successful.
     */
    public boolean doGet(String key, Bucket data, int htl) {
        try {
            return doGet(new FreenetURI(key), data, htl);
        } catch (java.net.MalformedURLException e) {
            error = "Malformed URL: " + e;
            return false;
        }
    }


    /**
     * Requests data for the given key from Freenet.
     * @param key     The key to request
     * @param data    The bucket in which to place the resulting data.
     * @param htl     Hops to live to give request.
     * @return  true if the request was successful.
     */
    public boolean doGet(FreenetURI key, Bucket data, int htl) {
        
        error = "";
        metadata = null;
        return executeProcess( new GetRequestProcess(key, htl, 
                                                     data, 
                                                     new FileBucketFactory(),
                                                     0, doRedirect));
    }

   /**
     * Puts data for the given key into Freenet.
     * @param  key           The key to attempt to insert
     * @param  data          A bucket containing the data
     * @param htl     Hops to live to give request.
     * @param  contentType   The content type to label the data as.
     * @return  true if the request was successful.
     */
    public boolean doPut(String key, Bucket data, int htl, 
                         String contentType) {
        try {
            return doPut(new FreenetURI(key), data, htl, contentType);
        } catch (java.net.MalformedURLException e) {
            error = "Malformed key URI" + e;
            return false;
        }
    }

    /**
     * Puts data for the given key into Freenet.
     * @param  key           The key to attempt to insert
     * @param  data          A bucket containing the data
     * @param htl     Hops to live to give request.
     * @param  contentType   The content type to label the data as.
     * @return  true if the request was successful.
     */
    public boolean doPut(FreenetURI key, Bucket data, int htl, 
                         String contentType) {

        try {
            error = "";
            metadata = null;
            Metadata metadata = new Metadata();
            metadata = addRedirect(key, metadata);

            DocumentCommand mdc = metadata.getDefaultDocument();
            if (mdc == null) {
                mdc = new DocumentCommand((Metadata) null);
                metadata.addDocument(mdc);
            }
            mdc.addPart(new InfoPart("file", contentType));
            
            return 
                executeProcess(new PutRequestProcess(key, htl, "Rijndael",
                                                     metadata, data, 
                                                     new FileBucketFactory(), 
                                                     0, true));
        } catch (Throwable e) {
            error = "Internal error preparing insert metadata: " + e;
            return false;
        }
    }

    /**
     * Inserts a set of buckets under a map file. Each file is inserted with
     * as a redirect in the map file, named after the name of the Bucket 
     * @see Bucket.getName()
     * @param key     The key to the mapfile
     * @param data    A list of Buckets containing the data.
     * @param htl     The hops to live.
     * @return  true if the request was successful.
     */
    public boolean doPutSite(String key, Bucket[] data, int htl) {
        try {
            return doPutSite(new FreenetURI(key), data, htl);
        } catch (java.net.MalformedURLException e) {
            error = "Malformed URL: " + e;
            return false;
        }
    }

    /**
     * Inserts a set of buckets under a map file. Each file is inserted with
     * as a redirect in the map file, named after the name of the Bucket 
     * @see Bucket.getName()
     * @param key     The key to the mapfile
     * @param data    A list of Buckets containing the data.
     * @param htl     The hops to live.
     * @return  true if the request was successful.
     */
    public boolean doPutSite(FreenetURI key, Bucket[] data, int htl) {
        return updateSite(key, data, htl, new Metadata());
    }

    /**
     * Update an old map with files that have changed. 
     * @param key   The key to the mapfile.
     * @param data  A list of of Buckets containing the data of exactly
     *              those documents that have changed (and any new documents
     *              to add).
     * @param htl   The hops to live.
     * @param oldMap  The old map file, as returned from getMetadata() after
     *                it was last inserted.
     * @return  true if the request was successful.
     */
    public boolean updateSite(String key, Bucket[] data, int htl,
                              Metadata oldMap) {
        try {
            return updateSite(new FreenetURI(key), data, htl, oldMap);
        } catch (java.net.MalformedURLException e) {
            error = "Malformed URL: " + e;
            return false;
        }
    }
    /**
     * Update an old map with files that have changed. 
     * @param key   The key to the mapfile.
     * @param data  A list of of Buckets containing the data of exactly
     *              those documents that have changed (and any new documents
     *              to add).
     * @param htl   The hops to live.
     * @param oldMap  The old map file, as returned from getMetadata() after
     *                it was last inserted.
     * @return  true if the request was successful.
     */
    public boolean updateSite(FreenetURI key, Bucket[] data, int htl,
                              Metadata oldMap) {
        try {
            error = "";
            metadata = null;

            for (int i = 0 ; i < data.length ; i++) {
                //  buckets[0] = new FileBucket(new File(p.getArg(i)));
                String name = data[i].getName();
                
                if (metadata.getDocument(name) == null) {
                    DocumentCommand dc = new DocumentCommand(null, name);
                    dc.addPart(new Redirect(new FreenetURI("CHK@")));
                    String type = MimeTypeUtils.getExtType(name);
                    if (type != null)
                        dc.addPart(new InfoPart("file", type));
                    metadata.addCommand(dc);
                }
            }

            Metadata qual = addRedirect(key, metadata);

            boolean success 
                = executeProcess(new PutSiteProcess(key, htl, "Rijndael",
                                                    qual, data, 
                                                    new FileBucketFactory()));
            // we want to make sure it's the map not the doc nfo
            this.metadata = metadata;

            return success;

        } catch (Throwable e) {
            error = "Internal error preparing insert metadata: " + e;
            return false;
        }
    }

    /**
     * Calculate the CHK value of a piece of the data. No metadata will
     * be included, as it is generally a bad idea to use metadata with
     * CHKs anyways.
     * @param data  The data.
     * @return  true if the calculation was successful.
     */
    public boolean doComputeCHK(Bucket data) {
        return executeProcess(new ComputeCHKProcess("Rijndael", null, data));
    }
    
    /**
     * Generate an SVK key pair.
     * @return  { privateKey, publicKey }
     */
    public String[] generateSVKPair() {
        ComputeSVKPairProcess rp = new ComputeSVKPairProcess(null);
        boolean success = executeProcess(rp);
        if (success)
            return new String[] { rp.getPrivateKey(), rp.getPublicKey() };
        else
            throw new RuntimeException("Fatal error generating keys: " 
                                       + error);
    }

    /**
     * If a request fails, this will return an error string.
     * @return  Any error string generated during the last request.
     */
    public String getError() {
        return error;
    }

    /**
     * If a request is successful this will return the metadata.
     * @return  Any metadata generated during the last request.
     */
    public Metadata getMetadata() {
        return metadata;
    }

    /**
     * If a request is successful, this will return the final key value.
     * @return  The final key value genertaed from the last request.
     */
    public FreenetURI getKey() {
        return keyUri;
    }

    /**
     * Add a listener which receives events from all intermediate
     * requests executed by the AutoRequester.
     */
    public void addEventListener(ClientEventListener cel) {
	if (!listeners.contains(cel)) {
	    listeners.addElement(cel);
	}
    }

    /**
     * Removes a listener.
     **/
    public boolean removeEventListener(ClientEventListener cel) {
	return listeners.removeElement(cel);
    }

    // Adds the pertinent redirect to inserts.
    private Metadata addRedirect(FreenetURI key, Metadata metadata) 
        throws InvalidPartException {

        if (!key.getKeyType().equals("CHK") 
            && (doRedirect || doDateRedirect)) {
            
            DocumentCommand redirect = 
                new DocumentCommand((Metadata) null);
            try {
                redirect.addPart(doDateRedirect ? 
                                 new DateRedirect(drIncrement, drOffset,
                                                  key) :
                                 new Redirect(new FreenetURI("CHK@")));
            } catch (java.net.MalformedURLException e) {
                throw new RuntimeException("Error in URI code: " + e);
            }
            metadata = new Metadata();
            metadata.addCommand(redirect);
        } 
        return metadata;
    }

    private final void addListeners(SimpleEventProducer p) {
	if (p == null) {
	    return;
	}

	for (int i = 0; i < listeners.size(); i++) {
	    p.addEventListener((ClientEventListener)listeners.elementAt(i));
	} 
    }

    private final void removeListeners(SimpleEventProducer p) {
	if (p == null) {
	    return;
	}

	for (int i = 0; i < listeners.size(); i++) {
	    p.removeEventListener((ClientEventListener)listeners.elementAt(i));
	}
    }

    // This could be made smarter.
    private boolean executeProcess(RequestProcess rp) {

	Request r = null;
	Request prev = null;
        try {
            while ((r = rp.getNextRequest()) != null) {
		removeListeners(prev);
		prev = r;
                if (!clientFactory.supportsRequest(r.getClass())) {
                    rp.abort();
                    error = ("A request needed to be made " +
                             " that was not supported by the " +
                             "current protocol");
                    return false;
                }
                Client c = clientFactory.getClient(r);
		addListeners(r);
                c.start();
            }
	    removeListeners(prev);

        } catch (Throwable e) {
	    removeListeners(prev);
	    removeListeners(r);
            error = "Request failed due to error: " + e;
            return false;
        }

        if (rp.failed()) {
            error = "Request failed gracefully.";
            return false;
        } else {
            keyUri = rp.getURI();
            metadata = rp.getMetadata();
            return true;
        }
    }
}

