package freenet.client;

import freenet.*;
import freenet.client.events.*;
import freenet.client.listeners.*;
import freenet.crypt.*;
import freenet.message.*;
import freenet.node.*;
import freenet.node.states.request.*;
import freenet.support.Bucket;
import freenet.support.io.*;
import java.io.*;
import java.util.Enumeration;

/** Implementation of Client that uses the node's message handling
  * states to perform requests.
  * @author tavin
  */
public class InternalClient implements ClientFactory {

    private final Node node;
    private VirtualClient vc;
    
    /** Create a new ClientFactory for spawning Clients to process requests.
      * @param n  the running node implementation
      */    
    public InternalClient(Node node) {
        this.node = node;
        vc = new VirtualClient(node, node.randSource, node.bf);
    }

    public Client getClient(Request req) throws UnsupportedRequestException,
                                                IOException, KeyException {
        if (req instanceof GetRequest) {
            return new ClientMessageVector(
                new NewInternalGet(node.randSource.nextLong(),
                                   new InternalGetToken((GetRequest) req))
            );
        }
        else if (req instanceof PutRequest) {
            return new ClientMessageVector(
                new NewInternalPut(node.randSource.nextLong(),
                                   new InternalPutToken((PutRequest) req))
            );
        }
        // see if VirtualClient can handle it ..
        else return vc.getClient(req);
    }

    public boolean supportsRequest(Class req) {
        return GetRequest.class.isAssignableFrom(req)
            || PutRequest.class.isAssignableFrom(req)
            || vc.supportsRequest(req);
    }

    private final class ClientMessageVector implements Client, NodeMessageObject {
        
        private final InternalClientState s;
        
        public ClientMessageVector(InternalClientState s) {
            this.s = s;
        }
        
        public final long id() {
            return s.id();
        }

        public final boolean isExternal() {
            return true; // Changed by Oskar - can't get ReceivedData otherwise
            // (also it is correct, since this ID does leave the node).
        }
        
        // we're just a vector to get this State where it belongs
        public final State getInitialState() throws BadStateException {
            return s.ft.isAlive() ? s : null;
        }
        
        // node overflowed, inform the user
        public final void drop(Node n) {
            s.lost(n);
        }
        
        public final void start() {
            if (s.ft.isAlive()) node.schedule(this);
        }

        public final int blockingRun() {
            start();
            s.ft.dl.waitDone();
            return s.ft.state();
        }

        public final boolean cancel() {
            return s.ft.cancel();
        }

        public final Enumeration getEvents() {
            return s.ft.cel.events();
        }
    }

    
    //=== FeedbackToki =========================================================
    
    private abstract class InternalFeedbackToken implements FeedbackToken {
        
        final Request req;
        DoneListener dl;
        CollectingEventListener cel;
        
        ClientKey clientKey;

        public InternalFeedbackToken(Request req) {
            this.req = req;
            this.dl  = new DoneListener();
            this.cel = new CollectingEventListener();
            req.addEventListener(dl);
            req.addEventListener(cel);
        }

        final boolean isAlive() {
            return state() >= Request.INIT;
        }

        final void checkAlive() throws SendFailedException {
            if (!isAlive()) throw new SendFailedException(
                null, "Request reached state "+req.stateName()
            );
        }

        final int state() {
            return req.state();
        }

        
        // We return the StateReachedEvent so that it 
        // can be sent in an unlocked scope.
        synchronized final StateReachedEvent state(int state) {
            req.state(state); // ok, not locked for (Get/Put)Request
            return new StateReachedEvent(state);
        }
        
        // DO NOT call this while holding a lock on (this).
        // If you do, client event handlers may deadlock.
        //
        // e.g.:
        // Thread 0:
        // InternalFeedBackToken -- locked
        //    --> client event handler
        //        --> ClientObject -- waits for lock on ClientObject
        //
        // Thread 1:
        // ClientObject -- locked
        //   --> Client.cancel()
        //       --> InternalFeedBackToken.cancel -- waits for lock on 
        //                                           InternalFeedBackToken
        //
        //
        // Send an event to the listeners.
        final void unlockedProduceEvent(ClientEvent evt) {
            if (evt == null) {
                return;
            }
            if (req == null) {
                return;
            }
            req.produceEvent(evt);
        }
        
        final boolean cancel() {
            ClientEvent evt = null;
            synchronized (InternalFeedbackToken.this) {
                if (state() >= Request.INIT && state() < Request.DONE) {
                    evt = state(Request.CANCELLED);
                }
            }
            unlockedProduceEvent(evt);
            return evt != null;
        }
        
        final void fail(Exception e) {
            ClientEvent evt = null;
            synchronized (InternalFeedbackToken.this) {
                if (state() >= Request.INIT && state() < Request.DONE) {
                    evt = state(Request.FAILED);
                }
            }
            if (evt != null) {
                unlockedProduceEvent(new ExceptionEvent(e));
                unlockedProduceEvent(evt);
            }
        }
        
        public void queryRejected(Node n, int htl, String reason,
                                  FieldSet fs, int unreachable,
                                  int restarted, int rejected)
            throws SendFailedException {
            ClientEvent evt = null;
            synchronized (InternalFeedbackToken.this) {
                checkAlive();
                evt = state(Request.FAILED);
            }
            
            unlockedProduceEvent(new RouteNotFoundEvent(reason, unreachable,
                                                        restarted, rejected));
            unlockedProduceEvent(evt);
        }
        
        public synchronized void restarted(Node n, long millis) throws SendFailedException {
            synchronized (InternalFeedbackToken.this) {
                checkAlive();
            }
            unlockedProduceEvent(new RestartedEvent(millis));
        }
    }
    
    private class InternalGetToken extends InternalFeedbackToken {
        private final GetRequest req;
        
        public InternalGetToken(GetRequest req) {
            super(req);
            this.req = req;
            req.addEventListener(new TransferCompleteListener());
        }
        
        private class TransferCompleteListener implements ClientEventListener {
            public final void receive(ClientEvent ce) {
                // hrm.. not sure I like this but with the current framework it's
                // the only way to gauge when the request can be considered DONE
                if (ce instanceof TransferCompletedEvent) {
                    ClientEvent evt = null;
                    synchronized (InternalGetToken.this) {
                        if (isAlive()) { 
                            evt = state(Request.DONE);
                        }
                    }
                    unlockedProduceEvent(evt);
                }
            }
        }
        
        public final void insertReply(Node n, long millis) {}  // not
        public void storeData(Node n, NodeReference nr, long rate) {}  // don't care
        
        public void dataNotFound(Node n, long timeOfQuery) throws SendFailedException {
            ClientEvent evt = null;
            synchronized (InternalGetToken.this) {
                checkAlive();
                evt = state(Request.FAILED);
                
            }
            if (evt != null) {
                unlockedProduceEvent(new DataNotFoundEvent());
                unlockedProduceEvent(evt);
            }
        }

        public synchronized OutputStream dataFound(Node n, Storables storables,
                                                   long transLength)
            throws SendFailedException {
            Document doc;
            OutputStream ret = null;
            ClientEvent transferEvt = null;
            ClientEvent exceptionEvt = null;
            ClientEvent failedEvt = null;
            synchronized (InternalGetToken.this) {
                try {
                    checkAlive();
                    doc = clientKey.decode(storables, transLength);
                    
                    Bucket[] buckets = { req.meta, req.data };
                    long[] lengths   = { doc.metadataLength(), doc.dataLength(), doc.length() };
                    // a little sneaky here, we'll pass this to
                    // SegmentOutputStream too, knowing it will
                    // will ignore the last length
                    
                    transferEvt = new TransferStartedEvent(lengths);
                    
                    final OutputStream tmpOut =
                        doc.decipheringOutputStream(new SegmentOutputStream(req, 
                                                                            clientKey.getPartSize(),
                                                                            buckets, lengths, true));
                    
                    ret =
                        new CBStripOutputStream(tmpOut,
                                            clientKey.getPartSize(), clientKey.getControlLength());
                }
                catch (DataNotValidIOException e) {
                    exceptionEvt = new DocumentNotValidEvent(e);
                    failedEvt = state(Request.FAILED);
                    ret = null;
                }
                catch (IOException e) {
                    exceptionEvt = new ExceptionEvent(e);
                    failedEvt = state(Request.FAILED);
                    ret = null;
                }
            }
            unlockedProduceEvent(transferEvt);
            unlockedProduceEvent(exceptionEvt);
            unlockedProduceEvent(failedEvt);
            return ret;
        }
    }
    
    private class InternalPutToken extends InternalFeedbackToken {
        
        private final PutRequest req;
        private boolean keyCollision = false;
        
        public InternalPutToken(PutRequest req) {
            super(req);
            this.req = req;
        }

        public final void dataNotFound(Node n, long timeOfQuery) {}  // not
        
        public void insertReply(Node n, long millis)
                                    throws SendFailedException {
            ClientEvent uriEvt = null;
            ClientEvent pendingEvt = null;
            synchronized (InternalPutToken.this) {
                checkAlive();
                //keyCollision = true; REDFLAG: check with Oskar. I think this was just plain wrong...
                uriEvt = new GeneratedURIEvent("Insert URI",
                                               clientKey.getURI());
                pendingEvt = new PendingEvent(millis);
            }
            unlockedProduceEvent(uriEvt);
            unlockedProduceEvent(pendingEvt);
        }
        
        public void storeData(Node n, NodeReference nr, long rate)
                                        throws SendFailedException {
            ClientEvent evt = null;
            synchronized (InternalPutToken.this) {
                if (!keyCollision) {
                    checkAlive();
                    evt = state(Request.DONE);
                }
            }
            unlockedProduceEvent(evt);
        }
        
        public synchronized OutputStream dataFound(Node n, Storables storables,
                                                   long ctLength) throws SendFailedException {
            ClientEvent uriEvt = null;
            ClientEvent collisionEvt = null;
            ClientEvent failedEvt = null;
            synchronized (InternalPutToken.this) {
                checkAlive();
                
                // REDLFAG: correct?
                keyCollision = true; 
                
                uriEvt = new GeneratedURIEvent("Insert URI",
                                               clientKey.getURI());
                collisionEvt = new CollisionEvent(clientKey);
                failedEvt = state(Request.FAILED);
            }

            unlockedProduceEvent(uriEvt);
            unlockedProduceEvent(collisionEvt);
            unlockedProduceEvent(failedEvt);
            return null;
        }
    }


    //=== states ===============================================================
    
    private abstract class InternalClientState extends State {
        
        final InternalFeedbackToken ft;
        
        public InternalClientState(long id, InternalFeedbackToken ft) {
            super(id);
            this.ft = ft;
        }
        
        public final void lost(Node n) {
            ft.fail(new RuntimeException("Node states overflowed."));
        }
    }

    private class NewInternalGet extends InternalClientState {

        private InternalGetToken ft;

        public NewInternalGet(long id, InternalGetToken ft) throws KeyException,
                                                                    IOException {
            super(id, ft);
            this.ft = ft;
            ft.clientKey = AbstractClientKey.createFromRequestURI(ft.req.uri);
            if (ft.clientKey.getKey() == null)
                throw new KeyException("got null Key");
            ClientEvent evt = ft.state(Request.PREPARED);
            ft.unlockedProduceEvent(evt);
        }
    
        public final String getName() {
            return "New Internal Get";
        }
    
        public State received(Node n, MessageObject mo) throws StateException {
            if (!(mo instanceof ClientMessageVector)) {
                throw new BadStateException("Not a ClientMessageVector: "+mo);
            }

            node.diagnostics.occurrenceCounting("inboundClientRequests", 1);

            ClientEvent evt = null;
            State s = null;
            synchronized (ft) {
                if (!ft.isAlive()) return null;
                evt = ft.state(Request.REQUESTING);
                RequestInitiator ri = new RequestInitiator(id);
                Pending p = new DataPending(id, ft.req.htl, ft.clientKey.getKey(),
                                            null, ft, ri);
                // REDFLAG: do any calls within this scope
                //          cause events to be dispatched?????
                //          Back to this.

                s = p.received(n, ri);
            }
            ft.unlockedProduceEvent(evt);
            return s; // REDFLAG: return was synchronized, ok?
        }
    }

    private class NewInternalPut extends InternalClientState {

        private InternalPutToken ft;
        private DataInsert dim;
        
        public NewInternalPut(long id, InternalPutToken ft) throws KeyException,
                                                                    IOException {
            super(id, ft);
            this.ft = ft;
            
            // set up ClientKey
            ft.clientKey = AbstractClientKey.createFromInsertURI(node.randSource, ft.req.uri);
            try {
                ft.clientKey.setCipher(ft.req.cipherName);
            }
            catch (UnsupportedCipherException e) {
                throw new KeyException(""+e);
            }

            long total = ft.clientKey.getTotalLength(ft.req.meta.size() + ft.req.data.size());
            
            // set up encoded key data
            Bucket ctBucket = node.bf.makeBucket(total);
            InputStream ctStream = null;
            try {
                ctStream = ft.clientKey.encode(ft.req.meta, ft.req.data, ctBucket);
                ctStream = new FreeBucketInputStream(ctStream, node.bf, ctBucket);
                ctBucket = null;
            }
            finally {
                if (ctBucket != null)
                    node.bf.freeBucket(ctBucket);
            }
            EventInputStream ein = new EventInputStream(ctStream, ft.req,
                                                        total >> 4, total, true);
            // prepare the insert
            ft.req.clientKey = ft.clientKey;

            Storables storables = ft.clientKey.getStorables();
            FieldSet root = new FieldSet();
            storables.addTo(root);

            try {
                dim = new DataInsert(id, root, ein, 
                                     ft.clientKey.getTotalLength()); //total);
            }
            catch (BadAddressException e) {
                // should be impossible
                node.logger.log(this, "Got mystery exception", e, 
                                node.logger.ERROR);
                ctStream.close();
                throw new IOException("Got mystery exception: "+e);
            }
            ClientEvent evt = ft.state(Request.PREPARED);
            ft.unlockedProduceEvent(evt);
        }
        
        public final String getName() {
            return "New Internal Put";
        }
        
        public State received(Node n, MessageObject mo) throws StateException {
            if (!(mo instanceof ClientMessageVector)) {
                throw new BadStateException("Not a ClientMessageVector: "+mo);
            }

            node.diagnostics.occurrenceCounting("inboundClientRequests", 1);

            // walk through client event semantics while initiating
            // the InsertPending state and sending the DataInsert
            ClientEvent evt = null;
            State s = null;
            synchronized (ft) {
                // REDFLAG: do any calls within this scope
                //          cause events to be dispatched?????
                //          Back to this.
                if (!ft.isAlive()) return null;
                evt = ft.state(Request.REQUESTING);

                RequestInitiator ri = new RequestInitiator(id);

                Pending p = new InsertPending(id, ft.req.htl, ft.clientKey.getKey(),
                                              null, ft, ri);
                System.err.println(ft.clientKey.getURI());
                s = p.received(n, ri);

                try {
                    s = s.received(n, dim);
                }
                catch (BadStateException e) {
                    // key collision, or something
                    dim.drop(n);
                }
            }
            // REDFLAG: out of order events???
            ft.unlockedProduceEvent(evt);
            return s; // REDFLAG: return was synchronized, ok?
        }
    }
}









