package freenet.fs;

import freenet.crypt.*;
import java.io.*;
import java.util.Random;

/**
 * Adds an encryption layer around the FileSystem.  The streams
 * are PCFB-mode encrypted using a symmetric block cipher, with
 * the IV made from XORing a given base value with bytes from
 * the address of the storage fragment being read/written.
 * @author tavin
 */
public class EncryptedFileSystem extends FileSystem {

    /**
     * @param storage  the underlying unencrypted Storage
     * @param cipher   the (already initialized) BlockCipher
     * @param baseIV   base IV value
     */
    public EncryptedFileSystem(Storage storage, BlockCipher cipher, byte[] baseIV) {
        super(new EncryptedStorage(storage, cipher, baseIV));
    }
    

    /**
     * Writes a bunch of random junk to initialize the file-system.
     * Might take a really long time..
     * @param rand  crypto-strength entropy source
     * @param notifyMe  if non-null, will be notify()'d once the FS
     *                  is ready to be used
     */
    public void initialize(Random rand, Object notifyMe) throws IOException {
        long len = size() - truncation();
        if (len <= 0)
            throw new IllegalStateException("FS does not require initialization");
        
        OutputStream out = WriteLock.getOutputStream(this, truncation(),
                                                     size() - 1);

        if (notifyMe != null) {
            synchronized (notifyMe) {
                notifyMe.notify();
            }
        }

        try {
            int i = 0;
            byte[] junk = new byte[0x10000];
            byte[] fill = new byte[0x100];
            while (len > 0) {
                rand.nextBytes(fill);
                System.arraycopy(fill, 0,
                                 junk, (i++ * fill.length) % junk.length,
                                 fill.length);
                out.write(junk, 0, (int) Math.min(len, junk.length));
                len -= junk.length;
            }
        }
        finally {
            out.close();
        }
    }
    
    
    private static final class EncryptedStorage implements Storage {

        private final Storage raw;
    
        private final BlockCipher cipher;
        private final byte[] baseIV;
        
        EncryptedStorage(Storage raw, BlockCipher cipher, byte[] baseIV) {
            this.raw = raw;
            this.cipher = cipher;
            this.baseIV = new byte[(int) Math.max(16, baseIV.length)];
            System.arraycopy(baseIV, 0, this.baseIV, 0, baseIV.length);
        }
    
        public final long size() {
            return raw.size();
        }
    
        public final long truncation() {
            return raw.truncation();
        }
    
        public final InputStream getInputStream(long start, long end)
                                                    throws IOException {
            return new CipherInputStream(cipher,
                                         raw.getInputStream(start, end),
                                         buildIV(start, end));
        }
    
        public final OutputStream getOutputStream(long start, long end)
                                                    throws IOException {
            return new CipherOutputStream(cipher,
                                          raw.getOutputStream(start, end),
                                          buildIV(start, end));
        }
    
        private byte[] buildIV(long a, long b) {
            byte[] iv = new byte[baseIV.length];
            System.arraycopy(baseIV, 0, iv, 0, iv.length);
            iv[0]  ^= (byte) (0xff & (a >>  0));
            iv[0]  ^= (byte) (0xff & (a >>  0));
            iv[1]  ^= (byte) (0xff & (b >>  0));
            iv[2]  ^= (byte) (0xff & (a >>  8));
            iv[3]  ^= (byte) (0xff & (b >>  8));
            iv[4]  ^= (byte) (0xff & (a >> 16));
            iv[5]  ^= (byte) (0xff & (b >> 16));
            iv[6]  ^= (byte) (0xff & (a >> 24));
            iv[7]  ^= (byte) (0xff & (b >> 24));
            iv[8]  ^= (byte) (0xff & (a >> 32));
            iv[9]  ^= (byte) (0xff & (b >> 32));
            iv[10] ^= (byte) (0xff & (a >> 40));
            iv[11] ^= (byte) (0xff & (b >> 40));
            iv[12] ^= (byte) (0xff & (a >> 48));
            iv[13] ^= (byte) (0xff & (b >> 48));
            iv[14] ^= (byte) (0xff & (a >> 56));
            iv[15] ^= (byte) (0xff & (b >> 56));
            return iv;
        }
    }
}


