package freenet.node;

import freenet.support.DoublyLinkedListImpl;
import freenet.support.DoublyLinkedListImpl.Item;

import java.math.BigInteger;
import java.util.Hashtable;
import java.util.Enumeration;
import java.io.PrintWriter;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * Helper class to keep track of the data needed to
 * do intra-node load balancing.
 * @author giannij
 **/
public class LoadStats {
    private long requestCount = 0;
    private Object requestCountLock = new Object();
    private long startupTimeMs;
    private Hashtable table = new Hashtable();
    private DoublyLinkedListImpl lru = new DoublyLinkedListImpl();
    private float totalGlobal = 0;

    private int maxTableSize;

    public LoadStats(long startupTimeMs, int tableSize) {
        this.startupTimeMs = startupTimeMs;
        this.maxTableSize = tableSize;
    }

    // This may be called very often and
    // it doesn't matter if the count is
    // off by a little, so I don't synchronize
    // on this.
    
    /**
     * Call to increment the outbound request count.
     **/
    public final void incRequestCount() {
        synchronized (requestCountLock) {
            requestCount++;
        }
    }

    /**
     * Call to update the table used to estimate global network load.
     **/
    public final synchronized void storeRequestRateForNode(NodeReference nr, long requestsPerHour) {
        if (nr == null || (requestsPerHour == -1)) {
            // These cases are legal NOPs.
            return;
        }

        LoadEntry le = new LoadEntry(nr.getIdentity().fingerprint(),
                                     requestsPerHour);
        LoadEntry oldle = (LoadEntry) table.get(le.bi);
        if (oldle != null) {
            lru.remove(oldle);
            totalGlobal -= oldle.requestsPerHour;
        }

        table.put(le.bi, le);
        lru.push(le);
        totalGlobal += le.requestsPerHour;
        
        // Don't let the table grow without bound.
        if (lru.size() > maxTableSize) {
            LoadEntry last = (LoadEntry) lru.shift();
            table.remove(last.bi);
            totalGlobal -= last.requestsPerHour;
        }
    }

    /**
     * @return The number of outbound requests per hour made from
     *         this node.
     */
    public final synchronized float localRequestsPerHour() {
        final long deltat = System.currentTimeMillis() - startupTimeMs;
        if (deltat < 1) {
            return 0.0f;
        }

        return (float) (requestCount * 3600000 / (float)deltat);
    }

    /**
     * @return An estimate of the global per node network 
     *         load in requests per hour.
     **/
    public final synchronized float globalRequestsPerHour() {
        // Oskar, we are vulnerable to hostile nodes manipulating
        // the stats (e.g. RequestRate=1000000000000).  We should
        // be doing something more robust than just calculating
        // the mean.  I leave that "something" to you since I
        // was such a crappy stats student. --gj
        return lru.size() == 0 ? 0.0f : totalGlobal / lru.size();
    }

    // Oskar this is a rough first pass.  I am
    // sure you can offer suggestions for improvements.
    // --gj

    /**
     * Calculates the probability of resetting the DataSource
     * that should be used when sending StoreData messages.
     * <p>
     * This is how the node does intra-node load balancing.
     **/
    public final synchronized float resetProbability() {
        final float DEFAULT_PROBABILITY = 0.05f;
        final float HIGHER_PROBABILITY = 0.10f;
        final float LOWER_PROBABILITY = 0.025f;        
        
        final float MIN_DELTA = 0.25f;        

        // Don't have enough info on global network
        // conditions to make a useful guess.
        if (table.size() <  3) {
            return DEFAULT_PROBABILITY;
        }

        float local = localRequestsPerHour();
        float global = globalRequestsPerHour();
        float denom = local + global;
        
        // hmmm..
        if (denom == 0.0f) {
            return DEFAULT_PROBABILITY;
        }
        
        float percentDiff = 2.0f * (global - local) / denom;
        
        if (Math.abs(percentDiff) < MIN_DELTA) {
            return DEFAULT_PROBABILITY;
        }

        if (percentDiff > 0) {
            // Global traffic is higher so we want
            // more traffic.
            return HIGHER_PROBABILITY;
        }
        else {
            // Global traffic is lower so we want
            // less traffic.
            return LOWER_PROBABILITY;
        }
    }

    public final synchronized void dump(PrintWriter pw) {
        pw.println("# entries: " + table.size());
        pw.println("# globalRequestsPerHour: " + globalRequestsPerHour());
        pw.println("# localRequestsPerHour: " + localRequestsPerHour());
        pw.println("# format: <requests per hour> <node fingerprint>");                   
        for (Enumeration e = table.keys() ; e.hasMoreElements() ;) {
            BigInteger key = (BigInteger) e.nextElement();
            long rate = ((LoadEntry)table.get(key)).requestsPerHour;
            pw.println(rate + "\t" + "\"" +  key.toString(16) + "\"");
        }
    }

    public final synchronized void dumpHtml(PrintWriter pw) {
        pw.println("<ul>");
        pw.println("<li> entries: " + table.size() + "</li>");
        pw.println("<li> globalRequestsPerHour: " + globalRequestsPerHour()
                   + "</li>");
        pw.println("<li> localRequestsPerHour: " + localRequestsPerHour() +
                   "</li>");
        pw.println("</ul>");
        pw.println("<table border=1><tr><th>Requests per hour</th>" + 
                   "<th>Node fingerprint</th></tr>");
        for (Enumeration e = table.keys() ; e.hasMoreElements() ;) {
            BigInteger key = (BigInteger) e.nextElement();
            long rate = ((LoadEntry)table.get(key)).requestsPerHour;
            pw.println("<tr><td>" + rate + "</td><td>" +  key.toString(16) 
                       + "<td></tr>");
        }
        pw.println("</table>");
    }



    private class LoadEntry extends DoublyLinkedListImpl.Item {

        private BigInteger bi;
        private long requestsPerHour;

        private LoadEntry(byte[] b, long requestsPerHour) {
            this.bi = new BigInteger(b);
            this.requestsPerHour = requestsPerHour;
        }
    }

}





