package freenet.node;

import freenet.MessageObject;
import freenet.support.Logger;
import freenet.support.Fields;

/**
 * Acts as a handle for a state chain and contains the current state
 * for that chain.
 */
public class StateChain implements AggregatedState {
    
    protected State state;

    public String toString() {
        return state == null ? "(null)" : state.toString();
    }

    /**
     * @return  true if the message should go to this state
     */
    public synchronized boolean receives(MessageObject mo) {
        return state == null
            || !(state instanceof AggregatedState)
            || ((AggregatedState) state).receives(mo);
    }
    
    /**
     * @return  false if the state chain has ended (gone to state null)
     */
    public boolean alive() {
        return state != null;
    }

    /**
     * Sends the MO to the internal state for this chain,
     * and updates the chain to the new state.
     * @return  true if the chain is still alive
     */
    public synchronized boolean received(Node node, NodeMessageObject mo) {
        state = received(state, node, mo);
        return state != null;
    }

    /**
     * Calls lost() for the active state on this chain and
     * updates the internal state to null.
     * @return  true if a live state was lost
     */
    public synchronized boolean lost(Node node) {
        if (state == null)
            return false;
        lost(node, state);
        state = null;
        return true;
    }


    
    // static methods


    /**
     * Calls drop() on the MO, and eats and logs any uncaught throwable.
     */
    public static void drop(Node node, NodeMessageObject mo) {
        try {
            mo.drop(node);
        }
        catch (Throwable e) {
            node.logger.log(node, "Error dropping message: "+mo,
                            e, Logger.ERROR);
        }
    }

    /**
     * Calls lost() on the state, and eats and logs any uncaught throwable.
     */
    public static void lost(Node node, State state) {
        try {
            state.lost(node);
        }
        catch (Throwable e) {
            node.logger.log(node, "Error discarding state: "+state,
                            e, Logger.ERROR);
        }
    }
    
    /**
     * @return  the new State when the given State receives the MO
     */
    public static State received(State state, Node node,
                                 NodeMessageObject mo) {
        // new chain
        if (state == null) {
            try {
                state = mo.getInitialState();
            } catch (BadStateException e) {
                node.logger.log(node, "Bad state on new message: " + mo,
                                e, Logger.MINOR);
            } catch (Throwable e) {
                node.logger.log(node,
                                "Error getting initial state for message: "+mo,
                                e, Logger.ERROR);
            }
            if (state == null)  // still
                return null;
        }

        // pass message to state and record transition
        State newState = null;
        NodeMessageObject[] queue = null;
        
        try {
            newState = state.received(node, mo);
        }
        catch (BadStateException e) {
            node.logger.log(node,
                "Message " + mo + " received during state "
                + state + " was not welcome",
                e, Logger.MINOR);
            // clean up
            drop(node, mo);
            // no state change; exit
            return state;
        }
        catch (StateTransition e) {
            newState = e.state;
            if (e.exec) {
                queue = e.mess;
            }
            else {
                for (int i=0; i<e.mess.length; ++i)
                    drop(node, e.mess[i]);
            }
        }
        // we'll treat an abnormal exit the same as a null exit
        // (newState is null)
        catch (Throwable e) {
            node.logger.log(node,
                "Error while receiving message "+mo+" in state "+state,
                e, Logger.ERROR);
        }

        // null or abnormal exit
        if (newState == null) {
            node.logger.log(node,
                "Ending chain: " + Fields.longToHex(state.id()),
                Logger.DEBUG);
        }
        // normal transition
        else if (newState != state) {
            node.logger.log(node,
                "Chain " + Fields.longToHex(state.id()) + " state change: "
                + state.getName() + " -> " + newState.getName(),
                Logger.DEBUG);
        }
        
        // okay, watch out for weirdo recursion problems..
        if (queue != null) {
            for (int i=0; i<queue.length; ++i) {
                if (newState == null)
                    drop(node, queue[i]);
                else
                    newState = received(newState, node, queue[i]);
            }
        }
        
        return newState;
    }
}


