package freenet.node.rt;

import freenet.BadAddressException;
import freenet.Address;
import freenet.Key;
import freenet.Identity;
import freenet.node.NodeReference;
import freenet.support.StringMap;
import freenet.support.SimpleStringMap;
import freenet.support.sort.*;
import freenet.support.Comparable;
import freenet.transport.TCP;

import java.util.Random;
import java.util.Vector;
import java.util.Enumeration;
import java.io.IOException;

/**
 * RoutingTable implementation which uses the tested CP
 * algorithm that was in place before Tavin ripped out 
 * the old routing code.
 *
 * @author giannij
 */
public class CPAlgoRoutingTable extends TreeRoutingTable {

    protected final Random rand;
    
    public CPAlgoRoutingTable(RoutingStore routingStore,
                                   int maxNodes, int maxRefsPerNode,
                                   Random rand) throws IOException {
        super(routingStore, maxNodes, maxRefsPerNode);
        this.rand = rand;
    }

    
    public synchronized RTDiagSnapshot getSnapshot() {
        long nowMs = System.currentTimeMillis();

        // Reset ref aggregate counters
        totalRefs = 0;
        backedOffRefs = 0;
        contactedRefs = 0;

        // ^^^ the fact that you have to do this suggests these
        //     should be local variables not instance variables  -tc
        //
        // 6 of 1, 1/2 dozen of the other. makeRefInfo() updates
        // the counts as a side effect, if you were morally opposed
        // to side effects you could get rid of makeRefInfo
        // (decreasing readability).  --gj

        Vector nodeRefs = new Vector();
        Vector refInfos = new Vector();
        Vector keys = new Vector();

        Enumeration memories = routingStore.elements();
        while (memories.hasMoreElements()) {
            RoutingMemory memory = (RoutingMemory)memories.nextElement();
            NodeReference nodeRef = memory.getNodeReference();
            if (nodeRef == null) {
                continue;
            }

            ReferenceSet refSet = ReferenceSet.getProperty(memory, "keys");
            Enumeration refs = refSet.references();
            while (refs.hasMoreElements()) {
                Reference ref = (Reference) refs.nextElement();
                if (!keys.contains(ref.key)) {
                    // REDFLAG: Change to only exclude "0000" keys once code is 
                    //          fixed to mark all non-data keys with "0000"
                    final String keyAsString = ref.key.toString();
                    if ((keyAsString.endsWith("0302") || keyAsString.endsWith("0203"))) {
                        // Only keys ending in 0x0302 or 0x0203 are valid data
                        keys.addElement(ref.key);
                    }
                } 
            }
            if (!nodeRefs.contains(nodeRef)) {
                final StringMap refInfo = makeRefInfo(nodeRef, memory, nowMs); 
                refInfos.addElement(refInfo);
                nodeRefs.addElement(nodeRef);
            }
        }

        CPSortableStringMap[] refs = new CPSortableStringMap[refInfos.size()];
        refInfos.copyInto(refs);

        Object[] props = new Object[TABLE_PROPERTIES.length];
        props[0] = new Integer(totalRefs);
        props[1] = new Integer(contactedRefs);
        props[2] = new Integer(backedOffRefs);
        props[3] = new Integer(totalAttempts);
        props[4] = new Integer(totalSuccesses);
        props[5] = getClass().getName();
        
        Key[] keyList = new Key[keys.size()];
        keys.copyInto(keyList);

        HeapSorter.heapSort(new ArraySorter(refs));

        return new SimpleRTDiagSnapshot(new SimpleStringMap(TABLE_PROPERTIES, props),
                                        refs, keyList);
    }

    private final static String RMKEY = "cpdata";
 
    
    protected final Comparable getDiscardValue(RoutingMemory mem) {
        CPAlgoData cpd = CPAlgoData.getProperty(mem, RMKEY);
        return new DiscardValue(cpd.lastRetryMs);
        // probably, we should record a last successful contact time,
        // but this will do for now..
    }

    private static final class DiscardValue implements Comparable {
        final long time;
        DiscardValue(long time) {
            this.time = time;
        }
        public final int compareTo(Object o) {
            return compareTo((DiscardValue) o);
        }
        public final int compareTo(DiscardValue o) {
            return time == o.time ? 0 : (time < o.time ? 1 : -1);
        }
    }
    

    private final synchronized void incTrials(RoutingMemory mem) {
        CPAlgoData cpd = CPAlgoData.getProperty(mem, RMKEY);
        cpd.trials++;
        // We don't call mem.setProperty() because trials isn't
        // persisted.
    }
 
    private final synchronized void fail(RoutingMemory mem) {
        CPAlgoData cpd = CPAlgoData.getProperty(mem, RMKEY);
        if (cpd.decreaseContactProbability()) {
            routingStore.remove(mem.getIdentity());
            return;
        }
        mem.setProperty(RMKEY, cpd);
    }
 
    private final synchronized void succeed(RoutingMemory mem, boolean attenuated) {
        CPAlgoData cpd = CPAlgoData.getProperty(mem, RMKEY);
        cpd.successes++;
        cpd.increaseContactProbability();
        if (attenuated) {
            cpd.backoffRouting();
        }
        mem.setProperty(RMKEY, cpd);
    }

    protected synchronized boolean isRoutable(RoutingMemory mem) {
        CPAlgoData cpd = CPAlgoData.getProperty(mem, RMKEY);

        if (cpd.routingBackedOff()) {
            return false;
        }

        return rand.nextFloat() <= cpd.contactProbability();
    }
    
    protected synchronized void routeConnected(RoutingMemory mem) {
        incTrials(mem);
        totalAttempts++;
    }

    protected synchronized void routeAccepted(RoutingMemory mem) {
        // hmm.. what?
    }

    protected synchronized void routeSucceeded(RoutingMemory mem) {
        succeed(mem, false);
        totalSuccesses++;
    }

    protected synchronized void connectFailed(RoutingMemory mem) {
        incTrials(mem);
        totalAttempts++;
        fail(mem);
    }

    protected synchronized void authFailed(RoutingMemory mem) {
        // dereference node
        routingStore.remove(mem.getIdentity()); 
    }

    protected synchronized void timedOut(RoutingMemory mem) {
        fail(mem);
    }

    protected synchronized void transferFailed(RoutingMemory mem) {
        fail(mem);
    }

    protected synchronized void verityFailed(RoutingMemory mem) {
        // dereference node
        routingStore.remove(mem.getIdentity()); 
    }

    private final synchronized boolean deleteReferences(ReferenceSet refSet, long number) {
        if (refSet.size() == 0) {
            return false;
        }
        
        while (number > 0 ) {
            Reference ref = refSet.pop();
            if (ref == null) {
                break;
            }
            refTree.treeRemove(ref);
            number--;
        }

        return true;
    }

    // Reduce the chances of routing to nodes that always 
    // replies with QueryRejected.
    protected synchronized void queryRejected(RoutingMemory mem, long attenuation) {
        
        // I would have preferred to make queryRejected() in addition
        // to routeSucceeded() to keep transport and application
        // levels separate. --gj
        
        // Counter-intutitive.  The routing has succeeded at the 
        // transport layer,  but we got a QueryRejected 
        // back at the application layer.
        //
        succeed(mem, attenuation > 0);
        totalSuccesses++;

        
        ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
        int refCount = refSet.size();

        if (refCount < 4) {
            // Never remove the very last three refs.
            return;
        }

        if (refCount <= attenuation) {
            attenuation = refCount - 1;
        }

        if (attenuation < 1) {
            return;
        }
         
        // We want to remove refs more aggressively when 
        // there are many.
        if (rand.nextFloat() < (1.0f - (1.0f / refCount))) {
            if (deleteReferences(refSet, attenuation)) {
                mem.setProperty("keys", refSet);
            }
        }
    }

    ////////////////////////////////////////////////////////////
    // Helper functions used to implement diagnostic snapshots.
    private final static String[] REF_PROPERTIES 
        = {"Address", "Contact Probablitity", "Failure Intervals", 
           "Connection Attempts", "Successful Connections", "Backed Off", 
           "Last Attempt", "NodeReference", "Node Version", "Routing Backed Off", "Key Count"};

    private final static String[] TABLE_PROPERTIES 
        = {"Number of node references", "Contacted node references",
           "Backed off node references", "Total Trials", "Total Successes",
           "Implementation" };

    private final static TCP tcp = new TCP(100);

    private final static String getAddr(NodeReference ref) {
        String addrAsString = "BAD ADDRESS!";
        try {
            // I know Oskar is going to hate this, but we
            // need the information.  Is there a cleaner way to
            // get it?
            Address addr = ref.getAddress(tcp);
            addrAsString =addr.toString();
        }
        catch (BadAddressException bae) {
            // NOP
        }
        return addrAsString;
    }

    // ^^^ i don't understand why you are doing this here
    //     instead of just passing the NodeReference back  -tc
    //
    // The point is to present data that the client
    // (i.e. the caller of getSnapShot()) can immediately 
    // display without doing any more work, or even having
    // to know the type of the value.  
    // NodeReference.toString() doesn't give a value that 
    // makes sense to an end user. --gj


    private final static Long lastAttemptSecs(CPAlgoData cpd, long nowMs, long thenMs) {
        long ret = -1; // never
        if (cpd.trials > 0) {
            ret = (nowMs - cpd.lastRetryMs) / 1000;
            if (ret < 1) {
                ret = 1;
            }
        }
        return new Long(ret);
    }

    private final static Long backedOffSecs(CPAlgoData cpd, long nowMs) {
        long ret = -1;
        
        // Real backoff.
        if (cpd.failuresThisInterval > CPAlgoData.MAXFAILURESPERINTERVAL) {
            ret = (cpd.intervalTimeoutMs - nowMs) / 1000;
        }
        // Short timeout immediately after failure.
        else if ((cpd.failureIntervals > 0) && (nowMs - cpd.lastRetryMs < CPAlgoData.SHORTBACKOFFMS)) {
            ret = (CPAlgoData.SHORTBACKOFFMS - (nowMs -  cpd.lastRetryMs)) / 1000;
        }

        if (ret == 0) {
            ret = 1;
        }

        return new Long(ret);
    }

    private int totalAttempts = 0;
    private int totalSuccesses = 0;

    // Diagnostic stats set by makeRefInfo
    private int totalRefs = 0;
    private int backedOffRefs = 0;
    private int contactedRefs = 0;

    private final Long LONG_MINUS_ONE = new Long(-1);
    private final Integer INT_ZERO = new Integer(0);

    ////////////////////////////////////////////////////////////
    static class CPSortableStringMap extends SimpleStringMap implements Comparable {
        public CPSortableStringMap(String[] keys, Object[] objs) {
            super(keys, objs);
        }

        public int compareTo(Object o) {
            if (!(o instanceof CPSortableStringMap)) {
                throw new IllegalArgumentException("Not a CPSortableStringMap"); 
            }

            final float cpA = ((Float)objs[1]).floatValue();
            final float cpB = ((Float)((CPSortableStringMap)o).objs[1]).floatValue();

            final float delta = cpB - cpA;

            if (delta > 0.0f) {
                return 1;
            }
            else if (delta < 0.0f) {
                return -1;
            }
            else {
                return 0;
            }
        }
    }

    ////////////////////////////////////////////////////////////
    private final CPSortableStringMap makeRefInfo(NodeReference ref, RoutingMemory mem, long nowMs) {
        CPAlgoData cpd = CPAlgoData.getProperty(mem, RMKEY);
        Object[] values = new Object[REF_PROPERTIES.length];
        values[0] = getAddr(ref);
        // Don't phreak out cluebies with 0.0 cp during backoff.
        values[1] = new Float(cpd.effectiveCP(nowMs));
        values[2] = new Integer(cpd.failureIntervals);
        values[3] = new Integer(cpd.trials);
        values[4] = new Integer(cpd.successes);
        values[5] = backedOffSecs(cpd, nowMs);
        values[6] = lastAttemptSecs(cpd, nowMs, cpd.lastRetryMs);
        values[7] = ref; // hmmm.... clone?
        values[8] = ref.getVersion();
        values[9] = new Boolean(cpd.routingBackedOff());
        ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
        values[10] = new Integer(refSet.size());

        // Update aggregate stats
        totalRefs++;
        if (!values[5].equals(LONG_MINUS_ONE)) {
            backedOffRefs++;
        }

        if (!values[4].equals(INT_ZERO)) {
            contactedRefs++;
        }

        return new CPSortableStringMap(REF_PROPERTIES, values); 
    }
    ////////////////////////////////////////////////////////////
}







