package freenet.node.states.announcement;

import freenet.*;
import freenet.node.*;
import freenet.message.*;
import freenet.crypt.*;
import freenet.support.Logger;
import freenet.support.Fields;
import java.math.BigInteger;
import java.util.Enumeration;

/**
 * Initial state for the announcements.
 *
 * @author oskar
 */

public class NewAnnouncement extends AnnouncementState {

    public NewAnnouncement(long id, NodeReference announcee,
                           int depth, int hopsToLive, byte[] commitVal) {
        super(id, announcee, depth, hopsToLive, commitVal);
    }

    NewAnnouncement(ReplyPending p) {
        super(p);
    }

    public String getName() {
        return "New Announcement";
    }

    public State receivedMessage(Node n, NodeAnnouncement na) {
        try {
            origRec = n.getPeer(na.getSource());
            if (origRec == null) {
                n.logger.log(this, "Failed to determine Peer for NodeReference.", 
                             Logger.NORMAL);
                return null;
            }
            // enforce aloofness
            String vers = na.getSource().getVersion();
            if (vers != null && !Version.checkGoodVersion(vers)) {
                String reason = Version.explainBadVersion(vers);
                n.logger.log(this,
                    "Rejecting query from host of type " + vers
                    +": "+reason,
                    Logger.MINOR);
                n.sendMessage(new QueryRejected(id, na.hopsToLive(),
                                                reason, null), origRec);
                return null;
            }
            
            if (n.rt.references(announcee.getIdentity())) {
                n.logger.log(this, "Previously knew Announcee, rejecting.",
                             Core.logger.DEBUG);
                n.sendMessage(
                    new AnnouncementFailed(id, AnnouncementFailed.KNOWN_ANNOUNCEE),
                    origRec);
                return null;
            } else if ((depth + hopsToLive) > Node.maxHopsToLive) {
                n.logger.log(this, "Too high HTL on Announcement, rejecting.",
                             Core.logger.DEBUG);
                n.sendMessage(
                    new AnnouncementFailed(id, AnnouncementFailed.UNACCEPTABLE_HTL),
                    origRec);
                return null;
            }


            na.setSource(n.myRef);

            na.incDepth();
            na.decHopsToLive();
            
            // Return Accepted
            n.sendMessage(new Accepted(id), origRec);
            
        } catch (CommunicationException e) {
            n.logger.log(this, "Failed to return reply to Announcement",
                         n.logger.MINOR);
            return null;
        }
        // Our random commit value
        myVal = new byte[20];
        n.randSource.nextBytes(myVal);
        
        // update commitVal
        synchronized (ctx) {
            ctx.update(myVal);
            ctx.update(commitVal);
            ctx.digest(true, commitVal, 0); 
        }
        //na.setCommitVal(commitVal);   side effect...

        if (hopsToLive > 0) {
            return sendOn(n, na);

        } else { // end of the line
            try {
                AnnouncementReply ar = new AnnouncementReply(id, myVal);
                NoExecute ne = new NoExecute(id);
                n.schedule(getTime((int)depth), ne);
                n.sendMessage(ar, origRec);
                //System.err.println("MYVAL: " + Fields.bytesToHex(myVal));
                return new LastNode(this, ne);
            } catch (CommunicationException e) {
                n.logger.log(this, "Failed to return reply to Announcement: "+e,
                             Logger.MINOR);
                return null;
            }
        }
    }

    /**
     * Routes a forwards an announcement. 
     * @param n  The node
     * @param na  The nodeAnnouncement message
     * @return  The state we are in after sending.
     */
    State sendOn(Node n, NodeAnnouncement na) {

        if (routes == null) {  // first time
            // Route. No sense in taking too much entropy. (And this way
            // I can test Tavin's routing table :-) ).
            long l = n.randSource.nextLong();
            Key k = new Key(new byte[] {
                (byte) l, (byte) (l >> 8), (byte) (l >> 16), 
                (byte) (l >> 24), (byte) (l >> 32), (byte) (l >> 40), 
                (byte) (l >> 48), (byte) (l >> 56)
            });
            routes = n.rt.route(k);
        }

        while (true) {

            NodeReference nr = routes.getNextRoute();
            if (nr == null)
                break;
            
            if (origRec != null && origRec.equalsIdent(nr.getIdentity()))
                // don't loop back to the sender
                continue;
            
            n.logger.log(this, "Forwarding query to: "+nr, Logger.DEBUG);
            try {
                n.sendMessage(na, nr, n.routeConnectTimeout);
                routes.routeConnected();
                
                lastAddr = n.getPeer(nr);
                NoReply nrm = new NoReply(id);
                n.schedule(getTime(1), nrm);
                return new ReplyPending(this, nrm);
                
            } catch (AuthenticationFailedException e) {
                routes.routeConnected();
                routes.authFailed();
            } catch (CommunicationException ce) {
                routes.connectFailed();
            }
        }
        try {
            n.sendMessage(new QueryRejected(id, hopsToLive,
                                            na.otherFields), origRec);
        } catch (CommunicationException e) {
            n.logger.log(this, "Failed to send back QueryRejected.",
                         e, n.logger.MINOR);
        }
        return null;
    }
}




