/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: Resources.cc,v 1.15 2001/12/18 22:48:29 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/* $Id: Resources.cc,v 1.15 2001/12/18 22:48:29 vkurland Exp $ */

#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <fstream>
#include <cstdlib>

#include <parser.h>
#include <tree.h>

#include <config.h>


#include "Resources.hh"

#include "fwbuilder/FWObject.hh"
#include "fwbuilder/FWObjectDatabase.hh"
#include "fwbuilder/RuleElement.hh"

#include "fwbuilder/Tools.hh"
#include "helpers.hh"


using namespace libfwbuilder;

const string Resources::RES_FILE_NAME = "resources.xml";

Resources*  Resources::global_res = NULL;

Resources::Resources() throw(FWException)
{
    g_assert(global_res==NULL);

    global_res=this;

    struct passwd *pwd=getpwuid(getuid());
    g_assert(pwd!=NULL);

    resFile = TEMPLATE_DIR "/" +RES_FILE_NAME;
    
    doc=NULL;

    if(access(resFile.c_str() , R_OK )!=0)
            throw FWException("Could not load resources: "+resFile);

    loadRes();
}

void Resources::loadRes() throw(FWException)
{
    doc = xmlParseFile(resFile.c_str()); 
    
    if(!doc)
        throw FWException("Error parsing resources file");

    root=xmlDocGetRootElement(doc);
    
    if(!root || 
       !root->name || 
       strcmp(FROMXMLCAST(root->name), "FWBuilderResources")!=0) {

        xmlFreeDoc(doc);
        throw FWException("Invalid resources file");
    }
}

xmlNodePtr Resources::getXmlNode(const string& path)
{
    return XMLTools::getXmlNodeByPath(root,path);
}


string  Resources::getResourceStr(const char* resource_path)
{
    xmlNodePtr node=XMLTools::getXmlNodeByPath(root,resource_path);
    const char *res=NULL;

    if (node) 
	res= (char*)( xmlNodeGetContent(node) );

    if (res) return(res);
    else     return("");
}

string  Resources::getResourceStr(const string& resource_path)
{
    return getResourceStr(resource_path.c_str());
}

int     Resources::getResourceInt(const char*   resource_path)
{
    return atoi(getResourceStr(resource_path).c_str());
}

int     Resources::getResourceInt(const string& resource_path)
{
    return getResourceInt(resource_path.c_str());
}

string  Resources::getIconPath(const char* icon)
{
    string icn;

    icn= getResourceStr("/FWBuilderResources/Paths/Icndir");
    icn += "/";
    icn += getResourceStr(string("/FWBuilderResources/UI/Icons/")+icon);

    return icn;
}

string  Resources::getIconPath(const string& icon)
{
    return getIconPath(icon.c_str());
}

/**
 *
 *    Resources can be defined for the whole type and for individual
 *    object. This method first tries to find resource defined for
 *    the whole type and then for this particular object. The latter,
 *    if exists, override the former.
 *
 *    There is also a "default" entry for resources in the resources
 *    file. If neither per type or per object resources could be found,
 *    the "default" ones are used
 *
 *    Here is an example:
 *

  <Resources>
    <TypeResources Type="DEFAULT">
      <icon>blank.xpm</icon>
      <icon_dlg>blank.xpm</icon_dlg>
      <hidden>false</hidden>
      <system>false</system>
      <read>true</read>
      <delete>true</delete>
      <write>true</write>
      <rename>true</rename>
    </TypeResources>

    <TypeResources Type="Host">
      <description>Host</description>
      <icon>host.png</icon>
      <icon_dlg>host_64.png</icon_dlg>
    </TypeResources>

    <ObjectResources ID="sysid0">
      <description>Any</description>
      <hidden>true</hidden>
      <system>true</system>
      <read>false</read>
      <delete>false</delete>
      <write>false</write>
    </ObjectResources>

 *
 *
 */

string  Resources::getObjResourceStr(const FWObject *obj,
				     const char* resource_name)
{
    char         *cptr    = NULL;
    const char   *res     = NULL;

    string objid   = obj->getId();
    string objtype = obj->getTypeName();

    xmlNodePtr pn = XMLTools::getXmlNodeByPath(root,"/FWBuilderResources");

    xmlNodePtr opt;
    xmlNodePtr defopt=NULL;


    for(opt=pn->xmlChildrenNode; opt; opt=opt->next) {
	if ( xmlIsBlankNode(opt) ) continue;
	if ( strcmp(FROMXMLCAST(opt->name),"ObjectResources")==SAME) {
	    cptr=(char*)( xmlGetProp(opt,TOXMLCAST("ID")));
	    if (cptr && objid==cptr) {
		res=_getResourceFromXML(opt,resource_name);
		if (res && *res!='\0') return res;
	    }
	}
    }

    for(opt=pn->xmlChildrenNode; opt; opt=opt->next) {
	if ( xmlIsBlankNode(opt) ) continue;
	if ( strcmp(FROMXMLCAST(opt->name),"TypeResources")==SAME) {
	    cptr=(char*)( xmlGetProp(opt,TOXMLCAST("Type")));
	    if (cptr && strcmp("DEFAULT",cptr)==SAME) defopt=opt;
	    if (cptr && objtype==cptr) {
		res=_getResourceFromXML(opt,resource_name);
		if (res && *res!='\0') return res;
	    }
	}
    }


    if (defopt) 
	return _getResourceFromXML(defopt,resource_name);

    return "";
}

string  Resources::getObjResourceStr(const FWObject *obj,
					  const string& resource_name)
{
    return getObjResourceStr(obj,resource_name.c_str());
}

bool  Resources::getObjResourceBool(const FWObject *obj,const char* resource_name)
{
    string res=getObjResourceStr(obj,resource_name);
    if (res=="true") return true;
    return false;
}

bool  Resources::getObjResourceBool(const FWObject *obj,
				    const string&  resource_name)
{
    return getObjResourceBool(obj,resource_name.c_str());
}

const char*  Resources::_getResourceFromXML(xmlNodePtr res_node,
					    const char* res_name)
{
    xmlNodePtr opt;
    char      *res;

    for(opt=res_node->xmlChildrenNode; opt; opt=opt->next) {
	if ( xmlIsBlankNode(opt) ) continue;
	if ( strcmp(FROMXMLCAST(opt->name),res_name)==SAME)  {
	    res= (char*)( xmlNodeGetContent(opt) );
	    return res;
	}
    }
    return "";
}

string  Resources::getCompiler(const string& platform)
{
    char* r;
    xmlNodePtr pn=
	Resources::global_res->getXmlNode("/FWBuilderResources/Platforms/");
    xmlNodePtr opt;
		
    for(opt=pn->xmlChildrenNode; opt; opt=opt->next) {
	if ( xmlIsBlankNode(opt) ) continue;
	r= (char*) xmlGetProp(opt,TOXMLCAST("name"));
	if (r && platform==r) {

	    for(opt=opt->xmlChildrenNode; opt; opt=opt->next) {
		if ( xmlIsBlankNode(opt) ) continue;
		if ( strcmp(FROMXMLCAST(opt->name),"compiler")==SAME) {
		    r=(char*)xmlNodeGetContent(opt);
		    if (r) return r;
		}
	    }
	}
    }
    return ("");
}

vector<string>    Resources::getPlatforms()
{
    vector<string> vs;
    xmlNodePtr pn=
	Resources::global_res->getXmlNode("/FWBuilderResources/Platforms/");
    xmlNodePtr opt;
    char      *r;

    for(opt=pn->xmlChildrenNode; opt; opt=opt->next) {
	if ( xmlIsBlankNode(opt) ) continue;
	r= (char*) xmlGetProp(opt,TOXMLCAST("name"));
	if (r) vs.push_back(r);
    }
    return vs;
}


const char* Resources::getRuleElementResourceStr(const char* rel, 
						 const char* resource_name)

{
    xmlNodePtr  c,d;
    const char       *s;

    xmlNodePtr  dptr=Resources::global_res->getXmlNode("FWBuilderResources/RuleElements");

    assert (dptr!=NULL);

    for(c=dptr->xmlChildrenNode; c; c=c->next) {
	if ( xmlIsBlankNode(c) ) continue;

	if ( strcmp(rel,(char*)xmlGetProp(c,TOXMLCAST("RuleElement")) )==SAME ) {

	    d=XMLTools::getXmlChildNode(c,resource_name);
	    if (d) {
		s=(const char*)xmlNodeGetContent(d);
		return s;
	    }
	}
    }
    return NULL;
}

bool Resources::isSystem(const FWObject *o)
{
    return global_res->getObjResourceBool(o, "system");
}

string  Resources::getXPMIconFileName(const FWObject *o, const string& icon_sfx)
{ 
    string res;
    
    res=global_res->getResourceStr("/FWBuilderResources/Paths/Icndir");
    res += "/";
    res += global_res->getObjResourceStr(o, "icon"+icon_sfx);
    
    return res;
}

