/* 

                          Firewall Builder

                 Copyright (C) 2004 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: FWBSettings.cpp,v 1.38 2004/08/29 04:38:56 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "../../config.h"
#include "global.h"

#include "FWBSettings.h"
#include "FWWindow.h"
#include "ObjectManipulator.h"

#include "fwbuilder/FWObjectDatabase.h"

#include <qtextbrowser.h>
#include <qmessagebox.h>
#include <qapplication.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#ifdef _WIN32
#  include <direct.h>
#  include <stdlib.h>
#  include <io.h>
#else
#  include <unistd.h>     // for access(2)
#endif

#include <iostream>

using namespace std;
using namespace libfwbuilder;

/*
 * Note:
 *
 * We need to keep installation data and program settings in registry
 * folders with different names. QSettings always looks into Current
 * User registry first, so if the folders have the same names, then we
 * store evaluation key in Current User, while it is better to put it
 * in the Local Machine branch.
 *
 * So, installation data goes to HKLM Software\NetCitadel\FirewallBuilder
 * and settings to HKCU Software\NetCitadel\FirewallBuilder2
 * 
 * fwbuilder-lm determines folder path for the license file by
 * reading key Install_Dir under HKLM Software\NetCitadel\FirewallBuilder
 */

const char* DTDSetpath           = "/FirewallBuilder2/System/DTDPath";
const char* ResSetpath           = "/FirewallBuilder2/System/ResPath";
const char* wdirSetpath          = "/FirewallBuilder2/Environment/WDir";
const char* ofdirSetpath         = "/FirewallBuilder2/Environment/OpenFileDir";
const char* sfdirSetpath         = "/FirewallBuilder2/Environment/SaveFileDir";
const char* startupActionSetpath = "/FirewallBuilder2/Environment/StartupAction";
const char* labelColorPath       = "/FirewallBuilder2/ColorLabels/color_";
const char* labelTextPath        = "/FirewallBuilder2/ColorLabels/text_";
const char* lastEditedSetpath    = "/FirewallBuilder2/Environment/LastEdited";
const char* autoSave             = "/FirewallBuilder2/Environment/autoSave";
const char* expandTreeSetpath    = "/FirewallBuilder2/UI/ExpandTree";
const char* MergeLibsSetpath     = "/FirewallBuilder2/UI/MergeLibraries";
const char* infoStyleSetpath     = "/FirewallBuilder2/UI/InfoWindowStyle";
const char* infoWindowHSetpath   = "/FirewallBuilder2/UI/InfoWindowHeight";
const char* groupModeSetpath     = "/FirewallBuilder2/UI/GroupViewMode";
const char* groupColsSetpath     = "/FirewallBuilder2/UI/GroupViewColumns";
const char* objTooltips          = "/FirewallBuilder2/UI/objTooltips";
const char* tooltipDelay         = "/FirewallBuilder2/UI/tooltipDelay";
const char* emptyRCSLog          = "/FirewallBuilder2/RCS/emptyLog";
const char* dontSaveStdLib       = "/FirewallBuilder2/DataFormat/dontSaveStdLib";
const char* WindowGeometrySetpath = "/FirewallBuilder2/Layout/";
const char* screenPositionSetpath = "/FirewallBuilder2/ScreenPos/";

const char* SSHPath           = "/FirewallBuilder2/SSH/SSHPath";
const char* SCPPath           = "/FirewallBuilder2/SSH/SCPPath";

FWBSettings::FWBSettings() : QSettings()
{
    setPath("netcitadel.com", "Firewall Builder", QSettings::User);
    insertSearchPath( QSettings::Windows, "/NetCitadel" );

//    writeEntry( DTDSetpath, librespath.c_str() );
//    writeEntry( ResSetpath, respath.c_str()   );
}

/**
 *  to preserve behavior of the old versions of fwbuilder on Unix, the
 *  default working dir is set to "." - current dir.
 *
 *  On Windows default working dir is set to
 *             "Documents and settings/USERNAME/Firewalls"
 */
void FWBSettings::init()
{
    bool ok=false;
/*
    QString defwd = 
#ifdef _WIN32
        QString(getenv("HOMEPATH"))+"/Firewalls";
#else
        "";
#endif
    QString wd = readEntry(wdirSetpath,defwd, &ok);
    if (!ok)  writeEntry(wdirSetpath, defwd );
*/

    readNumEntry(infoStyleSetpath,2,&ok);
    if (!ok) writeEntry(infoStyleSetpath,2);

    readNumEntry(infoWindowHSetpath,200,&ok);
    if (!ok) writeEntry(infoWindowHSetpath,200);

    readBoolEntry(dontSaveStdLib,true,&ok);
    if (!ok) setDontSaveStdLib(true);

#ifdef _WIN32
    QString wd = getWDir().replace('/','\\');
#else
    QString wd = getWDir();
#endif
    if ( ! wd.isEmpty())
    {
#ifdef _WIN32
        if ( _access(wd.latin1(), F_OK)!=0 )
#else
        if ( access(wd.latin1(), F_OK)!=0 )
#endif
        {
            if ( QMessageBox::warning(
                     0,"Firewall Builder", 
                     QObject::tr("Working directory %1 does not seem to exist.\nDo you want to create it ?")
                     .arg(wd.latin1()),
                     "&Create", "&Ignore", QString::null,
                     0, 1 )==0)
            {
#ifdef _WIN32
                _mkdir(wd.latin1() );
#else
                mkdir(wd.latin1(), 0775 );
#endif
            }
        }
    }

    QString c;
    if (getLabelColor(RED   ).isEmpty())
    { setLabelColor(RED   ,"#C86E6E"); setLabelText(RED   ,"Red"); }
    if (getLabelColor(ORANGE).isEmpty())
    { setLabelColor(ORANGE,"#C08B5A"); setLabelText(ORANGE,"Orange"); }
    if (getLabelColor(YELLOW).isEmpty())
    { setLabelColor(YELLOW,"#C0BA44"); setLabelText(YELLOW,"Yellow"); }
    if (getLabelColor(GREEN ).isEmpty())
    { setLabelColor(GREEN ,"#8BC065"); setLabelText(GREEN ,"Green"); }
    if (getLabelColor(BLUE  ).isEmpty())
    { setLabelColor(BLUE  ,"#7694C0"); setLabelText(BLUE  ,"Blue"); }
    if (getLabelColor(PURPLE).isEmpty())
    { setLabelColor(PURPLE,"#A37EC0"); setLabelText(PURPLE,"Purple"); }
    if (getLabelColor(GRAY  ).isEmpty())
    { setLabelColor(GRAY  ,"#C0C0C0"); setLabelText(GRAY  ,"Gray"); }

#ifndef _WIN32
    if (getSCPPath().isEmpty())  setSCPPath("scp");
    if (getSSHPath().isEmpty())  setSSHPath("ssh");
#endif
}

QString FWBSettings::getStr(const QString &attribute)
{
    QString path="/FirewallBuilder2/"+attribute;
    return readEntry(path);
}

void    FWBSettings::setStr(const QString &attribute,
                            const QString &val)
{
    QString path="/FirewallBuilder2/"+attribute;
    writeEntry(path,val);
}

bool    FWBSettings::getBool(const QString &attribute)
{
    QString path="/FirewallBuilder2/"+attribute;
    return readBoolEntry(path);
}

void    FWBSettings::setBool(const QString &attribute, bool val )
{
    QString path="/FirewallBuilder2/"+attribute;
    writeEntry(path,val);
}

    
int     FWBSettings::getInt(const QString &attribute)
{
    QString path="/FirewallBuilder2/"+attribute;
    return readNumEntry(path);
}

void    FWBSettings::setInt(const QString &attribute, int val )
{
    QString path="/FirewallBuilder2/"+attribute;
    writeEntry(path,val);
}

    

QString FWBSettings::getWDir() { return readEntry(wdirSetpath);}
void    FWBSettings::setWDir( const QString &wd ) { writeEntry(wdirSetpath,wd);}
int     FWBSettings::getInfoStyle() { return readNumEntry(infoStyleSetpath);}
void    FWBSettings::setInfoStyle(int s) { writeEntry(infoStyleSetpath,s);}
int     FWBSettings::getInfoWindowHeight() { return readNumEntry(infoWindowHSetpath);}
void    FWBSettings::setInfoWindowHeight(int h) { writeEntry(infoWindowHSetpath,h);}

QString FWBSettings::getGroupViewMode() { return readEntry(groupModeSetpath);}
void    FWBSettings::setGroupViewMode(const QString &m) { writeEntry(groupModeSetpath,m);}

QString FWBSettings::getGroupViewColumns() { return readEntry(groupColsSetpath);}
void    FWBSettings::setGroupViewColumns(const QString &m) { writeEntry(groupColsSetpath,m);}


int     FWBSettings::getStartupAction() { return readNumEntry(startupActionSetpath);}
void    FWBSettings::setStartupAction(int sa) { writeEntry( startupActionSetpath , sa );}

int     FWBSettings::getExpandTree() {  return readNumEntry(expandTreeSetpath); }
void    FWBSettings::setExpandTree(int f) { writeEntry( expandTreeSetpath , f ); }

int     FWBSettings::getMergeLibs() {  return readNumEntry(MergeLibsSetpath); }
void    FWBSettings::setMergeLibs(int f) { writeEntry( MergeLibsSetpath , f ); }

bool    FWBSettings::getObjTooltips() {  return readBoolEntry( objTooltips );}
void    FWBSettings::setObjTooltips(bool f) {  writeEntry( objTooltips, f); }

int     FWBSettings::getTooltipDelay() { return readNumEntry( tooltipDelay ); }
void    FWBSettings::setTooltipDelay(int v) { writeEntry( tooltipDelay, v); }
    

QString FWBSettings::getLastEdited() { return readEntry(lastEditedSetpath);}
void    FWBSettings::setLastEdited(const QString &file) { writeEntry(lastEditedSetpath,file);}

QString FWBSettings::getOpenFileDir()
{
    return readEntry(ofdirSetpath);
}

void    FWBSettings::setOpenFileDir( const QString &d )
{
     writeEntry(ofdirSetpath,d);
}

QString FWBSettings::getSaveFileDir()
{
    return readEntry(sfdirSetpath);
}

void    FWBSettings::setSaveFileDir( const QString &d )
{
     writeEntry(sfdirSetpath,d);
}


void    FWBSettings::save()
{
    setLastEdited( FWObjectDatabase::db->getFileName().c_str() );

    if (getInfoStyle()!=0) setInfoWindowHeight(oi->geometry().height());
}

bool    FWBSettings::getRCSLogState() { return readBoolEntry( emptyRCSLog ); }
void    FWBSettings::setRCSLogState(bool f) { writeEntry( emptyRCSLog , f ); }

bool    FWBSettings::getAutoSave() { return readBoolEntry( autoSave ); }
void    FWBSettings::setAutoSave(bool f) { writeEntry( autoSave, f); }

bool    FWBSettings::getDontSaveStdLib() {return readBoolEntry(dontSaveStdLib);}
void    FWBSettings::setDontSaveStdLib( bool f) { writeEntry(dontSaveStdLib,f);}

bool    FWBSettings::haveScreenPosition(const QString &wname)
{
    QString val = readEntry(QString(screenPositionSetpath)+wname );
    bool res=(!val.isEmpty());

    if (fwbdebug)
    {
	qDebug("FWBSettings::haveScreenPosition wname '%s' ret=%d",
		wname.latin1(), res);
    }

    return res;
}

QPoint  FWBSettings::getScreenPosition(const QString &wname)
{
    QString val = readEntry(QString(screenPositionSetpath)+wname );
    int     x  = val.section(',',0,0).toInt();
    int     y  = val.section(',',1,1).toInt();
    int  width = 150;  // won't get closer to the screen edge than this
    int height = 150;

    QDesktopWidget *d = QApplication::desktop();
// get geometry of the screen that contains mw
    QRect sg = d->screenGeometry(mw);

    if (x+width > sg.width())   x=sg.width()-width;
    if (y+height > sg.height()) y=sg.height()-height;
    if (x<0)                    x=(sg.width()-width)/2;
    if (y<0)                    y=(sg.height()-height)/2;

    if (fwbdebug)
    {
	qDebug("FWBSettings::getScreenPosition wname '%s' x=%d y=%d",
		wname.latin1(), x,y );
    }

    return QPoint(x,y);
}

void    FWBSettings::saveScreenPosition(const QString &wname, const QPoint &p)
{
    int x = p.x();
    int y = p.y();
    if (x<0) x=0;
    if (y<0) y=0;

    QString val =QString("%1,%2").arg(x).arg(y);

    if (fwbdebug)
    {
	qDebug("FWBSettings::saveScreenPosition wname '%s' x=%d y=%d",
		wname.latin1(), x,y );
    }

    writeEntry(QString(screenPositionSetpath)+wname, val );
}

bool    FWBSettings::haveGeometry(QWidget *w)
{
    QString name=w->name();
    QString val = readEntry(QString(WindowGeometrySetpath)+name,"");
    return (!val.isEmpty());
}

void    FWBSettings::restoreGeometry(QWidget *w)
{
    QString name=w->name();
    QString val = readEntry(QString(WindowGeometrySetpath)+name );
    int     x      = val.section(',',0,0).toInt();
    int     y      = val.section(',',1,1).toInt();
    int     width  = val.section(',',2,2).toInt();
    int     height = val.section(',',3,3).toInt();

    QDesktopWidget *d = QApplication::desktop();
// get geometry of the screen that contains mw
    QRect sg = d->screenGeometry(mw);

    if (x+width > sg.width())   x=sg.width()-width;
    if (y+height > sg.height()) y=sg.height()-height;
    if (x<0)                    x=(sg.width()-width)/2;
    if (y<0)                    y=(sg.height()-height)/2;

    if (fwbdebug)
    {
	qDebug("FWBSettings::restoreGeometry  widget '%s' vis=%d x=%d y=%d",
               w->name(), w->isVisible(), x,y);
    }

    w->resize( QSize(width,height) );
    w->move( QPoint(x,y) );
}

void    FWBSettings::restoreGeometry(QWidget *w, const QRect &dg)
{
    QString name=w->name();
    QString defval =QString("%1,%2,%3,%4")
        .arg(dg.x()).arg(dg.y()).arg(dg.width()).arg(dg.height());

    QString val = readEntry(QString(WindowGeometrySetpath)+name , defval );
    int     x      = val.section(',',0,0).toInt();
    int     y      = val.section(',',1,1).toInt();
    int     width  = val.section(',',2,2).toInt();
    int     height = val.section(',',3,3).toInt();

    QDesktopWidget *d = QApplication::desktop();
// get geometry of the screen that contains mw
    QRect sg = d->screenGeometry(mw);

    if (x+width > sg.width())   x=sg.width()-width;
    if (y+height > sg.height()) y=sg.height()-height;
    if (x<0)                    x=(sg.width()-width)/2;
    if (y<0)                    y=(sg.height()-height)/2;

    if (fwbdebug)
    {
	qDebug("FWBSettings::restoreGeometry  widget '%s' vis=%d x=%d y=%d",
               w->name(), w->isVisible(), x,y);
    }

//    w->setGeometry( QRect(x,y,width,height) );

    w->resize( QSize(width,height) );
    w->move( QPoint(x,y) );
}

void    FWBSettings::saveGeometry(QWidget *w)
{
    QString name = w->name();
//    QRect   g = w->geometry();
//    g.moveTopLeft(w->frameGeometry().topLeft());
    QPoint p = w->pos();
    QSize  s = w->size();

    int x = p.x();
    int y = p.y();
    if (x<0) x=0;
    if (y<0) y=0;

    QString val =QString("%1,%2,%3,%4")
        .arg(p.x())
        .arg(p.y())
        .arg(s.width())
        .arg(s.height());
    
    if (fwbdebug)
    {
	qDebug("FWBSettings::saveGeometry  widget '%s' vis=%d val=%s",
               w->name(), w->isVisible(), val.ascii());
    }

    writeEntry(QString(WindowGeometrySetpath)+name, val );
}


QString FWBSettings::getLabelColorStr(enum LabelColors c)
{
    switch (c) 
    {
    case RED:    return "red";
    case ORANGE: return "orange";
    case YELLOW: return "yellow";
    case GREEN:  return "green";
    case BLUE:   return "blue";
    case PURPLE: return "purple";
    case GRAY:   return "gray";
    default:     return "default";
    }
}

QString FWBSettings::getLabelColor(enum LabelColors c)
{
    return readEntry(QString(labelColorPath) + getLabelColorStr(c));
}

void    FWBSettings::setLabelColor(enum LabelColors c,const QString &s)
{
    writeEntry(QString(labelColorPath) + getLabelColorStr(c), s);
}

QString FWBSettings::getLabelText(enum LabelColors c)
{
    return readEntry(QString(labelTextPath) + getLabelColorStr(c));
}

void    FWBSettings::setLabelText(enum LabelColors c, const QString &s)
{
    writeEntry(QString(labelTextPath) + getLabelColorStr(c),s);
}

QString FWBSettings::getSCPPath()
{
    return readEntry(SCPPath);
}

void    FWBSettings::setSCPPath(const QString &path)
{
    writeEntry(SCPPath,path);
}
    
QString FWBSettings::getSSHPath()
{
    return readEntry(SSHPath);
}

void    FWBSettings::setSSHPath(const QString &path)
{
    writeEntry(SSHPath,path);
}
    
