/**
 * @file meta-service.c Meta-Service API
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "meta-service.h"
#include "service-list.h"
#include "filter.h"

GalagodMetaService *
galagod_meta_service_new(const char *id, const char *name,
						 GalagoServiceFlags flags)
{
	GalagodMetaService *meta_service;
	char *obj_path;

	g_return_val_if_fail(id   != NULL, NULL);
	g_return_val_if_fail(name != NULL, NULL);

	galago_context_push(galagod_get_context());

	meta_service = g_new0(GalagodMetaService, 1);

	meta_service->service = galago_create_service(id, name, flags);
	meta_service->account_list = galagod_account_list_new(meta_service);

	obj_path = galagod_service_generate_path(meta_service->service);
	galago_object_set_dbus_path(GALAGO_OBJECT(meta_service->service),
								obj_path);
	g_free(obj_path);

	galago_context_pop();

	g_object_set_data(G_OBJECT(meta_service->service),
					  "meta-service", meta_service);

	galagod_services_add_meta_service(meta_service);
	galagod_service_add_filter(meta_service->service);

	return meta_service;
}

void
galagod_meta_service_destroy(GalagodMetaService *meta_service)
{
	g_return_if_fail(meta_service != NULL);

	while (meta_service->services != NULL)
	{
		galagod_meta_service_remove_service(meta_service,
			(GalagoService *)meta_service->services->data);
	}

	galagod_services_remove_meta_service(meta_service);
	galagod_service_remove_filter(meta_service->service);
	galagod_account_list_destroy(meta_service->account_list);
	g_object_unref(meta_service->service);

	g_free(meta_service);
}

GalagoService *
galagod_meta_service_get_service(const GalagodMetaService *meta_service)
{
	g_return_val_if_fail(meta_service != NULL, NULL);

	return meta_service->service;
}

GList *
galagod_meta_service_get_services(const GalagodMetaService *meta_service)
{
	g_return_val_if_fail(meta_service != NULL, NULL);

	return meta_service->services;
}

GalagodAccountList *
galagod_meta_service_get_account_list(const GalagodMetaService *meta_service)
{
	g_return_val_if_fail(meta_service != NULL, NULL);

	return meta_service->account_list;
}

void
galagod_meta_service_add_service(GalagodMetaService *meta_service,
								 GalagoService *service)
{
	g_return_if_fail(meta_service != NULL);
	g_return_if_fail(service      != NULL);

	meta_service->services = g_list_append(meta_service->services, service);

	g_object_set_data(G_OBJECT(service), "meta-service", meta_service);

	galagod_service_add_filter(service);
}

void
galagod_meta_service_remove_service(GalagodMetaService *meta_service,
									GalagoService *service)
{
	g_return_if_fail(meta_service != NULL);
	g_return_if_fail(service      != NULL);

	galagod_service_remove_filter(service);

	g_object_set_data(G_OBJECT(service), "meta-service", NULL);

	meta_service->services = g_list_remove(meta_service->services, service);

	galagod_account_list_remove_with_service(meta_service->account_list,
											 service);
}
