/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "window.h"
#include "session.h"
#include "embed.h"
#include "dialog.h"
#include "misc_callbacks.h"
#include "misc_gui.h"
#include "misc_string.h"
#include "menubar.h"
#include "toolbar.h"
#include "history.h"
#include "bookmarks.h"
#include "bookmarks_editor.h"
#include "prefs.h"
#include "find.h"
#include "glade.h"
#include "gfilepicker.h"
#include "mozilla_i18n.h"
#include "spinner.h"
#include "window_notifiers.h"
#include "themes.h"
#include "link_interfaces.h"
#include "mozilla.h"
#include "favicon.h"

#include <string.h>
#include <time.h>
#include <stdio.h>

#include <gtk/gtkdnd.h>
#include <gtk/gtkpaned.h>
#include <gtkmozembed.h>

/* GNOME includes */
#include <libgnomeui/gnome-animator.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-winhints.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomeui/gtkpixmapmenuitem.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-uri.h>

#ifdef HAVE_XINERAMA
#include <X11/X.h>
#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <X11/extensions/Xinerama.h>
#endif

/* local function prototypes */
static void window_init_data (GaleonWindow *window);
static void window_set_menu_data (GaleonWindow *window);
static void window_menubar_create (GaleonWindow *window);
static void window_toolbars_create (GaleonWindow *window);
static void window_create_embed_container (GaleonWindow *window);
static void window_set_chrome (GaleonWindow *window, guint32 chrome_mask);
static void window_init_menu_data (GaleonWindow *window);
static void window_statusbar_create (GaleonWindow *window);
static gint strcasestr_cb (gconstpointer a, gconstpointer b);
static GtkWidget *window_notebook_tab_new (GaleonEmbed *embed,
					   const gchar *title);
static GtkWidget *window_create_recent_menu_item (GaleonWindow *window,
						  RecentMenuEntry *entry,
						  gint num);
static void window_add_recent_menu_entry_add_menuitem (GaleonWindow *window,
						       RecentMenuEntry *entry,
						       gint num);
static void window_remove_recent_menu_entry (GaleonWindow *window,
					     RecentMenuEntry *entry);
static void window_recent_menu_entry_update_number (GaleonWindow *window, 
						    RecentMenuEntry *entry,
						    gint num);

/* callbacks used here */
void file_open_ok_button_clicked_cb (const gchar *file, gpointer data);
static void window_menu_recent_select_cb (GtkMenuItem *menuitem,
					  GaleonWindow *window);
static void window_menu_recent_deselect_cb (GtkMenuItem *menuitem,
					    GaleonWindow *window);

/* the types which can be dropped either into the GtkMozEmbed or onto
 * the location entry field */
const GtkTargetEntry drop_types[] =
{
	{ "GALEON_URL",    0, DND_TARGET_GALEON_URL    },
	{ "text/uri-list", 0, DND_TARGET_TEXT_URI_LIST },
	{ "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL  },
	{ "STRING",        0, DND_TARGET_STRING        }
};
const gint drop_types_num_items = (sizeof (drop_types) / 
				   sizeof (GtkTargetEntry));

/* the types which can be dragged from various places in Galeon, such as
 * the bookmarks editor, the toolbar bookmark icon, and the history */
const GtkTargetEntry url_drag_types[] = 
{
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK },
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint url_drag_types_num_items = (sizeof (url_drag_types) /
				       sizeof (GtkTargetEntry));

/** Where does the recent menu start? */
#define GO_RECENT_POS 9

/** How many items to put in the recent menu */
#define GO_RECENT_MAX 10

/** Is fullscreen active? */
static gboolean fullscreen_active = FALSE;

/**
 * window_load_url: load a URL into the active embed of a window
 */
void
window_load_url (GaleonWindow *window, const gchar *url)
{
	GaleonEmbed *embed;

	/* check arguments */
	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	/* load into the active GaleonEmbed */
	embed_load_url (embed, url);
}

/**
 * window_set_location_entry_text: sets the location entry to the given url
 */
void
window_set_location_entry_text (GaleonWindow *window, const gchar *url)
{
	return_if_not_window (window);

	if (window->location_entry)
	{	
		/* clear text */
		gtk_editable_delete_text (
				GTK_EDITABLE (window->location_entry), 0, -1);
		/* set text */
		if (url) gtk_entry_set_text (GTK_ENTRY (
						window->location_entry), url);
	}
}

/**
 * window_update_location_entry_text: update the window's location entry
 * text to reflect the active embed
 */
void
window_update_location_entry_text (GaleonWindow *window)
{
	GaleonEmbed *embed;
	gchar *location;

	embed = window->active_embed;
	return_if_not_embed (embed);

	/* if we have a saved modified location, it means that we just
	 * switched back to this embed.  use the modified text. */
	if (embed->modified_location)
		location = embed->modified_location;
	/* otherwise, use the actual location */
	else location = embed->location;

	/* clear if we'd be setting to about:blank */
	if (location && !strcmp (location, "about:blank"))
		location = NULL;

	/* finally, set the location entry text */
	window_set_location_entry_text (window, location);
}

/**
 * window_update_title: update the window's title to reflect the active
 * embed
 */
void
window_update_title (GaleonWindow *window)
{
	gchar *title_string, *full_title;
	GaleonEmbed *embed;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);
	
	/* get the format string */
	title_string = eel_gconf_get_string (CONF_WINDOWS_TITLE);
	if (!title_string) return;

	/* format the full title */
	full_title = misc_string_strdup_replace (title_string, "%s",
						 embed->title);
	
	/* set the toplevel window title to the document title */
	gtk_window_set_title (GTK_WINDOW (window->wmain), full_title);
	
	/* free allocated strings */
	g_free (title_string);  
	if (full_title) g_free (full_title);
}

/**
 * window_create: create a browser structure and main window, 
 * but without an embedding widget. Should only be called from
 * embed_create
 */
GaleonWindow *
window_create (guint32 chrome_mask, GaleonWindow *previous)
{
        GaleonWindow *window;
	gchar *file;
	
	/* allocate and initialise the GaleonWindow structure */
	window = g_new0 (GaleonWindow, 1);

	/* set magic */
	window->magic = GALEON_WINDOW_MAGIC;

	window_set_chrome (window, chrome_mask);

	/* create the browser window */
	window->wmain = gnome_app_new ("galeon", _("Galeon"));
	gtk_window_set_wmclass (GTK_WINDOW (window->wmain), "Galeon",
				"galeon_browser");

	window->layered = FALSE;

	/* initialize GaleonWindow data */
	window_init_data (window);

	/* create tooltips */
	window->bookmarks_tooltips = gtk_tooltips_new ();
	gtk_tooltips_enable (window->bookmarks_tooltips);
	
	/* create the statusbar */
	window_statusbar_create (window);

	/* get the dock */
	window->dock = GNOME_APP (window->wmain)->dock;
	window->dock_type = DOCK_NONE;

	/* clear drag stuff */
	window->drag_motion_signal = 0;
	window->drag_leave_signal = 0;
	window->in_predrag = FALSE;
	window->in_drag = FALSE;

	/* the spinner isn't running yet */
	window->spinner_running = 0;

	/* save the toolbar and menubar layout */
	window_save_layout (window);

	/* create the toolbar */
	window_toolbars_create (window);

	/* create the menubar */
	window_menubar_create (window);

	/* add notifiers for the window */
	window_init_notifiers (window);

	/* restore the toolbar and menubar layout */
	window_restore_layout (window);

	/* set mini icon */
	file = gnome_pixmap_file ("galeon.png");
	gnome_window_icon_set_from_file (GTK_WINDOW (window->wmain), file);
	g_free (file);

	/* set window policies regarding resizing */
	gtk_window_set_policy (GTK_WINDOW (window->wmain), TRUE, TRUE, TRUE);

	/* so widgets can look the data up */
	gtk_object_set_data (GTK_OBJECT (window->wmain), "GaleonWindow",
			     window);

	/* connect the delete signal handler*/
	gtk_signal_connect(GTK_OBJECT(window->wmain), "delete-event",
			   GTK_SIGNAL_FUNC(window_delete_cb),
			   window);

	gtk_signal_connect(GTK_OBJECT(window->wmain), "key-press-event",
			   GTK_SIGNAL_FUNC(window_key_press_event),
			   window);

 	gtk_signal_connect (GTK_OBJECT(window->wmain), "selection_received",
 			    GTK_SIGNAL_FUNC (window_selection_received_cb),
 			    window);

	/* NB: window isn't shown until we add an embed */
	/* don't know the size at which to show it yet -- if we discover
	 * a size (e.g. for a popup) then we'll use that, otherwise we'll
	 * find a default, but not until the window is made visible */
	window->set_size = FALSE;
	
	/* set size to match that of the last used window */
	if (previous)
	{
		window->requested_width = previous->wmain->allocation.width;
		window->requested_height = previous->wmain->allocation.height;
	}

	/* add to session */
	session_add_window (window);

	/* create widgets used to pack embeds in */
	window_create_embed_container (window);

	/* setup visibility according to config */
	if (!eel_gconf_get_boolean (CONF_WINDOWS_SHOW_TOOLBARS) ||
	    !window->toolbar_on)
	{
		window_toolbar_hide (window);
	}
	if (!eel_gconf_get_boolean (CONF_WINDOWS_SHOW_MENUBAR) ||
	    !window->menubar_on)
	{
		window_menubar_hide (window);
	}
	if (!eel_gconf_get_boolean (CONF_WINDOWS_SHOW_STATUSBAR) ||
	    !window->statusbar_on)
	{
		window_statusbar_hide (window);
	}
	if (!eel_gconf_get_boolean (CONF_WINDOWS_SHOW_BOOKMARKS) ||
	    !window->toolbar_on)
	{
		bookmarks_toolbars_set_visibility (window, FALSE);
		if (window->hbox)
		{
			gtk_widget_queue_resize (GTK_WIDGET (window->hbox));
		}
	}
	if (eel_gconf_get_integer (CONF_WINDOWS_SHOW_DOCK) == 1)
	{
		history_show_dock (window);
	}
	if (eel_gconf_get_integer (CONF_WINDOWS_SHOW_DOCK) == 2)
	{
		bookmarks_editor_show_dock (window);
	}
	
	/* return the completed GaleonWindow */
	return window;
}

/**
 * window_init_data: init GaleonWindow data structure
 */
static void
window_init_data (GaleonWindow *window)
{
	/* these are all nullified */
	window->spinner = NULL;
	window->bookmarks_toolbars = NULL;
	window->bookmarks_tooltips = NULL;
	window->recent_menu_entries = NULL;
}

static void
window_init_menu_data (GaleonWindow *window)
{
	/* menus */
	window->file_menu =
		GTK_MENU_ITEM (menubar_uiinfo[0].widget)->submenu;
	window->bookmarks_menu = 
		GTK_MENU_ITEM (menubar_uiinfo[6].widget)->submenu;
	window->go_menu = 
		GTK_MENU_ITEM (menubar_uiinfo[5].widget)->submenu;
		
	/* file menu */
	window->recent_sessions_menuitem = file_menu_uiinfo[11].widget;
		
	/* edit menu */
	window->edit_cut = edit_menu_uiinfo[0].widget;
	window->edit_copy = edit_menu_uiinfo[1].widget;
	window->edit_paste = edit_menu_uiinfo[2].widget;

	/* view menu */
	window->view_menubar = view_menu_uiinfo[0].widget;
	window->view_toolbar = view_menu_uiinfo[1].widget;
	window->view_bookmarks = view_menu_uiinfo[2].widget;
	window->view_statusbar = view_menu_uiinfo[3].widget;
	window->view_fullscreen = view_menu_uiinfo[4].widget;
	window->encoding = view_menu_uiinfo[7].widget;
	window->csslist = view_menu_uiinfo[8].widget;

	/* tab menu */
	window->detach_tab = tab_menu_uiinfo[0].widget;
	window->move_tab_to_window_menu = tab_menu_uiinfo[1].widget;
	window->prev_tab = tab_menu_uiinfo[3].widget;
	window->next_tab = tab_menu_uiinfo[4].widget;
	window->move_tab_left = tab_menu_uiinfo[6].widget;
	window->move_tab_right = tab_menu_uiinfo[7].widget;

	/* settings menu */
	window->load_images_always = load_images_always_uiinfo[0].widget;
	window->load_images_from_current_server_only = 
					load_images_always_uiinfo[1].widget;
	window->load_images_never = load_images_always_uiinfo[2].widget;

	window->animate_always = animate_always_uiinfo[0].widget;
	window->animate_once_through = animate_always_uiinfo[1].widget;
	window->animate_never = animate_always_uiinfo[2].widget;

	window->no_proxy = proxies_always_uiinfo[0].widget;
	window->manual_proxy = proxies_always_uiinfo[1].widget;
	window->autoproxy = proxies_always_uiinfo[2].widget;
	  
	window->use_own_fonts =  settings_menu_uiinfo[4].widget;
	window->use_own_colors = settings_menu_uiinfo[5].widget;
	window->enable_java = settings_menu_uiinfo[6].widget;
	window->enable_javascript = settings_menu_uiinfo[7].widget;
	window->allow_popups = settings_menu_uiinfo[8].widget;

	/* bookmarks menu */
	window->add_bookmark = bookmarks_menu_uiinfo[1].widget;
	window->bookmarks_separator = bookmarks_menu_uiinfo[3].widget;

	/* go menu */
	window->back_menuitem = go_menu_uiinfo[0].widget;
	window->forward_menuitem = go_menu_uiinfo[1].widget;
	window->up_menuitem = go_menu_uiinfo[2].widget;
	window->refresh_menuitem = go_menu_uiinfo[3].widget;
	window->homepage_menuitem = go_menu_uiinfo[4].widget;
	window->related_links_menuitem = go_menu_uiinfo[7].widget;

	/* tools menu */
	window->java_console = tools_menu_uiinfo[6].widget;

	window->open_url_entry = NULL;

	/* connect signal for the move_tab_to_window_menu */
	gtk_signal_connect (GTK_OBJECT (window->move_tab_to_window_menu),
			    "activate", window_move_tab_to_window_menu_cb,
			    window);
	gtk_menu_item_set_submenu
		(GTK_MENU_ITEM (window->move_tab_to_window_menu),
		 gtk_menu_new ());

	/* create submenu for the related links menu */
	gtk_menu_item_set_submenu
		(GTK_MENU_ITEM (window->related_links_menuitem),
		 gtk_menu_new ());

	/* connect signals for go menu */
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[0].widget),
			    "button-press-event",
			    GTK_SIGNAL_FUNC
			   	(window_back_forward_button_press_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[1].widget),
			    "button-press-event",
			    GTK_SIGNAL_FUNC
			   	(window_back_forward_button_press_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[2].widget),
			    "button-press-event",
			    GTK_SIGNAL_FUNC
			   	(window_up_button_press_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[3].widget),
			    "button-press-event",
			    GTK_SIGNAL_FUNC
			   	(window_refresh_button_press_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[4].widget),
			    "button-press-event",
			    GTK_SIGNAL_FUNC
			   	(window_home_button_press_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[5].widget),
			    "button-press-event",
			    GTK_SIGNAL_FUNC
			   	(window_stop_button_press_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[0].widget),
			    "button-release-event",
			    GTK_SIGNAL_FUNC
			   	(window_back_forward_button_release_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[1].widget),
			    "button-release-event",
			    GTK_SIGNAL_FUNC
			    	(window_back_forward_button_release_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[2].widget),
			    "button-release-event",
			    GTK_SIGNAL_FUNC
			    	(window_up_button_release_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[3].widget),
			    "button-release-event",
			    GTK_SIGNAL_FUNC
			    	(window_refresh_button_release_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (go_menu_uiinfo[4].widget),
			    "button-release-event",
			    GTK_SIGNAL_FUNC
			    	(window_home_button_release_event_cb),
			    window);

	/* connect signals for help menu */
	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[0].widget),
			     "url", GALEON_FAQ_URL);
	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[2].widget),
			     "url", GALEON_HOMEPAGE_URL);
	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[3].widget),
			     "url", GALEON_BUGREPORT_URL);
	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[4].widget),
			     "url", "about:plugins");

	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[0].widget),
			     "title", help_menu_uiinfo[0].label);
	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[1].widget),
			     "title", help_menu_uiinfo[1].label);
	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[2].widget),
			     "title", help_menu_uiinfo[2].label);
	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[3].widget),
			     "title", help_menu_uiinfo[3].label);
	gtk_object_set_data (GTK_OBJECT (help_menu_uiinfo[4].widget),
			     "title", help_menu_uiinfo[4].label);
	
	gtk_signal_connect (GTK_OBJECT (help_menu_uiinfo[0].widget),
			    "button-release-event",
			    window_menu_help_generic_release_cb,
			    window);
	gtk_signal_connect (GTK_OBJECT (help_menu_uiinfo[2].widget),
			    "button-release-event",
			    window_menu_help_generic_release_cb,
			    window);
	gtk_signal_connect (GTK_OBJECT (help_menu_uiinfo[3].widget),
			    "button-release-event",
			    window_menu_help_generic_release_cb,
			    window);
	gtk_signal_connect (GTK_OBJECT (help_menu_uiinfo[4].widget),
			    "button-release-event",
			    window_menu_help_generic_release_cb,
			    window);

	gtk_signal_connect (GTK_OBJECT (help_menu_uiinfo[0].widget),
			    "button-press-event", GTK_SIGNAL_FUNC
			    (generic_link_button_press_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (help_menu_uiinfo[2].widget),
			    "button-press-event", GTK_SIGNAL_FUNC
			    (generic_link_button_press_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (help_menu_uiinfo[3].widget),
			    "button-press-event", GTK_SIGNAL_FUNC
			    (generic_link_button_press_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT (help_menu_uiinfo[4].widget),
			    "button-press-event", GTK_SIGNAL_FUNC
			    (generic_link_button_press_event_cb),
			    window);
}

static void
window_menubar_create (GaleonWindow *window)
{
        g_assert (window->wmain);

	/* need to set the user_data field in all the GnomeUIInfo structs */
	window_set_menu_data (window);

	gnome_app_create_menus (GNOME_APP(window->wmain), 
				menubar_uiinfo);

	window->menubar = GNOME_APP(window->wmain)->menubar;

	window_init_menu_data (window);

	gnome_app_install_menu_hints (GNOME_APP(window->wmain), 
				      menubar_uiinfo);

	bookmarks_menu_create (window);
	window_session_history_menu_create (window);

	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_menubar),
		 window->menubar_on ?
		 eel_gconf_get_boolean (CONF_WINDOWS_SHOW_MENUBAR) :
		 FALSE);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_toolbar),
		 window->toolbar_on ?
		 eel_gconf_get_boolean (CONF_WINDOWS_SHOW_TOOLBARS) :
		 FALSE);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_bookmarks),
		 window->toolbar_on ?
		 eel_gconf_get_boolean (CONF_WINDOWS_SHOW_BOOKMARKS) :
		 FALSE);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_statusbar),
		 window->statusbar_on ?
		 eel_gconf_get_boolean (CONF_WINDOWS_SHOW_STATUSBAR) :
		 FALSE);

	/* connect state-changed on the econding menuitem to catch the
	 * first entry and create the submenu, not using entry-notify-event
	 * because of a gtk+ bug */
	gtk_signal_connect (GTK_OBJECT (window->encoding), "state-changed",
			    GTK_SIGNAL_FUNC 
			    (window_menu_encoding_state_changed_cb),
			    window);
	/* also create a dummy submenu sa that the submenu arrow is shown */
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (window->encoding), 
				   gtk_menu_new ());
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (window->csslist),
				   gtk_menu_new ());

	/* handle expose for the edit menu, used to set the edit menuitems 
	 * sensitivity according to the current selection */
	gtk_signal_connect (GTK_OBJECT (window->edit_cut->parent), 
			    "expose_event",
			    GTK_SIGNAL_FUNC (window_menu_edit_expose_event_cb),
			    window);

	/* setup settings menu and install notifiers*/
	window_menubar_init_notifiers (window);
}

static void
window_toolbars_create (GaleonWindow *window)
{
	/* create the toolbars */
	toolbar_create (window, TRUE);
	bookmarks_toolbars_create (window);
}

void 
window_grab_location_focus (GaleonWindow *window)
{
	if (window->location_entry)
	{
		gtk_widget_grab_focus (
			GTK_WIDGET (window->location_entry));
	}
}

void
window_save_layout (GaleonWindow *window)
{
	g_return_if_fail (window->layout == NULL);

	/* get the saved layout string and store it in the window structure */
	window->layout = gnome_config_get_string ("/galeon/Placement/Dock");

	/* make sure there's a layout to build into */
	if (GNOME_APP (window->wmain)->layout == NULL)
	{
		GNOME_APP (window->wmain)->layout = gnome_dock_layout_new ();
	}
}

void
window_restore_layout (GaleonWindow *window)
{
	GnomeApp *app;
	GList *lp;
	
	/* get the app */
	app = GNOME_APP (window->wmain);

	/* restore placements of recreated items */
	/* have to do this manually to get the ordering right, since
	 * we don't want to disturb things which we haven't recreated,
	 * such as the menu bar */
	gnome_dock_layout_parse_string (app->layout, window->layout);
	for (lp = app->layout->items; lp != NULL; lp = g_list_next (lp))
	{
		GnomeDockLayoutItem *item = (GnomeDockLayoutItem *)(lp->data);
		
		if (item->placement == GNOME_DOCK_FLOATING)
		{
			gnome_dock_add_floating_item
				(GNOME_DOCK (app->dock),
				 item->item,
				 item->position.floating.x,
				 item->position.floating.y,
				 item->position.floating.orientation);
		}
		else
		{
			gnome_dock_add_item
				(GNOME_DOCK (app->dock),
				 item->item,
				 item->placement,
				 item->position.docked.band_num,
				 0,
				 item->position.docked.offset,
				 TRUE);
		}
	}

	/* get rid of the layout object, it's no longer needed */
	gtk_object_unref (GTK_OBJECT (app->layout));
	app->layout = NULL;

	/* free the layout string */
	g_free (window->layout);
	window->layout = NULL;
}

static void
window_statusbar_create (GaleonWindow *window)
{
	gnome_app_set_statusbar (
		GNOME_APP(window->wmain),
		gnome_appbar_new (TRUE, TRUE,
				  GNOME_PREFERENCES_NEVER));
	window->appbar = GNOME_APP(window->wmain)->statusbar;

	window->security_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(window->security_frame), 
				  GTK_SHADOW_IN);

	window_statusbar_set_security_icon (window, FALSE, NULL);

	gtk_box_pack_start(GTK_BOX(window->appbar), 
			   GTK_WIDGET(window->security_frame), 
			   FALSE, TRUE, 0);
	gtk_widget_show(window->security_frame);
}

void
window_statusbar_set_security_icon (GaleonWindow *window, gboolean state,
				    gchar *tooltip)
{
	PixmapData *pixmap = NULL;
	gchar *tmp = tooltip;

	return_if_not_window (window);

	if (!tooltip)
	{
		gchar *level = mozilla_security_level_string (0);
		tmp = g_strdup_printf (_("Security level: %s"), level);
		g_free (level);
	}

	if (!state)
		pixmap = themes_get_pixmap ("Insecure.png", FALSE);
	else
		pixmap = themes_get_pixmap ("Secure.png", FALSE);

	if (!window->security_icon)
	{

	  	window->security_icon = gtk_pixmap_new (pixmap->pixmap,
						        pixmap->mask);
		window->security_eventbox = gtk_event_box_new ();
		gtk_container_add (GTK_CONTAINER(window->security_frame), 
				   GTK_WIDGET(window->security_eventbox));
		gtk_container_add (GTK_CONTAINER(window->security_eventbox), 
				   GTK_WIDGET(window->security_icon));
		gtk_signal_connect (GTK_OBJECT (window->security_eventbox),
				    "button_release_event",
				    GTK_SIGNAL_FUNC
				    	(window_security_icon_button_release_cb),
				    window);
		gtk_widget_show_all (window->security_eventbox);
	}
	else
	{
		gtk_pixmap_set (GTK_PIXMAP (window->security_icon),
				pixmap->pixmap, pixmap->mask);
	}
	
	gtk_tooltips_set_tip (window->bookmarks_tooltips,
			      window->security_eventbox, tmp, NULL);

	if (!tooltip) g_free (tmp);
}

static void
window_create_embed_container (GaleonWindow *window)
{
	/* make the toplevel notebobok */
	window->notebook = gtk_notebook_new ();

	/* set some notebook properties */
	gtk_notebook_set_tab_hborder (GTK_NOTEBOOK (window->notebook), 0);
	gtk_notebook_set_tab_vborder (GTK_NOTEBOOK (window->notebook), 0);
	gtk_notebook_set_scrollable (GTK_NOTEBOOK (window->notebook), TRUE);
	gtk_signal_connect_after (GTK_OBJECT (window->notebook), "switch_page",
				  window_notebook_switch_page_cb, window);
	gtk_signal_connect_after (GTK_OBJECT (window->notebook), "add",
				  window_notebook_add_remove_page_cb, window);
	gtk_signal_connect_after (GTK_OBJECT (window->notebook), "remove",
				  window_notebook_add_remove_page_cb, window);

	/* set up drag target */
	gtk_signal_connect (GTK_OBJECT (window->notebook),
			    "drag_data_received",
			    GTK_SIGNAL_FUNC (
			            window_notebook_drag_data_received_cb),
			    window);
	gtk_drag_dest_set (window->notebook, GTK_DEST_DEFAULT_ALL,
			   embed_drop_types, embed_drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK);

	/* connect signals for tab drag events */
	gtk_widget_add_events (window->notebook, GDK_BUTTON1_MOTION_MASK |
			       GDK_LEAVE_NOTIFY_MASK);
	gtk_signal_connect (GTK_OBJECT (window->notebook),
			    "button_press_event", GTK_SIGNAL_FUNC
			    (window_notebook_button_press_cb), window);
	gtk_signal_connect (GTK_OBJECT (window->notebook),
			    "button_release_event", GTK_SIGNAL_FUNC
			    (window_notebook_button_release_cb), window);
			    
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (window->notebook),
				  eel_gconf_get_integer 
				  (CONF_TABS_TABBED_POSITION));
	gtk_notebook_set_show_border (GTK_NOTEBOOK (window->notebook), FALSE);
	gtk_widget_show (GTK_WIDGET (window->notebook));

	/* make the toplevel hpaned, for docks */
	window->hpaned = gtk_hpaned_new ();
	gtk_paned_set_gutter_size (GTK_PANED (window->hpaned), 12);
	gtk_paned_set_handle_size (GTK_PANED (window->hpaned), 10);

	/* make a toplevel hbox for inserting the paned into */
	window->hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (window->hbox),
			    GTK_WIDGET (window->hpaned),
			    TRUE, TRUE, 0);
	gtk_box_pack_end (GTK_BOX (window->hbox),
			  GTK_WIDGET (window->notebook),
			  TRUE, TRUE, 0);

	/* insert hbox into toplevel window */
	gnome_app_set_contents (GNOME_APP (window->wmain), 
				GTK_WIDGET (window->hbox));
}

static void
window_set_chrome (GaleonWindow *window, guint32 chrome_mask)
{
	/* set the chrome info */
	window->menubar_on = FALSE;
	window->toolbar_on = FALSE;
	window->location_bar_on = FALSE;
	window->statusbar_on = FALSE;

	if (chrome_mask & GTK_MOZ_EMBED_FLAG_MENUBARON)
	{
		window->menubar_on = TRUE;
	}
	if (chrome_mask & GTK_MOZ_EMBED_FLAG_TOOLBARON)
	{
		window->toolbar_on = TRUE;
	}
	if (chrome_mask & GTK_MOZ_EMBED_FLAG_LOCATIONBARON)
	{
		window->location_bar_on = TRUE;
	}
	if (chrome_mask & GTK_MOZ_EMBED_FLAG_STATUSBARON)
	{
		window->statusbar_on = TRUE;
	}
}

/**
 * window_update_tab_controls: updates the various tab-related items inside
 * of the "View" menu
 */
void
window_update_tab_controls (GaleonWindow *window)
{
	gboolean tabs_to_left, tabs_to_right;
	gboolean tabs, show_tabs;
	GaleonEmbed *embed;
	gint index, size;

	return_if_not_window (window);

	/* get active embed */
	embed = window->active_embed;
	return_if_not_embed (embed);

	/* set next/prev tab and shift tab menuitems */
	index = gtk_notebook_page_num (GTK_NOTEBOOK (window->notebook), 
				       GTK_WIDGET (embed->mozembed));
	size = g_list_length (GTK_NOTEBOOK (window->notebook)->children);
	tabs = (size > 1);
	tabs_to_left = (index > 0);
	tabs_to_right = (index < (size - 1));

	gtk_widget_set_sensitive (window->detach_tab, tabs);
	gtk_widget_set_sensitive (window->prev_tab, tabs);
	gtk_widget_set_sensitive (window->next_tab, tabs);
	gtk_widget_set_sensitive (window->move_tab_left, tabs_to_left);
	gtk_widget_set_sensitive (window->move_tab_right, tabs_to_right);

	/* update tab display */
	show_tabs = eel_gconf_get_boolean (CONF_TABS_TABBED_ALWAYS_SHOW);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook),
				    (show_tabs && !window->is_popup) ||
				    (size > 1));
}

/**
 * window_update_nav_controls: update back and forward toolbar buttons
 * status and status of items in go menu
 */
void
window_update_nav_controls (GaleonWindow *window)
{
	gboolean can_go_back;
	gboolean can_go_up;
	gboolean can_go_forward;
	gboolean can_stop;
	gint tearoff = 0;
	GaleonEmbed *embed;
	GtkWidget *menu;
	GList *children;

	return_if_not_window (window);

	/* get active embed */
	embed = window->active_embed;
	return_if_not_embed (embed);
		
	/* query mozilla / embed */
	can_go_back =
		gtk_moz_embed_can_go_back (GTK_MOZ_EMBED(embed->mozembed));
	can_go_up = embed_can_go_up (embed);
	can_go_forward =
		gtk_moz_embed_can_go_forward (GTK_MOZ_EMBED(embed->mozembed));
	can_stop = (embed->load_started > 0);

	/* gets gnome detachabilty of menus so we set sensitivity on
	   the right item */
	if (gnome_preferences_get_menus_have_tearoff ())
	{
		tearoff++;
	}
		
	/* get the go_menu widgets */
	menu = window->go_menu;
	children = gtk_container_children (GTK_CONTAINER(menu));
		
	/* set go_menu buttons */
	gtk_widget_set_sensitive (
		GTK_WIDGET(g_list_nth_data(children, 0 + tearoff)),
		can_go_back);
	gtk_widget_set_sensitive (
		GTK_WIDGET(g_list_nth_data(children, 1 + tearoff)),
		can_go_forward);
	gtk_widget_set_sensitive (
		GTK_WIDGET(g_list_nth_data(children, 2 + tearoff)),
		can_go_up);
	gtk_widget_set_sensitive (
		GTK_WIDGET(g_list_nth_data(children, 5 + tearoff)),
		can_stop);

	/* free the list of children */
	g_list_free (children);

	/* update back, forward, up, stop and refresh buttons */
	if (window->back_button != NULL)
	{
		gtk_widget_set_sensitive 
			(window->back_button, can_go_back);
	}
	if (window->back_history_button != NULL)
	{
		gtk_widget_set_sensitive 
			(window->back_history_button, can_go_back);
	}
	if (window->up_button != NULL)
	{
		gtk_widget_set_sensitive
			(window->up_button, can_go_up);
	}
	if (window->forward_button != NULL)
	{
		gtk_widget_set_sensitive
			(window->forward_button, can_go_forward);
	}
	if (window->forward_history_button != NULL)
	{
		gtk_widget_set_sensitive
			(window->forward_history_button, 
			 can_go_forward);
	}
	if (window->stop_button != NULL)
	{
		gtk_widget_set_sensitive 
			(window->stop_button, can_stop);
	}
}

/**
 * window_update_zoom: update the zoom spin control
 */
void
window_update_zoom (GaleonWindow *window)
{
	if (window->zoom_spin != NULL && window->active_embed != NULL)
	{
		gtk_spin_button_set_value 
			(GTK_SPIN_BUTTON (window->zoom_spin),
			 (gfloat)(window->active_embed->zoom));
	}
}

/**
 * window_show_open_dialog: show the open dialog
 */
void window_show_open_dialog (GaleonWindow *window)
{
	gchar *dir, *retDir;
	gchar *file = NULL;
	GnomeVFSURI *uri;

	dir = eel_gconf_get_string (CONF_STATE_OPEN_DIR);

	if (show_file_picker (window->wmain, _("Select the file to open"),
			      dir, NULL, modeOpen,
			      &file, NULL, NULL, NULL))
	{
		uri = gnome_vfs_uri_new (file);
		if (uri)
		{
			window_load_url(window, file);

			retDir = gnome_vfs_uri_extract_dirname (uri);

			/* set default open dir */
			eel_gconf_set_string (CONF_STATE_OPEN_DIR, 
						 retDir);

			g_free (retDir);
			gnome_vfs_uri_unref (uri);
		}


	}

	g_free (dir);	
	g_free (file);	
}

/**
 * window_show_openurl_dialog: show the open url dialog
 */
void window_show_openurl_dialog (GaleonWindow *window)
{
	GladeXML *gxml;
	GtkWidget *dialog = NULL;
	GtkWidget *entry;
	extern gchar *open_url_str;

	gxml = glade_widget_new ("galeon.glade", "open_url_dialog", 
				 &dialog, window);
	entry = glade_xml_get_widget (gxml, "open_url_entry");

	g_return_if_fail (dialog != NULL);
	g_return_if_fail (entry != NULL);

	/* unref the xml object - we don't need it any more */
	gtk_object_unref (GTK_OBJECT (gxml));

	/* this is done so that the enter key is not captured by the entry */
	gnome_dialog_editable_enters (GNOME_DIALOG (dialog),
				      GTK_EDITABLE (entry));

	/* FIXME: is this really desirable? */
	if (open_url_str)
	{
		gtk_entry_set_text (GTK_ENTRY (entry), open_url_str);
	}

	window->open_url_entry = entry;
	dialog_set_parent (dialog, window->wmain);
}

/**
 * window_toolbar_show: show the toolbar
 */
void window_toolbar_show (GaleonWindow *window)
{
	gtk_widget_show (window->main_dockitem);	
	if (!eel_gconf_get_boolean (CONF_TOOLBAR_URL_LOCATION))
	{
		gtk_widget_show (window->location_dockitem);
	}

	if (!GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		window->show_toolbar = TRUE;
}

/**
 * window_toolbar_hide: hide the toolbar
 */
void window_toolbar_hide (GaleonWindow *window)
{
	gtk_widget_hide (GTK_WIDGET (window->main_dockitem));
	if (window->location_dockitem)
	{
		gtk_widget_hide (GTK_WIDGET (window->location_dockitem));
	}
	gtk_widget_queue_resize (GTK_WIDGET (window->hbox));

	if (!GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		window->show_toolbar = FALSE;
}

/**
 * window_menubar_show: show the menubar
 */
void window_menubar_show (GaleonWindow *window)
{
	gtk_widget_show (GTK_WIDGET (window->menubar->parent));	

	if (!GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		window->show_menubar = TRUE;
}

/**
 * window_menubar_hide: hide the menubar
 */
void window_menubar_hide (GaleonWindow *window)
{
	gtk_widget_hide (GTK_WIDGET (window->menubar->parent));
	gtk_widget_queue_resize (GTK_WIDGET (window->hbox));

	if (!GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		window->show_menubar = FALSE;
}

/**
 * window_statusbar_show: show the statusbar
 */
void window_statusbar_show (GaleonWindow *window)
{
	gtk_widget_show (GTK_WIDGET (window->appbar->parent));

	if (!GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		window->show_statusbar = TRUE;
}

/**
 * window_statusbar_hide: hide the statusbar
 */
void window_statusbar_hide (GaleonWindow *window)
{
	gtk_widget_hide (window->appbar->parent);
	gtk_widget_queue_resize (GTK_WIDGET (window->hbox));

	if (!GTK_CHECK_MENU_ITEM(window->view_fullscreen)->active)
		window->show_statusbar = FALSE;
}

/**
 * window_statusbar_update_progress_bar: update the statusbar's progress bar
 */
void
window_statusbar_update_progress_bar (GaleonWindow *window)
{
	GaleonEmbed *embed;
	GtkWidget *progress;

	return_if_not_window (window);
	embed = window->active_embed;
	if (embed == NULL) return;
	return_if_not_embed (embed);

	/* get progress bar widget */
	progress = GTK_WIDGET (gnome_appbar_get_progress 
			       (GNOME_APPBAR(window->appbar)));

	/* have we started loading? */
	if (embed->load_started > 0 && embed->bytes_loaded > 0)
	{
		if (embed->load_percent == 0) 
		{
			/* not really, go into activity mode */
			gtk_progress_set_activity_mode 
				(GTK_PROGRESS (progress), TRUE);
			if (!(window->progress_timeout))
			{
				window->progress_timeout = TRUE;
				g_timeout_add (100, (GSourceFunc)
					       window_progress_action, 
					       window);
			}
		} 
		else
		{
			/* yes, show the progress in progress bar */
			window->progress_timeout = FALSE;
			gtk_progress_set_activity_mode 
				(GTK_PROGRESS (progress), FALSE);
			gnome_appbar_set_progress 
				(GNOME_APPBAR (window->appbar), 
				 embed->load_percent / 100.0);
		}
	}
	else
	{
		/* go to sleep */
		window->progress_timeout = FALSE;
		gtk_progress_set_activity_mode (GTK_PROGRESS (progress),
						FALSE);
		gnome_appbar_set_progress (GNOME_APPBAR (window->appbar), 0);
	}

}

/**
 * window_statusbar_update_message: update the statusbar's progress bar
 */
void
window_statusbar_update_message (GaleonWindow *window)
{
	GaleonEmbed *embed;
	const gchar *status;
	gchar message[256];

	return_if_not_window (window);
	embed = window->active_embed;
	if (embed == NULL) return;
	return_if_not_embed (embed);

	/* get rid of the messages */
	gnome_appbar_clear_stack (GNOME_APPBAR (window->appbar));

	/* if the active embed has a temp message, use that and return */
	if (embed->temp_statusbar_message != NULL)
	{
		gnome_appbar_push (GNOME_APPBAR (window->appbar),
				   embed->temp_statusbar_message);
		return;
	}

	/* otherwise, get the load status message */
	status = (embed->load_status_message == NULL ? "" :
		  embed->load_status_message);

	/* make a progress message */
	if (embed->bytes_loaded == 0)
	{
		g_snprintf (message, 255, "%s", status);
	}
	else if (embed->bytes_loaded <= embed->max_bytes_loaded)
	{
		gchar time_remaining[64] = "";
		gint hours, minutes, seconds;
		GTime time_delta;

		/* try to compute roughly the time remaining for the load */
		time_delta = (time (NULL) - embed->when_started);
		if (embed->load_percent > 5 && 
		    embed->load_percent < 100 && time_delta > 0)
		{
			time_delta *= ((double)embed->max_bytes_loaded / 
				       (double)embed->bytes_loaded) - 1.0;
			hours = time_delta / (60 * 60);
			time_delta %= (60 * 60);
			minutes = time_delta / 60;
			seconds = time_delta % 60;
			snprintf (time_remaining, 64,
				  _(", %02d:%02d:%02d remaining"),
				  hours, minutes, seconds);
		}
		
		/* fill the status bar text with progress info */
		g_snprintf (message, 255, 
			    _("%s (%d%% complete, %d kB of %d kB loaded%s)"),
			    status, 
			    embed->load_percent, 
			    embed->bytes_loaded / 1024,
			    embed->max_bytes_loaded / 1024,
			    time_remaining);
	}
	else
	{
		/* fill the status bar text with progress info: only kb */
		g_snprintf (message, 255, _("%s (%d kB loaded)"), 
			    status, 
			    embed->bytes_loaded / 1024);
	}

	/* show the new message */
	gnome_appbar_push (GNOME_APPBAR (window->appbar), message);
}

/**
 * window_close: close a GaleonWindow
 */
void 
window_close (GaleonWindow *window)
{
	GList *copy;

	return_if_not_window (window);

	/* hide the window, this makes the app seem more responsive */
	gtk_widget_hide (window->wmain);

	/* close any dock (in order to store width of dock) */
	window_undock (window);

	/* save window size if we are not in fullscreen and if 
	 * the window is not a popup */
	if (!window->is_popup &&
	    !GTK_CHECK_MENU_ITEM (window->view_fullscreen)->active)
	{	
		eel_gconf_set_integer(CONF_STATE_WINWIDTH, 
				     window->wmain->allocation.width);
		eel_gconf_set_integer(CONF_STATE_WINHEIGHT, 
				     window->wmain->allocation.height);
	}

	/* stop the spinner if it exists; I'm not sure if this is essential
	 * but it's an attempt to workaround GnomeAnimator crashes -- MattA */
	if (window->spinner != NULL)
        {
		spinner_destroy (window);
        }

	/* destroy tooltips */
	if (window->bookmarks_tooltips)
	{
		gtk_object_destroy (GTK_OBJECT (window->bookmarks_tooltips));
		window->bookmarks_tooltips = NULL;
	}

	/* free recent items */
	for (copy = g_list_copy (window->recent_menu_entries);
	     copy != NULL;
	     copy = copy->next)
	{
		window_remove_recent_menu_entry (window, copy->data);
	}
	g_assert (window->recent_menu_entries == NULL);

	/* remove notifiers */
	if (window->notifiers_id != NULL);
	{
		galeon_notification_remove(&window->notifiers_id);
	}

	/* unconnect the notebook motion and leave signals */
	if (window->drag_motion_signal)
	{
		gtk_signal_disconnect (GTK_OBJECT (window->notebook),
				       window->drag_motion_signal);
		gtk_signal_disconnect (GTK_OBJECT (window->notebook),
				       window->drag_leave_signal);
	}

	/* ungrab pointer if we're in a tab drag */
	if (window->in_drag)
	{
		gdk_pointer_ungrab (GDK_CURRENT_TIME);
		gtk_grab_remove (window->notebook);
	}

	/* any children? */
	if (window->embed_list != NULL)
	{
		gboolean prev_dont_autosave;

		/* get the current state of the dont_autosave flag */
		prev_dont_autosave = dont_autosave_session;

		/* enable it, so we don't save the session a bunch of times
		 * as we close all the embeds */
		dont_autosave_session = TRUE;

		/* close any embed children -- closing the last one will
		 * call window_close again, which will execute the else
		 * block below */
		copy = g_list_copy (window->embed_list);
		g_list_foreach (copy, (GFunc) embed_close, NULL);
		g_list_free (copy);

		/* reset the dont_autosave flag to its previous state */
		dont_autosave_session = prev_dont_autosave;

		/* save session now */
		session_autosave ();
	}
	else
	{
		/* stop the activity bar */
		window->progress_timeout = FALSE;

		/* destroy the window, and all it contains */
		gtk_widget_destroy (window->wmain);
		window->wmain = NULL;

		/* remove window from session */
		session_remove_window (window);

		/* destroy dummy widgets list */
		g_list_foreach (window->dummy_widgets, 
				(GFunc)gtk_widget_unref, NULL);
		g_list_free (window->dummy_widgets);

		/* destroy tooltips */
		if (window->bookmarks_tooltips)
		{
			gtk_object_destroy
				(GTK_OBJECT (window->bookmarks_tooltips));
			window->bookmarks_tooltips = NULL;
		}

		/* scrub and free the window structure */
		memset (window, 0, sizeof (GaleonWindow));
		g_free (window);
	}
}

/**
 * window_go_home:
 */
void
window_go_home (GaleonWindow *window, LinkState state)
{
	gchar *startpage;

	startpage = eel_gconf_get_string 
		(CONF_GENERAL_HOMEPAGE);

        if (startpage != NULL && strlen(startpage) != 0)
        {
		embed_activate_link (window->active_embed, NULL, 
				     startpage, state);
		g_free(startpage);
	}
        else
	{
		gnome_error_dialog (_("Please specify a Home Page "
				      "in the Preferences dialog."));
	}
}

/**
 * window_set_layer: make sure the window is on the right layer. Don't
 * use this for dialogs any more; instead use dialog_set_parent.
 */
void
window_set_layer (GaleonWindow *window)
{
	/* check if we're in fullscreen mode and stacked above dock */
	if (fullscreen_active && 
	    eel_gconf_get_boolean (CONF_WINDOWS_FULLSCREEN_STACKING))
	{
		/* get to the top */
		gnome_win_hints_set_layer (window->wmain, 
					   WIN_LAYER_ABOVE_DOCK);
	}
	else
	{
		/* down boy! */
		gnome_win_hints_set_layer (window->wmain, WIN_LAYER_NORMAL);
	}

	/* make sure it's raised */
	gdk_window_raise (window->wmain->window);

	window->layered = TRUE;
}

void
window_toggle_fullscreen_mode (GaleonWindow *window)
{
	window_set_fullscreen_mode (window, !window->fullscreen_active);
}

void
window_set_fullscreen_mode (GaleonWindow *window, gboolean active)
{
	GdkWindow *gdk_window;

	return_if_not_window (window);

	if (window->fullscreen_active == active) return;
	
	GTK_CHECK_MENU_ITEM (window->view_fullscreen)->active = active;

	if (active)
	{
		gboolean bool;

		bool = eel_gconf_get_boolean
			(CONF_WINDOWS_FS_SHOW_MENUBAR);
		if (bool != window->show_menubar) 
 		{
 			gtk_check_menu_item_set_active
 				(GTK_CHECK_MENU_ITEM
				 (window->view_menubar),
				 bool);
 		}
		bool = eel_gconf_get_boolean
			(CONF_WINDOWS_FS_SHOW_TOOLBARS);
		if (bool != window->show_toolbar) 
 		{
 			gtk_check_menu_item_set_active
 				(GTK_CHECK_MENU_ITEM
				 (window->view_toolbar),
				 bool);
 		}
		bool = eel_gconf_get_boolean
			(CONF_WINDOWS_FS_SHOW_BOOKMARKS);
		if (bool != window->show_bookmarks) 
 		{
			gtk_check_menu_item_set_active
 				(GTK_CHECK_MENU_ITEM
				 (window->view_bookmarks),
				 bool);
 		}
		bool = eel_gconf_get_boolean
			(CONF_WINDOWS_FS_SHOW_STATUSBAR);
		if (bool != window->show_statusbar)
 		{
 			gtk_check_menu_item_set_active
 				(GTK_CHECK_MENU_ITEM
				 (window->view_statusbar),
				 bool);
 		}
	}
	else
	{
		/* set things back to the state they were before 
		 * fullscreen */
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->view_menubar),
			 window->show_menubar);
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->view_toolbar),
			 window->show_toolbar);
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->view_bookmarks),
			 window->show_bookmarks);
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->view_statusbar),
			 window->show_statusbar);
	}

	window->fullscreen_active = fullscreen_active = active;
	gdk_window = window->wmain->window;
	window_set_layer (window);

	if (window->fullscreen_button)
	{
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (window->fullscreen_button),
			 window->fullscreen_active);
	}

	if (active)
	{
		gint client_x, client_y, root_x, root_y;
		gint width, height;
		gint screen_width, screen_height;

		/* get window position and geometry */
		gdk_window_get_origin (gdk_window, &root_x, &root_y);
		gdk_window_get_geometry (gdk_window, &client_x, &client_y,
					 &width, &height, NULL);

		/* store original position and size */
		window->x = root_x - client_x;
		window->y = root_y - client_y;
		window->width = width;
		window->height = height;

		/* get the size of the screen */
		screen_width = gdk_screen_width ();
		screen_height = gdk_screen_height ();

#ifdef HAVE_XINERAMA
		/* if the user is using Xinerama, adjust the area and
		 * positioning of the fullscreen window so it's not
		 * occupying any dead area or covering all of the heads */
		{
		Display *dpy;

		dpy = GDK_DISPLAY ();
#ifdef __sgi
		if (XineramaActive (dpy, 0))
#else
		if (XineramaIsActive (dpy))
#endif		
		{
			int num_heads, i;
			int max_area = 0, max_area_head = 0;
			int int_width, int_height, int_area;
#ifdef __sgi
			XRectangle *head_info;		
			XineramaGetInfo (dpy, 0, &head_info, &num_heads);
#else
			XineramaScreenInfo *head_info;

			/* query heads */
			head_info = XineramaQueryScreens (dpy, &num_heads);
#endif			
			g_return_if_fail (num_heads != 0);

			/* find the head with the largest area of the
			 * window in it */
			for (i = 0; i < num_heads; i++)
			{
				/* find the width and height of the
				 * intersecting area */
#ifdef __sgi				 
				int_width = MIN (head_info[i].x +
					head_info[i].width,
					window->x + window->width) -
					MAX (head_info[i].x, window->x);
				int_height = MIN (head_info[i].y +
					head_info[i].height,
					window->y + window->height) -
					MAX (head_info[i].y, window->y);
#else
				int_width = MIN (head_info[i].x_org +
					head_info[i].width,
					window->x + window->width) -
					MAX (head_info[i].x_org, window->x);
				int_height = MIN (head_info[i].y_org +
					head_info[i].height,
					window->y + window->height) -
					MAX (head_info[i].y_org, window->y);
#endif
				/* find the area of the intersection */
				if (int_width < 0 || int_height < 0)
					int_area = 0;
				else
					int_area = int_width * int_height;

				/* see if this head has a larger area */
				if (int_area > max_area)
				{
					max_area_head = i;
					max_area = int_area;
				}
			}

			/* update screen size and position offset */
			screen_width = head_info[max_area_head].width;
			screen_height = head_info[max_area_head].height;
#ifdef __sgi			
			client_x -= head_info[max_area_head].x;
			client_y -= head_info[max_area_head].y;
#else			
			client_x -= head_info[max_area_head].x_org;
			client_y -= head_info[max_area_head].y_org;
#endif			
			XFree (head_info);
		}
		}
#endif
		/* move and resize the window */
		gdk_window_move_resize (gdk_window, -client_x, -client_y,
					screen_width, screen_height);
	}
	else
	{
		/* restore the initial position and size */
		gdk_window_move_resize (gdk_window, 
					window->x, window->y,
					window->width, 
					window->height);
	}
}

static void 
window_set_menu_data (GaleonWindow *window)
{
	gint i, j;

	for (i = 0; i < num_menus; i++)
	{
		for (j = 0; j < menus_num_items[i]; j++)
		{
			all_menus[i][j].user_data = window;
		}
	}
}

/**
 * window_from_widget: get the GaleonWindow structure associated with
 * a widget within a main browser window. This is necessary in a few of
 * the tricker callbacks, and simply finds the wmain (by tracing up the
 * widget heirarchy), and the does an object data lookup.
 */
GaleonWindow *
window_from_widget (GtkWidget *widget)
{
	GaleonWindow *found_window = NULL;

	/* check */
	g_return_val_if_fail (widget != NULL, NULL);

	/* do the search */
	while (widget)
	{
		/* do the lookup */
		found_window = gtk_object_get_data (GTK_OBJECT (widget),
						    "GaleonWindow");

		/* get out if success */
		if (found_window != NULL)
		{
			break;
		}

		/* find parent */
		widget = (GTK_IS_MENU (widget) ? 
			  (gtk_menu_get_attach_widget (GTK_MENU (widget))) :
			  (widget->parent));
	}

	/* check -- FIXME: turn this critical */
	if (found_window == NULL)
	{
		g_warning ("GaleonWindow not found on widget");
	}

	/* return success */
	return found_window;
}

/**
 * window_reload_all: reload all tabs in a window
 */
void
window_reload_all (GaleonWindow *window)
{
	GList *l;

	/* check arguments */
	return_if_not_window (window);

	/* reload all tabs in the window */
	for (l = window->embed_list; l != NULL; l = g_list_next (l))
	{
		embed_reload ((GaleonEmbed *)(l->data), 
			      GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
	}
}

/**
 * window_stop_all: stop all tabs in a window
 */
void
window_stop_all (GaleonWindow *window)
{
	GList *l;

	/* check arguments */
	return_if_not_window (window);

	/* reload all tabs in the window */
	for (l = window->embed_list; l != NULL; l = g_list_next (l))
	{
		GaleonEmbed *e = (GaleonEmbed *)(l->data);
		gtk_moz_embed_stop_load (GTK_MOZ_EMBED (e->mozembed));
	}
}

/**
 * window_undock: remove the current dock
 */
void
window_undock (GaleonWindow *window)
{
	/* check something is docked */
	if (window->docked == NULL)
	{
		return;
	}

	/* store width in pref */
	if (window->docked_width_pref != NULL && 
	    GTK_PANED (window->hpaned)->position_set)
	{
	  eel_gconf_set_integer (window->docked_width_pref,
				 GTK_PANED (window->hpaned)->child1_size);
	}

	/* hide the hpaned and notebook */
	gtk_widget_hide (GTK_WIDGET (window->hpaned));
	gtk_widget_hide (GTK_WIDGET (window->notebook));
 
	/* remove the notebook from the hpaned and insert it into the box */
	gtk_widget_reparent (GTK_WIDGET (window->notebook),
			     GTK_WIDGET (window->hbox));

        /* destroy the docked widget */
	gtk_widget_destroy (GTK_WIDGET (window->docked));
	window->docked = NULL;

	/* update toolbar buttons */
	window_update_bm_and_hist_buttons (window, FALSE, FALSE);

	/* set dock type */
	window->dock_type = DOCK_NONE;
	
	/* show the notebook again */
	gtk_widget_show (GTK_WIDGET (window->notebook));
}

/**
 * window_dock: dock a widget into a window, destroying anything
 * that was there previously.
 */
void
window_dock (GaleonWindow *window, GtkWidget *widget, const gchar *width_pref)
{
	gint width;

	/* hide the notebook */
	gtk_widget_hide (GTK_WIDGET (window->notebook));

	/* destroy current dock */
	if (window->docked != NULL)
	{
		window_undock (window);
	}

	/* dock this */
	gtk_paned_add1 (GTK_PANED (window->hpaned), GTK_WIDGET (widget));
	window->docked = widget;
	window->docked_width_pref = width_pref;

	/* show the paned view */
	/* NB: this must be done before setting pane position! */
	gtk_widget_show (GTK_WIDGET (window->hpaned));

	/* move pane to accomodate width */
	if (width_pref != NULL)
	{
		width = eel_gconf_get_integer (width_pref);
		if (width != -1)
		{
			gtk_paned_set_position (GTK_PANED (window->hpaned),
						width);
		}
	}

	/* move notebook into the other side of the paned */
	gtk_widget_reparent (GTK_WIDGET (window->notebook),
			     GTK_WIDGET (window->hpaned));

	/* show the notebook again */
	gtk_widget_show (GTK_WIDGET (window->notebook));
}

/*
 * create the charset titles submenu structure 
 */
void
window_create_charset_submenus (GaleonWindow *window,
				GList *charset_titles)
{
	GtkWidget *encoding_menu, *tw, *tw2 = NULL, *tw3;
	GList *tl, *tl2 = NULL;
	gint i, j;
	gint lgroups_count = get_lgroups_count ();
	gint translated_cscount = get_translated_cscount ();
	
 	encoding_menu = gtk_menu_new();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (window->encoding), 
				   encoding_menu);

	tl = g_list_copy(charset_titles);
	
	for (i = 0;i < lgroups_count;i++) {
		tw = gtk_menu_item_new_with_label (_(lgroups[i]));
		gtk_menu_append (GTK_MENU(encoding_menu), tw);
		gtk_widget_lock_accelerators (tw);
		gtk_widget_show (tw);
		
		tw2 = gtk_menu_new();
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (tw), tw2);
		
		for (j = 0; j < translated_cscount; j++)
			if ((gint)charset_trans_array[j].lgroup == i) {
				tl2 = g_list_find_custom (
					tl, 
					_(charset_trans_array[j].charset_title),
					strcasestr_cb);
				if (tl2 != NULL) {
					tl = g_list_remove_link (tl, tl2);
					g_list_free_1 (tl2);
				} 
				else /* we dont want to add menuitems for
				      * charset titles not in mozilla */
					continue;
				
				tw3 = gtk_menu_item_new_with_label (
				       _(charset_trans_array[j].charset_title));
				gtk_menu_append (GTK_MENU (tw2), tw3);
				gtk_widget_lock_accelerators (tw3);
				gtk_widget_show (tw3);
				gtk_signal_connect (
					GTK_OBJECT (tw3), 
					"activate",
					window_menu_encoding_activate_cb, 
					window);
			}
	}
	/* add the leftovers /if any/ to the Other submenu */
	while (tl != NULL) { 
		tw = gtk_menu_item_new_with_label (tl->data);
		gtk_menu_append (GTK_MENU(tw2), tw);
		gtk_widget_lock_accelerators (tw);
		gtk_widget_show (tw);
		gtk_signal_connect (GTK_OBJECT (tw), "activate",
				    window_menu_encoding_activate_cb, window);
		tl2 = tl->next;
		g_list_free_1 (tl);
		tl = tl2;
	}
}

static gint 
strcasestr_cb (gconstpointer a, gconstpointer b)
{
	return misc_string_strcasestr (a, b) == NULL ? 1 : 0;
}

/**
 * window_add_embed: adds embed to window.  if force_append is true, the
 * embed will be placed at the end of the notebook; otherwise, the position
 * will be determined by the CONF_TABS_TABBED_INSERT_NEW_TABS pref. if
 * never_jump is true, the notebook will not switch to the page containing
 * the new tab; otherwise, the new page will be switched to if force_jump
 * or CONF_TABS_TABBED_AUTOJUMP is true.  if raise_window is true, the
 * window will be raised after the tab is added.
 */
void
window_add_embed (GaleonWindow *window, GaleonEmbed *embed, 
		  EmbedCreateFlags flags)
{
	GtkWidget *tab;
	gint page;

	/* set parent and store in parents list */
	embed->parent_window = window;
	window->embed_list = g_list_append (window->embed_list, embed);

	/* window can no longer be a popup window */
	if (g_list_length (window->embed_list) > 1 && window->is_popup)
		window->is_popup = FALSE;

	/* create tab */
	tab = window_notebook_tab_new (embed, embed->title);

	/* check if we should insert the page at the current location */
	if (!(flags & EMBED_CREATE_FORCE_APPEND) &&
	    (g_list_length (window->embed_list) > 1) &&
	    eel_gconf_get_boolean (CONF_TABS_TABBED_INSERT_NEW_TABS))
	{
		page = gtk_notebook_current_page 
			(GTK_NOTEBOOK (window->notebook));
		gtk_notebook_insert_page (GTK_NOTEBOOK (window->notebook),
				  	  GTK_WIDGET (embed->mozembed), tab,
					  page + 1);
		page++;
	}
	/* otherwise, append the page */
	else
	{
		gtk_notebook_append_page (GTK_NOTEBOOK (window->notebook),
				  	  GTK_WIDGET (embed->mozembed), tab);
		page = gtk_notebook_page_num (GTK_NOTEBOOK (window->notebook),
					      GTK_WIDGET (embed->mozembed));
	}

	/* update tab-related properties of the window */
	if (window->active_embed != NULL)
	{
		window_update_tab_controls (window);
	}

	/* switch to page if necessary */
	if (!(flags & EMBED_CREATE_NEVER_JUMP) &&
	    (eel_gconf_get_boolean (CONF_TABS_TABBED_AUTOJUMP) ||
	     (flags & EMBED_CREATE_FORCE_JUMP)))
	{
		gtk_widget_show (embed->mozembed);
		gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), page);
		embed->focus_type = FOCUS_ON_CREATE;
	}
	else
	{
		embed->focus_type = FOCUS_ON_REQUEST;
	}

	/* raise window if necessary */
	if ((flags & EMBED_CREATE_RAISE_WINDOW) &&
	    GTK_WIDGET (window->wmain)->window != NULL)
	{
		gdk_window_raise (GTK_WIDGET (window->wmain)->window);
	}
}

void
window_hide_one_embed (GaleonWindow *window)
{
	/* keep count */
	window->visible_embeds--;

	/* if this is the last one to be hidden, hide the main window */
	if (window->visible_embeds == 0)
	{
		gtk_widget_hide (GTK_WIDGET (window->wmain));
	}
}

void
window_show_one_embed (GaleonWindow *window)
{
	gint width, height;
	
	/* keepcount */
	window->visible_embeds++;

	/* if this isn't the first time this has been shown, skip the rest */
	if (window->visible_embeds != 1)
	{
		return;
	}

	/* find the set width and height if any */
	if (!window->set_size)
	{
                /* use system defaults */
		if (window->requested_height > 0 && window->requested_width > 0)
		{
			width = window->requested_width;
			height = window->requested_height;
		}
		else
		{
			width = eel_gconf_get_integer (CONF_STATE_WINWIDTH);
			height = eel_gconf_get_integer (CONF_STATE_WINHEIGHT);
		}
		gtk_window_set_default_size (GTK_WINDOW (window->wmain), 
					     width, height);
	}

	/* show it */
	gtk_widget_show (GTK_WIDGET (window->wmain));
}

static GtkWidget *
window_notebook_tab_new (GaleonEmbed *embed, const gchar *title)
{
	GtkWidget *label;
	GtkWidget *icon;
	GtkWidget *hbox;
	gboolean build_icon =
		(eel_gconf_get_integer (CONF_TABS_TABBED_FAVICONS) == 1);

	/* build the basic widgets */
	embed->notebook_hbox = hbox = gtk_hbox_new (FALSE, 0);
	embed->notebook_label = label = gtk_label_new (NULL);
	embed->notebook_icon = icon = gtk_pixmap_new (site_pixmap_data->pixmap, 
                                                      site_pixmap_data->mask);
	gtk_widget_set_usize (icon, 16, 16);

	/* set up drag target */
	gtk_signal_connect (GTK_OBJECT (hbox), "drag_data_received",
			    GTK_SIGNAL_FUNC (embed_drag_data_received_cb),
			    embed);
	gtk_drag_dest_set (hbox, GTK_DEST_DEFAULT_ALL, embed_drop_types,
			   embed_drop_types_num_items, GDK_ACTION_COPY |
			   GDK_ACTION_MOVE | GDK_ACTION_LINK | GDK_ACTION_ASK);

	/* setup icon */
	gtk_misc_set_alignment (GTK_MISC (icon), 0.5, 0.5);
	gtk_misc_set_padding (GTK_MISC (icon), 0, 0);
	gtk_box_pack_start (GTK_BOX (hbox), icon, TRUE, TRUE, 0);

	/* setup label */
	gtk_misc_set_alignment (GTK_MISC (label), 0.00, 0.5);
	gtk_misc_set_padding (GTK_MISC (label), 4, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

	/* set page title and add button */
	embed_update_tab_title (embed);
	embed_update_tab_close_button (embed);
	embed_update_tab_status (embed);
	embed_update_tab_size (embed);

	/* show label */
	if (build_icon) gtk_widget_show (embed->notebook_icon);
	gtk_widget_show (embed->notebook_label);

	/* attach data */
	gtk_object_set_data (GTK_OBJECT (hbox), "GaleonEmbed", embed);

	/* return overall widget */
	return hbox;
}

/**
 * window_session_history_menu_create: creates a new "recent sessions"
 * submenu for the given window, freeing the old one if it exists.
 */
void
window_session_history_menu_create (GaleonWindow *window)
{
	GList *l;
	GtkWidget *submenu = NULL;
	gboolean added = FALSE;

	/* remove old submenu */
	if (GTK_MENU_ITEM (window->recent_sessions_menuitem)->submenu)
	{
		gtk_menu_item_remove_submenu (
			GTK_MENU_ITEM (window->recent_sessions_menuitem));
	}

	/* add the items */
	for (l = recent_sessions; l; l = l->next) 
	{
		gchar *filename;
		GtkWidget *menuitem;

		/* does the file exist? */
		if (!g_file_exists (l->data))
		{
			continue;
		}

		/* create new submenu if we haven't already */
		if (!submenu)
		{
			submenu = gtk_menu_new ();
			gtk_menu_item_set_submenu (GTK_MENU_ITEM (
					window->recent_sessions_menuitem),
					submenu);
		}

		/* build menuitem */
		filename = g_basename (l->data);
		menuitem = gtk_menu_item_new_with_label (filename);
		gtk_widget_show_all (menuitem);
		gtk_object_set_data (GTK_OBJECT (menuitem), "filename",
				     l->data);
		gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			    window_session_history_menuitem_activate_cb, 
			    NULL);
		gtk_signal_connect (GTK_OBJECT (menuitem),
			"button-press-event", GTK_SIGNAL_FUNC (
			window_session_history_menuitem_button_press_event_cb),
			NULL);

		gtk_menu_append (GTK_MENU (submenu), menuitem);

		added = TRUE;
	}

	/* set item's sensitivity */
	if (added) gtk_widget_set_sensitive (window->recent_sessions_menuitem,
					     TRUE);
	else gtk_widget_set_sensitive (window->recent_sessions_menuitem, FALSE);
}

/**
 * window_session_history_update_all: updates the "recent sessions"
 * submenus for all windows
 */
void
window_session_history_update_all (void)
{
	GList *l;
	for (l = all_windows; l != NULL; l= l->next)
		window_session_history_menu_create (l->data);
}

/**
 * window_update_bm_and_hist_buttons: update bookmarks and history togglebuttons
 * on toolbar
 */
void
window_update_bm_and_hist_buttons (GaleonWindow *window, 
				   gboolean bm, gboolean hist)
{
	if (window->bookmarks_button != NULL)
	{
		gtk_toggle_button_set_active (
			GTK_TOGGLE_BUTTON (window->bookmarks_button),
			bm);
	}
	if (window->history_button != NULL)
	{
		gtk_toggle_button_set_active (
			GTK_TOGGLE_BUTTON (window->history_button),
			hist);
	}
}

void 
window_update_tabs_appearance (GaleonWindow *window)
{
	gint n_embeds, always_show, popup = 0;
	GList *w, *e;

	misc_gui_colors_init ();

	always_show =
		eel_gconf_get_boolean (CONF_TABS_TABBED_ALWAYS_SHOW);

	for (w = all_windows; w != NULL; w = g_list_next (w))
	{
		GaleonWindow *window = (GaleonWindow *)(w->data);

		/* check if the tab(s) should be visible */
		n_embeds = g_list_length (window->embed_list);
		if (n_embeds == 1)
		{
			GaleonEmbed *embed;

			e = window->embed_list;
 			embed = (GaleonEmbed *) (e->data);
			popup = window->is_popup;
		}

		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), 
					    (n_embeds > 1) | 
					    (always_show && !popup));

		/* update notebook tabs */
		for (e = window->embed_list; e != NULL; e = g_list_next (e))
		{
			GaleonEmbed *embed = (GaleonEmbed *) (e->data);

			embed_update_tab_title (embed);
			embed_update_tab_close_button (embed);
			embed_update_tab_status (embed);
			embed_update_tab_size (embed);
		}
	}
}

/**
 * Adds (or updates) a recent menu entry
 */
void
window_add_recent_menu_entry (GaleonWindow *window, const gchar *url,
			      const gchar *title_locale)
{
	RecentMenuEntry *new_entry;
	GList *l;
	gint count = 0;

	return_if_not_window (window);
	g_return_if_fail (url != NULL);
	if (title_locale == NULL) title_locale = g_strdup (url);

	new_entry = g_new0 (RecentMenuEntry, 1);
	new_entry->url = g_strdup (url);
	new_entry->title = g_strdup (title_locale);

	window->recent_menu_entries = g_list_prepend 
		(window->recent_menu_entries, new_entry);

	++count;
	window_add_recent_menu_entry_add_menuitem (window, new_entry, count);

	/* Update the numbers and remove other entry for the same url if any */
	for (l = window->recent_menu_entries->next; l != NULL; l = l->next)
	{
		RecentMenuEntry *current_entry = l->data;
		++count;
		if (!strcmp (current_entry->url, new_entry->url)
		    || (count > GO_RECENT_MAX))
		{
			window_remove_recent_menu_entry (window, 
							 current_entry);

			/* we can't have more items with this URL and
			   the list has been modified, so we can't continue
			   iterating. Also, we don't need to update more
			   menutitems. */
			break;
		}
		/* update the number */
		window_recent_menu_entry_update_number (window, current_entry,
							count);
	}
}

/**
 * Add a recent menu entry to the menu as the first one
 **/
static void
window_add_recent_menu_entry_add_menuitem (GaleonWindow *window, 
					   RecentMenuEntry *entry,
					   gint num)
{
	gint go_recent_pos = gnome_preferences_get_menus_have_tearoff () 
		? GO_RECENT_POS + 1 : GO_RECENT_POS;
	GtkWidget *menu;

	return_if_not_window (window);
	g_return_if_fail (entry != NULL);
	g_return_if_fail (entry->menuitem == NULL);

	menu = window->go_menu;
	/* if we don't have a Go menu, we don't need to add anything */
	if (!menu) return;
	
	entry->menuitem = GTK_MENU_ITEM (window_create_recent_menu_item
					 (window, entry, num));
	gtk_menu_insert (GTK_MENU (menu), GTK_WIDGET (entry->menuitem),
			 go_recent_pos);
	gtk_widget_show (GTK_WIDGET (entry->menuitem));
}

/**
 *  Create a recent menu item widget
 **/
static GtkWidget *
window_create_recent_menu_item (GaleonWindow *window, RecentMenuEntry *entry,
				gint num)
{
	GtkWidget *hb = gtk_hbox_new (FALSE, 0);
	const PixmapData *pixmap_data = favicon_get_pixmap (entry->url);
	GtkWidget *pixmap = gtk_pixmap_new (pixmap_data->pixmap,
					    pixmap_data->mask);
	GtkWidget *menuitem = gtk_pixmap_menu_item_new ();
	gchar *s, *title, *tmp = NULL;
	GtkWidget *label;
	gint length;

	s = entry->title;

	if (!strcmp(s, _("Untitled")))
	{
        	s = entry->url;
    	}

	length = strlen (s);
	
	/* shorten if needed and escape */
	if (length > 40) /* FIXME, this should be define somewhere */
	{
		tmp = g_strdup_printf ("%.29s...%s", s, s + length - 8);
	}

	title = misc_string_escape_uline_accel (tmp ? tmp : s);
	label = misc_gui_new_num_accel_label (num - 1, title, FALSE,
					      window->go_menu, menuitem);
	if (tmp) g_free (tmp);
	
	gtk_box_pack_start (GTK_BOX (hb), label, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (menuitem), hb);
	gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM (menuitem),
					 pixmap);
	gtk_widget_lock_accelerators (menuitem);
	
	gtk_widget_show_all (hb);
	if (gnome_preferences_get_menus_have_icons ())
		gtk_widget_show (pixmap);
	
	gtk_object_set_user_data (GTK_OBJECT (menuitem), entry);
	gtk_object_set_data (GTK_OBJECT (menuitem), "pixmap", pixmap);
	gtk_object_set_data (GTK_OBJECT (menuitem), "hb", hb);
	gtk_object_set_data (GTK_OBJECT (menuitem), "label", label);
	gtk_object_set_data (GTK_OBJECT (menuitem), "url", entry->url);
	gtk_object_set_data_full (GTK_OBJECT (menuitem), "title",
				  title, g_free);
	gtk_signal_connect (GTK_OBJECT(menuitem), "activate",
			    GTK_SIGNAL_FUNC (window_menu_recent_activate_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT(menuitem), "button-press-event",
			    GTK_SIGNAL_FUNC
			    (generic_link_button_press_event_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT(menuitem), "button-release-event",
			    GTK_SIGNAL_FUNC (window_menu_recent_release_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT(menuitem), "select",
			    GTK_SIGNAL_FUNC (window_menu_recent_select_cb),
			    window);
	gtk_signal_connect (GTK_OBJECT(menuitem), "deselect",
			    GTK_SIGNAL_FUNC (window_menu_recent_deselect_cb),
			    window);

	return menuitem;
}

/**
 * Remove an entry form the recent menu. Frees the struct and removes the
 * menu item.
 **/
static void
window_remove_recent_menu_entry (GaleonWindow *window, RecentMenuEntry *entry)
{
	GtkWidget *menu;
	return_if_not_window (window);
	g_return_if_fail (entry != NULL);

	window->recent_menu_entries = g_list_remove 
		(window->recent_menu_entries, entry);

	menu = window->go_menu;
	if (menu)
	{
		gtk_container_remove (GTK_CONTAINER (menu),
				      GTK_WIDGET (entry->menuitem));
	}

	g_free (entry->url);
	g_free (entry->title);
	g_free (entry);
}

/* FIXME why callbacks here ? we have window_callbacks.c */

static void
window_menu_recent_select_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	RecentMenuEntry *e = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	gnome_appbar_push (GNOME_APPBAR (window->appbar), e->url); 
}

static void
window_menu_recent_deselect_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	gnome_appbar_pop (GNOME_APPBAR (window->appbar));
}

/**
 * Updates the number of a recent menu item 
 **/
static void
window_recent_menu_entry_update_number (GaleonWindow *window, 
					RecentMenuEntry *entry, gint num)
{
	GtkWidget *menuitem = GTK_WIDGET (entry->menuitem);
	GtkWidget *hb = gtk_object_get_data (GTK_OBJECT (menuitem), "hb");
	gchar *title = gtk_object_get_data (GTK_OBJECT (menuitem), "title");
	GtkWidget *label = gtk_object_get_data (GTK_OBJECT (menuitem),
						"label");
	GtkWidget *new_label;

	new_label = misc_gui_new_num_accel_label (num - 1, title, FALSE,
						  window->go_menu, menuitem);
	
	gtk_box_pack_start (GTK_BOX (hb), new_label, FALSE, FALSE, 0);
	gtk_container_remove (GTK_CONTAINER (hb), label);
	gtk_widget_show_all (hb);

	gtk_object_set_data (GTK_OBJECT (menuitem), "label", new_label);

}
