/**********************************************
 *
 * $GAMGI/src/gtk/cell/gamgi_gtk_cell_shift.c
 *
 * Copyright (C) 2010 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_gtk.h"
#include "gamgi_math.h"
#include "gamgi_phys.h"
#include "gamgi_io.h"
#include "gamgi_global.h"

#include "gamgi_engine_create.h"
#include "gamgi_engine_start.h"
#include "gamgi_engine_remove.h"
#include "gamgi_engine_list.h"
#include "gamgi_engine_find.h"
#include "gamgi_gtk_dialog.h"
#include "gamgi_gtk_object.h"
#include "gamgi_gtk_cell_symmetry.h"
#include "gamgi_mesa_select.h"
#include "gamgi_math_vector.h"
#include "gamgi_math_matrix.h"
#include "gamgi_math_quaternion.h"
#include "gamgi_math_euler.h"
#include "gamgi_math_cell.h"
#include "gamgi_phys_cell.h"
#include "gamgi_phys_plane.h"
#include "gamgi_phys_direction.h"
#include "gamgi_io_token.h"

static struct {
double shift[3];
gamgi_enum vectors;
gamgi_bool active; } translation;

static struct {
double angle;
int axis[3];
gamgi_enum type;
gamgi_enum vectors;
gamgi_bool active; } rotation;

static void static_translation_remove (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button;

/***********************************************
 * this function presses the level 0 button    *
 * Translation, that closes the level 1 dialog *
 ***********************************************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_translation");
gtk_button_clicked (GTK_BUTTON (button));
}

static void static_translation_enable (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button;
GtkWidget *label;
GdkColor color;

translation.active = TRUE;

/******************** 
 * set active color *
 ********************/

gamgi_gtk_dialog_color (&color,
GAMGI_GTK_MOVE_R, GAMGI_GTK_MOVE_G, GAMGI_GTK_MOVE_B);

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_translation");
label = gtk_bin_get_child (GTK_BIN (button));
gamgi_gtk_dialog_color_fg (label, color);
}

static void static_translation_disable (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button;
GtkWidget *label;

translation.active = FALSE;

/*********************
 * set default color *
 *********************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_translation");
label = gtk_bin_get_child (GTK_BIN (button));
gamgi_gtk_dialog_color_default (label, window);
}

static void static_translation_cancel (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;

static_translation_disable (window);
static_translation_remove (window);
}

static void static_translation_delete (GtkWidget *widget, 
GdkEventButton *event, void *data)
{
static_translation_cancel (widget, data);
}

static void static_translation_ok (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog1;
GtkWidget *entry;
GtkWidget *combo;
const char *name;

/*************************
 * get shift coordinates *
 *************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_double_scan (name, &translation.shift[0], 
-DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Cell Data", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_double_scan (name, &translation.shift[1],
-DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Cell Data", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_double_scan (name, &translation.shift[2],
-DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Cell Data", window);
  return;
  }

/*********************
 * get shift vectors *
 *********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");
translation.vectors = 1 + gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

/********************************************************************
 * coordinates for absolute vectors must be real, coordinates for   *
 * conventional and primitive vectors must be in the range ]-1, +1[ *
 ********************************************************************/

if ((translation.vectors == GAMGI_PHYS_CONVENTIONAL ||
translation.vectors == GAMGI_PHYS_PRIMITIVE) &&
(abs (translation.shift[0]) >= 1.0 || abs (translation.shift[1]) >= 1.0 ||
abs (translation.shift[2]) >= 1.0))
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Vector Coordinates", window);
  return;
  }

static_translation_enable (window);
static_translation_remove (window);
}

static void static_translation_init (gamgi_window *window)
{
GtkWidget *dialog = window->dialog1;
GtkWidget *entry_x, *entry_y, *entry_z;
GtkWidget *combo;
char token[GAMGI_ENGINE_TOKEN];

entry_x = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
entry_y = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
entry_z = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");

/**********************
 * show empty default *
 **********************/

if (translation.active == FALSE)
  {
  gtk_entry_set_text (GTK_ENTRY (entry_x), "");
  gtk_entry_set_text (GTK_ENTRY (entry_y), "");
  gtk_entry_set_text (GTK_ENTRY (entry_z), "");
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
  }
else
  {
  /**********************************
   * show current translation shift *
   **********************************/

  sprintf (token, "%.*f", gamgi->gamgi->length, translation.shift[0]);
  gtk_entry_set_text (GTK_ENTRY (entry_x), token);
  sprintf (token, "%.*f", gamgi->gamgi->length, translation.shift[1]);
  gtk_entry_set_text (GTK_ENTRY (entry_y), token);
  sprintf (token, "%.*f", gamgi->gamgi->length, translation.shift[2]);
  gtk_entry_set_text (GTK_ENTRY (entry_z), token);
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), translation.vectors - 1);
  }
}

static void static_translation (gamgi_window *window)
{
GtkWidget *dialog;
GtkWidget *button;
GtkWidget *label;
GtkWidget *entry;
GtkWidget *table;
GtkWidget *vbox_dialog, *vbox_frame;
GtkWidget *hbox_center, *hbox_left;
GtkWidget *frame;
GtkWidget *combo;
GtkListStore *store;
GtkCellRenderer *renderer;
GtkTreeIter iter;

dialog = gamgi_gtk_dialog_task1_create ("Translation Shift", window);
gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
g_signal_connect (dialog, "delete_event",
G_CALLBACK (static_translation_delete), window);

vbox_dialog = gtk_vbox_new (FALSE, 5);
gtk_container_add (GTK_CONTAINER (dialog), vbox_dialog);
gtk_widget_show (vbox_dialog);

/********************
 * Cell coordinates *
 ********************/

frame = gtk_frame_new (NULL);
gtk_box_pack_start (GTK_BOX (vbox_dialog), frame, TRUE, TRUE, 0);
gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
gtk_widget_show (frame);

vbox_frame = gtk_vbox_new (FALSE, 5);
gtk_container_add (GTK_CONTAINER (frame), vbox_frame);
gtk_container_set_border_width (GTK_CONTAINER (vbox_frame), 10);
gtk_widget_show (vbox_frame);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_frame), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 6, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_widget_show (table);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_x", entry);
gtk_widget_show (entry);

label = gtk_label_new ("X");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_y", entry);
gtk_widget_show (entry);

label = gtk_label_new ("Y");
gtk_table_attach (GTK_TABLE (table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_z", entry);
gtk_widget_show (entry);

label = gtk_label_new ("Z");
gtk_table_attach (GTK_TABLE (table), label, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

/****************
 * Cell vectors *
 ****************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_frame), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Vectors");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "combo_vectors", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Conventional", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Primitive", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Absolute", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/*********************
 * Ok/Cancel buttons *
 *********************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_dialog), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

button = gamgi_gtk_dialog_button_create ("Ok", NULL);
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
g_signal_connect (button, "clicked",
G_CALLBACK (static_translation_ok), window);
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_show (button);

button = gamgi_gtk_dialog_button_create ("Cancel", "red");
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_grab_focus (button);
g_signal_connect (button, "clicked",
G_CALLBACK (static_translation_cancel), window);
gtk_widget_show (button);

static_translation_init (window);
gtk_widget_show (dialog);
}

static void static_rotation_remove (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button;

/********************************************
 * this function presses the level 0 button *
 * Rotation, that closes the level 1 dialog *
 ********************************************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_rotation");
gtk_button_clicked (GTK_BUTTON (button));
}

static void static_rotation_enable (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button;
GtkWidget *label;
GdkColor color;

rotation.active = TRUE;

/********************
 * set active color *
 ********************/

gamgi_gtk_dialog_color (&color,
GAMGI_GTK_MOVE_R, GAMGI_GTK_MOVE_G, GAMGI_GTK_MOVE_B);

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_rotation");
label = gtk_bin_get_child (GTK_BIN (button));
gamgi_gtk_dialog_color_fg (label, color);
}

static void static_rotation_disable (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *button;
GtkWidget *label;

rotation.active = FALSE;

/*********************
 * set default color *
 *********************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_rotation");
label = gtk_bin_get_child (GTK_BIN (button));
gamgi_gtk_dialog_color_default (label, window);
}

static void static_rotation_cancel (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;

static_rotation_disable (window);
static_rotation_remove (window);

/**************************
 * disable rotation shift *
 **************************/

}

static void static_rotation_delete (GtkWidget *widget, 
GdkEventButton *event, void *data)
{
static_rotation_cancel (widget, data);
}

static void static_rotation_type (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog1;
GtkWidget *label_u, *label_v, *label_w;
GtkWidget *combo;
int row;

/*************************************************************
 * this function updates the axis labels: plane or direction *
 *************************************************************/

label_u = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_u");
label_v = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_v");
label_w = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_w");

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
if (row == 0)
  {
  gtk_label_set_text (GTK_LABEL (label_u), "H");
  gtk_label_set_text (GTK_LABEL (label_v), "K");
  gtk_label_set_text (GTK_LABEL (label_w), "L");
  }
else
  {
  gtk_label_set_text (GTK_LABEL (label_u), "U");
  gtk_label_set_text (GTK_LABEL (label_v), "V");
  gtk_label_set_text (GTK_LABEL (label_w), "W");
  }
}

static void static_rotation_ok (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog1;
GtkWidget *entry;
GtkWidget *combo;
const char *name;

/**********************
 * get rotation angle *
 **********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_angle");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_double_scan (name, &rotation.angle, -360.0, 360.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Cell Angle", window);
  return;
  }

/*****************************************
 * get type of axis (direction or plane) *
 *****************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
rotation.type = 1 + gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

/************************
 * get axis coordinates *
 ************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_u");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_int_scan (name, &rotation.axis[0], -INT_MAX, INT_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Axis Data", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_v");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_int_scan (name, &rotation.axis[1], -INT_MAX, INT_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Axis Data", window);
  return;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_w");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_int_scan (name, &rotation.axis[2], -INT_MAX, INT_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Axis Data", window);
  return;
  }

if (gamgi_phys_direction_zero (rotation.axis) == TRUE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid Axis Data", window);
  return;
  }

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");
rotation.vectors = 1 + gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

static_rotation_enable (window);
static_rotation_remove (window);
}

static void static_rotation_init (gamgi_window *window)
{
GtkWidget *dialog = window->dialog1;
GtkWidget *entry;
GtkWidget *entry_u, *entry_v, *entry_w;
GtkWidget *combo_type, *combo_vectors;
GtkRequisition size;
char token[GAMGI_ENGINE_TOKEN];

/*******************************************************************
 * U V W labels are larger than H K L: set the largest window size *
 *******************************************************************/

gtk_widget_size_request (GTK_WIDGET (dialog), &size);
gtk_widget_set_size_request (dialog, size.width, size.height);

/*********************
 * get pointers data *
 *********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_angle");

entry_u = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_u");
entry_v = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_v");
entry_w = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_w");

combo_type = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
combo_vectors = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");

/**********************
 * show empty default *
 **********************/

if (rotation.active == FALSE)
  {
  gtk_entry_set_text (GTK_ENTRY (entry), "");

  gtk_entry_set_text (GTK_ENTRY (entry_u), "");
  gtk_entry_set_text (GTK_ENTRY (entry_v), "");
  gtk_entry_set_text (GTK_ENTRY (entry_w), "");
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo_type), 1);
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo_vectors), 0);
  }
else
  {
  /*******************
   * show angle data *
   *******************/

  sprintf (token, "%.*f", gamgi->gamgi->angle, rotation.angle);
  gtk_entry_set_text (GTK_ENTRY (entry), token);

  /******************
   * show axis data *
   ******************/

  sprintf (token, "%d", rotation.axis[0]);
  gtk_entry_set_text (GTK_ENTRY (entry_u), token);
  sprintf (token, "%d", rotation.axis[1]);
  gtk_entry_set_text (GTK_ENTRY (entry_v), token);
  sprintf (token, "%d", rotation.axis[2]);
  gtk_entry_set_text (GTK_ENTRY (entry_w), token);
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo_type), rotation.type - 1);
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo_vectors), rotation.vectors - 1);
  }
}

static void static_rotation (gamgi_window *window)
{
GtkWidget *dialog;
GtkWidget *button;
GtkWidget *label;
GtkWidget *entry;
GtkWidget *table;
GtkWidget *vbox_dialog, *vbox_frame, *vbox_top;
GtkWidget *hbox_center, *hbox_left;
GtkWidget *frame;
GtkWidget *combo;
GtkListStore *store;
GtkCellRenderer *renderer;
GtkTreeIter iter;

dialog = gamgi_gtk_dialog_task1_create ("Rotation Shift", window);
gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
g_signal_connect (dialog, "delete_event",
G_CALLBACK (static_rotation_delete), window);

vbox_dialog = gtk_vbox_new (FALSE, 5);
gtk_container_add (GTK_CONTAINER (dialog), vbox_dialog);
gtk_widget_show (vbox_dialog);

frame = gtk_frame_new (NULL);
gtk_box_pack_start (GTK_BOX (vbox_dialog), frame, TRUE, TRUE, 0);
gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
gtk_widget_show (frame);

vbox_frame = gtk_vbox_new (FALSE, 15);
gtk_container_add (GTK_CONTAINER (frame), vbox_frame);
gtk_container_set_border_width (GTK_CONTAINER (vbox_frame), 10);
gtk_widget_show (vbox_frame);

/*********
 * Angle *
 *********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_frame), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Angle");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_angle", entry);
gtk_widget_show (entry);

/********
 * Axis *
 ********/

vbox_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_frame), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Axis");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_signal_connect (combo, "changed",
G_CALLBACK (static_rotation_type), window);
g_object_set_data (G_OBJECT (dialog), "combo_type", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Plane", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Direction", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/****************
 * Axis indices *
 ****************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 7, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_widget_show (table);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_u", entry);
gtk_widget_show (entry);

label = gtk_label_new ("U");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "label_u", label);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_v", entry);
gtk_widget_show (entry);

label = gtk_label_new ("V");
gtk_table_attach (GTK_TABLE (table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "label_v", label);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_w", entry);
gtk_widget_show (entry);

label = gtk_label_new ("W");
gtk_table_attach (GTK_TABLE (table), label, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "label_w", label);
gtk_widget_show (label);

/****************
 * Axis vectors *
 ****************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Vectors");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "combo_vectors", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Conventional", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Primitive", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/*********************
 * Ok/Cancel buttons *
 *********************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_dialog), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

button = gamgi_gtk_dialog_button_create ("Ok", NULL);
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
g_signal_connect (button, "clicked",
G_CALLBACK (static_rotation_ok), window);
gtk_widget_show (button);

button = gamgi_gtk_dialog_button_create ("Cancel", "red");
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
gtk_widget_grab_focus (button);
g_signal_connect (button, "clicked",
G_CALLBACK (static_rotation_cancel), window);
gtk_widget_show (button);

gtk_widget_show (dialog);
static_rotation_init (window);
}

void gamgi_gtk_cell_shift_start (void)
{
/**************************************
 * disable translation,rotation shift *
 **************************************/

translation.active = FALSE;
rotation.active = FALSE;
}

void gamgi_gtk_cell_shift_translation_get (gamgi_cell *cell,
double *translate, gamgi_window *window)
{
double basis[9], inverse[9];
double position[3];

/***************************************************************
 * convert coordinates in primitive vectors ([-1,+1]) or       *
 * absolute vectors (any) to conventional vectors ([-1,+1]):   *
 *                                                             *
 * [conv. vectors][conv. coord] = [prim. vectors][prim. coord] *
 * [conv. vectors][conv. coord] = [axes coord]                 *
 *                                                             *
 * this conversion must be done before generating positions    *
 * with space symmetry information, as wyckoff positions       *
 * are always given as [-1,+1] conventional coordinates        *
 ***************************************************************/

if (translation.active == FALSE)
  { gamgi_math_vector_zero (translate); return; }

gamgi_math_vector_copy (translation.shift, translate);
if (translation.vectors != GAMGI_PHYS_CONVENTIONAL)
  {
  gamgi_math_matrix_absolute (basis,
  cell->a1[0], cell->a2[0], cell->a3[0],
  cell->a1[1], cell->a2[1], cell->a3[1],
  cell->a1[2], cell->a2[2], cell->a3[2]);
  gamgi_math_matrix_inverse (basis, inverse);

  if (translation.vectors == GAMGI_PHYS_PRIMITIVE)
    {
    gamgi_math_matrix_absolute (basis,
    cell->p1[0], cell->p2[0], cell->p3[0],
    cell->p1[1], cell->p2[1], cell->p3[1],
    cell->p1[2], cell->p2[2], cell->p3[2]);
    }
  else gamgi_math_matrix_unit (basis);

  /***********************************
   * convert to conventional vectors *
   ***********************************/

  gamgi_math_matrix_vector (basis, translate, position);
  gamgi_math_matrix_vector (inverse, position, translate);

  /*****************************************
   * use coordinates in the range ]-1, +1[ *
   *****************************************/

  gamgi_math_vector_fmod (translate, translate);
  }

}

void gamgi_gtk_cell_shift_rotation_get (gamgi_cell *cell,
double *rotate, gamgi_window *window)
{
double quaternion[4];
double axis[3];
double angle;

if (rotation.active == FALSE)
  { gamgi_math_matrix_unit (rotate); return; }

/***********************
 * get rotation matrix *
 ***********************/

angle = GAMGI_MATH_DEG_RAD * rotation.angle;

if (rotation.type == FALSE)
  gamgi_phys_plane_vector (cell, rotation.axis, rotation.vectors, axis);
else
  gamgi_phys_direction_vector (cell, rotation.axis, rotation.vectors, axis);

gamgi_math_quaternion_from_axis (angle, axis, quaternion);
gamgi_math_quaternion_to_matrix (quaternion, rotate);
}

void gamgi_gtk_cell_shift_rotate (gamgi_cell *cell,
int n_positions, double *positions)
{
if (rotation.active == FALSE) return;
}

void gamgi_gtk_cell_shift_translation (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *button;

if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)) == TRUE)
  {
  /*********************
   * reset Cell dialog *
   *********************/

  button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_rotation");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), FALSE);

  /***********************
   * create Align dialog *
   ***********************/

  static_translation (window);
  }

/***********************
 * remove Align dialog *
 ***********************/

else gamgi_gtk_dialog_task1_remove (widget, window);
}

void gamgi_gtk_cell_shift_rotation (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *button;

if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)) == TRUE)
  {
  /*********************
   * reset Cell dialog *
   *********************/

  button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_translation");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), FALSE);

  /***********************
   * Create Align Dialog *
   ***********************/

  static_rotation (window);
  }

/***********************
 * remove Align dialog *
 ***********************/

else gamgi_gtk_dialog_task1_remove (widget, window);
}
