/**
 * File:          $RCSfile: cameraf.c,v $
 * Module:        Camera definitions and building functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.8 $
 * Last edited:   $Date: 2002/05/24 12:00:46 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gandalf/vision/cameraf.h>
#include <gandalf/vision/cameraf_linear.h>
#include <gandalf/vision/cameraf_radial_dist1.h>
#include <gandalf/vision/cameraf_radial_dist2.h>
#include <gandalf/vision/cameraf_radial_dist3.h>
#include <gandalf/vision/cameraf_stereographic.h>
#include <gandalf/vision/cameraf_equidistant.h>
#include <gandalf/vision/cameraf_sine_law.h>
#include <gandalf/vision/cameraf_equi_solid_angle.h>
#include <gandalf/vision/cameraf_xy_dist4.h>
#include <gandalf/vision/cameraf_radial_dist1_inv.h>
#include <gandalf/common/numerics.h>

/**
 * \addtogroup Vision
 * \{
 */

/**
 * \addtogroup Camera
 * \{
 */

/**
 * \brief Constructs a structure representing a camera.
 * \param camera Pointer to the camera structure to be filled
 * \param type Type of camera e.g. #GAN_LINEAR_CAMERA
 * \param zh Third homogeneous image coordinate
 * \param fx Focal distance measured in x-pixels
 * \param fy Focal distance measured in y-pixels
 * \param x0 x-coordinate of image centre
 * \param y0 y-coordinate of image centre
 * \param ... Extra arguments depending on camera type
 * \return #GAN_TRUE on successfully building camera structure,
 *         #GAN_FALSE on failure.
 *
 * Constructs a structure representing a camera.
 *
 * \sa gan_camera_build_va().
 */
Gan_Bool
 gan_cameraf_build_va ( Gan_Camera_f *camera,
                        Gan_CameraType type,
                        float zh, float fx, float fy,
                        float x0, float y0, ... )
{
   va_list ap;
   Gan_Bool result;

   va_start ( ap, y0 );
   switch ( type )
   {
      case GAN_LINEAR_CAMERA:
        result = gan_cameraf_build_linear ( camera, zh, fx, fy, x0, y0 );
        break;

      case GAN_RADIAL_DISTORTION_1:
      {
         float K1 = (float)va_arg ( ap, double );

         result = gan_cameraf_build_radial_distortion_1 ( camera, zh, fx, fy,
                                                          x0, y0, K1 );
      }
      break;
        
      case GAN_RADIAL_DISTORTION_2:
      {
         float K1 = (float)va_arg ( ap, double );
         float K2 = (float)va_arg ( ap, double );

         result = gan_cameraf_build_radial_distortion_2 ( camera, zh, fx, fy,
                                                          x0, y0, K1, K2 );
      }
      break;
        
      case GAN_RADIAL_DISTORTION_3:
      {
         float K1 = (float)va_arg ( ap, double );
         float K2 = (float)va_arg ( ap, double );
         float K3 = (float)va_arg ( ap, double );

         result = gan_cameraf_build_radial_distortion_3 ( camera, zh, fx, fy,
                                                          x0, y0, K1, K2, K3 );
      }
      break;
        
      case GAN_RADIAL_DISTORTION_1_INV:
      {
         float K1 = (float)va_arg ( ap, double );

         result = gan_cameraf_build_radial_distortion_1_inv ( camera,
                                                              zh, fx, fy,
                                                              x0, y0, K1 );
      }
      break;
        
      case GAN_STEREOGRAPHIC_CAMERA:
        result = gan_cameraf_build_stereographic ( camera, zh, fx, fy, x0, y0);
        break;

      case GAN_EQUIDISTANT_CAMERA:
        result = gan_cameraf_build_equidistant ( camera, zh, fx, fy, x0, y0 );
        break;

      case GAN_SINE_LAW_CAMERA:
        result = gan_cameraf_build_sine_law ( camera, zh, fx, fy, x0, y0 );
        break;

      case GAN_EQUI_SOLID_ANGLE_CAMERA:
        result = gan_cameraf_build_equi_solid_angle ( camera,
                                                      zh, fx, fy, x0, y0 );
        break;

      case GAN_XY_DISTORTION_4:
      {
         float cxx = (float)va_arg ( ap, double );
         float cxy = (float)va_arg ( ap, double );
         float cyx = (float)va_arg ( ap, double );
         float cyy = (float)va_arg ( ap, double );

         result = gan_cameraf_build_xy_distortion_4 ( camera, zh, fx, fy,
                                                      x0, y0,
                                                      cxx, cxy, cyx, cyy );
      }
      break;
        
      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_cameraf_build_va", GAN_ERROR_ILLEGAL_TYPE, "");
        return GAN_FALSE;
   }
   
   va_end ( ap );

   if ( !result )
   {
      gan_err_register ( "gan_cameraf_build_va", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Builds the internals of a camera.
 * \param camera Pointer to the camera structure to be filled
 * \return #GAN_TRUE on successfully internalising camera structure,
 *         #GAN_FALSE on failure.
 *
 * Builds the internal thresholds and callbacks of a camera given that the
 * type and parameters have already been set.
 *
 * \sa gan_camera_internalize().
 */
Gan_Bool
 gan_cameraf_internalize ( Gan_Camera_f *camera )
{
   Gan_Bool result;

   switch ( camera->type )
   {
      case GAN_LINEAR_CAMERA:
        result = gan_cameraf_build_linear ( camera, camera->zh,
                                            camera->fx, camera->fy,
                                            camera->x0, camera->y0 );
        break;

      case GAN_RADIAL_DISTORTION_1:
        result = gan_cameraf_build_radial_distortion_1 (
                     camera, camera->zh,
                     camera->fx, camera->fy,
                     camera->x0, camera->y0,
                     camera->nonlinear.radial1.K1 );
        break;
        
      case GAN_RADIAL_DISTORTION_2:
        result = gan_cameraf_build_radial_distortion_2 (
                     camera, camera->zh,
                     camera->fx, camera->fy,
                     camera->x0, camera->y0,
                     camera->nonlinear.radial2.K1,
                     camera->nonlinear.radial2.K2 );
        break;
        
      case GAN_RADIAL_DISTORTION_3:
        result = gan_cameraf_build_radial_distortion_3 (
                     camera, camera->zh,
                     camera->fx, camera->fy,
                     camera->x0, camera->y0,
                     camera->nonlinear.radial3.K1,
                     camera->nonlinear.radial3.K2,
                     camera->nonlinear.radial3.K3 );
        break;
        
      case GAN_RADIAL_DISTORTION_1_INV:
        result = gan_cameraf_build_radial_distortion_1_inv (
                     camera, camera->zh,
                     camera->fx, camera->fy,
                     camera->x0, camera->y0,
                     camera->nonlinear.radial1.K1 );
        break;
        
      case GAN_STEREOGRAPHIC_CAMERA:
        result = gan_cameraf_build_stereographic ( camera, camera->zh,
                                                   camera->fx, camera->fy,
                                                   camera->x0, camera->y0 );
        break;

      case GAN_EQUIDISTANT_CAMERA:
        result = gan_cameraf_build_equidistant ( camera, camera->zh,
                                                 camera->fx, camera->fy,
                                                 camera->x0, camera->y0 );
        break;

      case GAN_SINE_LAW_CAMERA:
        result = gan_cameraf_build_sine_law ( camera, camera->zh,
                                              camera->fx, camera->fy,
                                              camera->x0, camera->y0 );
        break;

      case GAN_EQUI_SOLID_ANGLE_CAMERA:
        result = gan_cameraf_build_equi_solid_angle ( camera, camera->zh,
                                                      camera->fx, camera->fy,
                                                      camera->x0, camera->y0 );
        break;

      case GAN_XY_DISTORTION_4:
        result = gan_cameraf_build_xy_distortion_4 (
                     camera, camera->zh,
                     camera->fx, camera->fy,
                     camera->x0, camera->y0,
                     camera->nonlinear.xydist4.cxx,
                     camera->nonlinear.xydist4.cxy,
                     camera->nonlinear.xydist4.cyx,
                     camera->nonlinear.xydist4.cyy );
        break;
        
      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_cameraf_internalize", GAN_ERROR_ILLEGAL_TYPE,
                           "" );
        return GAN_FALSE;
   }
   
   if ( !result )
   {
      gan_err_register ( "gan_cameraf_internalize", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Sets the gamma for images taken with this camera.
 * \param camera Pointer to the camera structure
 * \param gamma The new gamma value
 *
 * Sets the gamma for images taken with this camera. The default value is one,
 * set when the camera structure was built. This is the single precision
 * version of the function.
 *
 * \return #GAN_TRUE on successfully setting the gamma, #GAN_FALSE on failure.
 */
Gan_Bool
 gan_cameraf_set_gamma ( Gan_Camera_f *camera, float gamma )
{
   camera->gamma = gamma;
   return GAN_TRUE;
}

/**
 * \brief Fills and returns a 3x3 upper triangular camera matrix.
 * \param camera Pointer to the camera structure to be read
 *
 * Fills and returns a 3x3 upper triangular camera matrix using the linear
 * parameters of the provided camera. Gandalf only stores lower triangular
 * matrices explicitly, so the transpose camera matrix is actually returned.
 *
 * Single precision version of gan_camera_fill_matrix_q().
 *
 * \return The filled 3x3 upper triangular matrix.
 */
Gan_SquMatrix33_f
 gan_cameraf_fill_matrix_s ( Gan_Camera_f *camera )
{
   return gan_ltmat33f_fill_s ( camera->fx,
                                      0.0F, camera->fy,
                                camera->x0, camera->y0, camera->zh );
}

/**
 * \}
 */

/**
 * \}
 */
