/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				rotate.c

Purpose:
	Rotate all caught macromolecular complexes.

Input:
	(1) Pointer to MolComplexS structure, with macromol. complexes.
	(2) Number of macromolecular complexes.
	(3) Pointer to ConfigS structure.
	(4) Rotation angle.
	(5) Rotation axis identifier (1 = x, 2 = y, 3 = z).

Output:
	(1) Atoms rotated in all caught macromolecular complexes.

Return value:
	No return value.

========includes:============================================================*/

#include <stdio.h>
#include <math.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

int		PrepareStereoData_ (MolComplexS *, ConfigS *);
void		RotatePlane_ (MolComplexS *, ConfigS *, double, int);

/*======rotate all caught macromol. complexes:===============================*/

void Rotate_ (MolComplexS *mol_complexSP, int mol_complexesN,
	      ConfigS *configSP,
	      double rotation_angle, int axisID)
{
double		cos_angle, sin_angle;
int		mol_complexI;
MolComplexS	*curr_mol_complexSP;
size_t		atomsN, atomI;
AtomS		*curr_atomSP;
double		x0, y0, z0, x, y, z, x_new = 0.0, y_new = 0.0, z_new = 0.0;

/* Calculate cosine and sine of the rotation angle: */
cos_angle = cos (rotation_angle);
sin_angle = sin (rotation_angle);

for (mol_complexI = 0; mol_complexI < mol_complexesN; mol_complexI++)
	{
	/* Pointer to the current macromolecular complex: */
	curr_mol_complexSP = mol_complexSP + mol_complexI;

	/* Check is the current macromolecular complex caught: */
	if (curr_mol_complexSP->catchF == 0) continue;

	/* Check is it necessary to rotate the plane: */
	if (curr_mol_complexSP->move_bits & PLANE_MASK)
		{
		RotatePlane_ (curr_mol_complexSP, configSP,
			      rotation_angle, axisID);
		}

	/* Check is it necessary to rotate the structure at all: */
	if (!(curr_mol_complexSP->move_bits & STRUCTURE_MASK)) continue;

	/* Copy the rotation center coordinates: */
	x0 = curr_mol_complexSP->rotation_center_vectorS.x;
	y0 = curr_mol_complexSP->rotation_center_vectorS.y;
	z0 = curr_mol_complexSP->rotation_center_vectorS.z;

	/* Number of atoms in a macromolecular complex: */
	atomsN = curr_mol_complexSP->atomsN;

	/* Scan all atoms in the current complex: */
	for (atomI = 0; atomI < atomsN; atomI++)
		{
		/* Pointer to the current atom: */
		curr_atomSP = curr_mol_complexSP->atomSP + atomI;

		/* Coordinates relative to the rotation center: */
		x = curr_atomSP->raw_atomS.x[0] - x0;
		y = curr_atomSP->raw_atomS.y    - y0;
		z = curr_atomSP->raw_atomS.z[0] - z0;

		/* Rotate these coordinates: */
		switch (axisID)
			{
			/* Rotate around x: */
			case 1:
				x_new =  x;
				y_new =  y * cos_angle - z * sin_angle;
				z_new =  y * sin_angle + z * cos_angle;
				break;

			/* Rotate around y: */
			case 2:
				x_new =  x * cos_angle + z * sin_angle;
				y_new =  y;
				z_new = -x * sin_angle + z * cos_angle;
				break;

			/* Rotate around z: */
			case 3:
				x_new =  x * cos_angle - y * sin_angle;
				y_new =  x * sin_angle + y * cos_angle;
				z_new =  z;
				break;

			/* The impossible case: */
			default:
				;
				break;

			}

		/* Translate and store new coordinates: */
		curr_atomSP->raw_atomS.x[0] = x_new + x0;
		curr_atomSP->raw_atomS.y    = y_new + y0;
		curr_atomSP->raw_atomS.z[0] = z_new + z0;
		}

	/* Prepare stereo data, if required: */
	if (configSP->stereoF)
		{
		PrepareStereoData_ (curr_mol_complexSP, configSP);
		}

	/* Set the position_changedF: */
	curr_mol_complexSP->position_changedF = 1;
	}
}

/*===========================================================================*/


