/* sqlquerygrid.c
 *
 * Copyright (C) 2001 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "sqlquerygrid.h"

static void sql_query_grid_class_init (SqlQueryGridClass * class);
static void sql_query_grid_init (SqlQueryGrid * qgr);
static void sql_query_grid_post_init (SqlQueryGrid * qgr);
static void sql_query_grid_destroy (GtkObject * object);

/*
 *
 * Main functions for the object
 *
 */

/* get a pointer to the parents to be able to call their destructor */
static GtkObject *parent_class = NULL;

enum
{
	ACTION_REFRESH,
	ACTION_INSERT,
	ACTION_EDIT,
	ACTION_DELETE,
	ACTION_CHOOSE,
	LAST_SIGNAL
};

static gint sql_query_grid_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0 };

guint
sql_query_grid_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Query_Grid",
			sizeof (SqlQueryGrid),
			sizeof (SqlQueryGridClass),
			(GtkClassInitFunc) sql_query_grid_class_init,
			(GtkObjectInitFunc) sql_query_grid_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_vbox_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_query_grid_class_init (SqlQueryGridClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	sql_query_grid_signals[ACTION_REFRESH] =
		gtk_signal_new ("action_refresh",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryGridClass,
						   action_refresh),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	sql_query_grid_signals[ACTION_INSERT] =
		gtk_signal_new ("action_insert",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryGridClass,
						   action_insert),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	sql_query_grid_signals[ACTION_EDIT] =
		gtk_signal_new ("action_edit",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryGridClass,
						   action_edit),
				gtk_marshal_NONE__UINT, GTK_TYPE_NONE, 1,
				GTK_TYPE_ULONG);

	sql_query_grid_signals[ACTION_DELETE] =
		gtk_signal_new ("action_delete",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryGridClass,
						   action_delete),
				gtk_marshal_NONE__UINT, GTK_TYPE_NONE, 1,
				GTK_TYPE_ULONG);

	sql_query_grid_signals[ACTION_CHOOSE] =
		gtk_signal_new ("action_choose",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryGridClass,
						   action_choose),
				gtk_marshal_NONE__UINT, GTK_TYPE_NONE, 1,
				GTK_TYPE_ULONG);


	gtk_object_class_add_signals (object_class, sql_query_grid_signals,
				      LAST_SIGNAL);
	class->action_refresh = NULL;
	class->action_insert = NULL;
	class->action_edit = NULL;
	class->action_delete = NULL;
	class->action_choose = NULL;

	object_class->destroy = sql_query_grid_destroy;
	parent_class = gtk_type_class (gtk_object_get_type ());
}

static void
sql_query_grid_init (SqlQueryGrid * qgr)
{
	qgr->qx = NULL;
	qgr->grid = NULL;
	qgr->private = NULL;
	qgr->sel_row = -1;
}

static void sql_query_grid_post_init (SqlQueryGrid * qgr);
static void qgr_do_action_insert (GtkWidget * button, SqlQueryGrid * qgr);
static void qgr_do_action_edit (GtkWidget * button, SqlQueryGrid * qgr);
static void qgr_do_action_refresh (GtkWidget * button, SqlQueryGrid * qgr);
static void qgr_do_action_delete (GtkWidget * button, SqlQueryGrid * qgr);
static void qgr_do_action_choose (GtkWidget * button, SqlQueryGrid * qgr);
GtkWidget *
sql_query_grid_new (SqlQueryExec * exec)
{
	GtkObject *obj;
	SqlQueryGrid *qgr;
	GtkWidget *wid;
	SqlQueryActions action;

	g_return_val_if_fail (exec != NULL, NULL);
	g_return_val_if_fail (IS_SQL_QUERY_EXEC (exec), NULL);

	obj = gtk_type_new (sql_query_grid_get_type ());
	qgr = SQL_QUERY_GRID (obj);
	qgr->qx = exec;

	/* get the action buttons */
	wid = sql_query_exec_get_action_buttons (exec);
	gtk_object_set_data (GTK_OBJECT (qgr), "buttons", wid);

	/* get the main display area */
	if (exec->qres) {
		sql_query_grid_post_init (qgr);	/* will set up the qgr->grid widget */
		gtk_box_pack_start_defaults (GTK_BOX (qgr), qgr->private);
	}
	else
		g_error ("SqlQueryRes is NULL! %s line %d", __FILE__,
			 __LINE__);

	/* update the buttons displayed as configured by the SqlQueryEnv */
	gtk_box_pack_start (GTK_BOX (qgr), wid, FALSE, FALSE, 0);
	for (action = QUERY_ACTION_FIRST; action < QUERY_ACTION_LAST_ENUM;
	     action *= 2) {
		GtkWidget *button;
		gchar *str;
		str = g_strdup_printf ("%ld", action);
		button = gtk_object_get_data (GTK_OBJECT (wid), str);
		if (button) {
			if (exec->q->text_only)
				switch (action) {
				case QUERY_ACTION_COMMIT:
				case QUERY_ACTION_FIRST:
				case QUERY_ACTION_PREV:
				case QUERY_ACTION_NEXT:
				case QUERY_ACTION_LAST:
				case QUERY_ACTION_DELETE:
				case QUERY_ACTION_INSERT:
				case QUERY_ACTION_EDIT:
				case QUERY_ACTION_VIEWALL:
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
					break;
				case QUERY_ACTION_REFRESH:
					gtk_widget_set_sensitive (button,
								  FALSE);
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qgr_do_action_refresh),
							    qgr);
					break;
				case QUERY_ACTION_CHOOSE:
					gtk_widget_set_sensitive (button,
								  FALSE);
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qgr_do_action_choose),
							    qgr);
					break;
				}
			else
				switch (action) {
				case QUERY_ACTION_COMMIT:
				case QUERY_ACTION_FIRST:
				case QUERY_ACTION_PREV:
				case QUERY_ACTION_NEXT:
				case QUERY_ACTION_LAST:
				case QUERY_ACTION_VIEWALL:
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
					break;
				case QUERY_ACTION_REFRESH:
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qgr_do_action_refresh),
							    qgr);
					break;
				case QUERY_ACTION_EDIT:
					gtk_widget_set_sensitive (button,
								  FALSE);
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qgr_do_action_edit),
							    qgr);
					break;
				case QUERY_ACTION_DELETE:
					gtk_widget_set_sensitive (button,
								  FALSE);
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qgr_do_action_delete),
							    qgr);
					break;
				case QUERY_ACTION_CHOOSE:
					gtk_widget_set_sensitive (button,
								  FALSE);
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qgr_do_action_choose),
							    qgr);
					break;
				case QUERY_ACTION_INSERT:
					if (qgr->qx->env->no_insert) {
						gtk_widget_destroy (button);
						gtk_object_set_data
							(GTK_OBJECT (wid),
							 str, NULL);
					}
					else
						gtk_signal_connect (GTK_OBJECT
								    (button),
								    "clicked",
								    GTK_SIGNAL_FUNC
								    (qgr_do_action_insert),
								    qgr);
					break;
				}
		}
		g_free (str);
	}


	/* FIXME: connect to the "not_valid" signal of SqlQueryExec object to freeze the widget */
	return GTK_WIDGET (obj);
}

static void
sql_query_grid_post_init (SqlQueryGrid * qgr)
{
	GtkWidget *sw;

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_ALWAYS);
	gtk_widget_show (sw);

	qgr->private = sw;
	sql_query_grid_refresh (qgr);
}

static void
sql_query_grid_destroy (GtkObject * object)
{
	SqlQueryGrid *qgr;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_QUERY_GRID (object));

	qgr = SQL_QUERY_GRID (object);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void update_titles_contents (SqlQueryGrid * qgr);
static void object_clist_select_row_cb (GtkWidget * wid,
					gint row, gint column,
					GdkEventButton * event,
					SqlQueryGrid * qgr);
static void object_clist_unselect_row_cb (GtkWidget * wid,
					  gint row, gint column,
					  GdkEventButton * event,
					  SqlQueryGrid * qgr);
void
sql_query_grid_refresh (SqlQueryGrid * qgr)
{
	gchar **entry;
	SqlQueryRes *res;
	GtkWidget *clist = NULL;
	gint i, j, nb, nrows;
	SqlDataType **types;
	SqlDataDisplayFns **fns;
	GSList *list;
	QueryElement *elt;
	GdaField *gdaf;

	res = qgr->qx->qres;

	if (!res) {
		clist = gtk_label_new (_
				       ("Grid not available anymore because an error has occured\n"
					"while refreshing..."));
	}
	else {
		nb = sql_query_res_get_nbcols (res);

		/* CList titles */
		entry = (gchar **) g_malloc (sizeof (gchar *) * nb);
		/* types of data being displayed in the clist */
		types = (SqlDataType **) g_malloc (sizeof (gpointer) * nb);
		fns = (SqlDataDisplayFns **) g_malloc (sizeof (gpointer) *
						       nb);
		list = qgr->qx->q->objects;
		j = 0;
		while (list) {
			elt = (QueryElement *) (list->data);
			if (elt->print_name) {
				entry[j] = elt->print_name;	/* sets the column titles */
				types[j] = NULL;
				switch (elt->type) {
				case QUERY_ELT_FIELD:
					types[j] =
						SQL_MEM_FIELD (elt->main)->
						type;
					break;
				case QUERY_ELT_AGGREGATE:
					types[j] = NULL;
					break;
				case QUERY_ELT_FUNCTION:
					types[j] =
						SQL_DATA_FUNCTION (elt->
								   main)->
						result_type;
					break;
				case QUERY_ELT_VALUE:
					types[j] =
						((QueryElementValue *) (elt->
									main))->
						data_type;
					break;
				}
				if (types[j])
					fns[j] = sql_access_get_object_display_fns (qgr->qx->q->conf->srv, GTK_OBJECT (types[j]));
				else
					fns[j] = sql_access_get_object_display_fns (qgr->qx->q->conf->srv, NULL);
				j++;
			}
			list = g_slist_next (list);
		}


		if ((qgr->grid && GTK_IS_CLIST (qgr->grid) &&
		     (nb != GTK_CLIST (qgr->grid)->columns)) ||
		    (qgr->grid && !GTK_IS_CLIST (qgr->grid)) ||
		    (!qgr->grid)) {

			/* create new CList */
			if (nb == 0)
				clist = gtk_clist_new (1);
			else
				clist = gtk_clist_new_with_titles (nb, entry);
			gtk_clist_freeze (GTK_CLIST (clist));
			for (i = 0; i < nb; i++)
				gtk_clist_set_column_auto_resize (GTK_CLIST
								  (clist), i,
								  TRUE);

			/*     gtk_signal_connect(GTK_OBJECT(clist), */
			/*                       "button_press_event", */
			/*                       GTK_SIGNAL_FUNC(press_handler_event), qgr); */

			if (qgr->grid) {
				gtk_container_remove (GTK_CONTAINER
						      (qgr->private),
						      qgr->grid);
				/* REM: the above function destroys the widget which was inside the container */
				/* => no need for gtk_object_destroy(GTK_OBJECT(qgr->grid)); */
			}
			gtk_container_add (GTK_CONTAINER (qgr->private),
					   clist);
			qgr->grid = clist;
		}
		else {		/* use the existing CList and refresh the titles */
			gint i;
			clist = qgr->grid;
			gtk_clist_freeze (GTK_CLIST (clist));
			for (i = 0; i < nb; i++)
				gtk_clist_set_column_title (GTK_CLIST (clist),
							    i, entry[i]);
			gtk_clist_clear (GTK_CLIST (clist));
		}


		/* CList content */
		if (nb > 0) {
			nrows = sql_query_res_get_nbtuples (res);
			for (i = 1; i <= nrows; i++) {
				for (j = 0; j < nb; j++) {
					if (types[j]) {
						gchar *str;
						gdaf = sql_query_res_get_gdafield (res, i, j);
						str = (fns[j]->
						       gdafield_to_str)
						(gdaf);
						if (str)
							entry[j] = str;
						else
							entry[j] =
								g_strdup ("");
					}
					else
						entry[j] =
							sql_query_res_get_item
							(res, i, j);
				}
				gtk_clist_append (GTK_CLIST (clist), entry);
				for (j = 0; j < nb; j++)
					g_free (entry[j]);
			}
			g_free (entry);
		}

		gtk_clist_thaw (GTK_CLIST (clist));
		g_free (types);
		g_free (fns);

		gtk_clist_set_reorderable (GTK_CLIST (clist), FALSE);
		gtk_clist_set_selection_mode (GTK_CLIST (clist),
					      GTK_SELECTION_SINGLE);
		gtk_clist_column_titles_passive (GTK_CLIST (clist));
		gtk_signal_connect (GTK_OBJECT (clist), "select_row",
				    GTK_SIGNAL_FUNC
				    (object_clist_select_row_cb), qgr);
		gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
				    GTK_SIGNAL_FUNC
				    (object_clist_unselect_row_cb), qgr);
		gtk_widget_set_usize (clist, 400, 300);
	}

	gtk_widget_show (clist);

	qgr->sel_row = -1;	/* because new CList and nothing selected yet */
	object_clist_unselect_row_cb (NULL, 0, 0, NULL, qgr);
}

static void
object_clist_select_row_cb (GtkWidget * wid,
			    gint row, gint column,
			    GdkEventButton * event, SqlQueryGrid * qgr)
{
	gchar *str;
	GtkWidget *buttons, *button;

	qgr->sel_row = row;
	if (row >= 0) {
		buttons = gtk_object_get_data (GTK_OBJECT (qgr), "buttons");
		str = g_strdup_printf ("%ld", QUERY_ACTION_EDIT);
		button = gtk_object_get_data (GTK_OBJECT (buttons), str);
		g_free (str);
		if (button)
			gtk_widget_set_sensitive (button, TRUE);

		str = g_strdup_printf ("%ld", QUERY_ACTION_DELETE);
		button = gtk_object_get_data (GTK_OBJECT (buttons), str);
		g_free (str);
		if (button)
			gtk_widget_set_sensitive (button, TRUE);

		str = g_strdup_printf ("%ld", QUERY_ACTION_CHOOSE);
		button = gtk_object_get_data (GTK_OBJECT (buttons), str);
		g_free (str);
		if (button)
			gtk_widget_set_sensitive (button, TRUE);
	}
}

static void
object_clist_unselect_row_cb (GtkWidget * wid,
			      gint row, gint column,
			      GdkEventButton * event, SqlQueryGrid * qgr)
{
	gchar *str;
	GtkWidget *buttons, *button;

	qgr->sel_row = -1;

	buttons = gtk_object_get_data (GTK_OBJECT (qgr), "buttons");
	str = g_strdup_printf ("%ld", QUERY_ACTION_EDIT);
	button = gtk_object_get_data (GTK_OBJECT (buttons), str);
	g_free (str);
	if (button)
		gtk_widget_set_sensitive (button, FALSE);

	str = g_strdup_printf ("%ld", QUERY_ACTION_DELETE);
	button = gtk_object_get_data (GTK_OBJECT (buttons), str);
	g_free (str);
	if (button)
		gtk_widget_set_sensitive (button, FALSE);

	str = g_strdup_printf ("%ld", QUERY_ACTION_CHOOSE);
	button = gtk_object_get_data (GTK_OBJECT (buttons), str);
	g_free (str);
	if (button)
		gtk_widget_set_sensitive (button, FALSE);
}



static void
qgr_do_action_insert (GtkWidget * button, SqlQueryGrid * qgr)
{
	gtk_signal_emit (GTK_OBJECT (qgr),
			 sql_query_grid_signals[ACTION_INSERT]);
}

static void
qgr_do_action_edit (GtkWidget * button, SqlQueryGrid * qgr)
{
	gtk_signal_emit (GTK_OBJECT (qgr),
			 sql_query_grid_signals[ACTION_EDIT],
			 qgr->sel_row + 1);
}

static void
qgr_do_action_refresh (GtkWidget * button, SqlQueryGrid * qgr)
{
	gtk_signal_emit (GTK_OBJECT (qgr),
			 sql_query_grid_signals[ACTION_REFRESH]);
}

static void
qgr_do_action_delete (GtkWidget * button, SqlQueryGrid * qgr)
{
	gtk_signal_emit (GTK_OBJECT (qgr),
			 sql_query_grid_signals[ACTION_DELETE],
			 qgr->sel_row + 1);
}

static void
qgr_do_action_choose (GtkWidget * button, SqlQueryGrid * qgr)
{
	gtk_signal_emit (GTK_OBJECT (qgr),
			 sql_query_grid_signals[ACTION_CHOOSE],
			 qgr->sel_row + 1);
}
