;; -*- coding: utf-8; mode: scheme -*-
;;
;; parser.scm
;; 
;;  Copyright (c) 2006 KOGURO, Naoki (naoki@koguro.net)
;; 
;;  Permission is hereby granted, free of charge, to any person 
;;  obtaining a copy of this software and associated 
;;  documentation files (the "Software"), to deal in the 
;;  Software without restriction, including without limitation 
;;  the rights to use, copy, modify, merge, publish, distribute, 
;;  sublicense, and/or sell copies of the Software, and to 
;;  permit persons to whom the Software is furnished to do so, 
;;  subject to the following conditions:
;; 
;;  The above copyright notice and this permission notice shall 
;;  be included in all copies or substantial portions of the 
;;  Software.
;; 
;;  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY 
;;  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE 
;;  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
;;  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS 
;;  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR 
;;  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
;;  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
;;  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
;; 
;;  $Id: $

(define-module c-wrapper.c-parser
  (use srfi-1)
  (use srfi-13)
  (use srfi-11)
  (use gauche.process)
  (use gauche.sequence)
  (use file.util)
  (use util.match)
  (use gauche.vport)
  (use gauche.uvector)
  (use gauche.regexp)
  (use gauche.config)
  (use util.queue)
  (use c-wrapper.config)
  (use c-wrapper.c-ffi)

  (export c-parse)

  (dynamic-load "c-parser")
  )

(select-module c-wrapper.c-parser)

(debug-print-width #f)

(define-macro (profiler-on)
  '(define-syntax profile
     (syntax-rules ()
       ((_ . body)
        (begin
          (profiler-start)
          (begin . body)
          (profiler-stop))))))

(define-macro (profiler-off)
  '(define-syntax profile
     (syntax-rules ()
       ((_ . body)
        (begin . body)))))

(profiler-off)

(define (warning fmt . args)
  (apply format
         (standard-error-port)
         (string-append "Warning: " fmt "~%")
         args))

(define (c-type->class-symbol type)
  (string->symbol (string-append "<" (symbol->string type) ">")))

;;
;;
;;
(define-class <parse-context> ()
  (
   ;; c-parse parameters
   (import-cond :init-value #f)
   (export? :init-value #f)
   (ignore-dlsym-check? :init-value #f)
   
   ;; parser states
   (side-effect? :init-value #f)
   (use-return? :init-value #f)
   (use-jump? :init-value #f)
   (use-iterator? :init-value #f)
   (typedefed-identifiers :init-form (let ((tbl (make-hash-table)))
                                       (for-each (cut hash-table-put! tbl <> #t)
                                                 '(__builtin_va_list
                                                   char
                                                   short
                                                   int
                                                   long
                                                   float
                                                   double
                                                   void
                                                   _Bool))
                                       tbl))
   (struct-pool :init-form (make-hash-table))
   (union-pool :init-form (make-hash-table))
   (value-pool :init-form (make-hash-table))
   (arg-pool :init-form (make-hash-table))
   (macro-queue :init-form (make-queue))
   (import-pool :init-form (make-hash-table 'equal?))
   (imported-pool :init-form (make-hash-table 'equal?))
   (code-queue :init-form (let ((tail-cons '(#f)))
                            (cons tail-cons tail-cons)))

   ;; lexer states
   (last-token :init-value #f)
   (lineno :init-value #f)
   (filename :init-value #f)
   (rest-chars :init-value '())
   (input-port :init-value #f)))

(define context #f)

(define-macro (define-context-accessor name)
  `(define-syntax ,name
     (syntax-rules ()
       ((_)
        (slot-ref context ',name))
       ((_ val)
        (slot-set! context ',name val)))))

(define-context-accessor side-effect?)
(define-context-accessor use-return?)
(define-context-accessor use-jump?)
(define-context-accessor use-iterator?)
(define-context-accessor import-cond)
(define-context-accessor export?)
(define-context-accessor last-token)
(define-context-accessor lineno)
(define-context-accessor filename)
(define-context-accessor rest-chars)
(define-context-accessor input-port)
(define-context-accessor typedefed-identifiers)
(define-context-accessor struct-pool)
(define-context-accessor union-pool)
(define-context-accessor value-pool)
(define-context-accessor arg-pool)
(define-context-accessor macro-queue)
(define-context-accessor import-pool)
(define-context-accessor ignore-dlsym-check?)
(define-context-accessor imported-pool)

(define (do-external-declaration decl-specs init-decl-list)
  (define (extern)
    (when init-decl-list
      (for-each (lambda (init-decl)
                  (emit-define-extern decl-specs init-decl))
                init-decl-list)))
  (match decl-specs
    ((('STRUCT tagname (elem-alist ...)))
     (emit-init-struct tagname elem-alist)
     (extern))
    ((('STRUCT tagname #f))
     (emit-alloc-struct tagname)
     (extern))
    ((('UNION tagname (elem-alist ...)))
     (emit-init-union tagname elem-alist)
     (extern))
    ((('UNION tagname #f))
     (emit-alloc-union tagname)
     (extern))
    ((('ENUM tagname (enum-alist ...)))
     (emit-define-enum tagname enum-alist)
     (extern))
    (('TYPEDEF type ...)
     (emit-typedef (make-var-list type init-decl-list)))
    (else
     (extern))))

(define (install-arg-pool init-decl-alist)
  (for-each (lambda (alist)
              (and-let* ((kv (assq 'identifier alist)))
                (hash-table-put! (arg-pool) (cadr kv) #t)))
            init-decl-alist))

(define (parameter-decl type-spec-list decl)
  (let ((v (make-var type-spec-list decl)))
    ;; ISO/IEC 9899:1999 6.7.5.3
    (match (type-of v)
      (('c-func ret-type arg-types)
       (set! (type-of v) `(make-c-func-ptr ,ret-type ,arg-types)))
      (('c-func-vaargs ret-type arg-types)
       (set! (type-of v) `(make-c-func-vaargs-ptr ,ret-type ,arg-types)))
      (else
       #t))
    (and-let* ((name (name-of v)))
      (hash-table-put! (arg-pool) name #t))
    v))

(define (declaration specifiers declarator-list)
  (install-arg-pool declarator-list)
  (cons specifiers declarator-list))

(define (decl-identifier v)
  (list (list 'identifier v)))

(define (decl-array v)
  (list (list 'array (and v (%INT v)))))

(define (decl-func args)
  (list (cons 'c-func args)))

(define (decl-func-vaargs args)
  (list (cons 'c-func-vaargs args)))

(define (decl-ptr)
  '((ptr)))

(define (decl-keyword selector . typename-list)
  (cons (list (string-append (x->string selector)
                             (if (null? typename-list) "" ":")))
        typename-list))

(define (decl-bitfield decl n)
  (if decl
      (cons (list 'bit-field n) decl)
      (decl-bitfield (decl-identifier (gensym "%")) n)))

(define (decl-enum identifier lst)
  (list 'ENUM (or identifier (gensym "%")) lst))

(define (decl-enumerator identifier expr)
  (cons identifier expr))

(define (decl-init-value v)
  (list 'init-value v))

(define (combine-decl-keyword decl1 . rest)
  (let-optionals* rest ((decl2 #f))
    (if decl2
        (cons (append (car decl1) (car decl2))
              (append (cdr decl1) (cdr decl2)))
        decl1)))

(define (decl-objc-method ret-type decl-arg)
  (list (car decl-arg) (cons ret-type (cdr decl-arg))))

(define (decl-struct-or-union struct-or-union identifier decl-list)
  (list struct-or-union (or identifier (gensym "%")) decl-list))

(define (var-id)
  (make-var '(id) '()))

(define (%INT v)
  (if (real? v)
      (x->integer v)
      `(cast <integer> ,v)))

(define (%REAL v)
  (if (real? v)
      v
      `(cast <real> ,v)))

(define (%SCM-CAST expr)
  (match expr
    ((? number? v)
     v)
    ((? string? v)
     v)
    (('begin (or (? number? v) (? string? v)))
     v)
    (('cast _ (? number? v))
     v)
    (else
     `(scm-cast ,expr))))

(define (%IDENTIFIER v)
  (if (registered-identifier? v) v #f))

(define-syntax define-maybe
  (syntax-rules ()
    ((_ (name . args) . body)
     (define (name . args)
       (if (and . args)
           (begin . body)
           #f)))))

(define-maybe (%MACRO-BODY body)
  (if (null? (cdr body))
      (%SCM-CAST (car body))
      (%SCM-CAST `(begin ,@body))))

(define-maybe (%FUNCTION-BODY body)
  `(call/cc (lambda (%return) ,body)))

(define-maybe (%OBJC-STRING v)
  `(@ ,v))

(define-maybe (%EXPR-IN-PARENS expr)
  (if (symbol? expr)
      `(identity ,expr)
      expr))

(define-maybe (%COMPOUND-STATEMENT statements)
  `(begin ,@statements))

(define-maybe (%COMPOUND-STATEMENT-WITH-DECL decl-list statements)
  (let ((var-list '())
        (init-list '()))
    (for-each (lambda (alist)
                (for-each (lambda (declarator)
                            (let* ((v (make-var (car alist)
                                                  declarator))
                                   (type (type-of v))
                                   (identifier (name-of v))
                                   (init-val (value-of v)))
                              ;; TODO: typedef in compound_statement is not supported
                              (push! var-list
                                     `(,identifier (make ,type)))
                              (when init-val
                                (push! init-list
                                       `(set! (ref ,identifier)
                                              ,init-val)))))
                          (cdr alist)))
              decl-list)
    `(let* ,(reverse var-list)
       ,@(reverse init-list)
       ,@statements)))

(define-maybe (%REF-ARRAY v index)
  `(ref ,v ,(%INT index)))

(define-maybe (%FUNCALL func names)
  (side-effect? #t)
  (cons (c-lookup-value func) names))

(define-maybe (%DOT-REF v name)
  `(raw-ref ,v ',name))

(define-maybe (%PTR-REF p name)
  `(raw-ref (deref ,p) ',name))

(define-maybe (%POST-INC v)
  (side-effect? #t)
  `(post++ ,v))

(define-maybe (%POST-DEC v)
  (side-effect? #t)
  `(post-- ,v))

(define-maybe (%LIST v)
  (list v))

(define-maybe (%ADD-LIST lst v)
  (append lst (list v)))

(define-maybe (%PRE-INC v)
  (side-effect? #t)
  `(pre++ ,v))

(define-maybe (%PRE-DEC v)
  (side-effect? #t)
  `(pre-- ,v))

(define-maybe (%UNARY op v)
  (case op
    ((+) v)
    ((-) (%SUB 0 v))
    ((!) (%IF v 0 1))
    ((~) (%BIT-NOT v))
    ((&) `(ptr ,v))
    ((*) `(deref ,v))
    (else #f)))

(define-maybe (%BIT-NOT v)
  (if (integer? v)
      (lognot v)
      `(lognot ,(%INT v))))

(define-maybe (%SIZEOF-EXPR v)
  `(c-sizeof ,v))

(define-maybe (%SIZEOF-TYPE v)
  `(c-sizeof ,(type-of v)))

(define-maybe (%CAST type-name expr)
  `(cast ,(type-of type-name) ,expr))

(define-maybe (%MUL expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (* expr1 expr2)
      `(* ,(%REAL expr1) ,(%REAL expr2))))

(define-maybe (%DIV expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (/ expr1 expr2)
      `(/ ,(%REAL expr1) ,(%REAL expr2))))

(define-maybe (%MOD expr1 expr2)
  (if (and (integer? expr1) (integer? expr2))
      (modulo expr1 expr2)
      `(modulo ,(%INT expr1) ,(%INT expr2))))

(define-maybe (%ADD expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (+ expr1 expr2)
      `(if (is-a? ,expr1 <c-ptr>)
           (c-ptr+ ,expr1 ,(%REAL expr2))
           (+ (cast <real> ,expr1) ,(%REAL expr2)))))

(define-maybe (%SUB expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (- expr1 expr2)
      `(if (is-a? ,expr1 <c-ptr>)
           (c-ptr- ,expr1 ,(%REAL expr2))
           (- (cast <real> ,expr1) ,(%REAL expr2)))))

(define-maybe (%SHIFT-LEFT expr1 expr2)
  (if (and (integer? expr1) (integer? expr2))
      (ash expr1 expr2)
      `(ash ,(%INT expr1) ,(%INT expr2))))

(define-maybe (%SHIFT-RIGHT expr1 expr2)
  (if (and (integer? expr1) (integer? expr2))
      (ash expr1 (- expr2))
      `(ash ,(%INT expr1) (- ,(%INT expr2)))))

(define-maybe (%LT expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (if (< expr1 expr2) 1 0)
      `(if (< ,(%REAL expr1) ,(%REAL expr2)) 1 0)))

(define-maybe (%GT expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (if (> expr1 expr2) 1 0)
      `(if (> ,(%REAL expr1) ,(%REAL expr2)) 1 0)))

(define-maybe (%LE expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (if (<= expr1 expr2) 1 0)
      `(if (<= ,(%REAL expr1) ,(%REAL expr2)) 1 0)))

(define-maybe (%GE expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (if (>= expr1 expr2) 1 0)
      `(if (>= ,(%REAL expr1) ,(%REAL expr2)) 1 0)))

(define-maybe (%EQ expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (if (eq? expr1 expr2) 1 0)
      `(if (equal? ,(%REAL expr1) ,(%REAL expr2)) 1 0)))

(define-maybe (%NE expr1 expr2)
  (if (and (real? expr1) (real? expr2))
      (if (eq? expr1 expr2) 0 1)
      `(if (equal? ,(%REAL expr1) ,(%REAL expr2)) 0 1)))

(define-maybe (%BIT-AND expr1 expr2)
  (if (and (integer? expr1) (integer? expr2))
      (logand expr1 expr2)
      `(logand ,(%INT expr1) ,(%INT expr2))))

(define-maybe (%BIT-XOR expr1 expr2)
  (if (and (integer? expr1) (integer? expr2))
      (logxor expr1 expr2)
      `(logxor ,(%INT expr1) ,(%INT expr2))))

(define-maybe (%BIT-OR expr1 expr2)
  (if (and (integer? expr1) (integer? expr2))
      (logior expr1 expr2)
      `(logior ,(%INT expr1) ,(%INT expr2))))

(define-maybe (%LOG-AND expr1 expr2)
  (let ((v (gensym "%")))
    `(let ((,v ,(%REAL expr1)))
       (if (eq? ,v 0) ,v ,expr2))))

(define-maybe (%LOG-OR expr1 expr2)
  (let ((v (gensym "%")))
    `(let ((,v ,(%REAL expr1)))
       (if (eq? ,v 0) ,expr2 ,v))))

(define-maybe (%IF test then else)
  `(if (eq? ,(%REAL test) 0) ,else ,then))

(define-maybe (%ASSIGN lhs rhs)
  (side-effect? #t)
  (match lhs
    ((? symbol? x)
     `(let ((%v ,rhs))
        (set! (ref ,x) %v)
        %v))
    (((or 'ref 'raw-ref) x ...)
     `(let ((%v ,rhs))
        (set! (ref ,@x) %v)
        %v))
    (('deref x)
     `(let ((%v ,rhs))
        (set! (deref ,x) %v)
        %v))
    (else
     #f)))

(define-maybe (%CONCAT-EXPR expr1 expr2)
  `(begin ,expr1 ,expr2))

(define-maybe (%FOR init test update body)
  (use-iterator? #t)
  `(call/cc (lambda (%break)
              ,init
              (let %continue ()
                ,(%IF test
                      `(begin
                         ,body
                         ,update
                         (%continue))
                      '(%break 0))))))

(define-maybe (%WHILE test statement)
  (use-iterator? #t)
  `(call/cc (lambda (%break)
              (let %continue ()
                ,(%IF test `(begin ,statement (%continue)) '(%break 0))))))

(define-maybe (%DO-WHILE test statement)
  (use-iterator? #t)
  `(call/cc (lambda (%break)
              (letrec ((%body (lambda () ,statement (%continue)))
                       (%continue (lambda ()
                                    ,(%IF test
                                          '(%body)
                                          '(%break 0)))))
                (%body)))))

(define (%CONTINUE)
  (use-jump? #t)
  '(%continue))

(define (%BREAK)
  (use-jump? #t)
  '(%break 0))

(define-maybe (%RETURN v)
  (use-return? #t)
  `(%return ,v))

(define-maybe (%QUOTE v)
  `(',v))

(define-maybe (%APPEND v1 v2)
  (append v1 v2))

(define-maybe (%KEYWORD-ARG selector expr)
  `(',selector ,expr))

(define-maybe (%KEYWORD-ARG-WITHOUT-SELECTOR expr)
  (list : expr))

(define-maybe (%OBJC-MESSAGE-EXPR receiver args)
  (side-effect? #t)
  `(,receiver ,@args))

(define-maybe (%SELECTOR selector)
  (side-effect? #t)
  `(@selector ,selector))

;;

(define (lexer-init port)
  (rest-chars '())
  (last-token #f)
  (input-port port))

(define (clear-arg-pool)
  (arg-pool (make-hash-table)))

(define (typedefed? symbol)
  (hash-table-exists? (typedefed-identifiers) symbol))

(define (install-type symbol)
  (hash-table-put! (typedefed-identifiers) symbol #t))

;;

(load "c-wrapper/c-lex")
(load "c-wrapper/c-grammar.yy")

(define (enqueue-code obj write?)
  (let ((new-tail (list (cons obj write?)))
        (queue (slot-ref context 'code-queue)))
    (set-cdr! (cdr queue) new-tail)
    (set-cdr! queue new-tail)
    (car new-tail)))

(define (code-queue->list)
  (cdar (slot-ref context 'code-queue)))

(define (register-identifier name value)
  (let ((pair (or (hash-table-get (value-pool) name #f)
                  (cons #f #f))))
    (set-car! pair value)
    (hash-table-put! (value-pool) name pair))
  value)

(define (c-lookup-value v)
  (match v
    (('identity (? symbol? name))
     (hash-table-get (value-pool) name #f))
    (else
     v)))

(define (register-macro name)
  (let ((pair (or (hash-table-get (value-pool) name #f)
                (cons #f #f))))
    (set-cdr! pair #t)
    (hash-table-put! (value-pool) name pair)))

(define (registered-identifier? name)
  (or (hash-table-exists? (value-pool) name)
      (hash-table-exists? (arg-pool) name)))

(define (enqueue-import-pool name data)
  (hash-table-update! (import-pool) name
                      (lambda (elem)
                        (append elem (list data)))
                      '()))

(define (dequeue-import-pool name)
  (let ((result #f))
    (hash-table-update! (import-pool) name
                        (lambda (elem)
                          (if (null? elem)
                              elem
                              (begin
                                (set! result (car elem))
                                (cdr elem))))
                        '())
    result))

(define (imp-sym v)
  (do ((data (dequeue-import-pool v) (dequeue-import-pool v)))
      ((not data))
    (traverse (cdr data))
    (set-cdr! (car data) #t)
    (when (and (export?) v)
      (enqueue-code `(export ,v) #t))
    (hash-table-put! (imported-pool) v #t)))

(define (traverse v)
  (cond
   ((null? v)
    #f)
   ((list? v)
    (if (memq (car v) '(c-struct c-union c-enum))
        (imp-sym v)
        (map traverse v)))
   (else
    (imp-sym v))))

(define (emit-definition name define-list . rest)
  (let-optionals* rest ((dependent-symbols define-list))
    (if (import-cond)
        (let ((pair (enqueue-code define-list #f)))
          (enqueue-import-pool name (cons pair dependent-symbols))
          (when (or (hash-table-exists? (imported-pool) name)
                    (need-import? name))
            (imp-sym name)))
        (begin
          (enqueue-code define-list #t)
          (when (and (export?) name)
            (enqueue-code `(export ,name) #t))))))

(define-method need-import? (name)
  (if name
      (need-import? name (import-cond))
      #t))

(define-method need-import? (name (re <regexp>))
  (re (x->string name)))

(define-method need-import? (name (col <collection>))
  (find (cut need-import? name <>) col))

(define-method need-import? (name (sym <symbol>))
  (eq? name sym))

(define-method need-import? (name (str <string>))
  (need-import? name (string->symbol str)))

(define-method need-import? (name (proc <procedure>))
  (apply proc (list (filename) name)))

(define (make-var type-spec-list qualifiers)
  (receive (typedef-list ts-list) (partition (cut eq? 'TYPEDEF <>) type-spec-list)
    (receive (type identifier) (qualified-type (typespec->c-type ts-list)
                                               qualifiers)
      (vector identifier
              type
              (or (and-let* ((kv (assq 'init-value qualifiers)))
                                 (cadr kv))
                               #f)
              (not (null? typedef-list))))))
                     
(define (name-of v)
  (vector-ref v 0))

(define type-of (getter-with-setter (lambda (v)
                                      (vector-ref v 1))

                                    (lambda (v type)
                                      (vector-set! v 1 type))))

(define (value-of v)
  (vector-ref v 2))

(define (typedef? v)
  (vector-ref v 3))

(define (make-var-list type-spec-list qualifiers-list)
  (map (cut make-var type-spec-list <>) qualifiers-list))

(define (qualified-type c-type qualifiers)
  (let loop ((ret-type c-type)
             (identifier #f)
             (rest qualifiers))
    (match rest
      (()
       (values ret-type identifier))
      ((('ptr) ('c-func args ...) x ...)
       (values `(c-func-ptr ,(receive (ret-type _) (qualified-type c-type x)
                                    ret-type)
                                 ,`(list ,@(map (lambda (v)
                                                  `(list ',(name-of v)
                                                         ,(type-of v)))
                                                args)))
               identifier))
      ((('ptr) ('c-func-vaargs args ...) x ...)
       (values `(c-func-vaargs-ptr ,(receive (ret-type _)
                                             (qualified-type c-type x)
                                           ret-type)
                                        ,`(list ,@(map (lambda (v)
                                                         `(list ',(name-of v)
                                                                ,(type-of v)))
                                                       args)))
               identifier))
      ((('c-func args ...) x ...)
       (values `(c-func ,(receive (ret-type _) (qualified-type c-type x)
                           ret-type)
                        ,`(list ,@(map (lambda (v)
                                         `(list ',(name-of v)
                                                ,(type-of v)))
                                       args)))
               identifier))
      ((('c-func-vaargs args ...) x ...)
       (values `(c-func-vaargs ,(receive (ret-type _) (qualified-type c-type x)
                                ret-type)
                             ,`(list ,@(map (lambda (v)
                                              `(list ',(name-of v)
                                                     ,(type-of v)))
                                            args)))
               identifier))
      ((('ptr) x ...)
       (values `(ptr ,(receive (ret-type _) (qualified-type c-type x)
                        ret-type))
               identifier))
      ((('array n) x ...)
       (values `(c-array ,(receive (ret-type _) (qualified-type c-type x)
                            ret-type)
                         ,n)
               identifier))
      ((('bit-field n) x ...)
       (loop `(c-bit-field ,ret-type ,n) identifier x))
      ((('init-value n) x ...)
       (loop ret-type identifier x))
      ((('identifier name) x ...)
       (loop ret-type name x)))))

(define (typespec->c-type type)
  (match type
    (('char) '<c-char>)
    (('SIGNED 'char) '<c-char>)
    (('UNSIGNED 'char) '<c-uchar>)
    (('short) '<c-short>)
    (('short 'int) '<c-short>)
    (('short 'UNSIGNED 'int) '<c-ushort>)
    (('SIGNED 'short) '<c-short>)
    (('SIGNED 'short 'int) '<c-short>)
    (('UNSIGNED 'short) '<c-ushort>)
    (('UNSIGNED 'short 'int) '<c-ushort>)
    (('int) '<c-int>)
    (('SIGNED 'int) '<c-int>)
    (('SIGNED) '<c-int>)
    (('UNSIGNED 'int) '<c-uint>)
    (('UNSIGNED) '<c-uint>)
    (('long) '<c-long>)
    (('long 'int) '<c-long>)
    (('SIGNED 'long) '<c-long>)
    (('SIGNED 'long 'int) '<c-long>)
    (('long 'SIGNED 'int) '<c-long>)
    (('UNSIGNED 'long) '<c-ulong>)
    (('UNSIGNED 'long 'int) '<c-ulong>)
    (('long 'UNSIGNED 'int) '<c-ulong>)
    (('long 'long) '<c-longlong>)
    (('long 'long 'int) '<c-longlong>)
    (('SIGNED 'long 'long) '<c-longlong>)
    (('SIGNED 'long 'long 'int) '<c-longlong>)
    (('long 'long 'SIGNED 'int) '<c-longlong>)
    (('UNSIGNED 'long 'long) '<c-ulonglong>)
    (('UNSIGNED 'long 'long 'int) '<c-ulonglong>)
    (('long 'long 'UNSIGNED 'int) '<c-ulonglong>)
    (('float) '<c-float>)
    (('double) '<c-double>)
    (('long 'double) '<c-longdouble>)
    (('void) '<c-void>)
    (('_Bool) '<c-int>)
    (('__builtin_va_list) '(ptr <c-void>))
    ((('STRUCT tagname (elem-alist ...)))
     (emit-init-struct tagname elem-alist)
     `(c-struct ',tagname))
    ((('STRUCT tagname #f))
     (emit-alloc-struct tagname)
     `(c-struct ',tagname))
    ((('UNION tagname (elem-alist ...)))
     (emit-init-union tagname elem-alist)
     `(c-union ',tagname))
    ((('UNION tagname #f))
     (emit-alloc-union tagname)
     `(c-union ',tagname))
    ((('ENUM tagname (enum-alist ...)))
     (emit-define-enum tagname enum-alist)
     `(c-enum ',tagname))
    (((? symbol? x))
     (c-type->class-symbol x))))

(define (make-member-alist elem-list)
  (map (lambda (elem)
         `(cons ',(name-of elem) ,(type-of elem)))
       elem-list))

(define (emit-alloc-struct tagname)
  (unless (hash-table-exists? (struct-pool) tagname)
    (hash-table-put! (struct-pool) tagname #t)
    (emit-definition `(c-struct ',tagname) `(define-c-struct ,tagname))))

(define (emit-init-struct tagname member-list)
  (emit-alloc-struct tagname)
  (emit-definition `(c-struct ',tagname)
                 `(init-c-struct! (c-struct ',tagname)
                                  (list ,@(make-member-alist member-list)))))

(define (emit-alloc-union tagname)
  (unless (hash-table-exists? (union-pool) tagname)
    (hash-table-put! (union-pool) tagname #t)
    (emit-definition `(c-union ',tagname) `(define-c-union ,tagname))))

(define (emit-init-union tagname member-list)
  (emit-alloc-union tagname)
  (emit-definition `(c-union ',tagname)
                 `(init-c-union! (c-union ',tagname)
                                 (list ,@(make-member-alist member-list)))))

(define (emit-define-enum tagname enum-alist)
  (fold (lambda (p prev)
          (let ((v (or (cdr p)
                       (if (number? prev)
                           (+ prev 1)
                           `(+ (scm-cast ,prev) 1)))))
            (register-identifier (car p) v)
            (emit-definition (car p) `((with-module c-wrapper define-enum) ,(car p) ,v))
            v))
        -1
        enum-alist)
  (emit-definition `(c-enum ',tagname)
                 `(init-c-enum! (c-enum ',tagname)
                                (list ,@(map car enum-alist)))))

(define (emit-typedef var-list)
  (for-each (lambda (v)
              (let ((obj (match (type-of v)
                           (('c-func-ptr ret-type ('list ('list _ arg-type) 
                                                         ...))
                            `(c-func-ptr ,ret-type (list ,@arg-type)))
                           (('c-func-vaargs-ptr ret-type 
                                                     ('list ('list _ arg-type) 
                                                            ...))
                            `(c-func-ptr ,ret-type (list ,@arg-type)))
                           (('c-func ret-type ('list ('list _ arg-type) 
                                                   ...))
                            `(c-func ,ret-type (list ,@arg-type)))
                           (('c-func-vaargs ret-type ('list ('list _ arg-type) 
                                                          ...))
                            `(c-func ,ret-type (list ,@arg-type)))
                           (else
                            (type-of v)))))
                (let ((sym (c-type->class-symbol (name-of v))))
                  (emit-definition sym `(define ,sym ,obj)))
                (install-type (name-of v))))
            var-list))

(define (emit-define-inline type declarator function-body)
  (define (make-bindings name-list type-list)
    (let loop ((bindings '())
               (i 0)
               (name-rest name-list)
               (type-rest type-list))
      (if (null? name-rest)
          (reverse bindings)
          (loop (cons `(,(car name-rest) (cast ,(car type-rest)
                                               (list-ref %args ,i)))
                      bindings)
                (+ i 1)
                (cdr name-rest)
                (cdr type-rest)))))
  (receive (c-type identifier)
      (qualified-type (typespec->c-type type) declarator)
    (match c-type
      (('c-func-ptr ret-type ('list ('list ('quote names) types) ...))
       (warning "'~a' is ignored. It appears in a function definition, but it is a pointer of a function in reality." identifier))
      (('c-func ret-type ('list ('list ('quote names) types) ...))
       (register-identifier identifier
                            `(lambda args
                               (apply ,identifier args)))
       (emit-definition identifier
                        `((with-module c-wrapper define-inline-cfunc)
                          ,identifier ,ret-type ,names ,types
                          ,(or function-body
                               `(errorf "~a is not supported. Try cwcompile if you want to use." ,identifier)))))
      (((? (lambda (v) (memq v '(c-func-vaargs-ptr func-vaargs))) v)
        ret-type ('list ('list ('quote names) types) ...))
       (warning "The inline function '~a' is ignored, because it gets variable arguments" identifier))
      (else
       (warning "'~a' is ignored, it is not a function." identifier)))))

(define (emit-define-extern type init-decl)
  (receive (c-type identifier)
      (qualified-type (typespec->c-type type) init-decl)
    (when (or (ignore-dlsym-check?)
              (c-lookup-symbol identifier))
      (and-let* ((obj (match c-type
                        (('c-func ret-type ('list ('list _ arg-type) ...))
                         `(make-c-func ',identifier
                                       ,ret-type
                                       (list ,@arg-type)))
                        (('c-func-vaargs ret-type ('list ('list _ arg-type) ...))
                         `(make-c-func-vaargs ',identifier
                                              ,ret-type
                                              (list ,@arg-type)))
                        (else
                         `(make-c-var ',identifier ,c-type)))))
        (register-identifier identifier obj)
        (emit-definition identifier `(define ,identifier ,obj))))))

(define (emit-define-objc-class classname-list)
  (for-each (lambda (classname)
              (when classname
                (install-type classname)
                (let ((sym (c-type->class-symbol classname)))
                  (emit-definition sym `(define ,sym (c-struct 'objc_object))))
                (emit-definition classname
                               `(define ,classname
                                  (objc-lookup-class ',classname)))))
            classname-list))

(define (emit-objc-method keywords type-list)
  (let ((name (apply string-append keywords)))
    (emit-definition #f
                   `(objc-register-method ,name
                                          (list ,@(map type-of type-list))))))

(define (emit-define-cmacro name body)
  (emit-definition name `(define ,name ,body)))

(define (emit-define-cfunclike-macro name args body)
  (emit-definition name
                   `((with-module c-wrapper define-cfunclike-macro)
                     ,name ,args ,body)))

(define (macro-parse include-dirs headers options)
  (call-with-process-io (cpp-command include-dirs headers options #f)
    (lambda (in out)
      (profile
       (let ((identifier-queue (make-queue)))
         (define (send-macro)
           (for-each (lambda (macro-def)
                       (display (car macro-def) out)
                       (newline out)
                       (enqueue! identifier-queue (cdr macro-def)))
                     (queue->list (macro-queue)))
           (close-output-port out)
           ;; skip the first line '# 1 "<stdin>"'
           (read-line in #t)
           (skip (read-line in #t)))
         (define (skip line)
           (cond
            ((eof-object? line)
             #f)
            ((eq? (string-size line) 0)
             (skip (read-line in #t)))
            ((not (eq? (string-byte-ref line 0) 35)) ;; '#' != 35 (ASCII)
             (skip (read-line in #t)))
            ((string-incomplete? line)
             (skip (read-line in #t)))
            ((string=? line "# 1 \"<stdin>\"")
             (parse-macro (read-line in #t)))
            (else
             (skip (read-line in #t)))))
         (define (parse-macro line)
           (cond
            ((eof-object? line)
             #f)
            ((queue-empty? identifier-queue)
             (error "[bug] lost macro body"))
            (else
             (let ((pos&name&args (dequeue! identifier-queue)))
               (filename (caar pos&name&args))
               (lineno (cdar pos&name&args))
               (parse-macro-body (string->symbol (cadr pos&name&args))
                                 (cddr pos&name&args)
                                 line))
             (parse-macro (read-line in #t)))))
         (send-macro))))))

(define (parse-macro-body name args body-str)
  (side-effect? #f)
  (use-return? #f)
  (use-iterator? #f)
  (use-jump? #f)
  (and-let* ((body (call/cc
                    (lambda (break)
                      (clear-arg-pool)
                      (when args
                        (for-each (lambda (arg)
                                    (hash-table-put! (arg-pool) arg #t))
                                  args))
                      (if body-str
                          (with-input-from-string (string-append body-str ";\n")
                            (lambda ()
                              (lexer-init (current-input-port))
                              (let ((first? #t))
                                (c-grammar (lambda ()
                                             (cond
                                              (first?
                                               (set! first? #f)
                                               'START_MACRO)
                                              (else
                                               (c-scan))))
                                           (lambda (msg . _)
                                             (break #f))))))
                          #f)))))
    (cond
     ((or (and (not (use-iterator?)) (use-jump?))
          (use-return?))
      #f)
     ((and (not args) (not (side-effect?)))
      (register-identifier name body)
      (emit-define-cmacro name body))
     (args
      (register-macro name)
      (emit-define-cfunclike-macro name args body))
     (else
      #f))))

;; FIXME!!
;; In Linux, /usr/include/sys/types.h uses __attribute__ for these type definitions, 
;; but c-wrapper ignores __attribute__.
(define (linux?)
  (not (not (#/-linux-/ (gauche-config "--arch")))))

(define (cpp-command include-dirs headers options . rest)
  (let-optionals* rest ((show-define? #t))
    (string-join (append (list GCC "'-D__attribute__(x)=' -E")
                         (if (linux?)
                             '("-D__int8_t_defined")
                             ())
                         (if show-define? 
                             '("-dD")
                             '())
                         options
                         (map (cut format "-I~a" <>) include-dirs)
                         (map (cut format "-include ~a" <>) headers)
                         (list "-"))
                 " ")))

(define (install-predefined-types)
  (when (linux?)
    (filename "/usr/include/sys/types.h")
    (for-each (lambda (pair)
                (emit-typedef (make-var-list (car pair)
                                             (%LIST (decl-identifier (cdr pair))))))
              '(((SIGNED char) . int8_t)
                ((UNSIGNED char) . u_int8_t)
                ((SIGNED short) . int16_t)
                ((UNSIGNED short) . u_int16_t)
                ((SIGNED int) . int32_t)
                ((UNSIGNED int) . u_int32_t)
                ((SIGNED long long) . int64_t)
                ((UNSIGNED long long) . u_int64_t)))))

(define (c-parse include-dirs headers options
                 import-cond-arg export?-arg ignore-dlsym-check?-arg)
  (set! context (make <parse-context>))
  (import-cond import-cond-arg)
  (export? export?-arg)
  (ignore-dlsym-check? ignore-dlsym-check?-arg)
  (call-with-output-string
    (lambda (wrapper-out)
      (call-with-process-io (cpp-command include-dirs headers options)
        (lambda (in out)
          (profile
           (close-output-port out)
           (with-input-from-port in
             (lambda ()
               (lexer-init in)
               (install-predefined-types)
               (c-grammar c-scan (lambda (msg . args)
                                   (errorf "~a:~a: ~a ~a (last token: ~s)"
                                           (filename)
                                           (lineno)
                                           msg
                                           (if (null? args) "" args)
                                           (last-token)))))))))
      (macro-parse include-dirs headers options)
      (when (export?)
        (enqueue-code '(export-all) #t))
      (for-each (lambda (elem)
                  (when (cdr elem)
                    (write (car elem) wrapper-out)))
                (code-queue->list))
      (set! context #f))))

(provide "c-wrapper/c-parser")

