/**
 *  c-ffi.c
 *  
 *   Copyright (c) 2006 KOGURO, Naoki (naoki@koguro.net)
 *  
 *   Permission is hereby granted, free of charge, to any person 
 *   obtaining a copy of this software and associated 
 *   documentation files (the "Software"), to deal in the 
 *   Software without restriction, including without limitation 
 *   the rights to use, copy, modify, merge, publish, distribute, 
 *   sublicense, and/or sell copies of the Software, and to 
 *   permit persons to whom the Software is furnished to do so, 
 *   subject to the following conditions:
 *  
 *   The above copyright notice and this permission notice shall 
 *   be included in all copies or substantial portions of the 
 *   Software.
 *  
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY 
 *   KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE 
 *   WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
 *   PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS 
 *   OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR 
 *   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 *   OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
 *   SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *  
 *   $Id: c-ffi.c 365 2006-12-22 23:24:14Z naoki $
 */

#include "c-ffi.h"

ScmObj Scm_GetUnsignedFFIType(int size)
{
    ffi_type *type;
    switch (size) {
    case 1:
        type = &ffi_type_uint8;
        break;
    case 2:
        type = &ffi_type_uint16;
        break;
    case 4:
        type = &ffi_type_uint32;
        break;
    case 8:
        type = &ffi_type_uint64;
        break;
    default:
        Scm_Error("unsupported type: ~S", size);
        break;
    }

    SCM_RETURN(SCM_MAKE_FFI_TYPE(type));
}

ScmObj Scm_GetSignedFFIType(int size)
{
    ffi_type *type;
    switch (size) {
    case 1:
        type = &ffi_type_sint8;
        break;
    case 2:
        type = &ffi_type_sint16;
        break;
    case 4:
        type = &ffi_type_sint32;
        break;
    case 8:
        type = &ffi_type_sint64;
        break;
    default:
        Scm_Error("unsupported type: ~S", size);
        break;
    }

    SCM_RETURN(SCM_MAKE_FFI_TYPE(type));
}

static ScmObj Scm_PointerBufferOf(ScmObj ptrObj)
{
    static ScmObj buffer_of_proc = SCM_FALSE;
    ScmObj uvec;

    if (SCM_FALSEP(buffer_of_proc)) {
        buffer_of_proc = SCM_SYMBOL_VALUE(MODULE_NAME, "buffer-of");
    }

    SCM_RETURN(Scm_ApplyRec(buffer_of_proc, SCM_LIST1(ptrObj)));
}

static void Scm_PointerSet(ScmObj ptrObj, void *ptr)
{
    ScmObj uvec = Scm_PointerBufferOf(ptrObj);
    memcpy(SCM_UVECTOR_ELEMENTS(uvec), &ptr, sizeof(void*));
}

void *Scm_PointerGet(ScmObj ptrObj)
{
    ScmObj uvec = Scm_PointerBufferOf(ptrObj);
    void *ptr;
    memcpy(&ptr, SCM_UVECTOR_ELEMENTS(uvec), sizeof(void*));
    return ptr;
}
        
ScmObj Scm_PtrClass(ScmObj cTypeClass)
{
    static ScmObj ptr_proc = SCM_FALSE;
    if (SCM_FALSEP(ptr_proc)) {
        ptr_proc = SCM_SYMBOL_VALUE(MODULE_NAME, "ptr");
    }
    SCM_RETURN(Scm_ApplyRec(ptr_proc, SCM_LIST1(cTypeClass)));
}

ScmObj Scm_BufferOf(ScmObj ptrObj)
{
    static ScmObj bufferof_proc = SCM_FALSE;
    if (SCM_FALSEP(bufferof_proc)) {
        bufferof_proc = SCM_SYMBOL_VALUE(MODULE_NAME, "buffer-of");
    }
    SCM_RETURN(Scm_ApplyRec(bufferof_proc, SCM_LIST1(ptrObj)));
}

ScmObj Scm_GetVoidPtrClass()
{
    static ScmObj voidptr_class = SCM_FALSE;
    if (SCM_FALSEP(voidptr_class)) {
        voidptr_class = PTR_CLASS(SCM_SYMBOL_VALUE(MODULE_NAME, "<c-void>"));
    }
    SCM_RETURN(voidptr_class);
}
    
ScmObj Scm_MakePointer(ScmObj klass, void *ptr)
{
    static ScmObj make_proc = SCM_FALSE;
    ScmObj ptrObj;
    ScmObj buf;

    if (SCM_FALSEP(make_proc)) {
        make_proc = SCM_SYMBOL_VALUE(MODULE_NAME, "make");
    }

    /*
           A (<uvector>)          (ptr A) (<uvector>)
      +----------+  +------+    +----------+  +---------------+
      |elements --->| data |    |elements --->|pointer of data|
      +----------+  +--^---+    +----------+  +-------|-------+
                       +------------------------------+  

      In this case, (ptr A) can be a dangling pointer when A is GCed and
      elements of (ptr A) are allocated with SCM_NEW_ATOMIC2.
      So we need to use SCM_NEW2 to allocate elements of (ptr A).
     */
    buf = Scm_MakeU8VectorFromArrayShared(sizeof(void*),
                                          SCM_NEW2(void*, sizeof(void*)));
    ptrObj = Scm_ApplyRec(make_proc, SCM_LIST3(klass,
                                            SCM_MAKE_KEYWORD("buffer"),
                                            buf));
    Scm_PointerSet(ptrObj, ptr);

    SCM_RETURN(ptrObj);
}

int PtrP(ScmObj obj)
{
    ScmClass *klass = NULL;

    if (!klass) {
        klass = SCM_CLASS(SCM_SYMBOL_VALUE(MODULE_NAME, "<c-ptr>"));
    }

    return SCM_ISA(obj, klass);
}

int BasicPtrP(ScmObj obj)
{
    ScmClass *klass = NULL;

    if (!klass) {
        klass = SCM_CLASS(SCM_SYMBOL_VALUE(MODULE_NAME, "<c-basic-ptr>"));
    }

    return SCM_ISA(obj, klass);
}

ScmObj Scm_Deref(ScmObj ptrObj)
{
    static ScmObj orig_c_type_of_proc = SCM_FALSE;
    static ScmObj make_proc = SCM_FALSE;
    static ScmObj csizeof_proc = SCM_FALSE;
    ScmObj klass;
    ScmObj buf;
    ScmObj size;
    ScmObj obj;
    unsigned char *p;

    if (SCM_FALSEP(orig_c_type_of_proc)) {
        orig_c_type_of_proc = SCM_SYMBOL_VALUE(MODULE_NAME, "orig-c-type-of");
        make_proc = SCM_SYMBOL_VALUE(MODULE_NAME, "make");
        csizeof_proc = SCM_SYMBOL_VALUE(MODULE_NAME, "c-sizeof");
    }

    klass = Scm_ApplyRec(orig_c_type_of_proc,
                      SCM_LIST1(SCM_OBJ(SCM_CLASS_OF(ptrObj))));
    size = Scm_ApplyRec(csizeof_proc, SCM_LIST1(klass));
    p = (unsigned char*)POINTER_DATA(ptrObj);
    buf = Scm_MakeU8VectorFromArrayShared(Scm_GetInteger(size), p);
    if (p == NULL) {
        SCM_UVECTOR_ELEMENTS(buf) = p;
    }
    obj = Scm_ApplyRec(make_proc, SCM_LIST3(klass,
                                        SCM_MAKE_KEYWORD("buffer"),
                                        buf));
    SCM_RETURN(obj);
}

ScmObj Scm_MakeFFIType(ffi_type *data)
{
    ScmFFIType *z = SCM_NEW(ScmFFIType);
    SCM_SET_CLASS(z, SCM_CLASS_FFI_TYPE);
    z->data = data;

    SCM_RETURN(SCM_OBJ(z));
}

ScmObj Scm_MakeFFICif(ffi_cif *data)
{
    ScmFFICif *z = SCM_NEW(ScmFFICif);
    SCM_SET_CLASS(z, SCM_CLASS_FFI_CIF);
    z->data = data;

    SCM_RETURN(SCM_OBJ(z));
}

ScmObj Scm_MakeFFIArrayType(ffi_type *elem_type, long size)
{
    ffi_type *array_type = SCM_NEW(ffi_type);
    array_type->size = elem_type->size * size;
    array_type->alignment = elem_type->alignment;
    array_type->type = elem_type->type;
    array_type->elements = elem_type->elements;
    
    SCM_RETURN(SCM_MAKE_FFI_TYPE(array_type));
}

ScmObj Scm_MakeFFIStructType(ScmObj elem_list)
{
    ffi_type *struct_type = SCM_NEW(ffi_type);
    ffi_cif dummy_cif;
    ScmObj p;
    int i = 0;

    if (Scm_Length(elem_list) == 0) {
        Scm_Error("can't define zero-member struct");
    }

    struct_type->size = 0;
    struct_type->alignment = 0;
    struct_type->type = FFI_TYPE_STRUCT;
    struct_type->elements = SCM_NEW_ARRAY(ffi_type*, Scm_Length(elem_list) + 1);
    SCM_FOR_EACH(p, elem_list) {
        if (!SCM_FFI_TYPEP(SCM_CAR(p))) {
            Scm_Error("<ffi-type> required, but got %S", SCM_CAR(p));
        }
        struct_type->elements[i] = SCM_FFI_TYPE_DATA(SCM_CAR(p));
        ++i;
    }
    struct_type->elements[i] = NULL;

    /* initialize aggregate */
    ffi_prep_cif(&dummy_cif, FFI_DEFAULT_ABI, 0, struct_type, NULL);

    SCM_RETURN(SCM_MAKE_FFI_TYPE(struct_type));
}
    
ScmObj Scm_FFIPrepCIF(ffi_type *rtype, ScmObj arglist)
{
    ffi_cif *cif = SCM_NEW(ffi_cif);
    int nargs = Scm_Length(arglist);
    ffi_type **atypes = SCM_NEW_ARRAY(ffi_type*, nargs);
    int i = 0;
    ffi_status status;
    ScmObj p;
    
    SCM_FOR_EACH(p, arglist) {
        atypes[i] = SCM_FFI_TYPE_DATA(SCM_CAR(p));
        ++i;
    }

    status = ffi_prep_cif(cif, FFI_DEFAULT_ABI, nargs, rtype, atypes);
    
    SCM_RETURN(Scm_Values2(SCM_MAKE_INT(status), SCM_MAKE_FFI_CIF(cif)));
}

ScmObj Scm_FFICall(ffi_cif *cif, ScmObj fnptr, ScmObj rvalueptr, ScmObj arglist)
{
    int nargs = Scm_Length(arglist);
    void **avalues = SCM_NEW_ARRAY(void*, nargs);
    void *fn;
    void *rvalue;
    int i = 0;
    ScmObj p;

    if (BASIC_POINTERP(fnptr)) {
        fn = POINTER_DATA(fnptr);
    } else {
        Scm_Error("<c-basic-ptr> required, but got %S", fnptr);
    }
    if (POINTERP(rvalueptr)) {
        rvalue = POINTER_DATA(rvalueptr);
    } else {
        Scm_Error("<c-ptr> required, but got %S", rvalueptr);
    }
    SCM_FOR_EACH(p, arglist) {
        if (!POINTERP(SCM_CAR(p))) {
            Scm_Error("<c-ptr> required, but got %S", SCM_CAR(p));
        }
        avalues[i] = POINTER_DATA(SCM_CAR(p));
        ++i;
    }

    TRY {
        if (cif->rtype->size < sizeof(ffi_arg)) {
            ffi_arg result;
            ffi_call(cif, FFI_FN(fn), &result, avalues);
#ifdef WORDS_BIGENDIAN
            if (cif->rtype->type == FFI_TYPE_STRUCT) {
                memcpy(rvalue, &result, cif->rtype->size);
            } else {
                memcpy(rvalue, 
                       ((void*) &result) + sizeof(result) - cif->rtype->size, 
                       cif->rtype->size);
            }
#else
            memcpy(rvalue, &result, cif->rtype->size);
#endif
        } else {
            ffi_call(cif, FFI_FN(fn), rvalue, avalues);
        }
    } CATCH(e) {
        THROW_EXCEPTION(e);
    }

    SCM_RETURN(SCM_UNDEFINED);
}

static void closure_func(ffi_cif *cif, void *rvalue, void **avalues, void *data)
{
    ScmObj proc = SCM_OBJ(data);
    ScmObj arglist = SCM_NIL;
    void *proc_result_ptr;
    int i;

    for (i = 0; i < cif->nargs; ++i) {
        arglist = Scm_Cons(MAKE_VOID_POINTER(avalues[i]), arglist);
    }
    arglist = Scm_Reverse(arglist);

    proc_result_ptr = POINTER_DATA(Scm_ApplyRec(proc, arglist));
    if (cif->rtype->size < sizeof(ffi_arg)) {
        memset(rvalue, 0, sizeof(ffi_arg));
#ifdef WORDS_BIGENDIAN
        if (cif->rtype->type == FFI_TYPE_STRUCT) {
            memcpy(rvalue, proc_result_ptr, cif->rtype->size);
        } else {
            memcpy(rvalue + sizeof(ffi_arg) - cif->rtype->size,
                   proc_result_ptr, cif->rtype->size);
        }
#else
        memcpy(rvalue, proc_result_ptr, cif->rtype->size);
#endif
    } else {
        memcpy(rvalue, proc_result_ptr, cif->rtype->size);
    }
}

ScmObj Scm_FFIPrepClosure(ffi_cif *cif, ScmProcedure *proc)
{
    ffi_closure *closure = SCM_NEW(ffi_closure);
    ffi_status status = ffi_prep_closure(closure, cif, closure_func,
                                         (void*) proc);
    void **ptr = SCM_NEW(void*);
    *ptr = (void*) closure;

    SCM_RETURN(Scm_Values2(SCM_MAKE_INT(status), MAKE_VOID_POINTER(*ptr)));
}

/*
 * Module initialization function.
 */
extern void Scm_Init_c_ffilib(ScmModule*);
#ifdef __OBJC__
extern void Scm_Init_ObjCError(ScmModule*);
#endif

ScmObj Scm_Init_c_ffi(void)
{
    ScmModule *mod;

    /* Register this DSO to Gauche */
    SCM_INIT_EXTENSION(c_ffi);

    /* Create the module if it doesn't exist yet. */
    mod = SCM_MODULE(SCM_FIND_MODULE(MODULE_NAME, TRUE));

    /* Register stub-generated procedures */
    Scm_Init_c_ffilib(mod);

#ifdef __OBJC__
    Scm_Init_ObjCError(mod);
#endif
}
