# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject

from gazpacho import parameter
from gazpacho import propertyclass as pc
from gazpacho.loader import tags
from gazpacho.signaleditor import SignalEditor

TABLE_ROW_SPACING = 2
TABLE_COLUMN_SPACING = 6

(EDITOR_INTEGER,
 EDITOR_FLOAT,
 EDITOR_DOUBLE) = range(3)

(FLAGS_COLUMN_SETTING,
 FLAGS_COLUMN_SYMBOL) = range(2)

class EditorProperty:
    """ For every PropertyClass we have a EditorProperty that is basically
    an input (GtkWidget) for that PropertyClass. """

    def __init__(self, property_class, app):
        # The class this property corresponds to
        self.klass = property_class
        
        self._app = app

        # The widget that modifies this property, the widget can be a
        # GtkSpinButton, a GtkEntry, GtkMenu, etc. depending on the property
        # type.
        self.input = None

        # The loaded property
        self.property = None

        # We set this flag when we are loading a new Property into this
        # EditorProperty. This flag is used so that when we receive a "changed"
        # signal we know that nothing has really changed, we just loaded a new
        # glade widget
        self.loading = False

    def load(self, widget):
        # Should be overridden
        return

    def set_property(self, value):
        self._app.command_manager.set_property(self.property, value)

    def set_translatable_property(self, text, comment_text, is_translatable,
                                  has_context):
        self._app.command_manager.set_translatable_property(self.property,
                                                            text,
                                                            comment_text,
                                                            is_translatable,
                                                            has_context)
        
    def _start_load(self, widget):
        self.property = widget.get_glade_property(self.klass.id)
        self.property.loading = True

    def _finish_load(self):
        self.property.loading = False

    def _internal_load(self, value):
        """This is the virtual method that actually put the data in the editor
        widget.
        Most subclasses implements this (But not EditorPropertySpecial)
        """
    
class EditorPropertyNumeric(EditorProperty):

    def __init__(self, property_class, app):
        EditorProperty.__init__(self, property_class, app)

    def load(self, widget):
        self._start_load(widget)
        
        self._internal_load(self.property.value)

        self.input.set_data('user_data', self)
        self._finish_load()

    def _internal_load(self, value):
        if self.klass.id in pc.UNICHAR_PROPERTIES:
            if isinstance(self.input, gtk.Editable):
                self.input.delete_text(0, -1)
                self.input.insert_text(value)
        elif self.klass.optional:
            button, spin = self.input.get_children()
            button.set_active(self.property.enabled)
            spin.set_sensitive(self.property.enabled)
            button.set_data('user_data', self)
            v = float(value)
            spin.set_value(v)
        else:
            v = float(value)
            self.input.set_value(v)
                
    def _changed_unichar(self, entry):
        if self.property.loading:
            return
        
        text = entry.get_chars(0, -1)
        unich = text[0]

        self.set_property(unich)
        
    def _create_input_unichar(self):
        entry = gtk.Entry()
        # it's to prevent spirious beeps ...
        entry.set_max_length(2)
        entry.connect('changed', self._changed_unichar)
        self.input = entry

    def _changed_numeric(self, spin):
        if self.property.loading:
            return
        
        numeric_type = spin.get_data('NumericType')
        if numeric_type == EDITOR_INTEGER:
            value = spin.get_value_as_int()
        elif numeric_type == EDITOR_FLOAT:
            value = spin.get_value()
        elif numeric_type == EDITOR_DOUBLE:
            value = spin.get_value()
        else:
            value = 0

        self.set_property(value)

    def _changed_enabled(self, check_button):
        if self.property.loading:
            return

        l = self.input.get_children()
        spin = l[1]
        state = check_button.get_active()
        spin.set_sensitive(state)
        prop = check_button.get_data('user_data')
        prop.property.enabled = state

    def _create_input_numeric(self, numeric_type):
        adjustment = parameter.adjustment_new(self.klass)
        if numeric_type == EDITOR_INTEGER:
            digits = 0
        else:
            digits = 2
            
        spin = gtk.SpinButton(adjustment, 0.2, digits)
        spin.set_data("NumericType", numeric_type)
        spin.connect('value_changed', self._changed_numeric)

        # Some numeric types are optional, for example the default window size,
        # so they have a toggle button right next to the spin button
        if self.klass.optional:
            check = gtk.CheckButton()
            hbox = gtk.HBox()
            hbox.pack_start(check, False, False)
            hbox.pack_start(spin)
            check.connect('toggled', self._changed_enabled)
            self.input = hbox
        else:
            self.input = spin

class EditorPropertyInteger(EditorPropertyNumeric):
    def __init__(self, property_class, app):
        EditorPropertyNumeric.__init__(self, property_class, app)

        if self.klass.id in pc.UNICHAR_PROPERTIES:
            self._create_input_unichar()
        else:
            self._create_input_numeric(EDITOR_INTEGER)

class EditorPropertyFloat(EditorPropertyNumeric):
    def __init__(self, property_class, app):
        EditorPropertyNumeric.__init__(self, property_class, app)
        self._create_input_numeric(EDITOR_FLOAT)        

class EditorPropertyDouble(EditorPropertyNumeric):
    def __init__(self, property_class, app):
        EditorPropertyNumeric.__init__(self, property_class, app)
        self._create_input_numeric(EDITOR_DOUBLE)
                                
class EditorPropertyText(EditorProperty):

    def __init__(self, property_class, app):
        EditorProperty.__init__(self, property_class, app)

        if self.klass.is_translatable:
            i18n_button = gtk.Button('...')
            i18n_button.connect('clicked', self._show_i18n_dialog)

        lines = parameter.get_integer(self.klass.parameters,
                                       tags.VISIBLE_LINES, 1)
        editable_widget = None
        hbox = gtk.HBox()

        # If the current data contains more than two lines,
        # use a TextView to display the data
        if lines >= 2:
            
            view = gtk.TextView()
            view.set_editable(True)
            text_buffer = view.get_buffer()
            text_buffer.connect('changed', self._changed_text_view)

            editable_widget = view
        # otherwise just use a normal entry
        else:
            entry = gtk.Entry()
            entry.connect('changed', self._changed_text)
            editable_widget = entry

        hbox.pack_start(editable_widget)
        if self.klass.is_translatable:
            hbox.pack_start(i18n_button, False, False)

        self.input = hbox
        
    def load(self, widget):
        self._start_load(widget)

        self._internal_load(self.property.value)

        self.input.set_data('user_data', self)

        self._finish_load()

    def _internal_load(self, value):
        # Text inputs are an entry or text view inside an hbox
        text_widget = self.input.get_children()[0]
        
        if isinstance(text_widget, gtk.Editable):
            pos = text_widget.get_position()
            text_widget.delete_text(0, -1)
            if value:
                text_widget.insert_text(value)

            text_widget.set_position(pos)

        elif isinstance(text_widget, gtk.TextView):
            text_buffer = text_widget.get_buffer()
            text_buffer.set_text(value)
        else:
            print _('Invalid Text Widget type')

    def _changed_text_common(self, text):
        self.set_property(text)

    def _changed_text(self, entry):
        if self.property.loading:
            return

        text = entry.get_chars(0, -1)
        self._changed_text_common(text)

    def _changed_text_view(self, text_buffer):
        if self.property.loading:
            return
        
        start = text_buffer.get_iter_at_offset(0)
        end = text_buffer.get_iter_at_offset(text_buffer.get_char_count())
        text = text_buffer.get_text(start, end, False)
        self._changed_text_common(text)

    def _show_i18n_dialog(self, button):
        editor = button.get_toplevel()

        is_translatable = self.property.is_translatable
        has_context = self.property.has_i18n_context
        comment_text = self.property.i18n_comment

        dialog = gtk.Dialog(_('Edit Text Property'), editor,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                             gtk.STOCK_OK, gtk.RESPONSE_OK))
        
        dialog.set_has_separator(False)

        vbox = gtk.VBox(False, 6)
        vbox.set_border_width(8)
        vbox.show()

        dialog.vbox.pack_start(vbox, True, True, 0)

        # The text content
        label = gtk.Label()
        label.set_markup_with_mnemonic(_('_Text:'))
        label.show()
        label.set_alignment(0, 0.5)
        vbox.pack_start(label, False, False, 0)

        scrolled_window = gtk.ScrolledWindow()
        scrolled_window.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrolled_window.set_shadow_type (gtk.SHADOW_IN)
        scrolled_window.set_size_request (400, 200)
        scrolled_window.show()

        vbox.pack_start(scrolled_window)

        text_view = gtk.TextView()
        text_view.show()
        label.set_mnemonic_widget(text_view)

        scrolled_window.add(text_view)

        text_buffer = text_view.get_buffer()

        text = self.property.value
        if text is not None:
            text_buffer.set_text(text)
       
        # Translatable and context prefix
        hbox = gtk.HBox(False, 12)
        hbox.show()

        vbox.pack_start (hbox, False, False, 0)

        translatable_button = gtk.CheckButton (_("T_ranslatable"))
        translatable_button.set_active (is_translatable)
        translatable_button.show()
        hbox.pack_start(translatable_button, False, False, 0)

        context_button = gtk.CheckButton (_("_Has context prefix"))
        context_button.set_active (has_context)
        context_button.show()
        hbox.pack_start(context_button, False, False, 0)

        # Comments
        label = gtk.Label()
        label.set_markup_with_mnemonic(_("Co_mments for translators:"))
        label.show()
        label.set_alignment(0, 0.5)
        vbox.pack_start (label, False, False, 0)

        scrolled_window = gtk.ScrolledWindow()
        scrolled_window.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrolled_window.set_shadow_type (gtk.SHADOW_IN)
        scrolled_window.set_size_request (400, 200)
        scrolled_window.show()

        vbox.pack_start(scrolled_window)

        comment_view = gtk.TextView()
        comment_view.show()
        label.set_mnemonic_widget(comment_view)

        scrolled_window.add(comment_view)

        comment_buffer = comment_view.get_buffer()

        if comment_text is not None:
            comment_buffer.set_text(comment_text)

        res = dialog.run()
        if res == gtk.RESPONSE_OK:
            is_translatable = translatable_button.get_active()
            has_context = context_button.get_active()

            (start, end) = text_buffer.get_bounds()
            text = text_buffer.get_text(start, end, False)

            (start, end) = comment_buffer.get_bounds()
            comment_text = comment_buffer.get_text(start, end, False)

            self.set_translatable_property(text, comment_text, is_translatable,
                                           has_context)

            # Update the entry in the property editor
            self.load(self.property.widget)

        dialog.destroy()

class EditorPropertyEnum(EditorProperty):
    def __init__(self, property_class, app):
        EditorProperty.__init__(self, property_class, app)

        model = gtk.ListStore(str, object)
        for choice in self.klass.choices:
            model.append((choice['name'], choice))

        combo_box = gtk.ComboBox(model)
        renderer = gtk.CellRendererText()
        combo_box.pack_start(renderer)
        combo_box.set_attributes(renderer, text=0)
        combo_box.connect('changed', self._changed_enum)
        
        self.input = combo_box

    def load(self, widget):
        self._start_load(widget)
        self._internal_load(self.property.value)
        self._finish_load()

    def _internal_load(self, value):
        idx = 0
        for choice in self.property.klass.choices:
            if choice['value'] == value:
                break
            idx += 1
        self.input.set_active(idx)
        
    def _changed_enum(self, combo_box):
        if self.property.loading:
            return
        
        active_iter = combo_box.get_active_iter()
        model = combo_box.get_model()
        choice = model.get_value(active_iter, 1)
        self.set_property(choice['value'])
   
class EditorPropertyFlags(EditorProperty):
    def __init__(self, property_class, app):
        EditorProperty.__init__(self, property_class, app)

        hbox = gtk.HBox()
        
        entry = gtk.Entry()
        entry.set_editable(False)
        entry.show()
        hbox.pack_start(entry)
        
        button = gtk.Button('...')
        button.connect('clicked', self._show_flags_dialog)
        button.show()
        hbox.pack_start(button, False, False)

        self.input = hbox

    def load(self, widget):
        self._start_load(widget)
        self._internal_load(self.property.value)
        self._finish_load()

    def _internal_load(self, value):
        # the entry should be the first child
        entry = self.input.get_children()[0]
        text = ''
        flags_class = self.klass.spec.flags_class
        
        # Populate the model with the flags
        for mask, flag in flags_class.__flags_values__.items():
            # we avoid composite flags
            if len(flag.value_names) > 1:
                continue

            if (value & mask) == mask:
                text += "%s | " % flag.first_value_name

        entry.set_text(text.strip(' |'))
        
    def _flag_toggled(self, cell, path_string, model):
        the_iter = model.get_iter_from_string(path_string)
        setting = not model.get_value(the_iter, FLAGS_COLUMN_SETTING)
        model.set_value(the_iter, FLAGS_COLUMN_SETTING, setting)

    def _show_flags_dialog(self, button):
        editor = button.get_toplevel()
        dialog = gtk.Dialog(_('Set Flags'), editor,
                             gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                              gtk.STOCK_OK, gtk.RESPONSE_OK))
        dialog.set_default_size(300, 400)
        scrolled_window = gtk.ScrolledWindow()
        scrolled_window.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrolled_window.show()
        dialog.vbox.pack_start(scrolled_window)
        # create the treeview using a simple list model with 2 columns
        model = gtk.ListStore(bool, str)
        treeview = gtk.TreeView(model)
        treeview.set_headers_visible(False)
        treeview.show()
        scrolled_window.add(treeview)

        column = gtk.TreeViewColumn()
        renderer = gtk.CellRendererToggle()
        renderer.connect('toggled', self._flag_toggled, model)
        column.pack_start(renderer, False)
        column.set_attributes(renderer, active=FLAGS_COLUMN_SETTING)
        renderer = gtk.CellRendererText()
        column.pack_start(renderer)
        column.set_attributes(renderer, text=FLAGS_COLUMN_SYMBOL)

        treeview.append_column(column)

        flags_class = self.klass.spec.flags_class
        flags = [(flag.first_value_name, mask)
                     for mask, flag in flags_class.__flags_values__.items()
                         # we avoid composite flags
                         if len(flag.value_names) == 1]
        flags.sort()
        
        # Populate the model with the flags 
        value = self.property.value
        for name, mask in flags:
            setting = (value & mask) == mask and True or False
            # add a row to represent the flag
            model.append((setting, name))

        response = dialog.run()
        
        if response == gtk.RESPONSE_OK:
            new_value = 0
            model_iter = model.get_iter_first()
            for name, mask in flags:
                setting = model.get_value(model_iter, FLAGS_COLUMN_SETTING)
                if setting:
                    new_value |= mask
                model_iter = model.iter_next(model_iter)
                
            # if the new value is different from the old value, we need to
            # update the property
            if new_value != value:
                self.set_property(new_value)
                # update the entry in the property editor
                self._internal_load(new_value)
                
        dialog.destroy()

class EditorPropertyBoolean(EditorProperty):
    def __init__(self, property_class, app):
        EditorProperty.__init__(self, property_class, app)

        button = gtk.ToggleButton(_('No'))
        button.connect('toggled', self._changed_boolean)

        self.input = button

    def load(self, widget):
        self._start_load(widget)

        self._internal_load(self.property.value)
        
        self.input.set_data('user_data', self)
        self._finish_load()

    def _internal_load(self, state):
        label = self.input.get_child()

        if state:
            value = tags.YES
        else:
            value = tags.NO
        label.set_text(value)
            
        self.input.set_active(state)
        
    def _changed_boolean(self, button):
        if self.property.loading:
            return
        
        if button.get_active():
            value = tags.YES
        else:
            value = tags.NO
            
        label = button.get_child()
        label.set_text(value)
        self.set_property(button.get_active())

class IncompleteFunctions(Exception): pass

class EditorPropertySpecial(EditorProperty):
    def __init__(self, property_class, app):
        EditorProperty.__init__(self, property_class, app)

        if self.klass._editor_create_function is None:
            raise IncompleteFunctions(_("EditorPropertySpecial without create function: %s") % self.klass.id)

        if self.klass._editor_update_function is None:
            raise IncompleteFunctions(_("EditorPropertySpecial without update function: %s") % self.klass.id)

        context = app.get_current_context()
        self.input = self.klass._editor_create_function(context)

    def load(self, widget):
        self._start_load(widget)
        self.widget = widget
        
        proxy = EditorValueProxy()
        proxy.connect('value-set', self._value_changed, widget)
        proxy.connect('property-set', self._property_set, widget)

        context = widget.project.context
        self.klass._editor_update_function(context, self.input,
                                           widget.gtk_widget, proxy)
        self._finish_load()

    def _value_changed(self, proxy, value, widget):
        if self.property.loading:
            return

        self.set_property(value)

    def _property_set(self, proxy, prop_id, value, widget):
        property = widget.get_glade_property(prop_id)

        if not property or property.loading: 
            return

        self.set_property(value)

class EditorValueProxy(gobject.GObject):
    __gsignals__ = {
        'value-set':(gobject.SIGNAL_RUN_LAST, None, (object,)),
        'property-set':(gobject.SIGNAL_RUN_LAST, None, (str, object))
        }

    def set_value(self, value):
        self.emit('value-set', value)

    def set_property(self, prop, value):
        self.emit('property-set', prop, value)
        
gobject.type_register(EditorValueProxy)

# this maps property types with EditorProperty classes
property_map = {
    gobject.TYPE_INT: EditorPropertyInteger,
    gobject.TYPE_UINT: EditorPropertyInteger,
    gobject.TYPE_FLOAT: EditorPropertyFloat,
    gobject.TYPE_DOUBLE: EditorPropertyDouble,
    gobject.TYPE_ENUM: EditorPropertyEnum,
    gobject.TYPE_FLAGS: EditorPropertyFlags,
    gobject.TYPE_BOOLEAN: EditorPropertyBoolean,
    gobject.TYPE_STRING: EditorPropertyText
    }

def create_editor_property(property_class, app):
    global property_map

    if property_class._editor_create_function:
        return EditorPropertySpecial(property_class, app)

    if property_map.has_key(property_class.type):
        return property_map[property_class.type](property_class, app)
    else:
        return None


class EditorTable:
    """ For each glade widget class that we have modified, we create a
    EditorTable. A EditorTable is mainly a gtk_table with all the widgets to
    edit a particular WidgetClass (well the first tab of the gtk notebook).
    When a widget of is selected and going to be edited, we create a
    EditorTable, when another widget of the same class is loaded so that it can
    be edited, we just update the contents of the editor table to reflect the
    values of that GladeWidget. """
    def __init__(self, editor, widget_klass, common):
        # Handy reference that avoids having to pass the editor arround.
        self.editor = editor

        # The WidgetClass this table belongs to.
        self.widget_klass = widget_klass

        # Is this table to be used in the common tab ? or the general tab ?
        self.common = common
        
        # This widget is a gtk_table that is displayed in the glade-editor when
        # a widget of this class is selected. It is hiden when another type is
        # selected. When we select a widget we load into the inputs inside this
        # table the information about the selected widget.
        self.table_widget = gtk.Table(1, 1, False)

        # A pointer to the gtk_entry that holds the name of the widget. This is
        # the first item pack'ed to the table_widget. We have a reference here
        # because it is an entry which will not be created from a GladeProperty
        # but rather from code.
        self.name_entry = None
        
        # A list of EditorPropery items. For each row in the gtk_table,
        # there is a corresponding EditorProperty.
        self.properties = []

        self.rows = 0

        self.packing = False

        self.table_widget.set_border_width(6)
        self.table_widget.set_row_spacings(TABLE_ROW_SPACING)
        self.table_widget.set_col_spacings(TABLE_COLUMN_SPACING)

    def _attach_row(self, label, editable, row):
        self.table_widget.attach(label, 0, 1, row, row+1,
                                 xoptions=gtk.FILL, yoptions=gtk.FILL)
        self.table_widget.attach(editable, 1, 2, row, row+1,
                                 xoptions=gtk.EXPAND|gtk.FILL,
                                 yoptions=gtk.EXPAND|gtk.FILL)

    def _create_item_label(self, property_class):
        label = gtk.Label("%s:" % property_class.name)
        label.set_size_request(-1, -1)
        label.set_alignment(1.0, 0.5)
        
        # we need to wrap the label in an event box to add tooltips
        eventbox = gtk.EventBox()
        eventbox.add(label)
        
        self.editor._tooltips.set_tip(eventbox, property_class.tooltip)

        return eventbox
    
    def _append_item(self, property_class):
        prop = create_editor_property(property_class, self.editor._app)

        if prop and prop.input:
            label = self._create_item_label(property_class)
            self._attach_row(label, prop.input, self.rows)
            self.rows += 1
        
        return prop
    
    def append_standard_fields(self):
        # Class
        label = gtk.Label(_('Class')+':'); label.set_alignment(1.0, 0.5)
        class_label = gtk.Label(self.widget_klass.name)
        class_label.set_alignment(0.0, 0.5)
        self._attach_row(label, class_label, self.rows)
        self.rows += 1

    def append_items(self, klass, thelist, common):
        for property_class in klass.properties:

            if not property_class.is_visible(klass):
                continue

            if common != property_class.common:
                continue

            if not property_class.editable:
                continue
            
            # don't add properties with type == object unless it has a custom
            # editor
            if property_class.type == gobject.TYPE_OBJECT and \
                   not property_class._editor_create_function:
                continue
            
            prop = self._append_item(property_class)
            if prop:
                thelist.insert(0, prop)
                
        return True
        
class Editor(gtk.Notebook):

    __gsignals__ = {
        'add_signal':(gobject.SIGNAL_RUN_LAST, None,
                      (str, long, int, str))
        }
    
    def __init__(self, app):
        gtk.Notebook.__init__(self)
        
        self._app = app

        # The editor has (at this moment) four tabs this are pointers to the
        # vboxes inside each tab. The vboxes are wrapped into a scrolled
        # window. The vbox_widget is deparented and parented with
        # GladeWidgetClass->table_widget when a widget is selecteed.
        self._vbox_widget = self._notebook_page(_('Widget'))
        self._vbox_packing = self._notebook_page(_('Packing'))
        self._vbox_common = self._notebook_page(_('Common'))
        self._vbox_signals = self._notebook_page(_('Signals'))

        # A list of GladeEditorTable. We have a table (gtktable) for each
        # GladeWidgetClass, if we don't have one yet, we create it when we are
        # asked to load a widget of a particular GladeWidgetClass
        self._widget_tables = []

        # Use when loading a GladeWidget into the editor we set this flag so
        # that we can ignore the "changed" signal of the name entry text since
        # the name has not really changed, just a new name was loaded.
        self._loading = False

        # A handy reference to the Widget that is loaded in the editor. None
        # if no widgets are selected
        self._loaded_widget = None

        #  A reference to the loaded WidgetClass. Note that we can have a class
        # loaded without a loaded_widget. For this reason we can't use
        # loaded_widget->class. When a widget is selected we load this class in
        # the editor first and then fill the values of the inputs with the
        # GladeProperty items. This is usefull for not having to
        # redraw/container_add the widgets when we switch from widgets of the
        # same class
        self._loaded_klass = None

        self._signal_editor = None

        self._tooltips = gtk.Tooltips()

        # Display some help
        help_label = gtk.Label(_("Select a widget to edit its properties"))
        help_label.show()
        self._vbox_widget.pack_start(help_label)
        
        self.set_size_request(350, 300)

    def _notebook_page(self, name):
        vbox = gtk.VBox()

        scrolled_window = gtk.ScrolledWindow()
        scrolled_window.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrolled_window.add_with_viewport(vbox)
        scrolled_window.set_shadow_type(gtk.SHADOW_NONE)

        self.insert_page(scrolled_window, gtk.Label(name), -1)

        return vbox
        
    def do_add_signal(self, id_widget, type_widget, id_signal, callback_name):
        pass

    def _table_create(self, klass, common):
        table = EditorTable(self, klass, common)
        if not common:
            table.append_standard_fields()
        if not table.append_items(klass, table.properties, common):
            return None

        table.table_widget.show_all()
        self._widget_tables.insert(0, table)

        return table
        
    def _get_table_from_klass(self, klass, common):
        for table in self._widget_tables:
            if common != table.common:
                continue
            if table.widget_klass == klass:
                return table

        # The table doesn't exist so we create it
        return self._table_create(klass, common)
        
    def _load_widget_page(self, klass):
        # Remove the old table that was in this container
        children = self._vbox_widget.get_children()
        map(self._vbox_widget.remove, children)

        if not klass:
            return

        table = self._get_table_from_klass(klass, False)

        # Attach the new table
        self._vbox_widget.pack_start(table.table_widget, False)
        
    def _load_common_page(self, klass):
        # Remove the old table that was in this container
        children = self._vbox_common.get_children()
        map(self._vbox_common.remove, children)

        if not klass:
            return

        table = self._get_table_from_klass(klass, True)
        # Attach the new table
        self._vbox_common.pack_start(table.table_widget, False)

    def _load_signal_page(self):
        if not self._signal_editor:
            self._signal_editor = SignalEditor(self, self._app)
            self._vbox_signals.pack_start(self._signal_editor)
    
    def _load_widget_klass(self, klass):
        self._load_widget_page(klass)
        self._load_common_page(klass)
        self._load_signal_page()

        self._loaded_klass = klass

    def _load_packing_page(self, widget):
        # Remove the old properties
        map(self._vbox_packing.remove,
            self._vbox_packing.get_children())

        if not widget:
            return

        # if the widget is a toplevel there are no packing properties
        parent = widget.get_parent()
        if not parent:
            return

        # Now add the new properties
        table = EditorTable(self, None, False)
        table.packing = True

        ancestor = parent
        while ancestor:
            for property_class in ancestor.klass.child_properties:
                if ancestor.klass.child_property_applies(ancestor.gtk_widget,
                                                         widget.gtk_widget,
                                                         property_class.id):
                    editor_property = table._append_item(property_class)
                    editor_property.load(widget)
            
            ancestor = ancestor.get_parent()

        table.table_widget.show_all()
        self._vbox_packing.pack_start(table.table_widget, False)
        
    def load_widget_real(self, widget):
        """ Load widget into the editor, if widget is None clear the editor """
        # Load the WidgetClass
        klass = widget and widget.klass or None
        if self._loaded_klass != klass:
            self._load_widget_klass(klass)

        self._signal_editor.load_widget(widget)
        self._load_packing_page(widget)

        # We are just clearing, we are done
        if not widget:
            self._loaded_widget = None
            return

        self._loading = True

        # Load each EditorProperty
        table = self._get_table_from_klass(klass, False)
        for widget_property in table.properties:
            widget_property.load(widget)

        # Load each EditorProperty for the common tab
        table = self._get_table_from_klass(klass, True)
        for widget_property in table.properties:
            widget_property.load(widget)

        self._loaded_widget = widget
        self._loading = False

    def load_widget(self, widget):
        """ Load widget into the editor, if widget is None clear the editor """
        # Skip widget if it's already loaded
        if self._loaded_widget == widget:
            return

        self.load_widget_real(widget)

    def refresh(self):
        """ Rereads properies and updates the editor """
        if self._loaded_widget is not None:
            self.load_widget_real(self._loaded_widget)

        
gobject.type_register(Editor)
