/*
 * Gnome-biff program
 * Version:  2.6
 * Copyright (C) 2000-2001  Nicolas Rougier
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#ifdef USE_GNOME
#  include <gnome.h>
#  include <applet-widget.h>
#endif
#include <popt.h>
#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "protocol.h"
#include "callbacks.h"
#include "interface.h"
#include "applet.h"
#include "window.h"
#include "support.h"
#include "gbiff.h"


/********************************************************************************
 * Function: main
 * Input   : argc, argv
 * Output  : /
 * Return  : /
 ********************************************************************************/
int main (int argc, char **argv) {
  Biff biff = {UNSET_TITLE,
	       UNSET_PROTOCOL,
	       UNSET_ADDRESS,
	       UNSET_USER,
	       UNSET_PASSWORD,
	       UNSET_POLLTIME,
	       UNSET_POPTIME,
	       UNSET_GEOM_POP_X,
	       UNSET_GEOM_POP_Y,
	       UNSET_GEOM_WIN_X,
	       UNSET_GEOM_WIN_Y,
	       UNSET_OPT_NO_POPUP,
	       UNSET_OPT_NO_TITLE,
	       UNSET_OPT_NO_DATE,
	       UNSET_OPT_NO_SOUND,
	       UNSET_OPT_NO_CONFIGURE,
	       UNSET_SOUND,
	       UNSET_FONT,
	       UNSET_MAIL_PIXMAP_FILENAME,
	       UNSET_NOMAIL_PIXMAP_FILENAME,
	       {{0,0}},0,0,0,0,0,0,0,
	       0,0,
	       0,0,0,0,0,0,0,0};

  /* Parse command line and connection to gnome session manager if necessary */
  parseCommandLine (argc, argv, &biff);


  /* Add directory where to look for pixmaps */
  /* Do not forget to add the "" directory,
      it allows to handle absolute path in filename */
  add_pixmap_directory ("");
  add_pixmap_directory ("/usr/share/pixmaps/gbiff");
  add_pixmap_directory (PACKAGE_DATA_DIR"/pixmaps/gbiff");


#ifdef USE_GNOME
  /* Creation of the applet */
  biff.applet = create_Applet (&biff, APPLET_VERTICAL);
#endif
  
  /* Setting default parameter after having load session within create_Applet */
  setDefaultParameters (&biff);

  /* Create widgets */
  biff.mailbox = create_Mailbox (&biff);
  biff.fontDialog = create_FontSelectionDialog (&biff);
  biff.fileDialog = create_FileSelectionDialog (&biff);
  biff.popup = create_Popup (&biff);
  biff.window = create_Window (&biff);

  /* Initialization of the mailbox widget */ 
  initializeMailbox (&biff);


  /* Show the Mailbox widget */
  if (biff.optNoConfigure) {
#ifdef USE_GNOME
    gtk_widget_show (biff.applet);
#else
    gtk_widget_show (biff.window);
#endif
    /* Timer for polling mail file */
    biff.polltag = gtk_timeout_add (biff.polltime*1000, on_TimeoutPoll_event, &biff);
  }
  else
    gtk_widget_show (biff.mailbox);

#ifdef USE_GNOME
  /* Special corba main loop */
  applet_widget_gtk_main ();
#else
  /* Gtk main loop */
  gtk_main ();
#endif

  /* Useless... */
  return (0);
};


/********************************************************************************
 * Function: Parse command line and connect to gnome session manager
 * Input   : argc, argv and a biff structure
 * Output  : Directly within the biff structure
 * Return  : /
 ********************************************************************************/
void parseCommandLine (int argc, char **argv, Biff *biff) {
  char *protocol = NULL;
  char *address = NULL;
  char *user = NULL;
  char *win_geom = NULL;
  char *pop_geom = NULL;
  int popupX = 0, popupY = 0;
  int windowX = 0, windowY = 0;
  char *title = NULL;
  char *mailpix = NULL;
  char *nomailpix = NULL;
  char *sound = NULL;
  char *font = NULL;
  int polltime = 1;
  int poptime =	1;
  int noPopup = UNSET_OPT_NO_POPUP;
  int noTitle = UNSET_OPT_NO_TITLE;
  int noDate = UNSET_OPT_NO_DATE;
  int noSound = UNSET_OPT_NO_SOUND;
  int noConfigure = UNSET_OPT_NO_CONFIGURE;
#ifndef USE_GNOME
  int status;
#endif
  poptContext poptcon;

  struct poptOption options[] = {
    {"protocol",   '\0', POPT_ARG_STRING, &protocol,   0, _("Protocol to use for checking mail"),              "[file,pop3,maildir,mh,imap4]"},
    {"address",    '\0', POPT_ARG_STRING, &address,    0, _("Location of the mailbox (path or address:port)"), "LOCATION"},
    {"user",       '\0', POPT_ARG_STRING, &user,       0, _("Username for login to a remote mailbox"),  "USERNAME"},
    {"polltime",   '\0', POPT_ARG_INT,    &polltime,   0, _("Check mail every POLLTIME seconds"),              "POLLTIME"},
    {"poptime",    '\0', POPT_ARG_INT,    &poptime,    0, _("Display popup windows for POPTIME seconds"),      "POPTIME"},
    {"pop_geom",   '\0', POPT_ARG_STRING, &pop_geom,   0, _("Display popup at specified coordinates"),         "GEOMETRY"},
#ifndef USE_GNOME
    {"win_geom",   '\0', POPT_ARG_STRING, &win_geom,   0, _("Display window at specified coordinates"),        "GEOMETRY"},
#endif
    {"title",      '\0', POPT_ARG_STRING, &title,      0, _("Title of the mailbox"),                           "NAME"},
    {"nosound",    '\0', POPT_ARG_NONE,   &noSound,    0, _("No sound when new mail has arrived"),             NULL},
    {"sndfile",    '\0', POPT_ARG_STRING, &sound,      0, _("Soundfile to play when new mail has arrived"),    "SOUNDFILE"},
    {"mailpix",    '\0', POPT_ARG_STRING, &mailpix,    0, _("Pixmap to use when new mail has arrived"),        "PIXMAP"},
    {"nomailpix",  '\0', POPT_ARG_STRING, &nomailpix,  0, _("Pixmap to use when no mail"),                     "PIXMAP"},
    {"nopopup",    '\0', POPT_ARG_NONE,   &noPopup,    0, _("No popup when new mail has arrived"),             NULL},
    {"notitle",    '\0', POPT_ARG_NONE,   &noTitle,    0, _("No title in popup window"),                       NULL},
    {"nodate",     '\0', POPT_ARG_NONE,   &noDate,     0, _("No date in popup window"),                        NULL},
    {"font",       '\0', POPT_ARG_STRING, &font,       0, _("Font to used in popup window"),                   "FONT"},
    {"noconfigure",'\0', POPT_ARG_NONE,   &noConfigure,0, _("Skip the configure process"),                     NULL},
#ifndef USE_GNOME
    POPT_AUTOHELP
#endif
    {NULL, '\0', 0, NULL, 0, NULL, NULL}
  };

#ifdef USE_GNOME  
#  ifdef ENABLE_NLS
  bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
  textdomain (PACKAGE);
#  endif
  /* Applet initialization */
  applet_widget_init (PACKAGE, VERSION, argc, argv, options, 0, &poptcon);
#else
  /* Argument parsing */
  poptcon = poptGetContext ("gbiff", argc,  (const char **) argv, options, 0);
  while ((status = poptGetNextOpt(poptcon)) >= 0);
  if (status < -1) {
    fprintf (stderr, "%s: %s\n\n", poptBadOption(poptcon, POPT_BADOPTION_NOALIAS),  poptStrerror(status));
    poptPrintHelp (poptcon, stderr, 0);
    exit (1);
  }
  poptGetNextOpt(poptcon);
  /* Gtk initialisation */
  gtk_init (&argc, &argv);
#endif

  /* Checking arguments */ 
  if (protocol != NULL) {
    if (strcmp (protocol, "file") == 0) {
      biff->protocol = PROTOCOL_FILE;
    }
    else if (strcmp (protocol, "maildir") == 0) {
      biff->protocol = PROTOCOL_MAILDIR;
    }
    else if (strcmp (protocol, "pop3") == 0) {
      biff->protocol = PROTOCOL_POP3;
    }
    else if (strcmp (protocol, "mh") == 0) {
      biff->protocol = PROTOCOL_MH;
    }
    else if (strcmp (protocol, "imap4") == 0) {
      biff->protocol = PROTOCOL_IMAP4;
    }
    else {
      g_error (_("Unknown protocol\n"));
    }
  }
  if (address != NULL) {
    if ((biff->protocol == PROTOCOL_FILE) || (biff->protocol == PROTOCOL_MAILDIR) || (biff->protocol == PROTOCOL_MH))
      biff->address = strdup (address);
    // In case of pop3 and imap4 protocol we have to make sure to supply the port
    else {
      if (strchr (address, (int) ':')) {
	biff->address = strdup (address);
      }
      else {
	biff->address = (char *) malloc (strlen(address)+1+4);
	if (biff->protocol == PROTOCOL_POP3)
	  sprintf (biff->address, "%s:%.4d", address, DEFAULT_POP3_PORT);
	if (biff->protocol == PROTOCOL_IMAP4)
	  sprintf (biff->address, "%s:%.4d", address, DEFAULT_IMAP4_PORT);
	printf ("%s\n", biff->address);
      }
    }
  }
  if (user != NULL)
    biff->user = strdup (user);
  if ((polltime >= 5) && (polltime <= 3600))
    biff->polltime = polltime;
  if ((poptime >= 2) && (poptime <= 30))
    biff->poptime = poptime;
  if (pop_geom != NULL) {
    sscanf (pop_geom, "%d %d", &popupX, &popupY);
    if ((popupX >= -1600) && (popupX <= 1600))
      biff->popupX = popupX;
    if ((popupY >= -1200) && (popupY <= 1200))
      biff->popupY = popupY;
  }
  if (win_geom != NULL) {
    sscanf (win_geom, "%d %d", &windowX, &windowY);
    if ((windowX >= -1600) && (windowX <= 1600))
      biff->windowX = windowX;
    if ((windowY >= -1200) && (windowY <= 1200))
      biff->windowY = windowY;
  }
  if (title != NULL)
    biff->title = strdup (title);
  biff->optNoPopup = noPopup;
  biff->optNoTitle = noTitle;
  biff->optNoDate = noDate;
  biff->optNoSound = noSound;
  biff->optNoConfigure = noConfigure;
  if (sound != NULL)
    biff->sound = strdup (sound);
  if (nomailpix != NULL)
    biff->nomailPixmapFilename = strdup (nomailpix);
  if (mailpix != NULL)
    biff->mailPixmapFilename = strdup (mailpix);
  if (font != NULL)
    biff->font = strdup (font);


  biff->polltag = -1;
  biff->poptag = -1;
};


/********************************************************************************
 * Function: Set all default parameters for all unset options
 * Input   : A biff structure
 * Output  : Directly within biff structure
 * Return  : /
 ********************************************************************************/
void setDefaultParameters (Biff *biff) {
  if (biff->title == UNSET_TITLE)
    biff->title = strdup ("mail");
  if (biff->protocol == UNSET_PROTOCOL)
    biff->protocol = DEFAULT_PROTOCOL;
  if (biff->address == UNSET_ADDRESS) {
    if (getenv("MAIL") != 0)
      biff->address = strdup (getenv("MAIL"));
    else
      biff->address = strdup ("");
  }
  if (biff->user == UNSET_USER) {
    if (getenv ("LOGNAME") != 0)
      biff->user = strdup (getenv("LOGNAME"));
    else
      biff->user = strdup ("");
  }
  if (biff->password == UNSET_PASSWORD)
    biff->password = strdup ("");
  if (biff->poptime == UNSET_POPTIME) 
    biff->poptime = DEFAULT_POPTIME;
  if (biff->polltime == UNSET_POLLTIME) 
    biff->polltime = DEFAULT_POLLTIME;
  if (biff->popupX == UNSET_GEOM_POP_X)
    biff->popupX = DEFAULT_GEOM_POP_X;
  if (biff->popupY == UNSET_GEOM_POP_Y)
    biff->popupY = DEFAULT_GEOM_POP_Y;
  if (biff->windowX == UNSET_GEOM_WIN_X)
    biff->windowX = DEFAULT_GEOM_WIN_X;
  if (biff->windowY == UNSET_GEOM_WIN_Y)
    biff->windowY = DEFAULT_GEOM_WIN_Y;
  if (biff->optNoPopup == UNSET_OPT_NO_POPUP)
    biff->optNoPopup = DEFAULT_OPT_NO_POPUP;
  if (biff->optNoTitle == UNSET_OPT_NO_TITLE)
    biff->optNoTitle = DEFAULT_OPT_NO_TITLE;
  if (biff->optNoDate == UNSET_OPT_NO_DATE)
    biff->optNoDate = DEFAULT_OPT_NO_DATE;
  if (biff->optNoSound == UNSET_OPT_NO_SOUND)
    biff->optNoSound = DEFAULT_OPT_NO_SOUND;
  if (biff->optNoConfigure == UNSET_OPT_NO_CONFIGURE)
    biff->optNoConfigure = DEFAULT_OPT_NO_CONFIGURE;
  if (biff->sound == UNSET_SOUND)
    biff->sound = strdup (DEFAULT_SOUND);
  if (biff->font == UNSET_FONT)
    biff->font = strdup (DEFAULT_FONT);
  if (biff->mailPixmapFilename == UNSET_MAIL_PIXMAP_FILENAME)
    biff->mailPixmapFilename = strdup (DEFAULT_MAIL_PIXMAP_FILENAME);
  if (biff->nomailPixmapFilename == UNSET_NOMAIL_PIXMAP_FILENAME)
    biff->nomailPixmapFilename = strdup (DEFAULT_NOMAIL_PIXMAP_FILENAME);
}


/********************************************************************************
 * Function: Initialize the Mailbox widget
 * Input   : A biff structure
 * Output  : Directly within widget fields
 * Return  : /
 ********************************************************************************/
void initializeMailbox (Biff *biff) {
  GtkWidget *tmp;
   
  tmp = biff->mailbox;
  gtk_entry_set_text (GTK_ENTRY (lookup_widget (tmp, "TitleEntry")), biff->title);
  if (biff->protocol == PROTOCOL_FILE)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "ProtocolRadioFile")), TRUE);
  else if (biff->protocol == PROTOCOL_MAILDIR)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "ProtocolRadioMaildir")), TRUE);
  else if (biff->protocol == PROTOCOL_POP3)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "ProtocolRadioPop3")), TRUE);
  else if (biff->protocol == PROTOCOL_MH)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "ProtocolRadioMh")), TRUE);
  else if (biff->protocol == PROTOCOL_IMAP4)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "ProtocolRadioImap4")), TRUE);
  if (biff->suspendMode == 0)
    gtk_label_set_text (GTK_LABEL (GTK_BIN (lookup_widget (biff->mailbox, "SuspendButton"))->child), "Suspend");
  else
    gtk_label_set_text (GTK_LABEL (GTK_BIN (lookup_widget (biff->mailbox, "SuspendButton"))->child), "Continue");
  gtk_entry_set_text (GTK_ENTRY(lookup_widget (tmp, "AddressEntry")), biff->address);
  gtk_entry_set_text (GTK_ENTRY(lookup_widget (tmp, "UserEntry")), biff->user);
  gtk_entry_set_text (GTK_ENTRY(lookup_widget (tmp, "PasswordEntry")), biff->password);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (tmp, "PollTimeSpin")), biff->polltime);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (tmp, "PopTimeSpin")), biff->poptime);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (tmp, "PopupGeometryXSpin")), biff->popupX);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (tmp, "PopupGeometryYSpin")), biff->popupY);
#ifndef USE_GNOME
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (tmp, "WindowGeometryXSpin")), biff->windowX);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (tmp, "WindowGeometryYSpin")), biff->windowY);
#endif
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "OptionsNoPopupCheck")), biff->optNoPopup); 
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "OptionsNoTitleCheck")), biff->optNoTitle); 
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "OptionsNoDateCheck")), biff->optNoDate); 
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (tmp, "OptionsNoSoundCheck")), biff->optNoSound); 
  gtk_label_set_text (GTK_LABEL(GTK_BIN(lookup_widget (tmp, "SoundButton"))->child), strrchr(biff->sound,'/')+1);
  gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG (biff->fontDialog), biff->font);
  gtk_font_selection_dialog_set_preview_text (GTK_FONT_SELECTION_DIALOG (biff->fontDialog), _("gbiff 2.5 by Nicolas Rougier"));
}
