/* PROGRAM:	eggsh
 * FILE:	$Header: /home/egg/src/RCS/eggui.c,v 1.8 1999/02/28 20:02:54 ghn Exp $
 * PURPOSE:	EGG site user interface
 * AUTHOR:	Greg Nelson
 * DATE:	98-06-28
 *
 * REVISED:
 * $Log: eggui.c,v $
 * Revision 1.8  1999/02/28 20:02:54  ghn
 * Version 5.1: The user interface now leaves the cursor at the bottom of
 * the screen and responds to ^L to redraw the entire screen.
 *
 * Revision 1.7  1999/01/01 23:58:56  ghn
 * Take out "CPU_BOUND" conditionals; we won't run CPU_BOUND any more.
 *
 * Revision 1.6  1998/12/31 22:07:56  ghn
 * Rev 5 code: includes multi-reg support, HTML, etc.
 *
 * Revision 1.5  1998/08/03 20:36:28  kelvin
 * Show time of last data collection by basket.
 *
 * Revision 1.4  1998/08/01  21:33:28  ghn
 * Added real implementation for ncursed() based UI for egg.
 *
 * Revision 1.3  1998/08/01 18:51:25  ghn
 * Added John's byte-order-independence changes.
 *
 * Revision 1.2  1998/08/01 17:05:48  ghn
 * Trivial additions of curses related code.
 *
 * Revision 1.1  1998/07/21 11:41:23  ghn
 * Initial revision
 *
 * Copyright 1998 - Greg Nelson
 */

#include <stdio.h>
#include <stdlib.h>
#include <curses.h>

/*  Curses takes it upon itself to define TRUE and FALSE,
    incompatibly, as it happens, with the definitions in
    global.h.  So, clear out the CURSES definitions.  */
#ifdef TRUE
#undef TRUE
#endif
#ifdef FALSE
#undef FALSE
#endif

#include "global.h"
#include "genlib.h"
#include "errnos.h"
#include "regs.h"
#include "version.h"

#ifndef NO_UI
static int32 inittm = 0;
#endif

extern int32 lastDataSent;
extern int32 time_latency, time_housekeeping;

/* Initialize user interface, as needed. */
int32 UIInit(void) {
#ifndef NO_UI
  initscr(); 
  cbreak(); noecho(); 
  nonl();
  intrflush(stdscr, FALSE);
  keypad(stdscr, TRUE);
  timeout(0);
  clear();
  move(0, 0);
  printw("Please wait, initializing...");
  refresh();
  inittm = 0;
#endif
  return ERR_NONE;
}

/* Close down user interface */
int32 UIClose(void) {
#ifndef NO_UI
  endwin();
#endif
  return ERR_NONE;
}

/* Update user interface once on every data collection, receiving the
   current collection result and CollectRecord. */
int32 UIUpdate(int32 cres, CollectRecord *coll) {
#ifndef NO_UI
  char *tmstr;
  double recavg;
  static double grandavg = 0;
  static int32 totsamp = 0;
  int32 i, line, now;

  if ((i = getch()) != ERR) {
    if (i == 12) clear();
  }

  if (inittm == 0) {
    inittm = getzulutime(NULL);
    clear();
  }

  /* Show protocol. */

  if (coll->sampct == 1) {
    line = 11;
    move(line++, 5); printw("Samples per record: %3d", (int)coll->opts.samp_rec);
    move(line++, 5); printw("Seconds per record: %3d", (int)coll->opts.sec_rec);
    move(line++, 5); printw("Records per packet: %3d", (int)coll->opts.rec_pkt);
    move(line++, 5); printw("Bits per trial:     %3d", (int)coll->opts.trialsz);
    refresh();
  }

  if (cres < 0) return ERR_NONE;

  line = 0;
  move(line++, 3);
  printw("EGG %s ID %d REG %s %s", eggtable[0].name, eggtable[0].id,
	 configuredREG->reg_name, Version);

  tmstr = ctime(&inittm); tmstr[strlen(tmstr)-1] = 0;
  move(line++, 5);
  printw("Up since       %25s", tmstr);

  move(line++, 5);
  now = getzulutime(NULL);
  tmstr = ctime(&now); tmstr[strlen(tmstr)-1] = 0;
  printw("Last sample at %25s", tmstr);

  move(line++, 5);
  tmstr = ctime(&lastDataSent);
  tmstr[strlen(tmstr)-1] = 0;
  printw("Last packet at %25s", lastDataSent == 0 ? "Never" : tmstr);

  /* Note that for a non-CPU_BOUND built, UIUpdate is passed a
     collection record filled with the last 10 samples collected.
     Since missing samples do not figure in this record, there
     is no need to test EGG_MISSING_DATA when computing the
     mean below. */

  /* Show mean since egg started and, each time 10 samples
     are collected, the mean for the last 10 samples. */

  line++;
  grandavg *= totsamp;
  grandavg += coll->data.trials[coll->sampct-1]; totsamp++;
  grandavg /= totsamp;
  move(line++, 5);
  printw("Grand mean:       %6.2f", grandavg);

  if (cres == 1) {
    for (recavg = 0, i = 0; i < coll->opts.samp_rec; i++)
      recavg += coll->data.trials[i];
    recavg /= coll->opts.samp_rec;
    move(line, 5); 
    printw("Last record mean: %6.2f (%d seconds)", recavg, coll->opts.sec_rec);
  }
  line++;			      /* Advance whether we show record mean or not */

  /* Show latency (lapse between start of second as measured by
     the computer's clock and when collection of the sample
     actually began) and time elapsed in housekeeping (saving
     packets in local files, talking to the basket, updating
     the user interface, etc.) following collection of the sample.
     If housekeeping time frequently consumes a substantial portion
     of the inter-sample interval, the probability of lost
     samples increases.  Both times are shown in milliseconds. */

  line++;
  move(line++, 5);
  printw("Sampling latency:   ");
  clrtoeol();
  printw("%.3f ms", time_latency / 1000.0);
  move(line++, 5);
  printw("Housekeeping time:  ");
  clrtoeol();
  printw("%.3f ms", time_housekeeping / 1000.0);

  move(16, 0);

  refresh();
#endif

  return ERR_NONE;
}
