/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <stdio.h>
#include <strings.h>
#include <stdlib.h>
#include <math.h>

#include "gdis.h"

/* data structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/****************/
/* SPECIAL OBJS */
/****************/
#define DEBUG_INIT 0
void init_objs(gint type, struct model_pak *data)
{
gint i;

g_return_if_fail(data != NULL);

switch(type)
  {
/* creation mesh */
  case MESH_ON:
    data->mesh_on = TRUE;
    break;
  case MESH_OFF:
    data->mesh_on = FALSE;
/* FIXME - conflict with selection highlighting here */
    for (i=data->num_atoms ; i-- ; )
      (data->atoms+i)->status &= ~SQUARE_HL;
    break;

/* set all atoms back to element colour */
  case REFRESH_COLOUR:
    for (i=data->num_atoms ; i-- ; )
      ARR3SET((data->atoms+i)->colour,
              elements[(data->atoms+i)->atom_code].colour);
    for (i=data->num_shells ; i-- ; )
      ARR3SET((data->shells+i)->colour,
              elements[(data->shells+i)->atom_code].colour);
    break;

/* scale all atom colours by their sof */
  case SOF_COLOUR:
    if (data->has_sof)
      for (i=data->num_atoms ; i-- ; )
        {
        VEC3MUL((data->atoms+i)->colour, (data->atoms+i)->sof);
        }
    break;

  case REFRESH_LITE:
/* refresh atom lighting */
    for (i=data->num_atoms ; i-- ; )
      {
      elem_seek(i, ATOM, data);
      if (!data->region[(data->atoms+i)->region])
        {
        VEC3MUL((data->atoms+i)->colour, 0.5);
        }
      }
/* refresh shell lighting */
    for (i=data->num_shells ; i-- ; )
      {
      elem_seek(i, SHELL, data);
      if (!data->region[(data->shells+i)->region])
        {
        VEC3MUL((data->shells+i)->colour, 0.5);
        }
      }
    break; 

/* atomic coordinates processing */
  case INIT_COORDS:
/* repetition here, since there is interdependence */
/* between centroid (cent_coords) and latmat (calc_coords) calc */
    cent_coords(data);
    calc_coords(INITIAL, data);
  case CENT_COORDS:
    cent_coords(data);
  case REDO_COORDS:
    calc_coords(REFRESH, data);
    update_shells(data);
    break;

  default:
    printf("Unknown object type requested!\n");
  }
}

/********/
/* AXES */
/********/
void make_axes(struct model_pak *data)
{
gfloat len[3];
#define LEN1 20.0
#define LEN2 25.0

/* 2D hack */
ARR3SET(len, data->pbc);
if (data->periodic == 2)
  len[2] = 1.0;

/* axes end points */
VEC3SET(data->axes[0].x, LEN1/len[0], 0.0, 0.0);
VEC3SET(data->axes[1].x, 0.0, LEN1/len[1], 0.0);
VEC3SET(data->axes[2].x, 0.0, 0.0, LEN1/len[2]);

/* label positions */
VEC3SET(data->axes[3].x, LEN2/len[0], 0.0, 0.0);
VEC3SET(data->axes[4].x, 0.0, LEN2/len[1], 0.0);
VEC3SET(data->axes[5].x, 0.0, 0.0, LEN2/len[2]);
}

/********/
/* CELL */
/********/
void make_cell(struct model_pak *data)
{
gfloat c = 1.0;

/* 2D hack */
if (data->periodic == 2)
  c = 0.0;

/* end face 1 */
VEC3SET(data->cell[0].x, 0.0, 0.0, 0.0);
VEC3SET(data->cell[1].x, 0.0, 1.0, 0.0);
VEC3SET(data->cell[2].x, 1.0, 1.0, 0.0);
VEC3SET(data->cell[3].x, 1.0, 0.0, 0.0);

/* end face 2 */
VEC3SET(data->cell[4].x, 0.0, 0.0, c);
VEC3SET(data->cell[5].x, 0.0, 1.0, c);
VEC3SET(data->cell[6].x, 1.0, 1.0, c);
VEC3SET(data->cell[7].x, 1.0, 0.0, c);
}

/**********/
/* SHELLS */
/**********/
#define DEBUG_SHELLS 0
void update_shells(struct model_pak *data)
{
gint i,n,match,tot_match;
gfloat dim[3], vec[3], len[3];

/* NEW - pbc calc with periodic image correction */
dim[0] = data->image_limit[0] + data->image_limit[1];
dim[1] = data->image_limit[2] + data->image_limit[3];
dim[2] = data->image_limit[4] + data->image_limit[5];

/* absolute lengths */
len[0] = fabs(data->pbc[0]);
len[1] = fabs(data->pbc[1]);
len[2] = fabs(data->pbc[2]);

tot_match = 0;
for (n=data->num_atoms ; n-- ; )
  {
/* default */
  (data->atoms+n)->has_shell = FALSE;
  match=0;

#if GOK
/* FIXME - should this be done first to avoid multiple core-shell matches? */
/* NB: x,y,z are in the lattice space, so pbc adjustment is easy */
  for (i=data->num_shells ; i-- ; )
    {
    if ((data->shells+i)->atom_code == (data->atoms+n)->atom_code)
      {
      vec[0] = fabs((data->shells+i)->x - (data->atoms+n)->x);
      vec[1] = fabs((data->shells+i)->y - (data->atoms+n)->y);
      vec[2] = fabs((data->shells+i)->z - (data->atoms+n)->z);
/* NEW - convert to cartesian */
      vecmat(data->latmat, vec);
/* FIXME - better cutoff? */
      if (VEC3MAGSQ(vec) < MAX_SHELL_DIST*MAX_SHELL_DIST)
        {
/* reference each other */
        (data->atoms+n)->has_shell = TRUE;
        (data->atoms+n)->idx_shell = i;
        (data->shells+i)->has_core = TRUE;
        (data->shells+i)->idx_core = n;
        match++;
        tot_match++;
        }
      }
    }
#endif

/* NEW - core/shell connection due to PBC's (corrected for images via dim) */
/* should this be done only if no smaller core/shell conection was found */
/* NB: x,y,z are in the lattice space, so pbc adjustment is easy */
  if (!match)
  for (i=data->num_shells ; i-- ; )
    {
    if ((data->shells+i)->atom_code == (data->atoms+n)->atom_code)
      {
      vec[0] = fabs((data->shells+i)->x - (data->atoms+n)->x);
      vec[1] = fabs((data->shells+i)->y - (data->atoms+n)->y);
      vec[2] = fabs((data->shells+i)->z - (data->atoms+n)->z);
      switch(data->periodic)
        {
        case 3:
          while(vec[2] > dim[2]*len[2]/2.0)
            vec[2] = fabs(vec[2] - len[2]);
        case 2:
          while(vec[0] > dim[0]*len[0]/2.0)
            vec[0] = fabs(vec[0] - len[0]);
          while(vec[1] > dim[1]*len[1]/2.0)
            vec[1] = fabs(vec[1] - len[1]);
          break;
        default:;
        }
/* NEW - convert to cartesian */
      vecmat(data->latmat, vec);
/* FIXME - better cutoff? */
      if (VEC3MAGSQ(vec) < MAX_SHELL_DIST*MAX_SHELL_DIST)
        {
/* reference each other */
        (data->atoms+n)->has_shell = TRUE;
        (data->atoms+n)->idx_shell = i;
        (data->shells+i)->has_core = TRUE;
        (data->shells+i)->idx_core = n;
        match++;
        tot_match++;
        }
      }
    }
/* TODO - how to handle this? */
#if DEBUG_SHELLS
  if (match > 1)
    printf("update_shells() warning: more than one core per shell.\n");
#endif
  }

/* do some shells float down here? */
#if DEBUG_SHELLS
printf("update_shells(): found %d core-shell connections.\n",tot_match);
#endif
/* NEW - delete shells with no cores */
/* TODO - is this a sensible way of handling the problem? */
for (i=data->num_shells ; i-- ; )
  if (!(data->shells+i)->has_core)
    (data->shells+i)->status |= DELETED;

if (tot_match != data->num_shells)
  {
#if DEBUG_SHELLS
printf("update_shells() warning: some shell(s) have no cores.\n");
#endif
/* mem shrink? */
  }
}

/**********/
/* COORDS */
/**********/
#define DEBUG_CENT 0
gint cent_coords(struct model_pak *data)
{
gint i, n;
gfloat vec[3], r, r2, rmax;

/* initialize */
VEC3SET(data->offset, 0, 0, 0);
VEC3SET(data->centroid, 0.0, 0.0, 0.0);
r2 = 0.0;

/* Morphology */
if (data->num_vertices)
  {
/* centroid is always the origin, so just calc. scaling */
  r2 = 0.0;
  for (i=data->num_vertices ; i-- ; )
    {
    vec[0] = (data->vertices+i)->x;
    vec[1] = (data->vertices+i)->y;
    vec[2] = (data->vertices+i)->z;
    vecmat(data->latmat, vec);
    r = VEC3MAGSQ(vec);
    if (r > r2)
      r2 = r; 
    }
  }

/* coord centroid calc */
n=0;
for (i=data->num_atoms ; i-- ; )
  {
/* don't include deleted atoms */
  if ((data->atoms+i)->status & DELETED)
    continue;
/* transformed coords */
  VEC3SET(vec, (data->atoms+i)->x, (data->atoms+i)->y, (data->atoms+i)->z);
/* NB: centroid should now always be fractional */
  ARR3ADD(data->centroid, vec);
  n++;
  }
/* calculate centroid */
if (n)
  {
  VEC3MUL(data->centroid, 1.0 / (gfloat) n);
  }

/* if centroid is substantially within pbc, make pbc/2 the centroid */
if (data->periodic)
  {
  for (i=data->periodic ; i-- ; )
    if (fabs(data->centroid[i] - 0.5) < 0.45)
      data->centroid[i] = 0.5; 
  }

/* molecular surface */
if (data->csurf_on)
  {
/* compensate for the surface drawing offset */
/* FIXME - this is a rough hack */
  data->centroid[data->csurf.ix[2]] += data->csurf.offset;
  }

/* get distance of furtherest displayed item from centroid */
/* check atoms */
for (i=data->num_atoms ; i-- ; )
  {
/* don't include deleted atoms */
  if ((data->atoms+i)->status & DELETED)
    continue;
/* compute distance from centroid to atoms */
  VEC3SET(vec, (data->atoms+i)->x, (data->atoms+i)->y, (data->atoms+i)->z);
/* centeroid removal */
  ARR3SUB(vec, data->centroid);
/* transform */
  vecmat(data->latmat,vec);

/* dist squared */
  r = VEC3MAGSQ(vec);
  if (r > r2)
    r2 = r; 
  }

/* check cell vertices */
if (data->periodic)
  {
  for (i=8 ; i-- ; )
    {
    ARR3SET(vec, data->cell[i].x);
/* centroid removal */
    ARR3SUB(vec, data->centroid);
/* transform */
    vecmat(data->latmat,vec);  
/* dist sq. */
    r = VEC3MAGSQ(vec);
    if (r > r2)
      r2 = r; 
    }
  }

rmax = RMAX_SCALE * sqrt(r2);

/* assign the calculated value, unless 0 */
if (rmax)
  data->rmax = rmax;
else
  data->rmax = RMAX_FUDGE;

/* hack to fit mol/sol surfaces on the screen */
/* FIXME - better (auto calc'd) inc vals */
if (data->solsurf_on && data->mysurf[SOLSURF].num_points)
  data->rmax += 3.0;
else
  if (data->molsurf_on && data->mysurf[MOLSURF].num_points)
    data->rmax += 2.0;

#if DEBUG_CENT
P3VEC("centroid = ",data->centroid);
printf("rmax = %f\n",data->rmax);
printf("cent_coords() done.\n");
#endif

return(0);
}

/**********************************************/
/* NEW - clean bond list of an atom reference */
/**********************************************/
void wipe_atom_bonds(struct model_pak *data, gint i)
{


}

/**************************************/
/* NEW - update bond list for an atom */
/**************************************/
#define DEBUG_ATOM_BONDS 0
void calc_atom_bonds(struct model_pak *data, gint i, gint start)
{
gint j, n;
gfloat r2, r2cut;
gfloat v1[3], v2[3];
struct elem_pak elem1, elem2;
struct bond_pak *bond;

/* number of bonds for atom i */
n = 0;

v1[0] = (data->atoms+i)->rx;
v1[1] = (data->atoms+i)->ry;
v1[2] = (data->atoms+i)->rz;
get_elem_data((data->atoms+i)->atom_code, &elem1, data);
for (j=start ; j<data->num_atoms ; j++)
  {
  if ((data->atoms+j)->status & DELETED)
    continue;
  v2[0] = (data->atoms+j)->rx;
  v2[1] = (data->atoms+j)->ry;
  v2[2] = (data->atoms+j)->rz;
  get_elem_data((data->atoms+j)->atom_code, &elem2, data);
/* get bond cutoff */
  r2cut = elem1.cova + elem2.cova;
  r2cut += BOND_FUDGE*r2cut;
  r2cut *= r2cut;
/* get separation */
  ARR3SUB(v2, v1);
  r2 = VEC3MAGSQ(v2);
/* compare */
  if (r2 < r2cut)
    {
    if (r2 < 0.1)
      {
#if DEBUG_ATOM_BONDS
printf("Ignoring atoms that are too close!\n");
#endif
      }
    else
      {
/* NEW - update bond list */
      bond = g_malloc(sizeof(struct bond_pak)); 
      bond->type = SINGLE;
      bond->atom1 = i;
      bond->atom2 = j;
      data->bonds = g_slist_prepend(data->bonds, bond);
/* bonded atoms - reference each other */
      (data->atoms+i)->bond[(data->atoms+i)->atom_bonds] = j;
      (data->atoms+j)->bond[(data->atoms+j)->atom_bonds] = i;
/* update counting */
      (data->atoms+i)->atom_bonds++;
      (data->atoms+j)->atom_bonds++;
/* FIXME - what if only one is exceeded? */
/* ie we may overwrite only half of another bond */
/* TODO - alloc, rather than use an array for atom reference */
/* to prevent maxbonds being exceeded */
      if ((data->atoms+i)->atom_bonds >= MAX_BONDS)
        {
#if DEBUG_ATOM_BONDS
printf("Warning: maxbonds exceeded.\n");
#endif
        (data->atoms+i)->atom_bonds = MAX_BONDS-1;
        }
      if ((data->atoms+j)->atom_bonds >= MAX_BONDS)
        {
#if DEBUG_ATOM_BONDS
printf("Warning: maxbonds exceeded.\n");
#endif
        (data->atoms+j)->atom_bonds = MAX_BONDS-1;
        }
/* model bond list */
      data->num_bonds++;
      n++;
      }
    }
  }         /* loop all other atoms */

}

/***********************************************/
/* NEW - update bonds for a list of atoms only */
/***********************************************/
#define DEBUG_REDO_BONDS 1
void redo_atom_bonds(struct model_pak *data, gint num, gint *list)
{
gint i;

#if DEBUG_REDO_BONDS
for (i=num ; i-- ; )
  {
  g_return_if_fail((list+i) != NULL);
  if (*(list+i) < 0 || *(list+i) > data->num_atoms)
    printf("Item %d (atom # %d) in list is invalid.\n", i, *(list+i));
  }
#endif

/* for each input atom */
for (i=num ; i-- ; )
  {
/* remove all bonding and recalculate */
  wipe_atom_bonds(data, *(list+i));
  calc_atom_bonds(data, *(list+i), 0);
  }

/* this may be fast enough not to need the same sort of incremental update */
calc_mols(data);
}

/*********/
/* BONDS */
/*********/
#define DEBUG_BONDS 0
void calc_bonds(struct model_pak *data)
{
gint i;

#ifdef TIMER
start_timer("calc_bonds");
#endif
#if DEBUG_BONDS
printf("update_bonds() start.\n");
#endif

/* clean the bond linked list */
wipe_bond_list(data);
/* clean the referencing list */
data->num_bonds=0;
for (i=data->num_atoms ; i-- ; )
  (data->atoms+i)->atom_bonds = 0;

/* calculate bonds - unique i,j */
for (i=0 ; i<((data->num_atoms)-1) ; i++)
  {
  if ((data->atoms+i)->status & DELETED)
    continue;
  calc_atom_bonds(data, i, i+1);
  }

/* finished */
#if DEBUG_BONDS
printf("update_bonds() done: %d bonds found.\n", data->num_bonds);
#endif
#ifdef TIMER
stop_timer("calc_bonds");
#endif
return;
}

/*************/
/* MOLECULES */
/*************/
#define DEBUG_MOLS 0
void calc_mols(struct model_pak *data)
{
gint i, j, k, n, ix, tmp, new_flag;
gint *mol_list, *mol_start;

/* need to do anything? */
data->num_mols=0;
if (!data->num_atoms)
  return;

/* molecule search scheme */
/* NB: limits like MAX_BOND_SIZE/MAX_MOL_SIZE can cause problems if overrun */
#if DEBUG_MOLS
printf("Generating molecule list.\n");
#endif
#ifdef TIMER
start_timer("calc_mols");
#endif

/* list of atom indices */
mol_list = (gint *) g_malloc(data->num_atoms*sizeof(gint));
/* 1 -> start */
mol_start = (gint *) g_malloc(data->num_atoms*sizeof(gint));

/* init */
for (i=data->num_atoms ; i-- ; )
  {
  *(mol_list+i) = i;
  *(mol_start+i) = 0;
  }

/* i - reference atom */
/* j - new atom index */
/* ix - position in list where atom j goes if bonded to i */
/* n - true atom # of i */

/* first molecule */
i=ix=0;
*(mol_start+i) = 1;
new_flag=1;
/* search for new atoms in current molecule */
while (i<data->num_atoms)
  {
/* if no substitutions made, and no more atoms in the */
/* molecule to compare with - start a new molecule */
  if (new_flag && ix==i)
    {
    *(mol_start+i) = 1;
    ix=i+1;
    }
/* scheme - test if unique i,j are bonded */
  n = *(mol_list+i);     /* current atom */
  j = ix;                /* compare atom */
/* default - no more bonds bound -> start new molecule */
  new_flag=1;
  while (j<data->num_atoms)
    {
/* are i and j bonded? */
    for (k=0 ; k<(data->atoms+n)->atom_bonds ; k++)
      {
      if ((data->atoms+n)->bond[k] == *(mol_list+j))
        {
/* yes, swap j and ix (substitution) atoms */
        tmp = *(mol_list+ix);
        *(mol_list+ix) = *(mol_list+j);
        *(mol_list+j) = tmp;
        new_flag=0;
        ix++;
        break;
        }
      }
/* next atom to compare with */
    j++;
    }
/* next atom to compare against */ 
  i++;
  }

/* use new molecule markers & molecule list to init mols */
for (i=data->num_atoms ; i-- ; )
  if (*(mol_start+i))
    data->num_mols++;

/* free old list */
g_free((data->mols)->atom);
/* alloc for the new */
data->mols = (struct mol_pak *) g_realloc
             (data->mols, data->num_mols*sizeof(struct mol_pak));

/* fill out the molecular atom lists */
j=0;
for (i=0 ; i<data->num_mols ; i++)
  {
/* k marks start of old - j seeks start of new */
  k=j++;
/* NEW - last j is data->num_atoms -> mol_start overrun */
  while (j<data->num_atoms)
    {
    if (*(mol_start+j) == 1)
      break;
    j++;
    }
/* got size - allocate */
  (data->mols+i)->num_atoms = j-k;
  (data->mols+i)->atom = (gint *) g_malloc((j-k)*sizeof(gint));
/* init */
  VEC3SET((data->mols+i)->centroid, 0.0, 0.0, 0.0);
/* read 'em in */
  for (n=0 ; n<(j-k) ; n++)
    {
    ix = *((data->mols+i)->atom+n) = *(mol_list+k+n);
    (data->mols+i)->centroid[0] += (data->atoms+ix)->x;
    (data->mols+i)->centroid[1] += (data->atoms+ix)->y;
    (data->mols+i)->centroid[2] += (data->atoms+ix)->z;
/* NEW - record assoc. molecule */
    (data->atoms+ix)->molecule = i;
    }
/* calc centroid */
  if (j-k)
    {
    (data->mols+i)->centroid[0] /= (gfloat) (j-k);
    (data->mols+i)->centroid[1] /= (gfloat) (j-k);
    (data->mols+i)->centroid[2] /= (gfloat) (j-k);
    }
  }

/* done */
g_free(mol_list);
g_free(mol_start);

#if DEBUG_MOLS
printf("Found %d molecules\n",data->num_mols);
for (i=0 ; i<data->num_mols ; i++)
  {
  printf("[%d]  ",i);
  for (j=0 ; j<(data->mols+i)->num_atoms ; j++)
    printf("%d ",*((data->mols+i)->atom+j));
  printf("\n");
  }
#endif
#ifdef TIMER
stop_timer("calc_mols");
#endif

return;
}

/**********************/
/* ROTATE A SELECTION */
/**********************/
void rotate_select(gfloat *mat, struct model_pak *data)
{
gint i, atom, shel;
gfloat scent[3], vec[3], invr[9], invl[9];

/* TODO - include this ability for fractional coords */
/* calculate selection's centroid */
VEC3SET(scent, 0.0, 0.0, 0.0);
for (i=0 ; i<data->select_size ; i++)
  {
/* NEW - rotate the *cartesian* version of the coords (back transform after) */
  atom = *(data->select_list+i);
  scent[0] += (data->atoms+atom)->rx;
  scent[1] += (data->atoms+atom)->ry;
  scent[2] += (data->atoms+atom)->rz;
  }
scent[0] /= data->select_size;
scent[1] /= data->select_size;
scent[2] /= data->select_size;

/* get inv of rotation and lattice matrices */
for (i=0 ; i<9 ; i++)
  invl[i] = data->latmat[i];
invmat(invl, 3);
ARR3SET(&invr[0], &data->rotmat[0]);
ARR3SET(&invr[3], &data->rotmat[3]);
ARR3SET(&invr[6], &data->rotmat[6]);
invmat(invr, 3);

for (i=0 ; i<data->select_size ; i++)
  {
  atom = *(data->select_list+i);
  vec[0] = (data->atoms+atom)->rx - scent[0];
  vec[1] = (data->atoms+atom)->ry - scent[1];
  vec[2] = (data->atoms+atom)->rz - scent[2];

/* rotate to get positions */
  vecmat(mat,vec);
  ARR3ADD(vec, scent);

/* CURRENT - non cartesian back transform */
  vecmat(invr, vec);
  vecmat(invl, vec);
  ARR3ADD(vec, data->centroid);

  (data->atoms+atom)->x = vec[0];
  (data->atoms+atom)->y = vec[1];
  (data->atoms+atom)->z = vec[2];

/* shells */
  if ((data->atoms+atom)->has_shell) 
    {
    shel = (data->atoms+atom)->idx_shell;
    vec[0] = (data->shells+shel)->x - scent[0];
    vec[1] = (data->shells+shel)->y - scent[1];
    vec[2] = (data->shells+shel)->z - scent[2];
/* rotate to get on screen positions, and store the values */
    vecmat(mat,vec);
    ARR3ADD(vec, scent);

/* CURRENT - non cartesian back transform */
    vecmat(invr, vec);
    vecmat(invl, vec);
    ARR3ADD(vec, data->centroid);

    (data->shells+shel)->x = vec[0];
    (data->shells+shel)->y = vec[1];
    (data->shells+shel)->z = vec[2];
    }
  }

/* recalc connectivity */
calc_bonds(data);
calc_mols(data);
}

/**********************/
/* CALC SCREEN COORDS */
/**********************/
#define DEBUG_UPDATE_COORDS 0
void calc_coords(gint mode, struct model_pak *data)
{
gint i,j,n,ghost;
gint ix[3];
gfloat cosa, sina, scale;
gfloat rot[9], g[3], vec[3];
struct model_pak *orig;
struct plane_pak *plane;
GSList *plist;

#ifdef TIMER
start_timer("calc_coords");
#endif
#if DEBUG_UPDATE_COORDS
printf("calc_coords() start.\n");
#endif

/* precalc sine & cosine */
cosa = cos(data->da);
sina = sin(data->da);

orig = data;

/* CURRENT */
ghost=0;
while(data)
  {

/* dependant matrix elements */
switch(mode)
  {
  case ROTATE:
    VEC3SET(&rot[0], cosa, 0.0,-sina);
    VEC3SET(&rot[3],  0.0, 1.0,  0.0);
    VEC3SET(&rot[6], sina, 0.0, cosa);
    break;
  case PITCH:
    VEC3SET(&rot[0], 1.0,  0.0, 0.0);
    VEC3SET(&rot[3], 0.0, cosa,-sina);
    VEC3SET(&rot[6], 0.0, sina, cosa);
    break;
  case ROLL:
    VEC3SET(&rot[0], cosa,-sina, 0.0);
    VEC3SET(&rot[3], sina, cosa, 0.0);
    VEC3SET(&rot[6],  0.0,  0.0, 1.0);
    break;
  case INITIAL:
/* set up initial orientation */
    init_rotmat(data->rotmat);
/* create lattice translation vectors */
    make_xlat(data);
/* all done - fall through to make rot[][] the identity */
/* FIXME - is this the best place for this??? */
    data->num_orig = data->num_atoms;
  case REFRESH:
/* do nothing but calculate pixel positions */
    VEC3SET(&rot[0], 1.0, 0.0, 0.0);
    VEC3SET(&rot[3], 0.0, 1.0, 0.0);
    VEC3SET(&rot[6], 0.0, 0.0, 1.0);
    break;
  default:
    printf("Unknown mode!\n");
    return;
  }
#if DEBUG_UPDATE_COORDS
P3MAT("rot matrix", rot);
#endif

/* selection only rotatation */
if (data->mode == ALTER_SELECT)
  rotate_select(rot, data); 
/* update cummulative rotation matrix */
else 
  matmat(rot,data->rotmat);

/* precalc scale */
scale = data->scale * sysenv.subscale * 0.5 / data->rmax;

/* update periodic image vectors */
/*
if (data->periodic)
*/
for (n=data->periodic ; n-- ; )
  {
/* setup appropriate pbc */
  VEC3SET(vec, 0.0, 0.0, 0.0);
  vec[n] = 1.0;
/* apply lattice vectors */
  vecmat(data->latmat,vec); 
/* apply rotation */
  vecmat(data->rotmat,vec);
/* save */
/* vectors in columns (not rows) */
  for (i=3 ; i-- ; )
    data->piv_real[3*i+n] = vec[i];
/* convert to pix */
 VEC2MUL(vec,scale);
/* TODO - use z coord for perspective view? */
/* save pixel coordinates for on screen selection AND drawing bonds */
  data->piv_pix[2*n] = (gint) vec[0];
  data->piv_pix[2*n+1] = (gint) vec[1];
  }

/* Rotate atoms */
for (n=data->num_atoms ; n-- ; )
  {
  if ((data->atoms+n)->status & DELETED)
    continue;
/* get coordinates */
  VEC3SET(vec, (data->atoms+n)->x,(data->atoms+n)->y,(data->atoms+n)->z);
/* center after lattice conversion */
  ARR3SUB(vec, data->centroid);
/* apply lattice vectors */
  vecmat(data->latmat,vec); 
/* apply rotation */
  vecmat(data->rotmat,vec);
/* save rotated coords */
  (data->atoms+n)->rx = vec[0];
  (data->atoms+n)->ry = vec[1];
  (data->atoms+n)->rz = vec[2];
/* scaling */
  VEC2MUL(vec,scale);
/* TODO - use z coord for perspective view? */
/* save pixel coordinates for on screen selection AND drawing bonds */
  (data->atoms+n)->px = (gint) vec[0];
  (data->atoms+n)->py = (gint) vec[1];
  }

/* rotate shells (if any ) */
for (n=data->num_shells ; n-- ; )
  {
  if ((data->shells+n)->status & DELETED)
    continue;
/* get coordinates */
  VEC3SET(vec, (data->shells+n)->x,(data->shells+n)->y,(data->shells+n)->z);
/* center after lattice conversion */
  ARR3SUB(vec, data->centroid);
/* apply lattice vectors */
  vecmat(data->latmat,vec);
/* rotation */
  vecmat(data->rotmat,&vec[0]);
/* save rotated coords */
  (data->shells+n)->rx = vec[0];
  (data->shells+n)->ry = vec[1];
  (data->shells+n)->rz = vec[2];
/* scaling */
  VEC2MUL(vec,scale);
/* TODO - use z coord for perspective view? */
/* save pixel coordinates for on screen selection AND drawing bonds */
  (data->shells+n)->px = (gint) vec[0];
  (data->shells+n)->py = (gint) vec[1];
  }

/* rotate special objects */

/********/
/* CELL */
/********/
for (n=8 ; n-- ; )
  {
/* get coordinates */
  ARR3SET(vec, data->cell[n].x);
/* CURRENT - center after lattice conversion */
  ARR3SUB(vec, data->centroid);
/* apply lattice vectors */
  vecmat(data->latmat,vec);
/* rotation */
  vecmat(data->rotmat,vec);
/* save rotated coords */
  ARR3SET(data->cell[n].rx, vec);
/* scaling */
  VEC2MUL(vec,scale);
/* save pixel coordinates for on screen selection AND drawing bonds */
/* generate pixel coordinates */
  ARR2SET(data->cell[n].px, vec);
  }

/********/
/* AXES */
/********/
/* FIXME - axes orientations won't change during animation */
/* don't do these to the axes */
for (n=6 ; n-- ; )
  {
/* do cartesian axes */
  ARR3SET(vec, data->axes[n].x);
/* NEW - scaling correction due to no latmat mult */
  for (i=data->periodic ; i-- ; )
    vec[i] *= data->pbc[i];
/* rotation */
  vecmat(data->rotmat,vec);
/* save rotated coords for rendering */
  ARR3SET(data->axes[n].rx, vec);
/* save pixel coordinates for on screen selection AND drawing bonds */
  ARR2SET(data->axes[n].px, vec);

/* lattice axes */
  ARR3SET(vec, data->axes[n].x);
  vecmat(data->latmat,vec);
  vecmat(data->rotmat,vec);
/* save rotated coords for rendering */
  ARR3SET(data->axes[n].rx, vec);
/* save pixel coordinates for on screen selection AND drawing bonds */
  ARR2SET(data->xlat[n].px, vec);
  }

/*****************/
/* CREATION MESH */
/*****************/
/* only doing highlighting here */
if (data->mesh_on)
  {
  for (n=0 ; n<data->num_atoms ; n++)
    {
/* not concerned with deleted atoms */
    if ((data->atoms+n)->status & (DELETED | HIDDEN))
      continue;
/* if inactive, atoms defaults to normal */
    (data->atoms+n)->status = NORMAL;
/* cutoff too arbitrary? */
    if (fabs((data->atoms+n)->rz) < 0.1)
      (data->atoms+n)->status |= SQUARE_HL;
    }
  }

/******************/
/* MYSURF updates */
/******************/
if (data->mysurf[SOLSURF].num_points)
  {
  for (i=0 ; i<data->mysurf[SOLSURF].num_points ; i++)
    {
    vec[0] = data->mysurf[SOLSURF].x[i];
    vec[1] = data->mysurf[SOLSURF].y[i];
    vec[2] = data->mysurf[SOLSURF].z[i];
/* centering */
    ARR3SUB(vec, data->centroid);
/* apply rotation */
    vecmat(data->rotmat,vec);
/* scaling */
    VEC2MUL(vec, scale);
/* save pixel coordinates for on screen selection AND drawing bonds */
    data->mysurf[SOLSURF].px[i] = (gint) vec[0];
    data->mysurf[SOLSURF].py[i] = (gint) vec[1];
    }
  }
if (data->mysurf[MOLSURF].num_points)
  {
  for (i=0 ; i<data->mysurf[MOLSURF].num_points ; i++)
    {
    vec[0] = data->mysurf[MOLSURF].x[i];
    vec[1] = data->mysurf[MOLSURF].y[i];
    vec[2] = data->mysurf[MOLSURF].z[i];
/* centering */
/*
    ARR3SUB(vec, data->centroid);
*/
/* apply rotation */
    vecmat(data->rotmat,vec);
/* scaling */
    VEC2MUL(vec, scale);
/* save pixel coordinates for on screen selection AND drawing bonds */
    data->mysurf[MOLSURF].px[i] = (gint) vec[0];
    data->mysurf[MOLSURF].py[i] = (gint) vec[1];
    }
  }

/***************/
/* VDW SURFACE */
/***************/
if (data->csurf_on)
  {
  n=0;
  ARR3SET(ix, data->csurf.ix);
  for (i=0 ; i<data->csurf.grid ; i++)
    {
    for (j=0 ; j<data->csurf.grid ; j++)
      {
/* get coords of surface (+ offset to 'height') */
      g[0] = (gfloat) i*data->csurf.step[ix[0]];
      g[1] = (gfloat) j*data->csurf.step[ix[1]];
      g[2] = 0.0;
      grid2cart(g,vec);
      vec[0] += data->csurf.min[ix[0]];
      vec[1] += data->csurf.min[ix[1]];
      vec[2] = *(data->csurf.h+n) + data->csurf.offset;
      ARR3SUB(vec, data->centroid);
/* apply translation vectors */
      vecmat(data->latmat,vec); 
/* apply rotation */
      vecmat(data->rotmat,vec);
/* save rotated coords */
      *(data->csurf.rx+n) = vec[0];
      *(data->csurf.ry+n) = vec[1];
      *(data->csurf.rz+n) = vec[2];
/* scaling */
      VEC2MUL(vec, scale);
/* save pixel coordinates for on screen selection AND drawing bonds */
      *(data->csurf.px+n) = (gint) vec[0];
      *(data->csurf.py+n) = (gint) vec[1];
      n++;
      }
    }
  }

/***************/
/* MORPH MODEL */
/***************/
  for (n=data->num_vertices ; n-- ; )
    {
/* do cartesian axes */
    vec[0] = (data->vertices+n)->x;
    vec[1] = (data->vertices+n)->y;
    vec[2] = (data->vertices+n)->z;
/* center */
    ARR3SUB(vec, data->centroid);
/* apply translation vectors */
    vecmat(data->latmat,vec);
/* rotation */
    vecmat(data->rotmat,vec);
/* save (for povray) */
    (data->vertices+n)->rx = vec[0];
    (data->vertices+n)->ry = vec[1];
    (data->vertices+n)->rz = vec[2];
/* scaling */
    VEC2MUL(vec, scale);
/* save pixel coordinates for on screen drawing */
    (data->vertices+n)->px = (gint) vec[0];
    (data->vertices+n)->py = (gint) vec[1];
    }
/* NEW - facet pixel center calc. */
  plist = data->planes;
  while (plist != NULL)
    {
    plane = (struct plane_pak *) plist->data;
    if (plane->present)
      {
/* set visible flag */
      if (facet_visible(data, plane))
        plane->visible = TRUE;
      else
        plane->visible = FALSE;
/* pixel coords of center */
      VEC3SET(vec, 0.0, 0.0, 0.0);
      g[0] = g[1] = 0.0;
      for (i=0 ; i<plane->num_points ; i++)
        {
        j = *(plane->points+i);
/* scan all facets connected to vertex */
/* NEW - screen pos and real pos (for povray) */
        vec[0] += (data->vertices+j)->rx;
        vec[1] += (data->vertices+j)->ry;
        vec[2] += (data->vertices+j)->rz;
        g[0] += (gfloat) (data->vertices+j)->px;
        g[1] += (gfloat) (data->vertices+j)->py;
        }
/* average the summation */
      VEC3MUL(vec, 1.0/plane->num_points);
      g[0] /= plane->num_points;
      g[1] /= plane->num_points;
/* assign */
      plane->rx = vec[0];
      plane->ry = vec[1];
      plane->rz = vec[2];
      plane->px = (gint) g[0];
      plane->py = (gint) g[1];
      }
    plist = g_slist_next(plist);
    }

/* NEW - update printed coord of selected atom in atom info mode */
  print_atom(data, data->atom_info);
  print_shell(data, data->shell_info);

/* CURRENT */
  if (ghost < data->num_ghosts)
    {
    data = model_ptr(*(orig->ghosts+ghost), RECALL);
    ghost++;
    }
  else
    data = NULL;
  }

#ifdef TIMER
stop_timer("calc_coords");
#endif
#if DEBUG_UPDATE_COORDS
printf("calc_coords() done.\n");
#endif
}

/*********************************/
/* unified atom location routine */
/*********************************/
#define DEBUG_SEEK_ATOM 0
/* NB: should always return -1 if no atom was found */
gint seek_atom(gint px, gint py, struct model_pak *data)
{
gint i, j, dx, dy, dr2;

#if GTK_GL
if (GTK_WIDGET_VISIBLE(sysenv.glarea))
  return(gl_seek_atom(px, py, data));
#endif

/* get the active subwin */
j=-1;
for (i=0 ; i<sysenv.num_displayed ; i++)
  if (sysenv.displayed[i] == sysenv.active)
    j=i;

/* failsafe */
if (j < 0)
  {
  printf("seek_atom() error: invalid subwin.\n");
  return(-1);
  }

/* NB: assume subwin is always the active one */
px -= sysenv.subcenx[j];
py -= sysenv.subceny[j];

for (i=0 ; i<data->num_atoms ; i++)
  {
  if ((data->atoms+i)->status & (DELETED | HIDDEN))
    continue;

  dx = ((data->atoms+i)->px - px) + data->offset[0]; 
  dy = ((data->atoms+i)->py - py) + data->offset[1]; 
  dr2 = dx*dx + dy*dy;
/* FIXME - cutoff too arbitrary? */
  if (dr2 < 36)
    return(i);
  }

/* NEW - (mainly for debugging) print if click on cell vertex */
#if DEBUG_SEEK_ATOM
for (i=8 ; i-- ; )
  {
  dx = (data->cell[i].px[0] - px) + data->offset[0]; 
  dy = (data->cell[i].px[1] - py) + data->offset[1]; 
  dr2 = dx*dx + dy*dy;
/* FIXME - cutoff too arbitrary? */
  if (dr2 < 36)
    {
    printf("Cell vertex %d: \n",i);
    P3VEC("orig: ", data->cell[i].x);
    P3VEC("xlat: ", data->cell[i].rx);
    }
  }
#endif

/* failed to find an atom at px,py */
return(-1);
}

/**********************************/
/* unified shell location routine */
/**********************************/
/* NB: should always return -1 if no atom was found */
gint seek_shell(gint px, gint py, struct model_pak *data)
{
gint i, j, dx, dy, dr2;

/* get the active subwin */
j=-1;
for (i=0 ; i<sysenv.num_displayed ; i++)
  if (sysenv.displayed[i] == sysenv.active)
    j=i;

/* failsafe */
if (j < 0)
  {
  printf("seek_shell() error: invalid subwin.\n");
  return(-1);
  }

/* NB: assume subwin is always the active one */
px -= sysenv.subcenx[j];
py -= sysenv.subceny[j];

for (i=0 ; i<data->num_shells ; i++)
  {
  if ((data->shells+i)->status & (DELETED | HIDDEN))
    continue;

  dx = ((data->shells+i)->px - px) + data->offset[0]; 
  dy = ((data->shells+i)->py - py) + data->offset[1]; 
  dr2 = dx*dx + dy*dy;
/* FIXME - cutoff too arbitrary? */
  if (dr2 < 36)
    return(i);
  }

/* failed to find an shell at px,py */
return(-1);
}

/***************************************/
/* unified pixel to coordinate routine */
/***************************************/
#define DEBUG_PCM 0
void pixel_coord_map(gint px, gint py, struct model_pak *data, gint atom)
{
gint i, model;
gfloat r[3], inv[9];

/* this could lead to trouble */
g_return_if_fail(sysenv.active == data->number);

/* find active subwin */
i=0;
while (i<sysenv.num_displayed)
  {
  if (sysenv.displayed[i] == sysenv.active)
     break;
  i++;
  }

/* the model should be displayed for this to make sense */
g_return_if_fail(i != sysenv.num_displayed);
model = sysenv.displayed[i];
g_return_if_fail(model == sysenv.active);

/* hack for selection altering */
if (data->mode != ALTER_SELECT)
  {
#if DEBUG_PCM
printf("(*) ");
#endif
/* remove subwin centre */
  px -= sysenv.subcenx[i];
  py -= sysenv.subceny[i];
/* account for user translation */
  px -= data->offset[0];
  py -= data->offset[1];
  }

#if DEBUG_PCM
printf("center [%d,%d] : offset [%d,%d] : pixel pos [%d,%d]\n",
  sysenv.subcenx[i],sysenv.subceny[i],data->offset[0],data->offset[1],px,py);
#endif

/* we know this for sure :-) */
(data->atoms+atom)->px = px;
(data->atoms+atom)->py = py;

/* convert pixel coords to screen values */
r[0] = (gfloat) px; 
r[1] = (gfloat) py; 
/* go up */
r[0] *= 2.0*data->rmax;
r[1] *= 2.0*data->rmax;
/* go down */
r[0] /= sysenv.subscale;
r[1] /= sysenv.subscale;
r[0] /= data->scale;
r[1] /= data->scale;
/* leave z where it is, make sure it's defined!!! */
r[2] = (data->atoms+atom)->rz;
/* assign the rotated vectors */
(data->atoms+atom)->rx = r[0];
(data->atoms+atom)->ry = r[1];

/* mult by inverse rotation matrix */
ARR3SET(&inv[0], &data->rotmat[0]);
ARR3SET(&inv[3], &data->rotmat[3]);
ARR3SET(&inv[6], &data->rotmat[6]);
invmat(inv,3);
vecmat(inv,r);

/* mult by inverse lattice matrix */
if (data->periodic)
  {
  for (i=0 ; i<9 ; i++)
    inv[i] = data->latmat[i];
  invmat(inv,3);
  vecmat(inv,r);
  }

/* generate and assign the unrotated coordinates */
ARR3ADD(r, data->centroid);
(data->atoms+atom)->x = r[0];
(data->atoms+atom)->y = r[1];
(data->atoms+atom)->z = r[2];
/* generate 'original' cartesian coordinates */
vecmat(data->latmat, r);
(data->atoms+atom)->cx = r[0];
(data->atoms+atom)->cy = r[1];
(data->atoms+atom)->cz = r[2];
}

/***********************************/
/* print atom coord data primitive */
/***********************************/
void print_atom(struct model_pak *data, gint i)
{
GString *txt;

/* check */
if (i < 0 || i >= data->num_atoms)
  return;

if ((data->atoms+i)->status & SELECT_HL)
  {

/* make string */
txt = g_string_new(NULL);

if (data->periodic)
  {
/* get cartesian coords */
  g_string_sprintf(txt,
  "p[%03d] %2s core at (%9.4f,%9.4f,%9.4f), orig frac (%9.4f,%9.4f,%9.4f).",
           i, (data->atoms+i)->element,
           data->centroid[0] + (data->atoms+i)->rx,
           data->centroid[1] + (data->atoms+i)->ry,
           data->centroid[2] + (data->atoms+i)->rz,
           (data->atoms+i)->x,
           (data->atoms+i)->y,
           (data->atoms+i)->z);
  }
else
  {
/* display */
  g_string_sprintf(txt,
  "[%03d] %2s core at (%9.4f,%9.4f,%9.4f), orig cart (%9.4f,%9.4f,%9.4f).",
           i,
           (data->atoms+i)->element,
           data->centroid[0] + (data->atoms+i)->rx,
           data->centroid[1] + (data->atoms+i)->ry,
           data->centroid[2] + (data->atoms+i)->rz,
           (data->atoms+i)->cx,
           (data->atoms+i)->cy,
           (data->atoms+i)->cz);
  }

/* print & exit */
show_text(txt->str);
g_string_free(txt, TRUE);
  }
}

/************************************/
/* print shell coord data primitive */
/************************************/
void print_shell(struct model_pak *data, gint i)
{
GString *txt;

/* check */
if (i < 0 || i >= data->num_shells)
  return;
if (!((data->shells+i)->status & SELECT_HL))
  return;

/* make string */
txt = g_string_new(NULL);

/* TODO - what if in funny 2D frac/cart coords? (transform x,y & leave z?) */
if (data->periodic)
  {
  g_string_sprintf(txt,
  "[%03d] %2s shell at (%9.4f,%9.4f,%9.4f), orig frac (%9.4f,%9.4f,%9.4f).",
           i, (data->shells+i)->element,
           data->centroid[0] + (data->shells+i)->rx,
           data->centroid[1] + (data->shells+i)->ry,
           data->centroid[2] + (data->shells+i)->rz,
           (data->shells+i)->x,
           (data->shells+i)->y,
           (data->shells+i)->z);
  }
else
  {
  g_string_sprintf(txt,
  "[%03d] %2s shell at (%9.4f,%9.4f,%9.4f), orig cart (%9.4f,%9.4f,%9.4f).",
           i, (data->shells+i)->element,
           data->centroid[0] + (data->shells+i)->rx,
           data->centroid[1] + (data->shells+i)->ry,
           data->centroid[2] + (data->shells+i)->rz,
           (data->shells+i)->cx,
           (data->shells+i)->cy,
           (data->shells+i)->cz);
  }

/* print & exit */
show_text(txt->str);
g_string_free(txt, TRUE);
}

