/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <GL/gl.h>

#include "gdis.h"
#include "file.h"
#include "parse.h"
#include "coords.h"
#include "matrix.h"
#include "molsurf.h"
#include "spatial.h"
#include "surface.h"
#include "sginfo.h"
#include "task.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"
#include "opengl.h"

/* main pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* molsurf globals */
gdouble ms_prad=1.0;
gdouble ms_blur=0.3;
gint ms_type=MS_MOLECULAR, ms_colour=MS_TOUCH;
gint ms_epot_autoscale=TRUE;
GtkWidget *ms_colour_combo;

/***************************************/
/* setup and run a surface calculation */
/***************************************/
#define DEBUG_CALC_MOLSURF 0
void ms_calculate(void)
{
struct model_pak *model;
const gchar *tmp;

model = sysenv.active_model;
g_assert(model != NULL);

spatial_delete_type(SPATIAL_MOLSURF, model);

/* get colouring type */
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(ms_colour_combo)->entry));
if (g_ascii_strncasecmp(tmp,"Touch",5) == 0)
  ms_colour = MS_TOUCH;
if (g_ascii_strncasecmp(tmp,"AFM",3) == 0)
  ms_colour = MS_AFM;
if (g_ascii_strncasecmp(tmp,"Electrostatic",13) == 0)
  ms_colour = MS_EPOT;
if (g_ascii_strncasecmp(tmp,"Hirshfeld",9) == 0)
  ms_colour = MS_HIRSHFELD;

ms_cube(ms_blur, ms_type, ms_colour, model);

init_objs(CENT_COORDS, model);
redraw_canvas(SINGLE);
}

/************************/
/* simple deletion hook */
/************************/
void ms_delete(void)
{
struct model_pak *model;

model = sysenv.active_model;
g_assert(model != NULL);

/* remove any previous surfaces */
spatial_delete_type(SPATIAL_MOLSURF, model);
model->ms_colour_scale = FALSE;
redraw_canvas(SINGLE);
}

/*******************************************/
/* Molecular surface colour mode selection */
/*******************************************/
void ms_colour_mode(GtkWidget *entry)
{
const gchar *tmp;

g_assert(entry != NULL);

tmp = gtk_entry_get_text(GTK_ENTRY(entry));

if (g_ascii_strncasecmp(tmp,"Touch",5) == 0)
  ms_colour = MS_TOUCH;
if (g_ascii_strncasecmp(tmp,"AFM",3) == 0)
  ms_colour = MS_AFM;
if (g_ascii_strncasecmp(tmp,"Electrostatic",13) == 0)
  ms_colour = MS_EPOT;
if (g_ascii_strncasecmp(tmp,"Hirshfeld",9) == 0)
  ms_colour = MS_HIRSHFELD;

redraw_canvas(ALL);
}

/*************/
/* callbacks */
/*************/
void cb_ms_accessible(void)
{
ms_type = MS_ACCESSIBLE;
}

void cb_ms_molecular(void)
{
ms_type = MS_MOLECULAR;
}

/**********************************/
/* molecular surface scaling type */
/**********************************/
GtkWidget *epot_vbox, *surf_epot_min, *surf_epot_max, *surf_epot_div;
GtkWidget *epot_pts;

/***************************/
/* Molecular surface setup */
/***************************/
void surf_dialog()
{
gpointer dialog;
GList *list;
GtkWidget *window, *frame, *vbox, *hbox, *button, *label;
struct model_pak *data;

/* checks */
data = sysenv.active_model;
if (!data)
  return;
if (data->id == MORPH)
  return;

/* create dialog */
dialog = dialog_request(SURF, "Gaussian Molecular Surfaces", NULL, NULL, data);
if (!dialog)
  return;
window = dialog_window(dialog);

/* frame for spinner setup */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox),frame,FALSE,TRUE,0); 
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

/*
gtksh_direct_spin("D size", &ms_dsize, 0.01, 1.0, 0.05, NULL, NULL, vbox);
gtksh_direct_spin("Triangulation depth", &ms_depth, 0, 10, 1, NULL, NULL, vbox);
gtksh_direct_spin("Probe radius", &ms_prad, 0.1, 50.0, 0.01, NULL, NULL, vbox);
*/
gtksh_direct_spin("Blurring factor", &ms_blur, 0.05, 1.0, 0.05, NULL, NULL, vbox);

/* surface contour type */
frame = gtk_frame_new ("Surface type");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);

new_radio_group(0, vbox, TT);

button = add_radio_button("Accessible surface", (gpointer) cb_ms_accessible, NULL);
if (ms_type == MS_ACCESSIBLE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("Molecular surface", (gpointer) cb_ms_molecular, NULL);
if (ms_type == MS_MOLECULAR)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* surface colour scheme */
frame = gtk_frame_new ("Colour method");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, PANEL_SPACING);

/* combo box for colour mode */
list = NULL;
list = g_list_prepend(list, "Touch");
list = g_list_prepend(list, "AFM");
list = g_list_prepend(list, "Electrostatic");
list = g_list_prepend(list, "Hirshfeld");
list = g_list_reverse(list);

ms_colour_combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(ms_colour_combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(ms_colour_combo), list);
gtk_box_pack_start(GTK_BOX(hbox), ms_colour_combo, FALSE, FALSE, PANEL_SPACING);

/* electrostatic potential scale setup */
gtksh_auto_check("Electrostatic autoscaling", NULL, NULL,
                 &data->gulp.epot_autoscale, vbox);

/* hide/show control */
epot_vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(vbox), epot_vbox, FALSE, TRUE,0);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(epot_vbox), hbox, FALSE, TRUE,0);
label = gtk_label_new("maximum ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
surf_epot_max = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), surf_epot_max, FALSE, FALSE, 0);
gtk_entry_set_text(GTK_ENTRY(surf_epot_max), 
                   g_strdup_printf("%f", data->gulp.epot_max));

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(epot_vbox), hbox, FALSE, TRUE,0);
label = gtk_label_new("minimum ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
surf_epot_min = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), surf_epot_min, FALSE, FALSE, 0);
gtk_entry_set_text(GTK_ENTRY(surf_epot_min), 
                   g_strdup_printf("%f", data->gulp.epot_min));

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(epot_vbox), hbox, FALSE, TRUE,0);
label = gtk_label_new("divisions ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
surf_epot_div = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), surf_epot_div, FALSE, FALSE, 0);
gtk_entry_set_text(GTK_ENTRY(surf_epot_div), 
                   g_strdup_printf("%d", data->gulp.epot_divisions));

/* make, hide, close - terminating buttons */
gtksh_stock_button(GTK_STOCK_EXECUTE, ms_calculate, NULL,
                   GTK_DIALOG(window)->action_area);

gtksh_stock_button(GTK_STOCK_REMOVE, ms_delete, NULL,
                   GTK_DIALOG(window)->action_area);

gtksh_stock_button(GTK_STOCK_CLOSE, dialog_destroy, dialog,
                   GTK_DIALOG(window)->action_area);

/* done */
gtk_widget_show_all(window);
}

