
/*
   factory.c -
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2000-2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: factory.c,v 1.17 2001/06/09 23:06:40 ntiffin Exp $
 
*/

/** \file factory.c
 *  \brief GEAS::ConnectionFactory implementation
 */

#include "config.h"

#include "geas-server.h"
#include "globals.h"
#include "geas-skeleton.h"
#include "factory.h"
#include "exceptions.h"
#include "datamonitor/datamonitor.h"

/* raises: AuthenticationError, ServerError */
GEAS_Connection
ConnectionFactory_guestLogin (CORBA_Environment * ev)
{
  return (ConnectionFactory_unsecureLogin ("guest", "guest", ev));
}

/* raises: AuthenticationError, ServerError */
GEAS_Connection
ConnectionFactory_unsecureLogin (const CORBA_char * username,
                                 const CORBA_char * password,
                                 CORBA_Environment * ev)
{
  GEAS_Connection c = CORBA_OBJECT_NIL;
  /* GEAS_Connection retval = CORBA_OBJECT_NIL; */
  GEAS_DataObject user;
  ObjectKey key;
  /* char *sessionid; */
  char *realpass;

  /* find and verify the user's record */
  /* message( "getting user record for '%s'" , username ); */
  user = GEAS_Connection_loadSingleObject (get_server_connection_object (),
                                           "geas::user", "username", username,
                                           ev);
  if (ev->_major != CORBA_NO_EXCEPTION || user == CORBA_OBJECT_NIL)
    {
      /* error finding user or unknown user */
      errormsg ("Unknown user '%s'", username);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      make_AuthenticationError_exception (ev, "Login failed.");
      dm_logentry (DM_EVENT_SECURITY_LOGON_FAIL, username, NULL);
      return (CORBA_OBJECT_NIL);
    }
  /* message("Validating password for user '%s' attempted with '%s')",username,password); */
  realpass = GEAS_DataObject_getField (user, "password", ev);
  if (ev->_major != CORBA_NO_EXCEPTION || strcmp (password, realpass) != 0)
    {
      /* error getting user or wrong password */
      errormsg ("Password failed '%s'", password);
      CORBA_exception_free (ev);
      CORBA_exception_init (ev);
      CORBA_Object_release (user, ev);
      if (realpass)
        {
          CORBA_free (realpass);
        }
      make_AuthenticationError_exception (ev, "Login failed.");
      dm_logentry (DM_EVENT_SECURITY_LOGON_FAIL, username, NULL);
      return (CORBA_OBJECT_NIL);
    }
  CORBA_Object_release (user, ev);
  if (realpass)
    {
      CORBA_free (realpass);
    }
  /* message( "Login succeeded" ); */
  dm_logentry (DM_EVENT_SECURITY_LOGON_SUCCESS, username, NULL);

  /* make a connection */
  key = create_new_object_key ();
  if (key)
    {
      const char *sessionid = object_key_as_string (key);
      if (sessionid)
        {
          c = make_connection_reference (username, sessionid, ev);
        }
      free_object_key (key);
    }
  if (c == CORBA_OBJECT_NIL)
    {
      make_ServerError_exception (ev, "Error allocating connection");
    }
  return (c);
}

/* raises: */
CORBA_char *
ConnectionFactory_getAuthenticationKey (CORBA_char * username,
                                        CORBA_Environment * ev)
{
  return (CORBA_string_dup ("guest"));
}

/* raises: AuthenticationError */
GEAS_Connection
ConnectionFactory_getConnection (CORBA_char * username, CORBA_char * passkey,
                                 CORBA_Environment * ev)
{
  return (CORBA_OBJECT_NIL);
}

/* raises: AuthenticationError */
GEAS_Connection
ConnectionFactory_adminLogin (CORBA_char * username, CORBA_char * passkey,
                              CORBA_Environment * ev)
{
  return (CORBA_OBJECT_NIL);
}
