
/*
   geas-internal-classes.h
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2000 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: geas-internal-classes.h,v 1.17 2001/06/10 12:56:12 ntiffin Exp $
 
*/

/** \file geas-internal-classes.h
 * \brief Server private class initialisation
 * A number of classes are used internally by GEAS, and are placed
 * in the 'geas' module, and given SERVER access (as opposed to PUBLIC)
 * These functions avoid the need to have an external GCD file, which
 * can easily be corrupted or lost.
*/

#ifndef GEAS_INTERNAL_CLASSES_H
#define GEAS_INTERNAL_CLASSES_H

/* normally a set of private functions */
#define IN_GCD_LIBRARY
#include "gcdparser.h"
#undef IN_GCD_LIBRARY

/* UUID values used to identify a class, created with 'uuidgen' */
#define GEAS_LISTITEM_UUID        "f6e52e6e-07a8-4f64-96fa-0bf1be51fec0"
#define GEAS_LISTHOLDER_UUID      "a759a9c5-27e1-45b4-ba98-e34b48ada4d3"
#define GEAS_USER_UUID            "ec3f41af-fd9c-4fe6-836e-3acfe037b76e"
#define GEAS_TRANSACTION_UUID     "6e2384cd-d8fc-4bb8-b681-d4afb784d564"
#define GEAS_SEARCHCRITERIA_UUID  "1fadb197-3149-44d6-9cfa-468a0dae4985"
#define GEAS_SEARCHFIELD_UUID     "0cf1a207-add6-4750-8ee8-be108a5ad3ac"

/* all classes should be placed in the 'GEAS' module unless there is */
/* a good reason to do otherwise ( ie: geas:: is the prefix for GEAS */
/* specific classes, so no conflict with non geas classes)           */

static void
make_server_security_classes (odl_module * m)
{
  odl_class *c;
  odl_field *f;
  /* odl_index *i; */
  GList *body = NULL;
  GList *fields = NULL;

  /* ************************************ */
  /* class GEAS::user */
  c = odl_new_class (ODL_ACCESS_SYSTEM, "user");
  odl_container_insert_container (m, c);
  c->base.parent = (odl_base *) m;
  c->base.access = ODL_ACCESS_PUBLIC;
  c->orderby = g_strdup ("username");
  body = NULL;

  /* add field char username[32] */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_PUBLIC;
  f->base.name = g_strdup ("username");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_char;
  f->format = g_strdup ("32");
  f->properties = ODL_PROP_NOTNULL;
  body = g_list_append (body, f);

  /* add field char username[32] */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_PUBLIC;
  f->base.name = g_strdup ("password");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_char;
  f->format = g_strdup ("32");
  body = g_list_append (body, f);

  /* add field object _currenttransaction */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("_currenttransaction");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_object;
  body = g_list_append (body, f);

  /* add fields to class */
  if (c->contents)
    {
      c->contents = g_list_concat (c->contents, body);
    }
  else
    {
      c->contents = body;
    }

  /* add indexes */
  fields = g_list_append (NULL, g_strdup ("username"));
  odl_make_index (c, TRUE, fields);
}

static void
make_server_transaction_classes (odl_module * m)
{
  odl_class *c;
  odl_field *f;
  GList *body = NULL;
  /* GList *fields = NULL; */

  /* ************************************ */
  /* make class GEAS::transaction */
  c = odl_new_class (ODL_ACCESS_SYSTEM, "transaction");
  odl_container_insert_container (m, c);
  c->base.parent = (odl_base *) m;
  body = NULL;

  /* add field object _user */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("_user");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_object;
  f->properties = ODL_PROP_NOTNULL;
  body = g_list_append (body, f);

  /* add field bool open */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("open");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_bool;
  f->properties = ODL_PROP_NOTNULL;
  body = g_list_append (body, f);

  /* add fields to class */
  if (c->contents)
    {
      c->contents = g_list_concat (c->contents, body);
    }
  else
    {
      c->contents = body;
    }
}

static void
make_server_list_handling_classes (odl_module * m)
{
  odl_class *c;
  odl_field *f;
  GList *body = NULL;
  GList *fields = NULL;

  /* ************************************ */
  /* make class GEAS::listitem */
  c = odl_new_class (ODL_ACCESS_SYSTEM, "listitem");
  odl_container_insert_container (m, c);
  c->base.parent = (odl_base *) m;
  body = NULL;
  c->orderby = g_strdup ("position");

  /* add field unsigned int position */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("position");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_unsignedint;
  f->properties = ODL_PROP_NOTNULL;
  body = g_list_append (body, f);

  /* add field object listid         */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("listid");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_object;
  f->properties = ODL_PROP_NOTNULL;
  body = g_list_append (body, f);

  /* add field object reference      */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("reference");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_object;
  f->properties = ODL_PROP_NOTNULL;
  body = g_list_append (body, f);

  /* add fields to class */
  if (c->contents)
    {
      c->contents = g_list_concat (c->contents, body);
    }
  else
    {
      c->contents = body;
    }

  /* add indexes */
  fields = g_list_append (NULL, g_strdup ("position"));
  fields = g_list_append (fields, g_strdup ("listid"));
  odl_make_index (c, TRUE, fields);

  /* ************************************ */
  /* make class GEAS::listholder */
  c = odl_new_class (ODL_ACCESS_SYSTEM, "listholder");
  odl_container_insert_container (m, c);
  c->base.parent = (odl_base *) m;
  body = NULL;

  /* add field unsigned int length; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("length");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_unsignedint;
  f->properties = ODL_PROP_NOTNULL;
  body = g_list_append (body, f);

  /* add field text classname; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("classname");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_text;
  f->properties = ODL_PROP_NOTNULL;
  body = g_list_append (body, f);

  /* add fields to class */
  if (c->contents)
    {
      c->contents = g_list_concat (c->contents, body);
    }
  else
    {
      c->contents = body;
    }
}

static void
make_server_query_handling_classes (odl_module * m)
{
  odl_class *c;
  odl_field *f;
  GList *body = NULL;
  /* GList *fields = NULL; */

  /* ************************************ */
  /* make class GEAS::searchcriteria */
  c = odl_new_class (ODL_ACCESS_SYSTEM, "searchcriteria");
  odl_container_insert_container (m, c);
  c->base.parent = (odl_base *) m;
  body = NULL;

  /* add field object parent; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("parent");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_object;
  f->properties = ODL_PROP_NONE;
  f->defaultval = g_strdup ("");
  body = g_list_append (body, f);

  /* add field char classname<128>; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("classname");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_char;
  f->format = g_strdup ("128");
  f->properties = ODL_PROP_NONE;
  f->defaultval = g_strdup ("unknown");
  body = g_list_append (body, f);

  /* add field char orderby<128>; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("orderby");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_char;
  f->format = g_strdup ("128");
  f->properties = ODL_PROP_NONE;
  f->defaultval = g_strdup ("");
  body = g_list_append (body, f);


  /* add field bool reverse; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("reverse");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_bool;
  f->properties = ODL_PROP_NONE;
  f->defaultval = g_strdup ("F");
  body = g_list_append (body, f);

  /* add field int logic; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("logic");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_int;
  f->properties = ODL_PROP_NONE;
  f->defaultval = g_strdup ("1");
  body = g_list_append (body, f);

  /* add fields to class */
  if (c->contents)
    {
      c->contents = g_list_concat (c->contents, body);
    }
  else
    {
      c->contents = body;
    }

  /* ************************************ */
  /* make class GEAS::searchfield */
  c = odl_new_class (ODL_ACCESS_SYSTEM, "searchfield");
  odl_container_insert_container (m, c);
  c->base.parent = (odl_base *) m;
  body = NULL;

  /* add field bool invert; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("invert");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_bool;
  f->properties = ODL_PROP_NONE;
  f->defaultval = g_strdup ("F");
  body = g_list_append (body, f);

  /* add field bool casesensitive; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("casesensitive");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_bool;
  f->properties = ODL_PROP_NONE;
  f->defaultval = g_strdup ("F");
  body = g_list_append (body, f);

  /* add field int test; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("test");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_unsignedint;
  f->properties = ODL_PROP_NONE;
  f->defaultval = g_strdup ("1");
  body = g_list_append (body, f);

  /* add field char field<128>; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("field");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_char;
  f->properties = ODL_PROP_NONE;
  f->format = g_strdup ("128");
  f->defaultval = g_strdup ("");
  body = g_list_append (body, f);

  /* add field char value<128>; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("value");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_char;
  f->properties = ODL_PROP_NONE;
  f->format = g_strdup ("128");
  f->defaultval = g_strdup ("");
  body = g_list_append (body, f);


  /* add field object constraint; */
  f = alloc_odl_item ();
  f->base.parent = (odl_base *) c;
  f->base.access = ODL_ACCESS_SYSTEM;
  f->base.name = g_strdup ("constraint");
  f->base.type = IT_field;
  f->fieldtype = FT_basic;
  f->datatype = DT_object;
  f->properties = ODL_PROP_NONE;
  body = g_list_append (body, f);

  /* add fields to class */
  if (c->contents)
    {
      c->contents = g_list_concat (c->contents, body);
    }
  else
    {
      c->contents = body;
    }
}

/* main function */
/** \brief Make private system classes
 * Calls a number of other functions to make classes used by different
 * parts of the system.
 */
static void
make_server_classes (void)
{
  odl_module *m;

  /* create GEAS module */
  m = odl_new_module (ODL_ACCESS_SYSTEM, "geas");
  if (m)
    {
      odl_container_insert_container (odl_tree_get_root (all_classes), m);
    }

  /* make classes */
  make_server_list_handling_classes (m);
  make_server_security_classes (m);
  make_server_transaction_classes (m);
  make_server_query_handling_classes (m);
}

#endif /* GEAS_INTERNAL_CLASSES_H */
