/* 
   Compare existing tables to class definition
   
   Part of GNU Enterprise Application Server (GEAS)

   Copyright (C) 2001 Free Software Foundation

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   $Id: compare.h,v 1.2 2001/07/25 12:51:08 reinhard Exp $
*/

/** \file classdata_database.h
 * \brief Do not include this directly
 * This contains function prototypes defines and structures used
 * by code dealing with converting classes to database tables
 */

#include <glib.h>
#include "classdata.h"
#include "oql/oql.h"

#ifndef CLASSDATA_DATABASE_H
#define CLASSDATA_DATABASE_H

/* store database related class data */
/* used for auto updating of database definitions, mostly */

#define DBCH_UNKNOWN          0
#define DBCH_ADD_TABLE        1
#define DBCH_MODIFY_TABLE     2
#define DBCH_ADD_COLUMN       3
#define DBCH_REMOVE_TABLE     4
#define DBCH_REMOVE_COLUMN    5

typedef struct
{
  int                   type;
  char                 *name;
  enum odl_datatype     datatype;
  char                 *format;
  gboolean              notnull;
  GList                *columns;
}
DBchange;

typedef struct
{
  GList                *adds;
  GList                *removes;
}
DatabaseChange;

typedef struct
{
  char                 *name;
  enum odl_datatype     datatype;       /** \brief GEAS datatype to store in this column */
}
DatabaseColumn;

typedef struct
{
  char                 *name;
  GList                *columns;
}
DatabaseTable;

typedef struct
{
  GList                *tables;
  char                 *name;
}
DatabaseDefinition;

void                  show_database_definition (DatabaseDefinition * def);

DatabaseDefinition   *create_database_definition (const char *name);
void                  free_database_definition (DatabaseDefinition * def);

gboolean              add_database_table (DatabaseDefinition * def,
                                          const char *name);
int                   count_database_tables (DatabaseDefinition * def);
DatabaseTable        *get_database_table (DatabaseDefinition * def,
                                          int which);
DatabaseColumn       *add_database_column (DatabaseTable * table,
                                           const char *name,
                                           enum odl_fieldtype type);

DatabaseChange       *create_database_change (void);
void                  free_database_change (DatabaseChange * change);

/** \brief Add a table and get ptr to DBchange for adding column info. */
DBchange             *dbchange_add_table (DatabaseChange * dbchange,
                                          const char *name);
/** \brief Remove a table. */
DBchange             *dbchange_remove_table (DatabaseChange * dbchange,
                                             const char *name);

/** \brief add a column to/from a table. */
DBchange             *dbchange_add_column (DBchange * change,
                                           const char *name,
                                           enum odl_fieldtype fieldtype,
                                           const char *format,
                                           gboolean notnull);
/** \brief Remove a column to/from a table. */
DBchange             *dbchange_remove_column (DBchange * change,
                                              const char *name);

/** \brief Scan a database definition and compare to the current class definitions. */
/* if there's a difference, produce a DatabaseChange object, which can be used to make */
/* database specific commands to update the database definitions */
DatabaseChange       *compare_classes_to_database (odl_tree * tree,
                                                   GList * requiredclasses,
                                                   DatabaseDefinition *
                                                   database, DBType db);

#endif
