/*
   mysql.c - Interface to MySql Database.
             ALL MySql code should go into this file.

   Copyright (C) 2001 Free Software Foundation

   This file is part of the GNU Enterprise Application Server (GEAS)

   GEAS is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GEAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GEAS; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   $Id: mysql.c,v 1.7 2001/07/25 15:40:18 reinhard Exp $
*/

#include "config.h"
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "objectstore.h"
#include "objectstore_private.h"
#include "geas-server.h"

#ifdef USE_MYSQL
#include <mysql.h>

/* Private Functions */
struct active_connection * MySQL_get_connection (struct database_handle *hnd);
void MySQL_notice_processor (void *arg, const char *msg);
void MySQL_add_column_data (GString *buf, GList *l, const char *separator);
struct query_result *
MySQL_update_tables (struct database_handle *hnd,
                          gboolean remove_items, int *errorcode,
                          char **errormsg);

struct MySQL_connection
{
  /* the first line must be: struct active_connection base; */
  struct active_connection base;

  /* any MySQL specific data goes here */
  MYSQL *handle;
};

struct MySQL_handle
{
  /* the first line must be: struct database_handle base; */
  struct database_handle base;

  /* any MySQL specific data goes here */
};

struct active_connection *
MySQL_get_connection (struct database_handle *hnd)
{
  struct MySQL_handle *h = (struct MySQL_handle *) hnd;
  struct active_connection *retval = NULL;
  /* struct active_connection *tmp = NULL; */
  unsigned int i;
  /* unsigned long int queue = UINT_MAX; */
  trace_functioncall ();

  /*
     find available connection:
     *    first available and connected connection
     *    else first available and non connected and connect it immediately
     *    else first connected with lowest queue 
     *    else fail
     */

  /* available and connected */
  /* TODO: make connection check depend on actual database state, to detect
     * unexpected disconnections */
  for (i = 0; i < h->base.num_connections; i++)
    {
      retval =
        (struct active_connection *) &((struct MySQL_connection *) h->
                                       base.connections)[i];
      if (retval->available && retval->connected)
        {
          retval->available = FALSE;
          return (retval);
        }
    }

  /* available and not connected, if connection succeeds */
  for (i = 0; i < h->base.num_connections; i++)
    {
      retval =
        (struct active_connection *) &((struct MySQL_connection *) h->
                                       base.connections)[i];
      if (retval->available && !retval->connected)
        {
          /* try to connect to database */
          if (hnd->connect (hnd, i))
            {
              return (retval);
            }
        }
    }

  fprintf (stderr, "\nTODO: handle no connection\n");
  fprintf (stderr,
           "needs to wait until a connection becomes available, as another\n");
  fprintf (stderr, "thread releases a connection\n\n");
  abort ();

  /* lowest queue */
  /* wait until the least used connection is available */
}

static gboolean
MySQL_connect (struct database_handle *hnd, unsigned long int index)
{
  /* struct MySQL_handle *h = (struct MySQL_handle *) hnd; */
  gboolean retval = FALSE;
  struct MySQL_connection *c;

  /* range check */
  if (index >= hnd->num_connections)
    {
      errormsg ("Invalid connection index");
      return (FALSE);
    }

  /* get connection */
  c =
    (struct MySQL_connection *) &((struct MySQL_connection *)
                                  hnd->connections)[index];

  /* TODO: thread safety */
  if (!c->base.connected)
    {
      c->handle = mysql_init (NULL);
      if (c->handle == NULL)
        {
          errormsg ("mysql_init() failed");
          return (FALSE);
        }

      if (!mysql_real_connect
          (c->handle, hnd->hostname, hnd->username, hnd->password,
           hnd->dbname, hnd->port, hnd->unixsocket, 0))
        {
          errormsg ("MySQL error: %s", mysql_error (c->handle));
          mysql_close (c->handle);
          c->handle = NULL;
          return (FALSE);
        }

      c->base.connected = TRUE;
      retval = TRUE;
    }
  return (retval);
}

static gboolean
MySQL_disconnect (struct database_handle *hnd, unsigned long int index)
{
  /* struct MySQL_handle *h = (struct MySQL_handle *) hnd; */
  struct MySQL_connection *c;

  /* range check */
  if (index >= hnd->num_connections)
    return (FALSE);

  /* get connection */
  c =
    (struct MySQL_connection *) &((struct MySQL_connection *)
                                  hnd->connections)[index];

  /* if not connected, 'succeed' at disconnecting */
  if (!c->base.connected)
    return (TRUE);

  /* TODO: thread safety */
  mysql_close (c->handle);
  c->handle = NULL;
  c->base.connected = FALSE;
  return (TRUE);
}

static struct query_result *
MySQL_execute_query (struct database_handle *ph,
                     QueryData * query, int *errorcode, char **errormsg)
{
  int dbtype;
  char *fieldname, *val;
  odl_class *c;

  /* result pointer */
  struct query_result *result = new_query_result ();

  /* convert to actual type */
  /* struct MySQL_handle *h = (struct MySQL_handle *) ph; */

  struct MySQL_connection *con;

  MYSQL_RES *res;
  MYSQL_ROW row;
  int err;

  /* claim a connection */
  con = (struct MySQL_connection *) ph->get_connection (ph);
  if (!con)
    {
      fprintf (stderr,
               "No available connection : should have aborted by now.\n");
      abort ();
    }

  if (!result)
    {
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup ("out of memory");
      return (NULL);
    }


  /* TODO: database dependant quote mode selection */
  dbtype = OQL_DBTYPE_MYSQL;
  debug_output (DEBUGLEVEL_HIGH, "SQL query: '%s'",
                oql_query_as_sql (query, dbtype));
  c = odl_find_class (all_classes, oql_query_get_classname (query), NULL);
  err = mysql_query (con->handle, oql_query_as_sql (query, dbtype));
  if (err == 0)
    {
      /* query appeared to succeed */
      res = mysql_store_result (con->handle);
      if (res == NULL)
        {
          if (mysql_field_count (con->handle) == 0)
            {
              /* query does not return rows, and didn't - ok */
              /* eg delete, insert, or replace SQL statemenmts */
              result->rows_affected = mysql_affected_rows (con->handle);
              result->success = TRUE;
            }
          else
            {
              /* should have returned rows of data, and didn't - error */
              con->base.available = TRUE;
              if (errorcode)
                *errorcode = (-1);      /* TODO: meaningful error codes */
              if (errormsg)
                *errormsg = g_strdup (mysql_error (con->handle));

              /* release this connection */
              con->base.available = TRUE;
              result->success = FALSE;
              return (result);
            }
        }
      else                      /* res != NULL */
        {
          /* returned some rows of data */
          result->rows_affected = 0;
          result->field_count = mysql_num_fields (res);
          result->success = TRUE;

          /* for each entry, extract object ID */
          row = mysql_fetch_row (res);
          while (row)
            {
              /* TODO: create list of objectIDs - need class
                 definitions */
              /* char *keydata = ""; */
              unsigned int i;
              DatabaseResultRow_t r = NULL;
              result->rows_affected++;

              /* the result set is a linked list of rows */
              /* each row is in turn a linked list of strings */
              /* where entry N is field N, listed left to right */
              /* in the SELECT query: */
              /* SELECT objectID,name,age : */
              /* objectID = field 0, name = field 1, age = field 2 */
              /* first row returned = row 0 (start of list), 2nd = row
                 1 (next in list) */
              for (i = 0; i < mysql_num_fields (res); i++)
                {
                  fieldname = (char *) oql_query_get_field_name (query, i);
                  val =
                    oql_translate_from_read (row[i], c, fieldname, dbtype);
                  /* add_field_to_result_row(r,val); */
                  r = g_list_append(r,val); /* add_field_to_result_row */
                }
              /* add entry to list */
              result->data = g_list_prepend (result->data, r);
              row = mysql_fetch_row (res);
            }
          mysql_free_result (res);
        }
    }
  else
    {
      /* error during query */
      /* message("error: ", mysql_error(con->handle)); */
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup (mysql_error (con->handle));
      result->success = FALSE;
    }
  /* release this connection */
  con->base.available = TRUE;
  result->data = g_list_reverse (result->data);
  return (result);
}

static struct query_result *MySQL_delete_object
  (struct database_handle *ph,
   const char *classname, const char *key, int *errorcode, char **errormsg)
{
  struct query_result *retval = NULL;
  QueryData *q = oql_delete_object (classname, key);

  trace_functioncall ();
  if (q)
    {
      retval = MySQL_execute_query (ph, q, errorcode, errormsg);
      oql_free_query (q);
    }
  else
    {
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup ("Could not create query");
    }
  return (retval);
}

static struct query_result *MySQL_delete_all_objects
  (struct database_handle *ph,
   const char *classname, const char *fieldname, const char *key,
   int *errorcode, char **errormsg)
{
  struct query_result *retval = NULL;
  QueryData *q = oql_delete_all_objects (classname, fieldname, key);

  trace_functioncall ();
  if (q)
    {
      retval = MySQL_execute_query (ph, q, errorcode, errormsg);
      oql_free_query (q);
    }
  else
    {
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup ("Could not create query");
    }
  return (retval);
}

static struct query_result *
MySQL_write_object (struct database_handle *ph, const gchar *classname,
                    const gchar *key, GHashTable *values, gboolean update,
                    int *errorcode, char **errormsg)
{
  struct query_result *retval = NULL;
  QueryData *q;
  q = oql_write_object (classname, key, values, update);
  if (q)
    {
      retval = MySQL_execute_query (ph, q, errorcode, errormsg);
      oql_free_query (q);
    }
  else
    {
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup ("Could not create write query");
    }
  return (retval);
}

/* free memory */
static void
MySQL_delete_database_handle (struct database_handle *hnd)
{
  unsigned int i;
  struct MySQL_handle *h = (struct MySQL_handle *) hnd;

  if (h)
    {
      free_generic_database (hnd);

      if (h->base.connections)
        {
          for (i = 0; i < h->base.num_connections; i++)
            {
              struct MySQL_connection *c
                =
                (struct MySQL_connection *) &((struct MySQL_connection *)
                                              hnd->connections)[i];

              if (c->handle)
                mysql_close (c->handle);
            }
          g_free (h->base.connections);
        }

      g_free (h);
    }
}

static DatabaseDefinition *
MySQL_read_database_definition (struct database_handle *hnd)
{
  struct MySQL_handle *h = (struct MySQL_handle *) hnd;
  MYSQL *handle;
  MYSQL_RES *res;
  MYSQL_ROW row;
  char *buf;
  /* GList *l; */
  int i;
  DatabaseDefinition *db = NULL;

  db = create_database_definition (h->base.name);
  if (!db)
    return (NULL);

  /* connect to database */
  handle = mysql_init (NULL);
  if (!handle)
    {
      errormsg ("Failed to initialise MySQL connection.");
      free_database_definition (db);
      return (NULL);
    }
  if (!mysql_real_connect
      (handle, hnd->hostname, hnd->username, hnd->password,
       hnd->dbname, hnd->port, hnd->unixsocket, 0))
    {
      errormsg ("Failed to connect to MySQL database %s.", hnd->name);
      mysql_close (handle);
      free_database_definition (db);
      return (NULL);
    }

  /* get list of tables */
  res = mysql_list_tables (handle, NULL);
  if (!res)
    {
      errormsg ("Failed to get MySQL table definitions from database %s.",
                hnd->name);
      mysql_close (handle);
      free_database_definition (db);
      return (NULL);
    }
  while ((row = mysql_fetch_row (res)) != NULL)
    {
      char *name;

      name = row[0];
      if (name[0] == '_' && name[1] == '_')
        name += 2;
      add_database_table (db, row[0]);
      /* message( " tab: %s" , row[0] ); */
    }

  /* for each table: */
  for (i = 0; i < count_database_tables (db); i++)
    {
      DatabaseTable *tab = get_database_table (db, i);
      /* DatabaseColumn *c; */
      enum odl_fieldtype type;
      int err;

      /* get list of fields and types */
      buf = g_strdup_printf ("SHOW COLUMNS FROM %s", tab->name);
      err = mysql_query (handle, buf);
      g_free (buf);
      if (err)
        {
          errormsg
            ("Failed to get MySQL table %s definition from database %s.",
             tab->name, hnd->name);
          mysql_close (handle);
          free_database_definition (db);
          return (NULL);
        }
      res = mysql_store_result (handle);
      if (!res)
        {
          errormsg ("Error storing query result");
          mysql_close (handle);
          free_database_definition (db);
          return (NULL);
        }

      if (mysql_num_rows (res) == 0)
        {
          /* no data in table? */
          message ("Warning: no fields in table %s in database %s",
                   tab->name, hnd->name);
        }
      else
        {
          /* store column details */
          while ((row = mysql_fetch_row (res)) != NULL)
            {
              char *t = strchr (row[1], '(');
              char *name;

              if (t)
                *t = '\0';

              /* information only, type is ignored */
              /* ie: accuracy is not critical */
              /* except when display will be too confusing */
              if (g_strcasecmp (row[1], "varchar") == 0)
                type = DT_char;
              else if (g_strcasecmp (row[1], "smallint") == 0)
                type = DT_int16;
              else if (g_strcasecmp (row[1], "int") == 0)
                type = DT_int32;
              else if (g_strcasecmp (row[1], "bigint") == 0)
                type = DT_int64;
              else if (g_strcasecmp (row[1], "tinytext") == 0)
                type = DT_text;
              else if (g_strcasecmp (row[1], "text") == 0)
                type = DT_text;
              else if (g_strcasecmp (row[1], "mediumtext") == 0)
                type = DT_text;
              else if (g_strcasecmp (row[1], "longtext") == 0)
                type = DT_text;
              else if (g_strcasecmp (row[1], "date") == 0)
                type = DT_date;
              else if (g_strcasecmp (row[1], "time") == 0)
                type = DT_time;
              else if (g_strcasecmp (row[1], "datetime") == 0)
                type = DT_datetime;
              else if (g_strcasecmp (row[1], "double") == 0)
                type = DT_float;
              else
                type = DT_unknown;

              name = row[0];
              if (name[0] == '_' && name[1] == '_')
                name += 2;

              add_database_column (tab, name, type);
            }
        }
    }
  /* close the database */
  mysql_close (handle);
  return (db);
}

void
MySQL_add_column_data (GString * buf, GList * l, const char *separator)
{
  char *name;
  DBchange *c;
  int len;

  while (l)
    {
      c = (DBchange *) l->data;
      name = oql_quote_column (NULL, c->name, OQL_DBTYPE_MYSQL);
      g_string_append (buf, name);
      g_free (name);
      g_string_append (buf, " ");
      switch (c->datatype)
        {
        case DT_char:
          g_string_sprintfa (buf, "VARCHAR");
          if (c->format)
            g_string_sprintfa (buf, "(%s)", c->format);
          else
            g_string_sprintfa (buf, "(1)");
          break;
        case DT_int16:
          g_string_sprintfa (buf, "SMALLINT");
          break;
        case DT_int32:
        case DT_int:
          g_string_sprintfa (buf, "INT");
          break;
        case DT_int64:
          g_string_sprintfa (buf, "BIGINT");
          break;
        case DT_boolean:
        case DT_bool:
          g_string_sprintfa (buf, "CHAR(1)");
          break;
        case DT_text:
          if (!c->format)
            g_string_sprintfa (buf, "TEXT");        /* <= 64k characters */
          else
            {
              len = atoi (c->format);
              if (len < 256)
                g_string_sprintfa (buf, "TINYTEXT");
              else if (len < 65536)
                g_string_sprintfa (buf, "TEXT");
              else if (len < 16777216)
                g_string_sprintfa (buf, "MEDIUMTEXT");
              else
                g_string_sprintfa (buf, "LONGTEXT");
            }
          break;
        case DT_date:
          g_string_sprintfa (buf, "DATE");
          break;
        case DT_time:
          g_string_sprintfa (buf, "TIME");
          break;
        case DT_datetime:
          g_string_sprintfa (buf, "DATETIME");
          break;
        case DT_float:
          g_string_sprintfa (buf, "DOUBLE");
          break;
        case DT_unsignedint:
          g_string_sprintfa (buf, "INT UNSIGNED");
          break;
        case DT_object:
          g_string_sprintfa (buf, "CHAR(32)");
          break;
        default:
          /* g_critical ("unknown data type: %d %s", c->datatype,
                      odl_datatype_name (c->datatype)); */
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
                 "unknown data type: %d %s", c->datatype,
                 odl_datatype_name (c->datatype));
          g_string_sprintfa (buf, "UNKNOWN_TYPE");
          break;
        }
      if (c->notnull)
        g_string_append (buf, " NOT NULL");
      else
        g_string_append (buf, " ");

      g_string_append (buf, separator);
      l = g_list_next (l);
    }
}

struct query_result *
MySQL_update_tables (struct database_handle *h,
                     gboolean remove_items, int *errorcode, char **errormsg)
{
  char *name;
  GString *buf = NULL;
  DatabaseChange *changes = NULL;
  GList *required = NULL;
  GList *tmp; 
  /* GList *l; */
  DatabaseDefinition *database = NULL;
  MYSQL *handle;
  /* MYSQL_RES *res; */
  /* MYSQL_ROW row; */
  int err;
  FILE *changefile;

  /* hard code this to OFF - is intended for remote admin */
  remove_items = FALSE;

  /* clear error indicators */
  if (errorcode)
    *errorcode = 0;
  if (errormsg)
    *errormsg = NULL;

  /* make a list of required classnames in this database */
  required = odl_tree_list_classes (all_classes);
  /* currently only a single SQL database is allowed at one time */
  /* so all classes are required */
  /* TODO: remove names from list if not required in */

     /* read current table definitions from database */
  database = MySQL_read_database_definition (h);
  /* show_database_definition( database ); */

  /* compare to current classes for this database */
  /* if table doesn't match a class, report the table as unnecessary     */
  /* if column doesn't match a class field, report it as unnecessary     */
  /* if class doesn't match a table, record the table as required        */
  /* if data field doesn't match a column, record the column as required */
  changes = compare_classes_to_database (all_classes, required, database,
                                         OQL_DBTYPE_MYSQL);
  odl_namelist_free (required);
  required = NULL;

  if (!changes)
    {
      /* no changes required!  yay!  */
      /* don't care about a result - error msgs alreayd printed */
      message ("No database changes.");
      return (NULL);
    }

  changefile =
    fopen (get_global_option_str
           (configdata, "databasechangefile", "database.changes.txt"), "a");
  if (!changefile)
    {
      errormsg ("Could not open log file '%s' for database changes.",
                get_global_option_str (configdata, "databasechangefile",
                                       "database.changes.txt"));
      return (NULL);
    }

  /* display remove suggestions */
  tmp = changes->removes;
  if (tmp)
    fprintf (changefile,
             "These changes should be made to database '%s', unless there has been an\n"
             "error in the configuration of GEAS:\n\n", h->name);
  while (tmp)
    {
      remove_message (changefile, (DBchange *) tmp->data);
      tmp = g_list_next (tmp);
    }
  fprintf (changefile, "\n\n");

  if (!changes->adds)
    {
      /* no additions, so don't both with the rest */
      fclose (changefile);
      return (NULL);
    }

  /* connect to database */
  handle = mysql_init (NULL);
  if (!handle)
    {
      errormsg ("Failed to initialise MySQL connection.");
      fclose (changefile);
      return (NULL);
    }
  if (!mysql_real_connect
      (handle, h->hostname, h->username, h->password,
       h->dbname, h->port, h->unixsocket, 0))
    {
      errormsg ("Failed to connect to MySQL database %s.", h->name);
      mysql_close (handle);
      fclose (changefile);
      return (NULL);
    }

  /* update tables on server */
  tmp = changes->adds;
  fprintf (changefile, "These changes are being made automatically:\n\n");
  while (tmp)
    {
      DBchange *add = (DBchange *) tmp->data;

      /* display add requirement */
      add_message (changefile, (DBchange *) tmp->data);

      if (add->type == DBCH_ADD_TABLE)
        {
          /* for each new table, construct a CREATE command */
          buf = g_string_new ("");
          name = oql_quote_column (add->name, NULL, OQL_DBTYPE_MYSQL);
          g_string_sprintf (buf, "CREATE TABLE %s (", name);
          g_free (name);
          MySQL_add_column_data (buf, add->columns, ", ");
          g_string_append (buf, "PRIMARY KEY (ObjectID) )");
        }
      else if (add->type == DBCH_MODIFY_TABLE)
        {
          /* for each table with new columns, construct a MODIFY command
           */
          buf = g_string_new ("");
          name = oql_quote_column (add->name, NULL, OQL_DBTYPE_MYSQL);
          g_string_sprintf (buf, "ALTER TABLE %s ADD ", name);
          g_free (name);
          MySQL_add_column_data (buf, add->columns, ", ADD ");
          buf->str[strlen (buf->str) - 5] = '\0';
          while (isspace (buf->str[strlen (buf->str) - 1]) ||
                 buf->str[strlen (buf->str) - 1] == ',')
            buf->str[strlen (buf->str) - 1] = '\0';
        }

      /* execute query */

      g_strdown (buf->str);
      /* printf( "CREATE/ALTER QUERY: [%s]\n" , buf->str ); */
      err = mysql_query (handle, buf->str);
      if (err == 0)
        {
          /* query appeared to succeed */
          message ("[%s] succeeded\n\n", buf->str);
        }
      else
        {
          /* TODO: error handling */
          message ("[%s] failed\n", buf->str);
          errormsg (mysql_error (handle));
          message ("\n");
        }

      tmp = g_list_next (tmp);
    }
  fprintf (changefile, "\n");
  mysql_close (handle);
  fclose (changefile);
  message ("Done updating database");

  /* don't care about a result - error msgs alreayd printed */
  return (NULL);
}

/* initialise a connection to a specific MySQL database */
struct database_handle *
MySQL_create_database_handle (configuration config, const char *dbname)
{
  unsigned int i;
  struct MySQL_handle *h =
    (struct MySQL_handle *) g_malloc (sizeof (struct MySQL_handle));

  if (h)
    {
      /* clear all pointers */
      /* none here */

      /* store general data */
      if (!generic_database_config
          (config, dbname, (struct database_handle *) h))
        {
          MySQL_delete_database_handle ((struct database_handle *) h);
          return (NULL);
        }

      /* h->whatever = MySQL specific options */
      /* if fail, delete handle and return NULL */

      /* store database operation functions */
      h->base.connect = MySQL_connect;
      h->base.disconnect = MySQL_disconnect;
      h->base.get_connection = MySQL_get_connection;
      h->base.execute = MySQL_execute_query;
      h->base.delete_database = MySQL_delete_database_handle;
      h->base.delete_object = MySQL_delete_object;
      h->base.delete_all_objects = MySQL_delete_all_objects;
      h->base.write_object = MySQL_write_object;
      h->base.update_tables = MySQL_update_tables;

      /* create connection data storage */
      h->base.connections =
        (struct active_connection *) g_new0 (struct MySQL_connection,
                                             h->base.num_connections);

      /* abort if resource acquisition failed */
      if (!h->base.connections /* || !other data */ )
        {
          MySQL_delete_database_handle ((struct database_handle *) h);
          return (NULL);
        }

      /* initialise connections */
      for (i = 0; i < h->base.num_connections; i++)
        {
          struct MySQL_connection *c
            =
            (struct MySQL_connection *) &((struct MySQL_connection *) h->base.
                                          connections)[i];

          /* set general state */
          c->base.available = TRUE;
          c->base.connected = FALSE;

          /* any MySQL specific data */
          c->handle = NULL;        /* no connection yet */
        }
    }
  return ((struct database_handle *) h);
}


#else

/* Dummy initialise a connection to a specific MySQL database */
struct database_handle *
MySQL_create_database_handle (configuration config, const char *dbname)
{
  g_assert_not_reached();
  return NULL;
}
#endif
