/*
   GNUE OID generator

   Copyright (C) 2001 Free Software Foundation

   This file is part of the GNU Enterprise Application Server (GEAS)

   GEAS is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GEAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GEAS; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   $Id: geas-oidgen.c,v 1.2 2001/08/23 16:21:16 reinhard Exp $
*/

#include "config.h"
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <uuid/uuid.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#include <unistd.h>
#endif

#define BUFFER_LENGTH 37

/* ------------------------------------------------------------------------- *\
 * Command line options
\* ------------------------------------------------------------------------- */

static long int opt_number = 25;

static char options_short[] = "n:hV";

#ifdef HAVE_GETOPT_LONG
static struct option options_long[] =
{
  { "number",  required_argument, NULL, 'n' },
  { "help",    no_argument,       NULL, 'h' },
  { "version", no_argument,       NULL, 'V' },
  { NULL }
};
#endif

/* ------------------------------------------------------------------------- *\
 * Output help list
\* ------------------------------------------------------------------------- */
static void
output_help (char *progname)
{
  printf ("Usage: %s [OPTION...]\n", progname);
  printf ("Generate OID's usable as primary key for new data records\n");
  printf ("\n");
#ifdef HAVE_GETOPT_LONG
  printf ("  -n, --number=NUM           Generate NUM OID's (default is 25)\n");
  printf ("  -h, --help                 Give this help list\n");
  printf ("  -V, --version              Print program version\n");
#else
  printf ("  -n NUM  Generate NUM OID's (default is 25)\n");
  printf ("  -h      Give this help list\n");
  printf ("  -V      Print program version\n");
#endif
  printf ("\n");
  printf ("Report bugs to bugs@gnue.org\n");
  exit (EXIT_SUCCESS);
}

/* ------------------------------------------------------------------------- *\
 * Output version information
\* ------------------------------------------------------------------------- */
static void
output_version (void)
{
  printf ("GEAS OID generator (GNU Enterprise Application Server) %s\n",
          VERSION);
  printf ("Copyright (c) 2001 Free Software Foundation\n");
  printf ("This is free software; see the source for copying conditions.\n");
  printf ("There is NO warranty; not even for MERCHANTABILITY\n");
  printf ("or FITNESS FOR A PARTICULAR PURPOSE.\n");
  exit (EXIT_SUCCESS);
}

/* ------------------------------------------------------------------------- *\
 * Parse command line options
\* ------------------------------------------------------------------------- */
static void
parse_options (int argc, char *argv[])
{
  char c;
#ifdef HAVE_GETOPT_LONG
  int i;

  while ((c = getopt_long (argc, argv, options_short, options_long, &i)) != -1)
#else
  while ((c = getopt (argc, argv, options_short)) != -1)
#endif
    {
      switch (c)
        {
        case 'n':
          opt_number = atoi (optarg);
          if (!opt_number)
            {
              fprintf (stderr, "%s: invalid number -- %s\n", argv[0], optarg);
              exit (EXIT_FAILURE);
            }
          break;
        case 'h':
          output_help (argv[0]);
          break;
        case 'V':
          output_version ();
          break;
        default:
          exit (EXIT_FAILURE);
        }
    }
}

/* ------------------------------------------------------------------------- *\
 * Change formatting from UUID style to OID style
\* ------------------------------------------------------------------------- */
static void
cleanup_oid (char *oid)
{
  char *p;

  /* convert to lower case */
  for (p = oid; p && *p != '\0'; p++)
    *p = tolower (*p);

  /* remove '-' characters */
  p = oid;
  while (p && *p != '\0')
    {
      while (*p == '-')
	memmove ((char *) p, (const char *) (p + 1), strlen (p));
      p++;
    }
}

/* ------------------------------------------------------------------------- *\
 * Main program
\* ------------------------------------------------------------------------- */
int
main (int argc, char *argv[])
{
  int i;
  uuid_t a_uuid;
  static char buf[BUFFER_LENGTH];
  
  parse_options (argc, argv);

  for (i = 0; i < opt_number; i++)
    {
      uuid_generate (a_uuid);
      uuid_unparse (a_uuid, buf);
      cleanup_oid (buf);
      printf ("%s\n", buf);
    }
  exit (EXIT_SUCCESS);
}
