/* $Id: xgifload.c,v 1.2 1994/05/25 13:44:12 bhalla Exp $ */
/* $Log: xgifload.c,v $
 * Revision 1.2  1994/05/25 13:44:12  bhalla
 * Uses SearchForNonScript instead of fopen for opening the text file. This
 * lets it find files anywhere on the genesis search path
 *
 * Revision 1.1  1994/03/22  15:24:32  bhalla
 * Initial revision
 * */
/*
 *
 * This file is from the xgif set of routines by John Bradley, U. Penn.
 * I have made a few minor changes, that is all.
 *		Upi Bhalla.
 *
 * Jason has made some additional changes to xgif that cleans it up
 * and fixes some of the colormap problems.  He also removed
 * Upi's black and white handling algorithm and installed a cleaner
 * thresholding algorithm.
 * It would be better to write a dithering algorithm for black and white
 * but time did not permit.  8/19/93
 *
 * GIF files have provisions for local as well as global colormaps
 * for an image.  With local colormaps a GIF file can have more
 * than 1 image.  In this implementation we are assuming
 * that only 1 image is in a gif file and local color maps
 * are not supported.
 *
 * xgifload.c  -  based strongly on...
 *
 * gif2ras.c - Converts from a Compuserve GIF (tm) image to a Sun Raster image.
 *
 * Copyright (c) 1988, 1989 by Patrick J. Naughton
 *
 * Author: Patrick J. Naughton
 * naughton@wind.sun.com
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 */

/* include files */
#include <stdio.h>
#include <math.h>
#include <ctype.h>

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

typedef unsigned char byte;

#define CENTERX(f,x,str) ((x)-XTextWidth(f,str,strlen(str))/2)
#define CENTERY(f,y) ((y)-((f->ascent+f->descent)/2)+f->ascent)
#define FatalError(str) {fprintf(stderr,"GIF load: %s",str); return(NULL);}

typedef int boolean;

/*#define NEXTBYTE (*ptr++) */
#define IMAGESEP 0x2c
#define INTERLACEMASK 0x40
#define COLORMAPMASK 0x80

FILE *fp;

int
    RWidth, RHeight,		/* screen dimensions */
    Width, Height,		/* image dimensions */
    LeftOfs, TopOfs,		/* image offset */
    BitsPerPixel,		/* Bits per pixel, read from GIF header */
    BytesPerScanline,		/* bytes per scanline in output raster */
    ColorMapSize,		/* number of colors */
    numcols,			/* number of colors */
    Background,			/* background color */
    CodeSize,			/* Code size, read from GIF header */
    InitCodeSize,		/* Starting code size, used during Clear */
    Code,			/* Value returned by ReadCode */
    MaxCode,			/* limiting value for current code size */
    ClearCode,			/* GIF clear code */
    EOFCode,			/* GIF end-of-information code */
    CurCode, OldCode, InCode,	/* Decompressor variables */
    FirstFree,			/* First free code, generated per GIF spec */
    FreeCode,			/* Decompressor, next free slot in hash
				 * table */
    FinChar,			/* Decompressor variable */
    BitMask,			/* AND mask for data size */
    ReadMask;			/* Code AND mask for current code size */

unsigned long cols[256];
XColor defs[256];

/* nostrip flags whether color approximations are done by stripping bits
 * or not.  nostrip means color approximation is by choosing nearest color
 * in the color map.
 * stripping means stripping down the colormap (bit by bit) to accomodate
 * the colors. By default I am setting it as non-stripping.  This may
 * be implemented later as a flag for the user the choose but for now it
 * will be force-set to non-stripping.
 */
int strip = 0, nostrip = 1;
int dispcells;			/* number of display cells in the current
				 * display's colormap.
				 */
Colormap theCmap;

boolean Interlace, HasColormap;
boolean Verbose = False;

byte *Image;			/* The result array */
byte *RawGIF;			/* The heap array to hold it, raw */
byte *Raster;			/* The raster data stream, unblocked */


 /* The color map, read from the GIF header */
byte Red[256], Green[256], Blue[256], used[256];
int numused;

char *id = "GIF87a";

static byte *filedataptr;
static int filesize;
byte ReverseBits();

/*****************************/
XImage *
UpiLoadGIF(fname, theDisp, theScreen)
char *fname;
Display *theDisp;
int theScreen;

/*****************************/
{
	int BitOffset = 0;	/* Bit Offset of next code */
	int XC = 0, YC = 0;	/* Output X and Y coords of current pixel */
	int Pass = 0;		/* Used by output routine if interlaced
				 * pic */
	int OutCount = 0;	/* Decompressor output 'stack count' */
	register byte ch, ch1;
	register byte  *ptr1;
	register int i;
	XImage *theImage;
	Visual *theVisual;
	int theDepth;

	/* The hash table used by the decompressor */
	int Prefix[4096];
	int Suffix[4096];

	/* An output array used by the decompressor */
	int OutCode[1025];

	/* Redeclaration of the Genesis shell utility func */
	FILE *SearchForNonScript();

	theVisual = XDefaultVisual(theDisp, theScreen);
	theCmap = DefaultColormap(theDisp, theScreen);
	theDepth = XDefaultDepth(theDisp, theScreen);
	dispcells = DisplayCells(theDisp, theScreen);


	if (strcmp(fname, "-") == 0) {
		fp = stdin;
		fname = "<stdin>";
	}
	else
		fp = SearchForNonScript(fname, "r");

	if (!fp)
		FatalError("file not found");

	/* find the size of the file */
	fseek(fp, 0L, 2);
	filesize = ftell(fp);
	fseek(fp, 0L, 0);


	/*******************************************************************************************/
	/* First read the GIF header information. */

	if (!(filedataptr = RawGIF = (byte *) malloc(filesize)))
		FatalError("not enough memory to read gif file");

	if (!(Raster = (byte *) malloc(filesize)))
		FatalError("not enough memory to read gif file");

	if (fread(filedataptr, filesize, 1, fp) != 1)
		FatalError("GIF data read failed");

	if (strncmp((char *)filedataptr, id, 6))
		FatalError("not a GIF87a format file");

	filedataptr += 6;

	/* Get variables from the GIF screen descriptor */

	ch = ((byte) get_byte());
	RWidth = ch + 0x100 * ((byte) get_byte());	/* screen dimensions... not used. */
	ch = ((byte) get_byte());
	RHeight = ch + 0x100 * ((byte) get_byte());

	if (Verbose)
		fprintf(stderr, "screen dims: %dx%d.\n", RWidth, RHeight);

	ch = ((byte) get_byte());
	HasColormap = ((ch & COLORMAPMASK) ? True : False);
	/* I.e. global color map follows descriptor */

	BitsPerPixel = (ch & 7) + 1;
	numcols = ColorMapSize = 1 << BitsPerPixel;
	BitMask = ColorMapSize - 1;

	Background = ((byte) get_byte());		/* background color... not used. */

	if (((byte) get_byte()))			/* supposed to be NULL */
		FatalError("corrupt GIF file (bad screen descriptor)");


	/* Read in global colormap. */

	if (HasColormap) {
		if (Verbose)
			fprintf(stderr, "%s is %dx%d, %d bits per pixel, (%d colors).\n",
			fname, Width, Height, BitsPerPixel, ColorMapSize);
		for (i = 0; i < ColorMapSize; i++) {
			Red[i] = ((byte) get_byte());
			Green[i] = ((byte) get_byte());
			Blue[i] = ((byte) get_byte());
			used[i] = 0;
		}
		numused = 0;

	}

	/* No GLOBAL colormap, does Upi look for a local colormap? Nope he doesn't */
	else {				/* no colormap in GIF file */
		fprintf(stderr, "warning!  no colortable in this file.  Winging it.\n");
		if (!numcols)
			numcols = 256;
		for (i = 0; i < numcols; i++)
			cols[i] = (unsigned long) i;
	}

	/* Check for image separator */

	if (((byte) get_byte()) != IMAGESEP)
		FatalError("corrupt GIF file (no image separator)");

	/* Now read in values from the image descriptor */

	ch = ((byte) get_byte());

	/* Start of image in pixels from the left side of the screen */
	LeftOfs = ch + 0x100 * ((byte) get_byte());
	ch = ((byte) get_byte());

	TopOfs = ch + 0x100 * ((byte) get_byte());
	ch = ((byte) get_byte());

	Width = ch + 0x100 * ((byte) get_byte());
	ch = ((byte) get_byte());

	Height = ch + 0x100 * ((byte) get_byte());

	/* Check interlace bit */
	Interlace = ((((byte) get_byte()) & INTERLACEMASK) ? True : False);


	if (Verbose)
		fprintf(stderr, "Reading a %d by %d %sinterlaced image...",
			Width, Height, (Interlace) ? "" : "non-");
/*
	else
		fprintf(stderr, "%s is %dx%d, %d colors  ",
			fname, Width, Height, ColorMapSize);
*/

	/* Note that I ignore the possible existence of a local color map.
	 * I'm told there aren't many files around that use them, and the spec
	 * says it's defined for future use.  This could lead to an error
	 * reading some files.
	 */

	if (*filedataptr & COLORMAPMASK)
		FatalError("cannot support images with local colormaps");

	/*******************************************************************************************/

	/* Start reading the raster data. First we get the intial code size
	 * and compute decompressor constant values, based on this code size.
	 */

	CodeSize = get_byte();
	ClearCode = (1 << CodeSize);
	EOFCode = ClearCode + 1;
	FreeCode = FirstFree = ClearCode + 2;

/* The GIF spec has it that the code size is the code size used to
 * compute the above values is the code size given in the file, but the
 * code size used in compression/decompression is the code size given in
 * the file plus one. (thus the ++).
 */

	CodeSize++;
	InitCodeSize = CodeSize;
	MaxCode = (1 << CodeSize);
	ReadMask = MaxCode - 1;

/* Read the raster data.  Here we just transpose it from the GIF array
 * to the Raster array, turning it from a series of blocks into one long
 * data stream, which makes life much easier for ReadCode().
 */

	ptr1 = Raster;
	do {
		ch = ch1 = get_byte();
		while (ch--)
			*ptr1++ = get_byte();
		if ((Raster - ptr1) > filesize)
			FatalError("corrupt GIF file (unblock)");
	} while (ch1);

	free(RawGIF);			/* We're done with the raw data
					 * now... */

	if (Verbose) {
		fprintf(stderr, "done.\n");
		fprintf(stderr, "Decompressing...");
	}


	/* Allocate the X Image */
	Image = (byte *) malloc(Width * Height);
	if (!Image)
		FatalError("not enough memory for XImage");

	theImage = XCreateImage(theDisp, theVisual, theDepth, ZPixmap, 0, Image,
				Width, Height, 8, Width);
	if (!theImage)
		FatalError("unable to create XImage");

	BytesPerScanline = Width;


	/* Decompress the file, continuing until you see the GIF EOF code.
	 * One obvious enhancement is to add checking for corrupt files here.
	 */

	Code = ReadCode(&BitOffset);
	while (Code != EOFCode) {

		/* Clear code sets everything back to its initial value, then reads the
		 * immediately subsequent code as uncompressed data.
		 */

		if (Code == ClearCode) {
			CodeSize = InitCodeSize;
			MaxCode = (1 << CodeSize);
			ReadMask = MaxCode - 1;
			FreeCode = FirstFree;
			CurCode = OldCode = Code = ReadCode(&BitOffset);
			FinChar = CurCode & BitMask;
			AddToPixel(FinChar, &XC, &YC, &Pass);
		}
		else {

			/* If not a clear code, then must be data: save same as CurCode and InCode */

			CurCode = InCode = Code;

			/* If greater or equal to FreeCode, not in the hash table yet;
			 * repeat the last character decoded
			 */

			if (CurCode >= FreeCode) {
				CurCode = OldCode;
				OutCode[OutCount++] = FinChar;
			}

			/* Unless this code is raw data, pursue the chain pointed to by CurCode
			 * through the hash table to its end; each code in the chain puts its
			 * associated output code on the output queue.
			 */

			while (CurCode > BitMask) {
				if (OutCount > 1024) {
					fprintf(stderr, "\nCorrupt GIF file (OutCount)!\n");
					_exit(-1);	/* calling 'exit(-1)'
							 * dumps core, so I
							 * don't */
				}
				OutCode[OutCount++] = Suffix[CurCode];
				CurCode = Prefix[CurCode];
			}

			/* The last code in the chain is treated as raw data. */

			FinChar = CurCode & BitMask;
			OutCode[OutCount++] = FinChar;

			/* Now we put the data out to the Output routine.
			 * It's been stacked LIFO, so deal with it that way...
			 */

			for (i = OutCount - 1; i >= 0; i--)
				AddToPixel(OutCode[i], &XC, &YC, &Pass);
			OutCount = 0;

			/* Build the hash table on-the-fly. No table is stored in the file. */

			Prefix[FreeCode] = OldCode;
			Suffix[FreeCode] = FinChar;
			OldCode = InCode;

			/* Point to the next slot in the table.  If we exceed the current
			 * MaxCode value, increment the code size unless it's already 12.  If it
			 * is, do nothing: the next code decompressed better be CLEAR
			 */

			FreeCode++;
			if (FreeCode >= MaxCode) {
				if (CodeSize < 12) {
					CodeSize++;
					MaxCode *= 2;
					ReadMask = (1 << CodeSize) - 1;
				}
			}
		}
		Code = ReadCode(&BitOffset);
	}

	free(Raster);

/*
	if (Verbose)
		fprintf(stderr, "done.\n");
	else
		fprintf(stderr, "(of which %d are used)\n", numused);
*/

	if (fp != stdin)
		fclose(fp);

	ColorDicking(fname, theDisp, theScreen, theDepth);

	return (theImage);
}


/* Fetch the next code from the raster data stream.  The codes can be
 * any length from 3 to 12 bits, packed into 8-bit bytes, so we have to
 * maintain our location in the Raster array as a BIT Offset.  We compute
 * the byte Offset into the raster array by dividing this by 8, pick up
 * three bytes, compute the bit Offset into our 24-bit chunk, shift to
 * bring the desired code to the bottom, then mask it off and return it.
 */
ReadCode(BitOffset)
int *BitOffset;
{
	int RawCode, ByteOffset;

	ByteOffset = (*BitOffset) / 8;
	RawCode = Raster[ByteOffset] + (0x100 * Raster[ByteOffset + 1]);
	if (CodeSize >= 8)
		RawCode += (0x10000 * Raster[ByteOffset + 2]);
	RawCode >>= ((*BitOffset) % 8);
	*BitOffset += CodeSize;
	return (RawCode & ReadMask);
}


AddToPixel(Index, pXC, pYC, Pass)
byte Index;
int *pXC, *pYC;
int *Pass;
{
	int XC = *pXC, YC = *pYC;

	if (YC < Height)
		*(Image + YC * BytesPerScanline + XC) = Index;

	if (!used[Index]) {
		used[Index] = 1;
		numused++;
	}

/* Update the X-coordinate, and if it overflows, update the Y-coordinate */

	if (++XC == Width) {

/* If a non-interlaced picture, just increment YC to the next scan line.
 * If it's interlaced, deal with the interlace as described in the GIF
 * spec.  Put the decoded scan line out to the screen if we haven't gone
 * past the bottom of it
 */

		XC = 0;
		if (!Interlace)
			YC++;
		else {
			switch (*Pass) {
			case 0:
				YC += 8;
				if (YC >= Height) {
					(*Pass)++;
					YC = 4;
				}
				break;
			case 1:
				YC += 8;
				if (YC >= Height) {
					(*Pass)++;
					YC = 2;
				}
				break;
			case 2:
				YC += 4;
				if (YC >= Height) {
					(*Pass)++;
					YC = 1;
				}
				break;
			case 3:
				YC += 2;
				break;
			default:
				break;
			}
		}
	}
	*pXC = XC;
	*pYC = YC;
}


/*
 * We've got the picture loaded, we know what colors are needed.
 * get 'em
 * First this function checks to see if the X window display is a monochrome
 * display.  If so then it will configure the image as a b/w image.
 * If not then an attempt will be made to allocate the colors in the
 * gif image to the display.  If exact colors cannot be allocated then
 * the closest colors are chosen.
 */
ColorDicking(fname, theDisp, theScreen, theDepth)
char *fname;
Display *theDisp;
int theScreen;
int theDepth;
{



	register int i, j, k, temp;
	int bitOrder;

	/* Masks for color stripping as a means to allocate colors */
	static byte lmasks[8] = {0xff, 0xfe, 0xfc, 0xf8, 0xf0, 0xe0, 0xc0, 0x80};
	byte lmask, *ptr;

	/* no need to allocate any colors if no colormap in GIF file */
	if (!HasColormap) return;


	/*******************************************************************************************/

	/* If the bitplane depth is 1 then the X display is monochrome. */
	if (theDepth == 1) {

		int chosenThreshold;
		int total;
		byte *bitptr, *byteptr, bitColor;
		int bitIncr;

		/* The histogram array is used to keep a count of the number of
		 * colors of a particular 'intensity' is used.
		 * intensity is given as the sum of the color's red, green, and blue components.
		 * This is not strictly true relative to how light really works
		 * but it is an effective compromise that's simple to implement.
		 */
		unsigned long histogram[768]; /* 768 = 256 * 3*/

		unsigned long i;
		int numUsed = 0;	/* Number of colors actually used */

		/* Get the way the bits are ordered in an X bitmap */
		bitOrder = XBitmapBitOrder(theDisp);

		/* Clear out the histogram */
		for (i = 0; i < 768; i++) histogram[i] = 0;

		/* Use histogram to find a reasonable threshold by going through
		 * all the colors in the colormap and recording their intensities.
		 * But you need only take into account those colors that actually
		 * get used in the GIF image.  Sometimes the GIF file will define
		 * colors for a colormap that won't get used in the image itself.
		 */
		for (i = 0; i < numcols; i++) {
			if (used[i]) {
				histogram[Red[i] + Green[i] + Blue[i]]++;
				numUsed++;
			}
		}

		/* Find halfway index; i.e. divide the number of colors used
		 * into 2 so that one half can be assigned black and the other
		 * as white.  This division will allow us to establish a
		 * threshold.  I realize that a dithering algorithm would be
		 * better but time did not permit.
		 */
		total = 0;
		i = 0;
		while(total < (int) (((float) numUsed)/2.0)) {
			total += histogram[i];
			i++;
		}

		if (i)	chosenThreshold = i-1;
		else chosenThreshold = 0;


		/* Note: Any sum of r,g,b > chosenThreshold is considered White
		 * else black.
		 */

		/* Now go through each byte in the Image array and
		 * find out what it's RGB value is and determine if it
		 * should be a black or white pixel according to the
		 * thresold.  Then reassign it to the image array.
		 * Note: you will have to take eight 1 byte color pixel
		 * values and compress it into 1 single byte representing
		 * the black and white bitmap.
		 */

		bitptr = Image;
		byteptr = Image;
		bitIncr = 0;


		/* Notice there is 2 for loops.  This is necessary because
		 * the number of bytes needed to represent a monochrome bitmap is
		 * less than that of a color image. And we are essentially writing
		 * over the color image memory with the black/white bitmap data.
		 * Hence there is a lot of memory that will be unused.  Now the
		 * 2 loops are necessary because inthe XCreateImage we set the
		 * bytes_per_line to Width.  Hence at the beginning of each line
		 * we have to set the bitptr to the byteptr's location on the new
		 * line.  This would not have been necessary if we had created
		 * an image that contained contiguous bytes. I.e. by setting
		 * bytes_per_line to 0 in the XCreateImage call instead of Width.
		 * I don't know why Upi didn't do that in the first place.
		 */
		for (i = 0; i < Height; i++) {

			/* Here the bitptr must be set to the beginning of each line */
			bitptr = byteptr;

			/* Used to keep track of how many bits we have assigned so far
			 * in a byte of a b/w bitmap.
			 */
			bitIncr = 0;

			for (j = 0; j < Width ; j++) {

				/* Here's the thresholding in action to choose between white or black. */
				if ((int)(Red[*byteptr] + Green[*byteptr] + Blue[*byteptr]) > chosenThreshold)
			      bitColor = 0;
				else bitColor = 1;

				/* First byte always gets zeroed out to start with */
				if (j == 0) *bitptr = 0;

				/* Shift the bits over and add the bitcolor to it. */
				*bitptr = *bitptr << 1;
				*bitptr += bitColor;
				bitIncr++;

				/* If 1 byte is filled then... */
				if (bitIncr == 8) {

					/* Check to see if the bitorder needs reversing. */
					if (bitOrder != MSBFirst) {

						/* Reverse the bits */
						*bitptr = ReverseBits(*bitptr);
					}

					/* Reset bit incrementor and move to next byte in bitmap */
					bitIncr = 0; bitptr++; *bitptr = 0;
				}

				/* Go to next byte (color pixel) in the image */
				byteptr++;
			}

			/* If the loop is terminated then check to see if we
			 * were in the middle of filling a byte.  If so
			 * the effect a reversal if necessary.
			 */
			if (bitIncr && (bitOrder != MSBFirst)) *bitptr = ReverseBits(*bitptr);
		}

		/* Done with black and white bitmap. Return. */
		return;
	}





	/*******************************************************************************************/

	/* If you are viewing image from a color display, allocate the X colors for this picture */

	/* Allocate colors without stripping down bits in the colors; that's the forced default. */
	if (nostrip) {
		for (i = j = 0; i < numcols; i++) {
			if (used[i]) {

				/* Select the desired color */
				defs[i].red = Red[i] << 8;
				defs[i].green = Green[i] << 8;
				defs[i].blue = Blue[i] << 8;
				defs[i].flags = DoRed | DoGreen | DoBlue;

				/* Ask for color.  Returns color index value
				 * in defs[i].pixel. 
				 */
				if (!XAllocColor(theDisp, theCmap, &defs[i])) {
					/* If unable to allocate color then make a note
					 * of it.
					 */
					j++;
					defs[i].pixel = 0xffff;
				}
				cols[i] = defs[i].pixel;
			}
		}

		/* If there are unallocated colors then try and allocate something close. */
		if (j) {
			XColor ctab[256];
			int dc;

			dc = (dispcells < 256) ? dispcells : 256;
			fprintf(stderr, "unable to allocate %d out of %d colors.  Trying extra hard.\n",
				j, numused);

			/* read in the current color table from the display */
			for (i = 0; i < dc; i++)
				ctab[i].pixel = i;
			XQueryColors(theDisp, theCmap, ctab, dc);

			/*
			 * run through the used colors.  any used color
			 * that has a pixel value of 0xffff wasn't
			 * allocated.  for such colors, run through the
			 * entire X colormap and pick the closest color
			 */

			for (i = 0; i < numcols; i++)
				if (used[i] && cols[i] == 0xffff) {	/* an unallocated pixel */
					int d, mdist, close;
					unsigned long r, g, b;


					close = -1;
					r = Red[i];
					g = Green[i];
					b = Blue[i];

					/* Start out by settting minimum distance
					 * to the first color's distance from
					 * the desired color.
					 */
					mdist = abs(r - (ctab[0].red >> 8)) +
							abs(g - (ctab[0].green >> 8)) +
							abs(b - (ctab[0].blue >> 8));

					for (j = 0; j < dc; j++) {
						d = abs(r - (ctab[j].red >> 8)) +
							abs(g - (ctab[j].green >> 8)) +
							abs(b - (ctab[j].blue >> 8));
						if (d <= mdist) {
							mdist = d;
							close = j;
						}
					}
					if (close < 0)
						FatalError("simply can't do it.  Sorry.");

					/* Otherwise copy that color */
					bcopy(&defs[close], &defs[i], sizeof(XColor));
					cols[i] = ctab[close].pixel;
				}
		}
	}

	/*******************************************************************************************/
	
	/* Alternatively use stripping to allocate colors. This has been turned off by the
	 * nostrip variable being set to 1.  Perhaps later we can use this as a user
	 * controllable option.
	 */

	else {
		/* stripping strips off bits at a time if the system cannot allocate
		 * the colors it wants.  Basically stripping zeros out the LSB's
		 * of the R,G and B values one at a time until a color can be allocated.
		 */
		j = 0;
		strip = 0;	/* Starting strip at zero means no stripping */
		while (strip < 8) {

			/* Strip masks */
			lmask = lmasks[strip];
			for (i = 0; i < numcols; i++)
				if (used[i]) {
					defs[i].red = (Red[i] & lmask) << 8;
					defs[i].green = (Green[i] & lmask) << 8;
					defs[i].blue = (Blue[i] & lmask) << 8;
					defs[i].flags = DoRed | DoGreen | DoBlue;
					if (!XAllocColor(theDisp, theCmap, &defs[i]))
						break;
					cols[i] = defs[i].pixel;
				}

			/* If failed color allocation then free up the colors,
			 * and strip down the bits using the next strip mask and try again.
			 */
			if (i < numcols) {	/* failed */
				strip++;
				j++;
				for (i--; i >= 0; i--)
					if (used[i])
						XFreeColors(theDisp, theCmap, cols + i, 1, 0L);
			}
			else
				break;
		}

		if (j && strip < 8)
			fprintf(stderr, "%s stripped %d bits\n", fname, strip);


		/* If all strip masks have been applied then give up */
		if (strip == 8) {
			fprintf(stderr, "UTTERLY failed to allocate the desired colors.\n");

			for (i = 0; i < numcols; i++)
				cols[i] = i;
		}
	}

		

	/*******************************************************************************************/

	/* Cols has been accumulating the newly allocated color maps.
	 * Now using cols, convert the values in the image to those
	 * that refer to the pixel values in cols.
	 */
	ptr = Image;
	for (i = 0; i < Height; i++) {
		for (j = 0; j < Width; j++, ptr++)
				*ptr = (byte) cols[*ptr];
	}
}

int get_byte()
{
	byte datum;

	if ((filedataptr - RawGIF) > (long) filesize) {
		printf("get_byte read error %ld %d\n",
		       filedataptr-RawGIF, filesize);
		return -1;
	}
	datum = *filedataptr++;
	return (int) datum;

}

/* Take the bits in a byte and reverse them (in the lisp sense,
 * i.e. not invert, but reverse).
 */
byte ReverseBits(temp)
byte temp;
{
	byte t2;
	int k;
	t2 = 0;
	for (k = 0; k < 8; k++) {
		if (temp & (1 << k))
	      t2 |= 1 << (7 - k);
	}
	return t2;
}
