// MM2ENG.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm2eng.h"

#include <fstream>
#include <strstream>
#include <algorithm>
using namespace std;

/*################################################################################################*/

#define NBTP_CSIGN  		0x01
#define NBTP_CHARGED		0x02

#define NBTP_POLAR  		0x04
#define NBTP_NEUTRAL		0x08

// the surface area code apparently contains some bugs, since it sometimes
// crashes. another possibility is that the surface area math contains some
// bad cases (like arcs/segments with zero length/area ???) which should be
// avoided somehow. either way, the numerical and analytical gradients of
// surface area seem to match.

// LOWLIMIT is a cheat to prevent zero-divisions in surface-area calculations.
// if a zero-division seems to happen, the values are just changed to LOWLIMIT
// in as early stage as possible, thus making minimum effects on results...

#define LOWLIMIT 0.0000001	// 0.0000001 seems to work quite well...

/*################################################################################################*/

void CopyCRD(mm2_mdl * p1, mm2_eng * p2, i32u p3)
{
	for (i32u n1 = 0;n1 < p2->index_vector.size();n1++)
	{
		mm2_atm tmp = p2->index_vector[n1];
		p2->crd[n1][0] = p1->chn_vector[tmp[0]].res_vector[tmp[1]].crd_vector[tmp[2]][p3][0];
		p2->crd[n1][1] = p1->chn_vector[tmp[0]].res_vector[tmp[1]].crd_vector[tmp[2]][p3][1];
		p2->crd[n1][2] = p1->chn_vector[tmp[0]].res_vector[tmp[1]].crd_vector[tmp[2]][p3][2];
	}
	
	for (i32u n1 = 0;n1 < p2->bt3_vector.size();n1++)
	{
		mm2_atm tmp = p2->index_vector[p2->bt3_vector[n1].atmi[1]];
		p2->bt3_vector[n1].pbdd = p1->chn_vector[tmp[0]].res_vector[tmp[1]].peptide_vector[p3];
	}
}

void CopyCRD(mm2_eng * p1, mm2_mdl * p2, i32u p3)
{
	for (i32u n1 = 0;n1 < p1->index_vector.size();n1++)
	{
		mm2_atm tmp = p1->index_vector[n1];
		p2->chn_vector[tmp[0]].res_vector[tmp[1]].crd_vector[tmp[2]][p3][0] = p1->crd[n1][0];
		p2->chn_vector[tmp[0]].res_vector[tmp[1]].crd_vector[tmp[2]][p3][1] = p1->crd[n1][1];
		p2->chn_vector[tmp[0]].res_vector[tmp[1]].crd_vector[tmp[2]][p3][2] = p1->crd[n1][2];
	}
	
	for (i32u n1 = 0;n1 < p1->bt3_vector.size();n1++)
	{
		mm2_atm tmp = p1->index_vector[p1->bt3_vector[n1].atmi[1]];
		p2->chn_vector[tmp[0]].res_vector[tmp[1]].peptide_vector[p3] = p1->bt3_vector[n1].pbdd;
	}
}

/*################################################################################################*/

// here we are dependent on secondary structure constraints -> if secondary structure is modified,
// the engine class must be discarded and a new one must be created to take effects into account...

// IMPORTANT!!! constraints are no longer a part of energy calculations (since they practically affect them,
// due to rounding errors)!!! you have to add constraints to energy to get the total energy!!!

mm2_eng::mm2_eng(mm2_mdl & p1) : engine(), mdl(p1)
{
	constraints = 0.0;
	
// constraints (not dependent on 3D-structure) should be outside the energy calculations, since they affect in all practical
// calculations due to rounding errors in intermediate results!!! originally, also an initial set of constraint parameters were
// accidentally included in calculations; if you wish to use *exactly* the original code, use -DJCC10231 at compiler command line.

constraints_OLD_FIXME = 0.0;

	for (i32u n1 = 0;n1 < GetModel()->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < GetModel()->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < GetModel()->chn_vector[n1].res_vector[n2].natm;n3++)
			{
				mm2_atm newatm(n1, n2, n3);
				index_vector.push_back(newatm);
			}
			
			if (GetModel()->chn_vector[n1].res_vector[n2].state == STATE_HELIX)
			{
				switch (GetModel()->chn_vector[n1].res_vector[n2].symbol)
				{
					case 'A':	constraints += +1.446e+01; break;
					case 'R':	constraints += -6.833e+00; break;
					case 'N':	constraints += +9.452e+00; break;
					case 'D':	constraints += +5.893e+00; break;
					case 'C':	constraints += +1.779e+00; break;
					case 'Q':	constraints += -1.667e+01; break;
					case 'E':	constraints += +7.172e+00; break;
					case 'G':	constraints += +1.646e+01; break;
					case 'H':	constraints += +7.052e+00; break;
					case 'I':	constraints += -1.236e+01; break;
					case 'L':	constraints += +1.775e-01; break;
					case 'K':	constraints += -8.890e-01; break;
					case 'M':	constraints += +4.644e+00; break;
					case 'F':	constraints += +5.412e+00; break;
					case 'P':	constraints += -3.191e+00; break;
					case 'S':	constraints += +1.971e+01; break;
					case 'T':	constraints += +4.572e+00; break;
					case 'W':	constraints += -6.210e+00; break;
					case 'Y':	constraints += -1.238e+01; break;
					case 'V':	constraints += -1.009e+01; break;
					
					default:
					cout << "problem : unknown residue!!!" << endl;
					exit(EXIT_FAILURE);
				}
				
#ifdef JCC10231
switch (GetModel()->chn_vector[n1].res_vector[n2].symbol)
{
	case 'A':	constraints_OLD_FIXME += -0.08338; break;
	case 'R':	constraints_OLD_FIXME += +0.27353; break;
	case 'N':	constraints_OLD_FIXME += +2.58063; break;
	case 'D':	constraints_OLD_FIXME += +2.09165; break;
	case 'C':	constraints_OLD_FIXME += +1.66854; break;
	case 'Q':	constraints_OLD_FIXME += +0.06163; break;
	case 'E':	constraints_OLD_FIXME += -0.16222; break;
	case 'G':	constraints_OLD_FIXME += +4.02585; break;
	case 'H':	constraints_OLD_FIXME += +1.37137; break;
	case 'I':	constraints_OLD_FIXME += -0.30487; break;
	case 'L':	constraints_OLD_FIXME += -0.41418; break;
	case 'K':	constraints_OLD_FIXME += +0.65021; break;
	case 'M':	constraints_OLD_FIXME += -0.21454; break;
	case 'F':	constraints_OLD_FIXME += +0.52234; break;
	case 'P':	constraints_OLD_FIXME += +15.0; break;
	case 'S':	constraints_OLD_FIXME += +2.25152; break;
	case 'T':	constraints_OLD_FIXME += +1.72040; break;
	case 'W':	constraints_OLD_FIXME += +0.44867; break;
	case 'Y':	constraints_OLD_FIXME += +0.61176; break;
	case 'V':	constraints_OLD_FIXME += +0.14525; break;
	
	default:
	cout << "problem : unknown residue!!!" << endl;
	exit(EXIT_FAILURE);
}
#endif

			}
			
			if (GetModel()->chn_vector[n1].res_vector[n2].state == STATE_STRAND)
			{
				switch (GetModel()->chn_vector[n1].res_vector[n2].symbol)
				{
					case 'A':	constraints += -8.465e+00; break;
					case 'R':	constraints += -7.428e+00; break;
					case 'N':	constraints += -1.505e+01; break;
					case 'D':	constraints += -9.706e+00; break;
					case 'C':	constraints += -9.969e+00; break;
					case 'Q':	constraints += -8.787e+00; break;
					case 'E':	constraints += -9.782e+00; break;
					case 'G':	constraints += -8.186e+00; break;
					case 'H':	constraints += -1.019e+01; break;
					case 'I':	constraints += -1.172e+01; break;
					case 'L':	constraints += -1.168e+01; break;
					case 'K':	constraints += -6.592e+00; break;
					case 'M':	constraints += -1.007e+01; break;
					case 'F':	constraints += -1.414e+01; break;
					case 'P':	constraints += -9.055e+00; break;
					case 'S':	constraints += -7.650e+00; break;
					case 'T':	constraints += -1.405e+01; break;
					case 'W':	constraints += -9.975e+00; break;
					case 'Y':	constraints += -1.431e+01; break;
					case 'V':	constraints += -1.285e+01; break;
					
					default:
					cout << "problem : unknown residue!!!" << endl;
					exit(EXIT_FAILURE);
				}
				
#ifdef JCC10231
switch (GetModel()->chn_vector[n1].res_vector[n2].symbol)
{
	case 'A':	constraints_OLD_FIXME += +2.59146; break;
	case 'R':	constraints_OLD_FIXME += +2.06707; break;
	case 'N':	constraints_OLD_FIXME += +4.13071; break;
	case 'D':	constraints_OLD_FIXME += +4.48329; break;
	case 'C':	constraints_OLD_FIXME += +1.37183; break;
	case 'Q':	constraints_OLD_FIXME += +2.31212; break;
	case 'E':	constraints_OLD_FIXME += +2.52577; break;
	case 'G':	constraints_OLD_FIXME += +4.35497; break;
	case 'H':	constraints_OLD_FIXME += +2.18158; break;
	case 'I':	constraints_OLD_FIXME += -0.46446; break;
	case 'L':	constraints_OLD_FIXME += +1.02332; break;
	case 'K':	constraints_OLD_FIXME += +2.61213; break;
	case 'M':	constraints_OLD_FIXME += +1.19623; break;
	case 'F':	constraints_OLD_FIXME += +0.55870; break;
	case 'P':	constraints_OLD_FIXME += +15.0; break;
	case 'S':	constraints_OLD_FIXME += +3.03858; break;
	case 'T':	constraints_OLD_FIXME += +1.78115; break;
	case 'W':	constraints_OLD_FIXME += +0.97847; break;
	case 'Y':	constraints_OLD_FIXME += +0.56763; break;
	case 'V':	constraints_OLD_FIXME += -0.48028; break;
	
	default:
	cout << "problem : unknown residue!!!" << endl;
	exit(EXIT_FAILURE);
}
#endif

			}
		}
	}
	
#ifdef JCC10231
// in the old code, old constraints were added to energy; now subtract them from real constraints to remove the effect!
constraints -= constraints_OLD_FIXME;	// ok, this trick should give us the same final energies, using the original code...
#endif

	mass = new f64[index_vector.size()];
	
	vdwr = new f64[index_vector.size()];
	vdwr1 = new f64[index_vector.size()];
	vdwr2 = new f64[index_vector.size()];
	
	nbtp = new i32s[index_vector.size()];
	sasa = new f64[index_vector.size()];
	
	for (i32u n1 = 0;n1 < GetModel()->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < GetModel()->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < GetModel()->chn_vector[n1].res_vector[n2].natm;n3++)
			{
				i32s n4 = GetIndex(index_vector, mm2_atm(n1, n2, n3));
				
				mass[n4] = GetModel()->chn_vector[n1].res_vector[n2].mass[n3];
				
				vdwr[n4] = GetModel()->chn_vector[n1].res_vector[n2].vdwr[n3];
				vdwr1[n4] = 0.275 + GetModel()->prm.solvrad; vdwr2[n4] = vdwr1[n4] * vdwr1[n4];
				
				// average radius is used here ->
				// sasa-parameters are independent of VA radii!!!
				
				switch (GetModel()->chn_vector[n1].res_vector[n2].type[n3])
				{
					case TYPE_CPOS:
					nbtp[n4] = NBTP_CHARGED;
					break;
					
					case TYPE_CNEG:
					nbtp[n4] = NBTP_CHARGED | NBTP_CSIGN;
					break;
					
					case TYPE_POLAR:
					nbtp[n4] = NBTP_POLAR;
					break;
					
					default:
					nbtp[n4] = NBTP_NEUTRAL;
				}
				
				sasa[n4] = GetModel()->chn_vector[n1].res_vector[n2].sasa[n3] * vdwr2[n4];
			}
		}
	}
	
	crd = new f64_a3[index_vector.size()];
	d1 = new f64_a3[index_vector.size()];
	
/*##############################################*/
/*##############################################*/

	// main-chain t1-terms: direction is always from the N- to the C-terminal.
	//                      ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	for (i32s n1 = 0;n1 < (i32s) GetModel()->chn_vector.size();n1++)
	{
		for (i32s n2 = 0;n2 < ((i32s) GetModel()->chn_vector[n1].res_vector.size()) - 1;n2++)
		{
			mm2_bt1 newbt1;
			newbt1.atmi[0] = GetIndex(index_vector, mm2_atm(n1, n2, 0));
			newbt1.atmi[1] = GetIndex(index_vector, mm2_atm(n1, n2 + 1, 0));
			
			bool proline = (GetModel()->chn_vector[n1].res_vector[n2 + 1].symbol == 'P');
			newbt1.opt = (proline ? 0.353084 : 0.380564);

			newbt1.fc = 60.0e+03;

			bt1_vector.push_back(newbt1);
		}
	}
	
/*##############################################*/
/*##############################################*/

	// main-chain t2-terms: add like t1-terms -> directions for the t1-terms
	// will always be FALSE for the first one and TRUE for the second one.
	//                ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	for (i32s n1 = 0;n1 < ((i32s) bt1_vector.size()) - 1;n1++)
	{
		i32s test1 = bt1_vector[n1].GetIndex(0, false);
		i32s test2 = bt1_vector[n1 + 1].GetIndex(0, true);
		
		if (test1 == test2)
		{
			mm2_bt2 newbt2;
			newbt2.index1[0] = n1; newbt2.dir1[0] = false;
			newbt2.index1[1] = n1 + 1; newbt2.dir1[1] = true;
			
			newbt2.atmi[0] = bt1_vector[newbt2.index1[0]].GetIndex(1, newbt2.dir1[0]);
			newbt2.atmi[1] = bt1_vector[newbt2.index1[0]].GetIndex(0, newbt2.dir1[0]);
			newbt2.atmi[2] = bt1_vector[newbt2.index1[1]].GetIndex(1, newbt2.dir1[1]);
			
			newbt2.type = TYPE_LOOP; mm2_atm tmp1 = index_vector[newbt2.atmi[1]];
			if (GetModel()->chn_vector[tmp1[0]].res_vector[tmp1[1]].state == STATE_HELIX) newbt2.type = TYPE_HELIX;
			if (GetModel()->chn_vector[tmp1[0]].res_vector[tmp1[1]].state == STATE_STRAND) newbt2.type = TYPE_STRAND;
			
			switch (newbt2.type)
			{
				case TYPE_HELIX:
				newbt2.opt = -3.0000e-02;			// 91.7 deg
				newbt2.fc[0] = 360.0;
				newbt2.fc[1] = GetModel()->prm.wtmp3;
				break;
				
				case TYPE_STRAND:
				newbt2.opt = -5.1805e-01;			// 121.2 deg
				newbt2.fc[0] = 50.0 * GetModel()->prm.wtmp1;
				newbt2.fc[1] = GetModel()->prm.wtmp3;
				break;
				
				// the loop-values are averages of helix and strand...
				// the loop-values are averages of helix and strand...
				// the loop-values are averages of helix and strand...
				
				default:
				newbt2.opt = -2.7403e-01;			// 105.9 deg
				newbt2.fc[0] = 25.0 * GetModel()->prm.wtmp1;
				newbt2.fc[1] = GetModel()->prm.wtmp3;
				break;
			}
			
			bt2_vector.push_back(newbt2);
		}
	}
	
/*##############################################*/
/*##############################################*/

	// main-chain t3-terms: again add like t1-terms -> both t2-terms will
	// always have directions TRUE+TRUE -> implementation can take care of them.
	//                        ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	for (i32s n1 = 0;n1 < ((i32s) bt2_vector.size()) - 1;n1++)
	{
		i32s test1i = bt2_vector[n1].index1[1];
		bool test1d = bt2_vector[n1].dir1[1];
		
		i32s test2i = bt2_vector[n1 + 1].index1[0];
		bool test2d = bt2_vector[n1 + 1].dir1[0];
		
		if (test1i == test2i && test1d != test2d)
		{
			mm2_bt3 newbt3;
			newbt3.index2[0] = n1; newbt3.index2[1] = n1 + 1;
			
			newbt3.index1[0] = bt2_vector[newbt3.index2[0]].index1[0];
			newbt3.dir1[0] = bt2_vector[newbt3.index2[0]].dir1[0];
			
			newbt3.index1[1] = bt2_vector[newbt3.index2[0]].index1[1];
			newbt3.dir1[1] = bt2_vector[newbt3.index2[0]].dir1[1];
			
			newbt3.index1[2] = bt2_vector[newbt3.index2[1]].index1[0];
			newbt3.dir1[2] = bt2_vector[newbt3.index2[1]].dir1[0];
			
			newbt3.index1[3] = bt2_vector[newbt3.index2[1]].index1[1];
			newbt3.dir1[3] = bt2_vector[newbt3.index2[1]].dir1[1];
			
			newbt3.atmi[0] = bt1_vector[newbt3.index1[0]].GetIndex(1, newbt3.dir1[0]);
			newbt3.atmi[1] = bt1_vector[newbt3.index1[0]].GetIndex(0, newbt3.dir1[0]);
			newbt3.atmi[2] = bt1_vector[newbt3.index1[3]].GetIndex(0, newbt3.dir1[3]);
			newbt3.atmi[3] = bt1_vector[newbt3.index1[3]].GetIndex(1, newbt3.dir1[3]);
			
			ifstream file;
			
			mm2_atm tmp1 = index_vector[newbt3.atmi[1]];
			mm2_atm tmp2 = index_vector[newbt3.atmi[2]];
			
			bool th1 = (GetModel()->chn_vector[tmp1[0]].res_vector[tmp1[1]].state == STATE_HELIX);
			bool th2 = (GetModel()->chn_vector[tmp2[0]].res_vector[tmp2[1]].state == STATE_HELIX);
			
			bool ts1 = (GetModel()->chn_vector[tmp1[0]].res_vector[tmp1[1]].state == STATE_STRAND);
			bool ts2 = (GetModel()->chn_vector[tmp2[0]].res_vector[tmp2[1]].state == STATE_STRAND);
			
			// tor_type... tor_type... tor_type... tor_type... tor_type...
			// tor_type... tor_type... tor_type... tor_type... tor_type...
			// tor_type... tor_type... tor_type... tor_type... tor_type...
			
			newbt3.tor_type = TYPE_LOOP;
			if (th1 && th2) newbt3.tor_type = TYPE_HELIX;
			if (ts1 && ts2) newbt3.tor_type = TYPE_STRAND;
			
			switch (newbt3.tor_type)
			{
				case TYPE_HELIX:
				newbt3.tors[0] = +0.802356;
				newbt3.tors[1] = 100.0 * sqrt(GetModel()->prm.wtor1);
				break;
				
				case TYPE_STRAND:
				newbt3.tors[0] = -2.839065;
				newbt3.tors[1] = 100.0 * sqrt(GetModel()->prm.wtor1);
				break;
								
				default:
				model_simple::OpenParameterFile(file, false, "mm2param/looptor.txt");
				
				while (true)
				{
					if (file.peek() == 'e') exit(EXIT_FAILURE);
					
					char buffer[256]; char tp1; char tp2; file >> tp1 >> tp2;
					bool test1 = (tp1 != GetModel()->chn_vector[tmp1[0]].res_vector[tmp1[1]].symbol);
					bool test2 = (tp2 != GetModel()->chn_vector[tmp2[0]].res_vector[tmp2[1]].symbol);
					if (test1 || test2) file.getline(buffer, sizeof(buffer));
					else
					{
						f64 value;
						file >> value; newbt3.torc[0] = value * GetModel()->prm.wtor1;
						file >> value; newbt3.torc[1] = value * GetModel()->prm.wtor1;
						file >> value; newbt3.torc[2] = value * GetModel()->prm.wtor1;
						file >> value; newbt3.tors[0] = value * GetModel()->prm.wtor1;
						file >> value; newbt3.tors[1] = value * GetModel()->prm.wtor1;
						file >> value; newbt3.tors[2] = value * GetModel()->prm.wtor1;
						
						break;
					}
				}
				
				file.close();		// looptor.txt
				break;
			}
			
			// dip_type... dip_type... dip_type... dip_type... dip_type...
			// dip_type... dip_type... dip_type... dip_type... dip_type...
			// dip_type... dip_type... dip_type... dip_type... dip_type...
			
			newbt3.dip_type = TYPE_LOOP;
			if (th1 || th2) newbt3.dip_type = TYPE_HELIX;
			if (ts1 || ts2) newbt3.dip_type = TYPE_STRAND;
			
			switch (newbt3.dip_type)
			{
				case TYPE_HELIX:	break;
				case TYPE_STRAND:	break;
				
				default:
				model_simple::OpenParameterFile(file, false, "mm2param/loopdip.txt");
				
				while (true)
				{
					if (file.peek() == 'e') exit(EXIT_FAILURE);
					
					char buffer[256]; char tp1; char tp2; file >> tp1 >> tp2;
					bool test1 = (tp1 != GetModel()->chn_vector[tmp1[0]].res_vector[tmp1[1]].symbol);
					bool test2 = (tp2 != GetModel()->chn_vector[tmp2[0]].res_vector[tmp2[1]].symbol);
					if (test1 || test2) file.getline(buffer, sizeof(buffer));
					else
					{
						f64 value;
						file >> value; newbt3.dipc[0] = value;
						file >> value; newbt3.dipc[1] = value;
						file >> value; newbt3.dipc[2] = value;
						file >> value; newbt3.dips[0] = value;
						file >> value; newbt3.dips[1] = value;
						file >> value; newbt3.dips[2] = value;
						file >> value; newbt3.dipk[0] = value;
						file >> value; newbt3.dipk[1] = value;
						
						break;
					}
				}
				
				file.close();		// loopdip.txt
				break;
			}
			
			// set the skip flag if this is a X-pro case !!!!!!!!!!!!!
			// set the skip flag if this is a X-pro case !!!!!!!!!!!!!
			// set the skip flag if this is a X-pro case !!!!!!!!!!!!!
			
			newbt3.skip = (GetModel()->chn_vector[tmp2[0]].res_vector[tmp2[1]].symbol == 'P');
			
			bt3_vector.push_back(newbt3);
		}
	}
	
/*##############################################*/
/*##############################################*/

	// side-chain t1-terms: directions are always MAIN -> SIDE1 -> SIDE2.
	//                      ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	for (i32u n1 = 0;n1 < GetModel()->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < GetModel()->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32s n3 = 0;n3 < ((i32s) GetModel()->chn_vector[n1].res_vector[n2].natm) - 1;n3++)
			{
				mm2_bt1 newbt1;
				newbt1.atmi[0] = GetIndex(index_vector, mm2_atm(n1, n2, n3));
				newbt1.atmi[1] = GetIndex(index_vector, mm2_atm(n1, n2, n3 + 1));
				
				if (!n3)
				{
					switch (GetModel()->chn_vector[n1].res_vector[n2].symbol)
					{
						case 'R':
						newbt1.opt = 0.299941; newbt1.fc = 20.0e+03;
						break;
						
						case 'N':
						newbt1.opt = 0.253636; newbt1.fc = 60.0e+03;
						break;
						
						case 'D':
						newbt1.opt = 0.254969; newbt1.fc = 60.0e+03;
						break;
						
						case 'C':
						newbt1.opt = 0.279874; newbt1.fc = 60.0e+03;
						break;
						
						case 'Q':
						newbt1.opt = 0.298058; newbt1.fc = 20.0e+03;
						break;
						
						case 'E':
						newbt1.opt = 0.298556; newbt1.fc = 20.0e+03;
						break;
						
						case 'H':
						newbt1.opt = 0.355106; newbt1.fc = 40.0e+03;
						break;
						
						case 'I':
						newbt1.opt = 0.252456; newbt1.fc = 60.0e+03;
						break;
						
						case 'L':
						newbt1.opt = 0.260709; newbt1.fc = 60.0e+03;
						break;
						
						case 'K':
						newbt1.opt = 0.255603; newbt1.fc = 60.0e+03;
						break;
						
						case 'M':
						newbt1.opt = 0.355919; newbt1.fc = 20.0e+03;
						break;
						
						case 'F':
						newbt1.opt = 0.378612; newbt1.fc = 40.0e+03;
						break;
						
						case 'W':
						newbt1.opt = 0.342257; newbt1.fc = 40.0e+03;
						break;
						
						case 'Y':
						newbt1.opt = 0.378437; newbt1.fc = 40.0e+03;
						break;
						
						default:
						cout << "problems: unknown residue!!! S-T1-A" << endl;
						exit(EXIT_FAILURE);
					}
				}
				else
				{
					switch (GetModel()->chn_vector[n1].res_vector[n2].symbol)
					{
						case 'R':
						newbt1.opt = 0.291060; newbt1.fc = 20.0e+03;
						break;
						
						case 'K':
						newbt1.opt = 0.252464; newbt1.fc = 60.0e+03;
						break;
						
						case 'W':
						newbt1.opt = 0.209983; newbt1.fc = 60.0e+03;
						break;
						
						default:
						cout << "problems: unknown residue!!! S-T1-B" << endl;
						exit(EXIT_FAILURE);
					}
				}
				
				bt1_vector.push_back(newbt1);
			}
		}
	}
	
/*##############################################*/
/*##############################################*/

	// side-chain t2-terms: directions are always SIDE1 -> SIDE2 -> SIDE3.
	//                      ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	for (i32s n1 = 0;n1 < ((i32s) bt1_vector.size()) - 1;n1++)
	{
		i32s test1 = bt1_vector[n1].GetIndex(0, false);
		i32s test2 = bt1_vector[n1 + 1].GetIndex(0, true);
		
		mm2_atm tmp1 = index_vector[test1];
		
		if (test1 == test2 && tmp1[2] != 0)	// zero means the main-chain...
		{
			mm2_bt2 newbt2;
			newbt2.index1[0] = n1; newbt2.dir1[0] = false;
			newbt2.index1[1] = n1 + 1; newbt2.dir1[1] = true;
			
			newbt2.atmi[0] = bt1_vector[newbt2.index1[0]].GetIndex(1, newbt2.dir1[0]);
			newbt2.atmi[1] = bt1_vector[newbt2.index1[0]].GetIndex(0, newbt2.dir1[0]);
			newbt2.atmi[2] = bt1_vector[newbt2.index1[1]].GetIndex(1, newbt2.dir1[1]);
			
			newbt2.type = TYPE_SIDE;
			
			switch (GetModel()->chn_vector[tmp1[0]].res_vector[tmp1[1]].symbol)
			{
				case 'R':
				newbt2.opt = -0.717276;
				newbt2.fc[0] = 25.0;
				newbt2.fc[1] = 0.0;
				break;
				
				case 'K':
				newbt2.opt = -0.826295;
				newbt2.fc[0] = 25.0;
				newbt2.fc[1] = 0.0;
				break;
				
				case 'W':
				newbt2.opt = -0.487250;
				newbt2.fc[0] = 75.0;
				newbt2.fc[1] = 0.0;
				break;
				
				default:
				cout << "problems: unknown residue!!! S-T2" << endl;
				exit(EXIT_FAILURE);
			}
			
			bt2_vector.push_back(newbt2);
		}
	}
	
/*##############################################*/
/*##############################################*/

	// t4-terms for all non-terminal residues with side-chains: just add
	// t1- and t2-subterms like defined before; all directions are TRUE.
	//                                          ^^^^^^^^^^^^^^^^^^^^^^^^
	
	for (i32u n1 = 0;n1 < GetModel()->chn_vector.size();n1++)
	{
		for (i32s n2 = 1;n2 < ((i32s) GetModel()->chn_vector[n1].res_vector.size()) - 1;n2++)
		{
			if (GetModel()->chn_vector[n1].res_vector[n2].natm > 1)
			{
				i32s index[2]; mm2_bt4 newbt4;
				index[0] = GetIndex(index_vector, mm2_atm(n1, n2, 0));
				index[1] = GetIndex(index_vector, mm2_atm(n1, n2, 1));
				
				newbt4.index1 = 0;	// find the first side-chain t1-term:
				while (true)
				{
					bool test1 = (bt1_vector[newbt4.index1].atmi[0] == index[0]);
					bool test2 = (bt1_vector[newbt4.index1].atmi[1] == index[1]);
					if (test1 && test2) break; else newbt4.index1++;
				}
				
				newbt4.index2 = 0;	// find the first t2-term (main-chain):
				while (true)
				{
					bool test1 = (bt2_vector[newbt4.index2].atmi[1] == index[0]);
					if (test1) break; else newbt4.index2++;
				}
				
				char symbol = GetModel()->chn_vector[n1].res_vector[n2].symbol;
				i32s state = GetModel()->chn_vector[n1].res_vector[n2].state;
				
				switch (symbol)
				{
					case 'R':
					newbt4.opt = 0.831950; newbt4.fc = 170.7397;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +1.63143;
						newbt4.fscos[1] = +2.73464;
						newbt4.fscos[2] = -1.03777;
						newbt4.fssin[0] = -0.40058;
						newbt4.fssin[1] = -0.69431;
						newbt4.fssin[2] = +0.05773;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +2.78280;
						newbt4.fscos[1] = -2.09507;
						newbt4.fscos[2] = -1.59947;
						newbt4.fssin[0] = +0.79738;
						newbt4.fssin[1] = +0.44323;
						newbt4.fssin[2] = +0.01855;
						break;
						
						default:
						newbt4.fscos[0] = +1.93312;
						newbt4.fscos[1] = -0.20439;
						newbt4.fscos[2] = -1.37930;
						newbt4.fssin[0] = -1.03771;
						newbt4.fssin[1] = +0.67642;
						newbt4.fssin[2] = +0.37414;
					}
					break;
					
					case 'N':
					newbt4.opt = 0.814866; newbt4.fc = 150.5479;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +2.88276;
						newbt4.fscos[1] = +2.60143;
						newbt4.fscos[2] = -2.78638;
						newbt4.fssin[0] = -1.77857;
						newbt4.fssin[1] = +0.55007;
						newbt4.fssin[2] = +1.77600;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +3.52914;
						newbt4.fscos[1] = -3.39513;
						newbt4.fscos[2] = -3.22182;
						newbt4.fssin[0] = +1.15495;
						newbt4.fssin[1] = +0.95165;
						newbt4.fssin[2] = +0.43547;
						break;
						
						default:
						newbt4.fscos[0] = +1.86622;
						newbt4.fscos[1] = -0.95325;
						newbt4.fscos[2] = -2.76931;
						newbt4.fssin[0] = -1.06226;
						newbt4.fssin[1] = +0.43020;
						newbt4.fssin[2] = +1.55835;
					}
					break;
					
					case 'D':
					newbt4.opt = 0.808208; newbt4.fc = 172.7237;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +1.49439;
						newbt4.fscos[1] = +2.12629;
						newbt4.fscos[2] = -3.01556;
						newbt4.fssin[0] = -2.77468;
						newbt4.fssin[1] = -0.64116;
						newbt4.fssin[2] = +1.40245;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +3.55939;
						newbt4.fscos[1] = -3.45628;
						newbt4.fscos[2] = -2.54698;
						newbt4.fssin[0] = +1.71713;
						newbt4.fssin[1] = -0.00349;
						newbt4.fssin[2] = -0.22173;
						break;
						
						default:
						newbt4.fscos[0] = +1.53874;
						newbt4.fscos[1] = -0.81516;
						newbt4.fscos[2] = -2.51332;
						newbt4.fssin[0] = -0.91387;
						newbt4.fssin[1] = +0.02749;
						newbt4.fssin[2] = +1.69879;
					}
					break;
					
					case 'C':
					newbt4.opt = 0.770372; newbt4.fc = 147.7602;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +0.34551;
						newbt4.fscos[1] = +3.54192;
						newbt4.fscos[2] = -3.19414;
						newbt4.fssin[0] = -2.04954;
						newbt4.fssin[1] = +0.47390;
						newbt4.fssin[2] = +1.38320;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +1.90270;
						newbt4.fscos[1] = -3.11550;
						newbt4.fscos[2] = -3.98876;
						newbt4.fssin[0] = +0.79080;
						newbt4.fssin[1] = +0.60295;
						newbt4.fssin[2] = -0.55119;
						break;
						
						default:
						newbt4.fscos[0] = +1.28752;
						newbt4.fscos[1] = -0.43311;
						newbt4.fscos[2] = -3.06322;
						newbt4.fssin[0] = -1.41254;
						newbt4.fssin[1] = +0.64893;
						newbt4.fssin[2] = +1.20569;
					}
					break;
					
					case 'Q':
					newbt4.opt = 0.829190; newbt4.fc = 180.7584;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +1.65725;
						newbt4.fscos[1] = +1.97787;
						newbt4.fscos[2] = -0.86685;
						newbt4.fssin[0] = -1.19629;
						newbt4.fssin[1] = -1.30646;
						newbt4.fssin[2] = +0.14858;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +2.98437;
						newbt4.fscos[1] = -2.44128;
						newbt4.fscos[2] = -2.25933;
						newbt4.fssin[0] = +0.31694;
						newbt4.fssin[1] = +0.07723;
						newbt4.fssin[2] = -0.01357;
						break;
						
						default:
						newbt4.fscos[0] = +1.89292;
						newbt4.fscos[1] = -0.21472;
						newbt4.fscos[2] = -1.39680;
						newbt4.fssin[0] = -1.43613;
						newbt4.fssin[1] = +0.35431;
						newbt4.fssin[2] = +0.36846;
					}
					break;
					
					case 'E':
					newbt4.opt = 0.823989; newbt4.fc = 184.3073;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +1.29559;
						newbt4.fscos[1] = +1.80600;
						newbt4.fscos[2] = -0.84464;
						newbt4.fssin[0] = -0.95983;
						newbt4.fssin[1] = -1.02559;
						newbt4.fssin[2] = -0.03870;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +3.05382;
						newbt4.fscos[1] = -2.32924;
						newbt4.fscos[2] = -1.71923;
						newbt4.fssin[0] = +0.63232;
						newbt4.fssin[1] = -0.09958;
						newbt4.fssin[2] = +0.14660;
						break;
						
						default:
						newbt4.fscos[0] = +1.54207;
						newbt4.fscos[1] = -0.48636;
						newbt4.fscos[2] = -1.36813;
						newbt4.fssin[0] = -1.37130;
						newbt4.fssin[1] = -0.05944;
						newbt4.fssin[2] = +0.32152;
					}
					break;
					
					case 'H':
					newbt4.opt = 0.700807; newbt4.fc = 101.5500;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +3.10021;
						newbt4.fscos[1] = +2.92106;
						newbt4.fscos[2] = -4.21712;
						newbt4.fssin[0] = -0.47290;
						newbt4.fssin[1] = +0.60643;
						newbt4.fssin[2] = +2.87543;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +2.01313;
						newbt4.fscos[1] = -2.58253;
						newbt4.fscos[2] = -5.23890;
						newbt4.fssin[0] = -0.44308;
						newbt4.fssin[1] = +0.22866;
						newbt4.fssin[2] = -0.14427;
						break;
						
						default:
						newbt4.fscos[0] = +1.88037;
						newbt4.fscos[1] = -0.35703;
						newbt4.fscos[2] = -3.91341;
						newbt4.fssin[0] = -1.27627;
						newbt4.fssin[1] = +0.94613;
						newbt4.fssin[2] = +1.24182;
					}
					break;
					
					case 'I':
					newbt4.opt = 0.836257; newbt4.fc = 175.0000;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +0.37840;
						newbt4.fscos[1] = +2.64754;
						newbt4.fscos[2] = -1.18774;
						newbt4.fssin[0] = -3.00566;
						newbt4.fssin[1] = -0.15588;
						newbt4.fssin[2] = +3.11327;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +2.29466;
						newbt4.fscos[1] = -2.37337;
						newbt4.fscos[2] = -3.79914;
						newbt4.fssin[0] = -0.07048;
						newbt4.fssin[1] = +2.60053;
						newbt4.fssin[2] = -0.31214;
						break;
						
						default:
						newbt4.fscos[0] = +0.80982;
						newbt4.fscos[1] = -0.99030;
						newbt4.fscos[2] = -2.66346;
						newbt4.fssin[0] = -2.02548;
						newbt4.fssin[1] = +0.60076;
						newbt4.fssin[2] = +0.90423;
					}
					break;
					
					case 'L':
					newbt4.opt = 0.830300; newbt4.fc = 184.5275;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +4.51664;
						newbt4.fscos[1] = +3.52721;
						newbt4.fscos[2] = -1.26140;
						newbt4.fssin[0] = -1.84121;
						newbt4.fssin[1] = -1.04472;
						newbt4.fssin[2] = +0.62448;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +5.07843;
						newbt4.fscos[1] = -2.87863;
						newbt4.fscos[2] = -3.44333;
						newbt4.fssin[0] = +0.56280;
						newbt4.fssin[1] = +0.29704;
						newbt4.fssin[2] = -0.08214;
						break;
						
						default:
						newbt4.fscos[0] = +4.39417;
						newbt4.fscos[1] = +0.53557;
						newbt4.fscos[2] = -1.90615;
						newbt4.fssin[0] = -2.67041;
						newbt4.fssin[1] = -0.13768;
						newbt4.fssin[2] = +0.51817;
					}
					break;
					
					case 'K':
					newbt4.opt = 0.808124; newbt4.fc = 174.9278;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +2.07049;
						newbt4.fscos[1] = +2.91613;
						newbt4.fscos[2] = -1.50320;
						newbt4.fssin[0] = -0.56703;
						newbt4.fssin[1] = -0.55586;
						newbt4.fssin[2] = +0.86432;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +3.43466;
						newbt4.fscos[1] = -2.43258;
						newbt4.fscos[2] = -2.61991;
						newbt4.fssin[0] = +0.78553;
						newbt4.fssin[1] = +0.37040;
						newbt4.fssin[2] = -0.15226;
						break;
						
						default:
						newbt4.fscos[0] = +2.09391;
						newbt4.fscos[1] = +0.27371;
						newbt4.fscos[2] = -1.96935;
						newbt4.fssin[0] = -1.47660;
						newbt4.fssin[1] = +0.40649;
						newbt4.fssin[2] = +0.60613;
					}
					break;
					
					case 'M':
					newbt4.opt = 0.831470; newbt4.fc = 100.0000;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +1.77679;
						newbt4.fscos[1] = +0.35577;
						newbt4.fscos[2] = +0.32907;
						newbt4.fssin[0] = -0.57727;
						newbt4.fssin[1] = -0.31228;
						newbt4.fssin[2] = -0.91888;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +3.21443;
						newbt4.fscos[1] = -1.25277;
						newbt4.fscos[2] = -0.83565;
						newbt4.fssin[0] = +0.60945;
						newbt4.fssin[1] = +0.02275;
						newbt4.fssin[2] = -0.07607;
						break;
						
						default:
						newbt4.fscos[0] = +2.72211;
						newbt4.fscos[1] = -0.72316;
						newbt4.fscos[2] = -0.08704;
						newbt4.fssin[0] = -0.81098;
						newbt4.fssin[1] = +0.54678;
						newbt4.fssin[2] = -0.13277;
					}
					break;
					
					case 'F':
					newbt4.opt = 0.687122; newbt4.fc = 109.5348;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +3.66097;
						newbt4.fscos[1] = +3.29575;
						newbt4.fscos[2] = -4.68664;
						newbt4.fssin[0] = +0.32275;
						newbt4.fssin[1] = +0.96452;
						newbt4.fssin[2] = +1.72057;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +2.21422;
						newbt4.fscos[1] = -3.35839;
						newbt4.fscos[2] = -5.40674;
						newbt4.fssin[0] = +0.23309;
						newbt4.fssin[1] = +1.24766;
						newbt4.fssin[2] = -0.49037;
						break;
						
						default:
						newbt4.fscos[0] = +2.53714;
						newbt4.fscos[1] = -0.48366;
						newbt4.fscos[2] = -4.18275;
						newbt4.fssin[0] = -1.52692;
						newbt4.fssin[1] = +0.82532;
						newbt4.fssin[2] = +0.84911;
					}
					break;
					
					case 'W':
					newbt4.opt = 0.719438; newbt4.fc = 108.6690;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +2.96897;
						newbt4.fscos[1] = +3.10578;
						newbt4.fscos[2] = -3.30420;
						newbt4.fssin[0] = +0.33580;
						newbt4.fssin[1] = +1.49448;
						newbt4.fssin[2] = +2.12032;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +2.55791;
						newbt4.fscos[1] = -2.97853;
						newbt4.fscos[2] = -4.18877;
						newbt4.fssin[0] = +0.60708;
						newbt4.fssin[1] = +1.12960;
						newbt4.fssin[2] = -0.41920;
						break;
						
						default:
						newbt4.fscos[0] = +1.91765;
						newbt4.fscos[1] = -0.93406;
						newbt4.fscos[2] = -3.93063;
						newbt4.fssin[0] = -0.46467;
						newbt4.fssin[1] = +1.67154;
						newbt4.fssin[2] = +1.23776;
					}
					break;
					
					case 'Y':
					newbt4.opt = 0.680813; newbt4.fc = 101.1190;
					
					switch (state)
					{
						case STATE_HELIX:
						newbt4.fscos[0] = +3.43699;
						newbt4.fscos[1] = +3.80013;
						newbt4.fscos[2] = -4.11741;
						newbt4.fssin[0] = +0.09932;
						newbt4.fssin[1] = +0.79649;
						newbt4.fssin[2] = +2.29246;
						break;
						
						case STATE_STRAND:
						newbt4.fscos[0] = +2.12536;
						newbt4.fscos[1] = -3.45230;
						newbt4.fscos[2] = -6.21118;
						newbt4.fssin[0] = -0.24071;
						newbt4.fssin[1] = +0.70545;
						newbt4.fssin[2] = -0.81146;
						break;
						
						default:
						newbt4.fscos[0] = +2.13478;
						newbt4.fscos[1] = -0.39627;
						newbt4.fscos[2] = -4.15397;
						newbt4.fssin[0] = -1.33143;
						newbt4.fssin[1] = +1.11577;
						newbt4.fssin[2] = +1.16319;
					}
					break;
					
					default:
					cout << "problems: unknown residue!!! S-T4" << endl;
					exit(EXIT_FAILURE);
				}

				newbt4.fc *= GetModel()->prm.wtmp2;
				
				newbt4.fscos[0] *= GetModel()->prm.wtor2;
				newbt4.fscos[1] *= GetModel()->prm.wtor2;
				newbt4.fscos[2] *= GetModel()->prm.wtor2; 
				
				newbt4.fssin[0] *= GetModel()->prm.wtor2;
				newbt4.fssin[1] *= GetModel()->prm.wtor2;
				newbt4.fssin[2] *= GetModel()->prm.wtor2; 
				
				bt4_vector.push_back(newbt4);
			}
		}
	}
	
/*##############################################*/
/*##############################################*/

	// dsb-terms, both bt1 and bt2...
	
	for (i32u n1 = 0;n1 < GetModel()->dsb_vector.size();n1++)
	{
		i32s index1[2];
		index1[0] = GetIndex(index_vector, mm2_atm(GetModel()->dsb_vector[n1].chn[0], GetModel()->dsb_vector[n1].res[0], 0));
		index1[1] = GetIndex(index_vector, mm2_atm(GetModel()->dsb_vector[n1].chn[0], GetModel()->dsb_vector[n1].res[0], 1));
		
		i32s ind1 = 0;
		while (true)
		{
			bool test1 = (bt1_vector[ind1].atmi[0] == index1[0]);
			bool test2 = (bt1_vector[ind1].atmi[1] == index1[1]);
			if (test1 && test2) break;
			
			ind1++;
			if (ind1 == (i32s) bt1_vector.size())
			{
				cout << "DSB error #1" << endl;
				exit(EXIT_FAILURE);
			}
		}
		
		i32s index2[2];
		index2[0] = GetIndex(index_vector, mm2_atm(GetModel()->dsb_vector[n1].chn[1], GetModel()->dsb_vector[n1].res[1], 0));
		index2[1] = GetIndex(index_vector, mm2_atm(GetModel()->dsb_vector[n1].chn[1], GetModel()->dsb_vector[n1].res[1], 1));
		
		i32s ind2 = 0;
		while (true)
		{
			bool test1 = (bt1_vector[ind2].atmi[0] == index2[0]);
			bool test2 = (bt1_vector[ind2].atmi[1] == index2[1]);
			if (test1 && test2) break;
			
			ind2++;
			if (ind2 == (i32s) bt1_vector.size())
			{
				cout << "DSB error #2" << endl;
				exit(EXIT_FAILURE);
			}
		}
		
		mm2_bt1 newbt1;
		newbt1.atmi[0] = index1[1];
		newbt1.atmi[1] = index2[1];
		
		newbt1.opt = 0.202121;
		newbt1.fc = 95000.0;
		
		i32s ind3 = bt1_vector.size();
		bt1_vector.push_back(newbt1);
		
		mm2_bt2 newbt2;
		newbt2.type = TYPE_BRIDGE;
		newbt2.opt = -0.23533853;
		newbt2.fc[0] = 30.0;
		newbt2.fc[1] = 0.0;
		
		// first angle...
		
		newbt2.index1[0] = ind1; newbt2.dir1[0] = false;
		newbt2.index1[1] = ind3; newbt2.dir1[1] = true;
		
		newbt2.atmi[0] = bt1_vector[newbt2.index1[0]].GetIndex(1, newbt2.dir1[0]);
		newbt2.atmi[1] = bt1_vector[newbt2.index1[0]].GetIndex(0, newbt2.dir1[0]);
		newbt2.atmi[2] = bt1_vector[newbt2.index1[1]].GetIndex(1, newbt2.dir1[1]);
		
		bt2_vector.push_back(newbt2);
		
		// second angle...
		
		newbt2.index1[0] = ind2; newbt2.dir1[0] = false;
		newbt2.index1[1] = ind3; newbt2.dir1[1] = false;
		
		newbt2.atmi[0] = bt1_vector[newbt2.index1[0]].GetIndex(1, newbt2.dir1[0]);
		newbt2.atmi[1] = bt1_vector[newbt2.index1[0]].GetIndex(0, newbt2.dir1[0]);
		newbt2.atmi[2] = bt1_vector[newbt2.index1[1]].GetIndex(1, newbt2.dir1[1]);
		
		bt2_vector.push_back(newbt2);
	}
	
/*##############################################*/
/*##############################################*/

	// terminal angles...
	
	for (i32u n1 = 0;n1 < GetModel()->chn_vector.size();n1++)
	{
		if (GetModel()->chn_vector[n1].res_vector[0].natm > 1)
		{
			i32s index[3];
			index[0] = GetIndex(index_vector, mm2_atm(n1, 0, 1));
			index[1] = GetIndex(index_vector, mm2_atm(n1, 0, 0));
			index[2] = GetIndex(index_vector, mm2_atm(n1, 1, 0));
			
			i32s ind1 = 0;
			while (true)
			{
				bool test1 = (bt1_vector[ind1].atmi[0] == index[1]);
				bool test2 = (bt1_vector[ind1].atmi[1] == index[0]);
				if (test1 && test2) break;
				
				ind1++;
				if (ind1 == (i32s) bt1_vector.size())
				{
					cout << "NT error #1" << endl;
					exit(EXIT_FAILURE);
				}
			}
			
			i32s ind2 = 0;
			while (true)
			{
				bool test1 = (bt1_vector[ind2].atmi[0] == index[1]);
				bool test2 = (bt1_vector[ind2].atmi[1] == index[2]);
				if (test1 && test2) break;
				
				ind2++;
				if (ind2 == (i32s) bt1_vector.size())
				{
					cout << "NT error #2" << endl;
					exit(EXIT_FAILURE);
				}
			}
			
			mm2_bt2 newbt2;
			newbt2.index1[0] = ind1; newbt2.dir1[0] = true;
			newbt2.index1[1] = ind2; newbt2.dir1[1] = true;
			
			newbt2.atmi[0] = bt1_vector[newbt2.index1[0]].GetIndex(1, newbt2.dir1[0]);
			newbt2.atmi[1] = bt1_vector[newbt2.index1[0]].GetIndex(0, newbt2.dir1[0]);
			newbt2.atmi[2] = bt1_vector[newbt2.index1[1]].GetIndex(1, newbt2.dir1[1]);
			
			newbt2.type = TYPE_TERM;
			newbt2.opt = -0.16223398;
			newbt2.fc[0] = 35.0;
			newbt2.fc[1] = GetModel()->prm.wtmp3;
			
			bt2_vector.push_back(newbt2);
		}
		
		i32s sz = GetModel()->chn_vector[n1].res_vector.size();
		if (GetModel()->chn_vector[n1].res_vector[sz - 1].natm > 1)
		{
			i32s index[3];
			index[0] = GetIndex(index_vector, mm2_atm(n1, sz - 1, 1));
			index[1] = GetIndex(index_vector, mm2_atm(n1, sz - 1, 0));
			index[2] = GetIndex(index_vector, mm2_atm(n1, sz - 2, 0));
			
			i32s ind1 = 0;
			while (true)
			{
				bool test1 = (bt1_vector[ind1].atmi[0] == index[1]);
				bool test2 = (bt1_vector[ind1].atmi[1] == index[0]);
				if (test1 && test2) break;
				
				ind1++;
				if (ind1 == (i32s) bt1_vector.size())
				{
					cout << "CT error #1" << endl;
					exit(EXIT_FAILURE);
				}
			}
			
			i32s ind2 = 0;
			while (true)
			{
				bool test1 = (bt1_vector[ind2].atmi[0] == index[2]);
				bool test2 = (bt1_vector[ind2].atmi[1] == index[1]);
				if (test1 && test2) break;
				
				ind2++;
				if (ind2 == (i32s) bt1_vector.size())
				{
					cout << "CT error #2" << endl;
					exit(EXIT_FAILURE);
				}
			}
			
			mm2_bt2 newbt2;
			newbt2.index1[0] = ind1; newbt2.dir1[0] = true;
			newbt2.index1[1] = ind2; newbt2.dir1[1] = false;
			
			newbt2.atmi[0] = bt1_vector[newbt2.index1[0]].GetIndex(1, newbt2.dir1[0]);
			newbt2.atmi[1] = bt1_vector[newbt2.index1[0]].GetIndex(0, newbt2.dir1[0]);
			newbt2.atmi[2] = bt1_vector[newbt2.index1[1]].GetIndex(1, newbt2.dir1[1]);
			
			newbt2.type = TYPE_TERM;
			newbt2.opt = -0.16223398;
			newbt2.fc[0] = 35.0;
			newbt2.fc[1] = GetModel()->prm.wtmp3;
			
			bt2_vector.push_back(newbt2);
		}
	}
	
/*##############################################*/
/*##############################################*/

	// nb1-terms...
	
	for (i32s n1 = 0;n1 < ((i32s) index_vector.size()) - 1;n1++)
	{
		for (i32s n2 = n1 + 1;n2 < (i32s) index_vector.size();n2++)
		{
			mm2_nbt1 newnbt1;
			newnbt1.atmi[0] = n1;
			newnbt1.atmi[1] = n2;
			
			if (!InitNBT1(& newnbt1)) continue;
			else nbt1_vector.push_back(newnbt1);
		}
	}
	
/*##############################################*/
/*##############################################*/

	// the solvent term... nbt1 are needed for all VA's separated by more than 1 bonds !!!
	// the solvent term... nbt1 are needed for all VA's separated by more than 1 bonds !!!
	// the solvent term... nbt1 are needed for all VA's separated by more than 1 bonds !!!
	
	nbt3_nl = new mm2_nbt3_nl[index_vector.size()];
	
	dist1 = new i32s[index_vector.size()];
	dist2 = new f64[index_vector.size() * (index_vector.size() - 1) / 2];
	
	i32s n1 = 0; i32s n2 = 0;
	while (n2 < (i32s) index_vector.size())
	{
		dist1[n2++] = n1;
		n1 += ((i32s) index_vector.size()) - n2;
	}
	
/*##############################################*/
/*##############################################*/

	bt1data.resize(bt1_vector.size());
	bt2data.resize(bt2_vector.size());
}

mm2_eng::~mm2_eng(void)
{
	delete[] mass;
	
	delete[] vdwr;
	delete[] vdwr1;
	delete[] vdwr2;
	
	delete[] crd;
	delete[] d1;
	
	delete[] nbt3_nl;
	delete[] dist1; delete[] dist2;
	
	delete[] nbtp;
	delete[] sasa;
}

bool mm2_eng::InitNBT1(mm2_nbt1 * ref)
{
	mm2_atm atm[2];
	atm[0] = index_vector[ref->atmi[0]];
	atm[1] = index_vector[ref->atmi[1]];
	
// these are used to save memory when measuring vdw-radii areas... 2.0 * (0.3 + 0.2) = 1.0
//fGL * c1 = GetModel()->chn_vector[atm[0][0]].res_vector[atm[0][1]].crd_vector[atm[0][2]][0].data;
//fGL * c2 = GetModel()->chn_vector[atm[1][0]].res_vector[atm[1][1]].crd_vector[atm[1][2]][0].data;
//v3d<fGL> v1 = v3d<fGL>(c1, c2); if (v1.len() > 1.5) return false;

// these are used to save memory when measuring surface areas... radii -> "limit"
//fGL limit = vdwr[ref->atmi[0]] + vdwr[ref->atmi[1]] + 2.0 * GetModel()->prm.solvrad + 0.5;
//fGL * c1 = GetModel()->chn_vector[atm[0][0]].res_vector[atm[0][1]].crd_vector[atm[0][2]][0].data;
//fGL * c2 = GetModel()->chn_vector[atm[1][0]].res_vector[atm[1][1]].crd_vector[atm[1][2]][0].data;
//v3d<fGL> v1 = v3d<fGL>(c1, c2); if (v1.len() > limit) return false;

	if (atm[0][2] && atm[1][2])		// bridge
	{
		for (i32u n1 = 0;n1 < GetModel()->dsb_vector.size();n1++)
		{
			bool test1a = ((i32s) atm[0][0] == GetModel()->dsb_vector[n1].chn[0] &&
				(i32s) atm[0][1] == GetModel()->dsb_vector[n1].res[0]);
			bool test1b = ((i32s) atm[1][0] == GetModel()->dsb_vector[n1].chn[1] &&
				(i32s) atm[1][1] == GetModel()->dsb_vector[n1].res[1]);
			if (test1a && test1b) return false;
			
			bool test2a = ((i32s) atm[0][0] == GetModel()->dsb_vector[n1].chn[1] &&
				(i32s) atm[0][1] == GetModel()->dsb_vector[n1].res[1]);
			bool test2b = ((i32s) atm[1][0] == GetModel()->dsb_vector[n1].chn[0] &&
				(i32s) atm[1][1] == GetModel()->dsb_vector[n1].res[0]);
			if (test2a && test2b) return false;
		}
	}
	
	bool same_chn = (atm[0][0] == atm[1][0]);
	
	bool rdist_is_0 = same_chn && ((atm[1][1] - atm[0][1]) == 0);
	bool rdist_is_1 = same_chn && ((atm[1][1] - atm[0][1]) == 1);
	
	if (rdist_is_0 && ((atm[1][2] - atm[0][2]) < 2)) return false;
	if (rdist_is_1 && !atm[1][2] && !atm[0][2]) return false;
	
	bool rdist_is_2 = same_chn && ((atm[1][1] - atm[0][1]) == 2);
	bool rdist_is_3 = same_chn && ((atm[1][1] - atm[0][1]) == 3);
	bool rdist_is_4 = same_chn && ((atm[1][1] - atm[0][1]) == 4);
	
	bool both_in_main_chn = (!atm[0][2] && !atm[1][2]);
	
	bool check3 = (atm[0][1] + 1 < GetModel()->chn_vector[atm[0][0]].res_vector.size());
	bool check4 = (atm[0][1] + 2 < GetModel()->chn_vector[atm[0][0]].res_vector.size());
	bool check5 = (atm[0][1] + 3 < GetModel()->chn_vector[atm[0][0]].res_vector.size());
	
	bool helix3 = check3 && (GetModel()->chn_vector[atm[0][0]].res_vector[atm[0][1] + 1].state == STATE_HELIX);
	bool helix4 = check4 && helix3 && (GetModel()->chn_vector[atm[0][0]].res_vector[atm[0][1] + 2].state == STATE_HELIX);
	bool helix5 = check5 && helix4 && (GetModel()->chn_vector[atm[0][0]].res_vector[atm[0][1] + 3].state == STATE_HELIX);
	
	f64 vdwr[2];
	vdwr[0] = GetModel()->chn_vector[atm[0][0]].res_vector[atm[0][1]].vdwr[atm[0][2]];
	vdwr[1] = GetModel()->chn_vector[atm[1][0]].res_vector[atm[1][1]].vdwr[atm[1][2]];
	
	f64 opt = vdwr[0] + vdwr[1];
	f64 fc = GetModel()->prm.lenjon;
	
	if (both_in_main_chn)
	{
		bool test3 = (rdist_is_2 && helix3);
		bool test4 = (rdist_is_3 && helix4);
		bool test5 = (rdist_is_4 && helix5);
		
		if (test3 || test4 || test5)
		{
			opt = 0.650;
			fc = GetModel()->prm.lenjon;
		}
	}
	
	InitLenJon(ref, opt, fc);
	return true;
}

void mm2_eng::InitLenJon(mm2_nbt1 * ref, f64 opt, f64 fc)
{
	ref->data[0] = opt * pow(2.0 * fc, 1.0 / 9.0);
	ref->data[1] = opt * pow(3.0 * fc, 1.0 / 6.0);
}

i32s mm2_eng::GetIndex(vector<mm2_atm> & p1, mm2_atm p2)
{
	vector<mm2_atm>::iterator it1;
	it1 = find(p1.begin(), p1.end(), p2);
	return it1 - p1.begin();
}

void mm2_eng::ComputeBT1(i32s p1)
{
	energy_bt1 = 0.0;
	
	// data1 -> length of the bond vector, in nanometers [nm].
	// data2[0] -> grad[0-2]: for atom 1 when direction = 0, for atom 0 when direction = 1
	// data2[1] -> grad[0-2]: for atom 0 when direction = 0, for atom 1 when direction = 1
	
	// mm1_eng_exp1::ComputeBT1() works in a similar way...
	
	for (i32u n1 = 0;n1 < bt1_vector.size();n1++)
	{
		i32s * atmi = bt1_vector[n1].atmi;
		
		f64 t1a[3]; f64 t1b = 0.0;
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 t9a = crd[atmi[0]][n2];
			f64 t9b = crd[atmi[1]][n2];
			
			t1a[n2] = t9a - t9b;
			t1b += t1a[n2] * t1a[n2];
		}
		
		f64 t1c = sqrt(t1b);
		bt1data[n1].data1 = t1c;
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 t9a = t1a[n2] / t1c;
			bt1data[n1].data2[0][n2] = +t9a;
			bt1data[n1].data2[1][n2] = -t9a;
		}
		
		/*##############################################*/
		/*##############################################*/
		// this is for the surface area term...
		
		bool first = (atmi[0] > atmi[1]);
		dist2[dist1[atmi[first]] + (atmi[!first] - atmi[first]) - 1] = t1c;
		
		if (t1c < (vdwr1[atmi[0]] + vdwr1[atmi[1]]))
		{
			nbt3_nl[atmi[0]].index[nbt3_nl[atmi[0]].index_count++] = atmi[1];
			if (nbt3_nl[atmi[0]].index_count >= SIZE_NL) { cout << "NL overflow!!!" << endl; exit(EXIT_FAILURE); }
			
			nbt3_nl[atmi[1]].index[nbt3_nl[atmi[1]].index_count++] = atmi[0];
			if (nbt3_nl[atmi[1]].index_count >= SIZE_NL) { cout << "NL overflow!!!" << endl; exit(EXIT_FAILURE); }
		}
		
		// this is for the surface area term...
		/*##############################################*/
		/*##############################################*/
		
		// f = a(x-b)^2
		// df/dx = 2a(x-b)
		
		f64 t2a = t1c - bt1_vector[n1].opt;
		energy_bt1 += bt1_vector[n1].fc * t2a * t2a;
		
		if (p1 > 0)
		{
			f64 t2b = 2.0 * bt1_vector[n1].fc * t2a;
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				f64 t2c = bt1data[n1].data2[0][n2] * t2b;
				
				d1[atmi[0]][n2] += t2c;
				d1[atmi[1]][n2] -= t2c;
			}
		}
	}
}

void mm2_eng::ComputeBT2(i32s p1)
{
	energy_bt2 = 0.0;
	
	// data1 -> cosine of the bond angle, in the usual range [-1.0, +1.0]
	// data2[0] -> grad[0-2]: for atom 2 when direction = 0, for atom 0 when direction = 1
	// data2[1] -> grad[0-2]: for atom 1 when direction = 0, for atom 1 when direction = 1
	// data2[2] -> grad[0-2]: for atom 0 when direction = 0, for atom 2 when direction = 1
	
	// mm1_eng_exp1::ComputeBT2() works in a similar way...
	
	for (i32u n1 = 0;n1 < bt2_vector.size();n1++)
	{
		i32s * atmi = bt2_vector[n1].atmi;
		
		i32s * index1 = bt2_vector[n1].index1;
		bool * dir1 = bt2_vector[n1].dir1;
		
		f64 * t1a = bt1data[index1[0]].data2[dir1[0]];
		f64 * t1b = bt1data[index1[1]].data2[dir1[1]];
		
		f64 t1c = t1a[0] * t1b[0] + t1a[1] * t1b[1] + t1a[2] * t1b[2];
		
		if (t1c < -1.0) t1c = -1.0;		// domain check...
		if (t1c > +1.0) t1c = +1.0;		// domain check...
		
// if the main-chain angles approach 180.0 deg we will have some serious numerical problems in later
// stages. fc[1] must be large enough to prevent such problems. here is how we can monitor this...

bool problem1 = (t1c < -0.999);				// is it too close to 180 deg ???
bool problem2 = (bt2_vector[n1].fc[1] > 0.0);		// is it really a main-chain angle???
if (problem1 && problem2) { cout << "BT2 problem!!!" << endl; exit(EXIT_FAILURE); }

		bt2data[n1].data1 = t1c;
		
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 t9a = (t1b[n2] - t1c * t1a[n2]) / bt1data[index1[0]].data1;
			f64 t9b = (t1a[n2] - t1c * t1b[n2]) / bt1data[index1[1]].data1;
			
			bt2data[n1].data2[0][n2] = t9a;
			bt2data[n1].data2[1][n2] = -(t9a + t9b);
			bt2data[n1].data2[2][n2] = t9b;
		}
		
		// f = a(x-b)^2
		// df/dx = 2a(x-b)
		
		f64 t3b = t1c - bt2_vector[n1].opt;
		energy_bt2 += bt2_vector[n1].fc[0] * t3b * t3b;
		
		// f = a/(x+1)^4
		// df/dx = -4a/(x+1)^5
		
		f64 t3c = t1c + 1.0;
		
		f64 t3d = t3c * t3c; f64 t3e = t3d * t3d;
		energy_bt2 += bt2_vector[n1].fc[1] / t3e;
		
		if (p1 > 0)
		{
			f64 t2a = 2.0 * bt2_vector[n1].fc[0] * t3b;
			
			f64 t3f = t3e * t3c;
			f64 t2b = 4.0 * bt2_vector[n1].fc[1] / t3f;
			
			f64 t2c = t2a - t2b;
			
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				d1[atmi[0]][n2] += bt2data[n1].data2[0][n2] * t2c;
				d1[atmi[1]][n2] += bt2data[n1].data2[1][n2] * t2c;
				d1[atmi[2]][n2] += bt2data[n1].data2[2][n2] * t2c;
			}
		}
	}
}

void mm2_eng::ComputeBT3(i32s p1)
{
	energy_bt3 = 0.0;
	
	for (i32u n1 = 0;n1 < bt3_vector.size();n1++)
	{
		i32s * atmi = bt3_vector[n1].atmi;
		
		i32s * index2 = bt3_vector[n1].index2;
		i32s * index1 = bt3_vector[n1].index1;
		bool * dir1 = bt3_vector[n1].dir1;
		
		// ang is never > 180.0 deg -> sin(ang) is never < 0.0 -> no need to check signs!!!
		// ang is never > 180.0 deg -> sin(ang) is never < 0.0 -> no need to check signs!!!
		// ang is never > 180.0 deg -> sin(ang) is never < 0.0 -> no need to check signs!!!
		
		f64 t1a[2] = { bt2data[index2[0]].data1, bt2data[index2[1]].data1 };
		f64 t1b[2] = { 1.0 - t1a[0] * t1a[0], 1.0 - t1a[1] * t1a[1] };
		
		f64 t1c[2][3];
		t1c[0][0] = bt1data[index1[0]].data2[dir1[0]][0] - t1a[0] * bt1data[index1[1]].data2[dir1[1]][0];
		t1c[0][1] = bt1data[index1[0]].data2[dir1[0]][1] - t1a[0] * bt1data[index1[1]].data2[dir1[1]][1];
		t1c[0][2] = bt1data[index1[0]].data2[dir1[0]][2] - t1a[0] * bt1data[index1[1]].data2[dir1[1]][2];
		t1c[1][0] = bt1data[index1[3]].data2[dir1[3]][0] - t1a[1] * bt1data[index1[2]].data2[dir1[2]][0];
		t1c[1][1] = bt1data[index1[3]].data2[dir1[3]][1] - t1a[1] * bt1data[index1[2]].data2[dir1[2]][1];
		t1c[1][2] = bt1data[index1[3]].data2[dir1[3]][2] - t1a[1] * bt1data[index1[2]].data2[dir1[2]][2];
		
		f64 t9a = t1c[0][0] * t1c[1][0] + t1c[0][1] * t1c[1][1] + t1c[0][2] * t1c[1][2];
		f64 t1d = t9a / sqrt(t1b[0] * t1b[1]);
		
		if (t1d < -1.0) t1d = -1.0;		// domain check...
		if (t1d > +1.0) t1d = +1.0;		// domain check...
		
		f64 t1e = acos(t1d);
		
		// now we still have to determine sign of the result...
		
		v3d<f64> tmpv1 = v3d<f64>(t1c[0]);
		v3d<f64> tmpv2 = v3d<f64>(bt1data[index1[2]].data2[dir1[2]]);
		v3d<f64> tmpv3 = v3d<f64>(bt1data[index1[3]].data2[dir1[3]]);
		v3d<f64> tmpv4 = tmpv2.vpr(tmpv3);
		
		if (tmpv1.spr(tmpv4) < 0.0) t1e = -t1e;
		
		f64 t1f; f64 t1g;		// f and df/dx !!!
		f64 cs[3]; f64 sn[3];
		
		if (bt3_vector[n1].tor_type != TYPE_LOOP)	// helix + strand
		{
			// Dx = x-b			| for the following formulas...
			
			// f = a(2PI-Dx)^4		| if Dx > +PI
			// df/fx = -4a(2PI-Dx)^3
			
			// f = a(2PI+Dx)^4		| if Dx < -PI
			// df/fx = +4a(2PI+Dx)^3
			
			// f = a(Dx)^4			| otherwise
			// df/fx = 4a(Dx)^3
			
			f64 t1h = t1e - bt3_vector[n1].tors[0];
			if (t1h > +M_PI)
			{
				t1h = 2.0 * M_PI - t1h; f64 t1z = t1h * t1h;
				
				t1f = bt3_vector[n1].tors[1] * t1z * t1z;
				t1g = -4.0 * bt3_vector[n1].tors[1] * t1z * t1h;
			}
			else if (t1h < -M_PI)
			{
				t1h = 2.0 * M_PI + t1h; f64 t1z = t1h * t1h;
				
				t1f = bt3_vector[n1].tors[1] * t1z * t1z;
				t1g = +4.0 * bt3_vector[n1].tors[1] * t1z * t1h;
			}
			else
			{
				f64 t1z = t1h * t1h;
				
				t1f = bt3_vector[n1].tors[1] * t1z * t1z;
				t1g = 4.0 * bt3_vector[n1].tors[1] * t1z * t1h;
			}
		}
		else		// loop!!!
		{
			// f = a*cos(x) + b*cos(2x) + c*cos(3x) + d*sin(x) + e*sin(2x) + f*sin(3x)
			// df/dx = d*cos(x) + 2e*cos(2x) + 3f*cos(3x) - a*sin(x) - 2b*sin(2x) - 3c*sin(3x)
			
			f64 t1m = t1e + t1e; f64 t1n = t1m + t1e;
			cs[0] = cos(t1e); cs[1] = cos(t1m); cs[2] = cos(t1n);
			sn[0] = sin(t1e); sn[1] = sin(t1m); sn[2] = sin(t1n);
			
			f64 * fsc = bt3_vector[n1].torc; f64 * fss = bt3_vector[n1].tors;
			
			t1f = fsc[0] * cs[0] + fsc[1] * cs[1] + fsc[2] * cs[2];
			t1f += fss[0] * sn[0] + fss[1] * sn[1] + fss[2] * sn[2];
			
			t1g = fss[0] * cs[0] + 2.0 * fss[1] * cs[1] + 3.0 * fss[2] * cs[2];
			t1g -= fsc[0] * sn[0] + 2.0 * fsc[1] * sn[1] + 3.0 * fsc[2] * sn[2];
		}
		
		energy_bt3 += t1f;
		
		f64 t4c[3]; f64 t5c[3]; f64 t6a[3]; f64 t7a[3];		// these are needed later...
		
		if (p1 > 0)
		{
			f64 t2a = bt1data[index1[0]].data1 * t1b[0];
			f64 t2b = bt1data[index1[0]].data1 * t1a[0] / bt1data[index1[1]].data1;
			
			f64 t3a = bt1data[index1[3]].data1 * t1b[1];
			f64 t3b = bt1data[index1[3]].data1 * t1a[1] / bt1data[index1[2]].data1;
			
			const i32s cp[3][3] = { { 0, 1, 2 }, { 1, 2, 0 }, { 2, 0, 1 } };
			
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				f64 t4a = bt1data[index1[0]].data2[dir1[0]][cp[n2][1]] * bt1data[index1[1]].data2[dir1[1]][cp[n2][2]];
				f64 t4b = bt1data[index1[0]].data2[dir1[0]][cp[n2][2]] * bt1data[index1[1]].data2[dir1[1]][cp[n2][1]];
				t4c[n2] = (t4a - t4b) / t2a;
				
				f64 t5a = bt1data[index1[2]].data2[dir1[2]][cp[n2][2]] * bt1data[index1[3]].data2[dir1[3]][cp[n2][1]];
				f64 t5b = bt1data[index1[2]].data2[dir1[2]][cp[n2][1]] * bt1data[index1[3]].data2[dir1[3]][cp[n2][2]];
				t5c[n2] = (t5a - t5b) / t3a;
				
				d1[atmi[0]][n2] += t1g * t4c[n2];
				d1[atmi[3]][n2] += t1g * t5c[n2];
				
				t6a[n2] = (t2b - 1.0) * t4c[n2] - t3b * t5c[n2];
				t7a[n2] = (t3b - 1.0) * t5c[n2] - t2b * t4c[n2];
				
				d1[atmi[1]][n2] += t1g * t6a[n2];
				d1[atmi[2]][n2] += t1g * t7a[n2];
			}
		}
		
		// calculate dipole directions (with derivatives if needed)...
		// calculate dipole directions (with derivatives if needed)...
		// calculate dipole directions (with derivatives if needed)...
		
		// f = a*cos(x) + b*cos(2x) + c*cos(3x) + d*sin(x) + e*sin(2x) + f*sin(3x) + g*x + h
		// df/dx = d*cos(x) + 2e*cos(2x) + 3f*cos(3x) - a*sin(x) - 2b*sin(2x) - 3c*sin(3x) + g
		
		f64 loop1; f64 loop2; f64 dpbdd[4][3];
		f64 * fsc = bt3_vector[n1].dipc; f64 * fss = bt3_vector[n1].dips;
		
		switch (bt3_vector[n1].dip_type)
		{
			case TYPE_HELIX:
			bt3_vector[n1].pbdd = -0.888278;
			break;
			
			case TYPE_STRAND:
			bt3_vector[n1].pbdd = +1.915213;
			break;
			
			default:	// loop!!!
			loop1 = bt3_vector[n1].dipk[0] * t1e + bt3_vector[n1].dipk[1];
			loop1 += fsc[0] * cs[0] + fsc[1] * cs[1] + fsc[2] * cs[2];
			loop1 += fss[0] * sn[0] + fss[1] * sn[1] + fss[2] * sn[2];
			bt3_vector[n1].pbdd = loop1;
			
			if (p1 > 0)
			{
				loop2 = bt3_vector[n1].dipk[0];
				loop2 += fss[0] * cs[0] + 2.0 * fss[1] * cs[1] + 3.0 * fss[2] * cs[2];
				loop2 -= fsc[0] * sn[0] + 2.0 * fsc[1] * sn[1] + 3.0 * fsc[2] * sn[2];
				
				for (i32s n2 = 0;n2 < 3;n2++)
				{
					dpbdd[0][n2] = loop2 * t4c[n2];
					dpbdd[3][n2] = loop2 * t5c[n2];
					
					dpbdd[1][n2] = loop2 * t6a[n2];
					dpbdd[2][n2] = loop2 * t7a[n2];
				}
			}
		}
		
		// calculate some dipole results already here...
		// calculate some dipole results already here...
		// calculate some dipole results already here...
		
		if (bt3_vector[n1].skip) continue;	// skip all X-pro cases...
		
		f64 t2a[3]; f64 t2b[3][3];		// 1st unit bond vector + derivatives...
		f64 t2c[3]; f64 t2d[3][3];		// 2nd unit bond vector + derivatives...
		
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			t2a[n2] = bt1data[index1[0]].data2[dir1[0]][n2];
			t2c[n2] = bt1data[index1[1]].data2[dir1[1]][n2];
		}
		
		if (p1 > 0)
		{
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					if (n2 != n3)
					{
						t2b[n2][n3] = -t2a[n2] * t2a[n3] / bt1data[index1[0]].data1;
						t2d[n2][n3] = -t2c[n2] * t2c[n3] / bt1data[index1[1]].data1;
					}
					else
					{
						t2b[n2][n2] = (1.0 - t2a[n2] * t2a[n2]) / bt1data[index1[0]].data1;
						t2d[n2][n2] = (1.0 - t2c[n2] * t2c[n2]) / bt1data[index1[1]].data1;
					}
				}
			}
		}
		
		f64 t2e = bt2data[index2[0]].data1;
		f64 t2f = t2e * t2e; f64 t2g = 1.0 - t2f;
		f64 t2h = sqrt(t2g);
		
		// 1st basis vector... 1st basis vector... 1st basis vector... 1st basis vector...
		// 1st basis vector... 1st basis vector... 1st basis vector... 1st basis vector...
		// 1st basis vector... 1st basis vector... 1st basis vector... 1st basis vector...
		
		bt3_vector[n1].bv[0][0] = (t2a[0] - t2e * t2c[0]) / t2h;
		bt3_vector[n1].bv[0][1] = (t2a[1] - t2e * t2c[1]) / t2h;
		bt3_vector[n1].bv[0][2] = (t2a[2] - t2e * t2c[2]) / t2h;
		
		if (p1 > 0)
		{
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				f64 t3a = -t2e * bt2data[index2[0]].data2[0][n2] / t2h;		// D sin(alpha)
				f64 t3b = -t2e * bt2data[index2[0]].data2[2][n2] / t2h;		// D sin(alpha)
				
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					f64 t4a = t2h * t2b[n2][n3] - t2a[n3] * t3a;
					f64 t4b = t2h * bt2data[index2[0]].data2[0][n2] - t2e * t3a;
					f64 t4c = (t4a - t2c[n3] * t4b) / t2g;
					
					f64 t5a = t3b * (t2a[n3] - t2e * t2c[n3]);
					f64 t5b = t2h * (t2e * t2d[n2][n3] + t2c[n3] * bt2data[index2[0]].data2[2][n2]);
					f64 t5c = -(t5a + t5b) / t2g;
					
					bt3_vector[n1].dbv[0][0][n2][n3] = t4c;
					bt3_vector[n1].dbv[0][1][n2][n3] = -(t4c + t5c);
					bt3_vector[n1].dbv[0][2][n2][n3] = t5c;
				}
			}
		}
		
		// 2nd basis vector... 2nd basis vector... 2nd basis vector... 2nd basis vector...
		// 2nd basis vector... 2nd basis vector... 2nd basis vector... 2nd basis vector...
		// 2nd basis vector... 2nd basis vector... 2nd basis vector... 2nd basis vector...
		
		f64 t2i[3];
		t2i[0] = bt1data[index1[0]].data2[dir1[0]][1] * bt1data[index1[1]].data2[dir1[1]][2] -
			bt1data[index1[0]].data2[dir1[0]][2] * bt1data[index1[1]].data2[dir1[1]][1];
		t2i[1] = bt1data[index1[0]].data2[dir1[0]][2] * bt1data[index1[1]].data2[dir1[1]][0] -
			bt1data[index1[0]].data2[dir1[0]][0] * bt1data[index1[1]].data2[dir1[1]][2];
		t2i[2] = bt1data[index1[0]].data2[dir1[0]][0] * bt1data[index1[1]].data2[dir1[1]][1] -
			bt1data[index1[0]].data2[dir1[0]][1] * bt1data[index1[1]].data2[dir1[1]][0];
		
		bt3_vector[n1].bv[1][0] = t2i[0] / t2h;
		bt3_vector[n1].bv[1][1] = t2i[1] / t2h;
		bt3_vector[n1].bv[1][2] = t2i[2] / t2h;
		
		if (p1 > 0)
		{
			for (i32s n2 = 0;n2 < 3;n2++)
			{
	f64 t3a[2];	// ???epsilon i
	t3a[0] = bt2data[index2[0]].data2[0][n2] * bt2data[index2[0]].data1 / t2g;
	t3a[1] = bt2data[index2[0]].data2[2][n2] * bt2data[index2[0]].data1 / t2g;
	
	f64 t3b[2];	// ???sigma ii / r2X
	t3b[0] = (1.0 - bt1data[index1[0]].data2[dir1[0]][n2] * bt1data[index1[0]].data2[dir1[0]][n2]) / bt1data[index1[0]].data1;
	t3b[1] = (1.0 - bt1data[index1[1]].data2[dir1[1]][n2] * bt1data[index1[1]].data2[dir1[1]][n2]) / bt1data[index1[1]].data1;
	
	i32s n9[2];
	n9[0] = (n2 + 1) % 3;	// index j
	n9[1] = (n2 + 2) % 3;	// index k
	
	f64 t3c[2];	// ???sigma ij / r2X
	t3c[0] = -bt1data[index1[0]].data2[dir1[0]][n2] * bt1data[index1[0]].data2[dir1[0]][n9[0]] / bt1data[index1[0]].data1;
	t3c[1] = -bt1data[index1[1]].data2[dir1[1]][n2] * bt1data[index1[1]].data2[dir1[1]][n9[0]] / bt1data[index1[1]].data1;
	
	f64 t3d[2];	// ???sigma ik / r2X
	t3d[0] = -bt1data[index1[0]].data2[dir1[0]][n2] * bt1data[index1[0]].data2[dir1[0]][n9[1]] / bt1data[index1[0]].data1;
	t3d[1] = -bt1data[index1[1]].data2[dir1[1]][n2] * bt1data[index1[1]].data2[dir1[1]][n9[1]] / bt1data[index1[1]].data1;
	
	bt3_vector[n1].dbv[1][0][n2][n2] = (t3c[0] * bt1data[index1[1]].data2[dir1[1]][n9[1]] -
		t3d[0] * bt1data[index1[1]].data2[dir1[1]][n9[0]] + t2i[n2] * t3a[0]) / t2h;
	bt3_vector[n1].dbv[1][0][n2][n9[0]] = (t3d[0] * bt1data[index1[1]].data2[dir1[1]][n2] -
		t3b[0] * bt1data[index1[1]].data2[dir1[1]][n9[1]] + t2i[n9[0]] * t3a[0]) / t2h;
	bt3_vector[n1].dbv[1][0][n2][n9[1]] = (t3b[0] * bt1data[index1[1]].data2[dir1[1]][n9[0]] -
		t3c[0] * bt1data[index1[1]].data2[dir1[1]][n2] + t2i[n9[1]] * t3a[0]) / t2h;
	
	bt3_vector[n1].dbv[1][2][n2][n2] = (t3d[1] * bt1data[index1[0]].data2[dir1[0]][n9[0]] -
		t3c[1] * bt1data[index1[0]].data2[dir1[0]][n9[1]] + t2i[n2] * t3a[1]) / t2h;
	bt3_vector[n1].dbv[1][2][n2][n9[0]] = (t3b[1] * bt1data[index1[0]].data2[dir1[0]][n9[1]] -
		t3d[1] * bt1data[index1[0]].data2[dir1[0]][n2] + t2i[n9[0]] * t3a[1]) / t2h;
	bt3_vector[n1].dbv[1][2][n2][n9[1]] = (t3c[1] * bt1data[index1[0]].data2[dir1[0]][n2] -
		t3b[1] * bt1data[index1[0]].data2[dir1[0]][n9[0]] + t2i[n9[1]] * t3a[1]) / t2h;
			}
			
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				bt3_vector[n1].dbv[1][1][n2][0] = -(bt3_vector[n1].dbv[1][0][n2][0] + bt3_vector[n1].dbv[1][2][n2][0]);
				bt3_vector[n1].dbv[1][1][n2][1] = -(bt3_vector[n1].dbv[1][0][n2][1] + bt3_vector[n1].dbv[1][2][n2][1]);
				bt3_vector[n1].dbv[1][1][n2][2] = -(bt3_vector[n1].dbv[1][0][n2][2] + bt3_vector[n1].dbv[1][2][n2][2]);
			}
		}
		
		// dipole vector... dipole vector... dipole vector... dipole vector...
		// dipole vector... dipole vector... dipole vector... dipole vector...
		// dipole vector... dipole vector... dipole vector... dipole vector...
		
		f64 csd = cos(bt3_vector[n1].pbdd);
		f64 snd = sin(bt3_vector[n1].pbdd);
		
		bt3_vector[n1].dv[0] = csd * bt3_vector[n1].bv[0][0] - snd * bt3_vector[n1].bv[1][0];
		bt3_vector[n1].dv[1] = csd * bt3_vector[n1].bv[0][1] - snd * bt3_vector[n1].bv[1][1];
		bt3_vector[n1].dv[2] = csd * bt3_vector[n1].bv[0][2] - snd * bt3_vector[n1].bv[1][2];
		
////////// debug!!!
//energy_bt3 += bt3_vector[n1].dv[0] + bt3_vector[n1].dv[1] + bt3_vector[n1].dv[2];
////////// debug!!!

		if (p1 > 0)
		{
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				for (i32s n3 = 0;n3 < 3;n3++)
				{
	bt3_vector[n1].ddv[0][n2][n3] = csd * bt3_vector[n1].dbv[0][0][n2][n3] - snd * bt3_vector[n1].dbv[1][0][n2][n3];
	bt3_vector[n1].ddv[1][n2][n3] = csd * bt3_vector[n1].dbv[0][1][n2][n3] - snd * bt3_vector[n1].dbv[1][1][n2][n3];
	bt3_vector[n1].ddv[2][n2][n3] = csd * bt3_vector[n1].dbv[0][2][n2][n3] - snd * bt3_vector[n1].dbv[1][2][n2][n3];
	bt3_vector[n1].ddv[3][n2][n3] = 0.0;
				}
			}
			
			if (bt3_vector[n1].dip_type == TYPE_LOOP)
			{
				for (i32s n2 = 0;n2 < 3;n2++)
				{
					for (i32s n3 = 0;n3 < 3;n3++)
					{
	bt3_vector[n1].ddv[0][n2][n3] -= snd * dpbdd[0][n2] * bt3_vector[n1].bv[0][n3] + csd * dpbdd[0][n2] * bt3_vector[n1].bv[1][n3];
	bt3_vector[n1].ddv[1][n2][n3] -= snd * dpbdd[1][n2] * bt3_vector[n1].bv[0][n3] + csd * dpbdd[1][n2] * bt3_vector[n1].bv[1][n3];
	bt3_vector[n1].ddv[2][n2][n3] -= snd * dpbdd[2][n2] * bt3_vector[n1].bv[0][n3] + csd * dpbdd[2][n2] * bt3_vector[n1].bv[1][n3];
	bt3_vector[n1].ddv[3][n2][n3] -= snd * dpbdd[3][n2] * bt3_vector[n1].bv[0][n3] + csd * dpbdd[3][n2] * bt3_vector[n1].bv[1][n3];
					}
				}
			}
			
////////// debug!!!
//for (i32s n2 = 0;n2 < 3;n2++) {
//for (i32s n3 = 0;n3 < 3;n3++) {
//d1[atmi[0]][n2] += bt3_vector[n1].ddv[0][n2][n3];
//d1[atmi[1]][n2] += bt3_vector[n1].ddv[1][n2][n3];
//d1[atmi[2]][n2] += bt3_vector[n1].ddv[2][n2][n3];
//d1[atmi[3]][n2] += bt3_vector[n1].ddv[3][n2][n3]; } }
////////// debug!!!

		}
	}
}

// it seems that we compute here the same cross product than in BT3... combine ??????????
// it seems that we compute here the same cross product than in BT3... combine ??????????
// it seems that we compute here the same cross product than in BT3... combine ??????????

void mm2_eng::ComputeBT4(i32s p1)
{
	energy_bt4 = 0.0;
	
	for (i32u n1 = 0;n1 < bt4_vector.size();n1++)
	{
		i32s * atma = bt2_vector[bt4_vector[n1].index2].atmi;
		i32s atmb = bt1_vector[bt4_vector[n1].index1].atmi[1];
		
		i32s * index1a = bt2_vector[bt4_vector[n1].index2].index1;
		bool * dir1a = bt2_vector[bt4_vector[n1].index2].dir1;
		
		i32s index1b = bt4_vector[n1].index1;
		i32s index2 = bt4_vector[n1].index2;
		
		f64 t1a[3];
		f64 t1b = 1.0 - bt2data[index2].data1 * bt2data[index2].data1; f64 t1c = sqrt(t1b);
		t1a[0] = bt1data[index1a[0]].data2[dir1a[0]][1] * bt1data[index1a[1]].data2[dir1a[1]][2] -
			bt1data[index1a[0]].data2[dir1a[0]][2] * bt1data[index1a[1]].data2[dir1a[1]][1];
		t1a[1] = bt1data[index1a[0]].data2[dir1a[0]][2] * bt1data[index1a[1]].data2[dir1a[1]][0] -
			bt1data[index1a[0]].data2[dir1a[0]][0] * bt1data[index1a[1]].data2[dir1a[1]][2];
		t1a[2] = bt1data[index1a[0]].data2[dir1a[0]][0] * bt1data[index1a[1]].data2[dir1a[1]][1] -
			bt1data[index1a[0]].data2[dir1a[0]][1] * bt1data[index1a[1]].data2[dir1a[1]][0];
		
		f64 t2a[3];
		f64 t2b = 2.0 * (bt2data[index2].data1 + 1.0); f64 t2c = sqrt(t2b);
		t2a[0] = bt1data[index1a[0]].data2[dir1a[0]][0] + bt1data[index1a[1]].data2[dir1a[1]][0];
		t2a[1] = bt1data[index1a[0]].data2[dir1a[0]][1] + bt1data[index1a[1]].data2[dir1a[1]][1];
		t2a[2] = bt1data[index1a[0]].data2[dir1a[0]][2] + bt1data[index1a[1]].data2[dir1a[1]][2];
		
		static const f64 sb = sin(M_PI * BETA / 180.0);		// sin(beta) = cos(gamma)
		static const f64 cb = cos(M_PI * BETA / 180.0);		// cos(beta) = sin(gamma)
		
		f64 t3a[3];		// vector e
		t3a[0] = cb * t1a[0] / t1c - sb * t2a[0] / t2c;
		t3a[1] = cb * t1a[1] / t1c - sb * t2a[1] / t2c;
		t3a[2] = cb * t1a[2] / t1c - sb * t2a[2] / t2c;
		
		f64 t3b[3];		// cos(kappa)
		t3b[0] = t3a[0] * bt1data[index1b].data2[1][0];
		t3b[1] = t3a[1] * bt1data[index1b].data2[1][1];
		t3b[2] = t3a[2] * bt1data[index1b].data2[1][2];
		f64 t3c = t3b[0] + t3b[1] + t3b[2];
		
		if (t3c < -1.0) t3c = -1.0;	// domain check...
		if (t3c > +1.0) t3c = +1.0;	// domain check...
		
// here like in BT2 if the kappa-angles approach 180.0 deg we will have some
// serious numerical problems in later stages. here is how we can monitor this...

bool problem1 = (t3c < -0.999);		// is it too close to 180 deg ???
if (problem1) { cout << "BT4 problem!!!" << endl; exit(EXIT_FAILURE); }

		// f = a(x-b)^2
		// df/dx = 2a(x-b)
		
		f64 t3g = t3c - bt4_vector[n1].opt;
		f64 t9a = bt4_vector[n1].fc * t3g * t3g;
		f64 t9b = 2.0 * bt4_vector[n1].fc * t3g;
		
		energy_bt4 += t9a;
		
		// f = a/(x-1)^4
		// df/dx = -4a/(x-1)^5
		
		// fc is here the same as in BT2..
		// fc is here the same as in BT2..
		// fc is here the same as in BT2..
		
		f64 t9x = t3c - 1.0;
		
		f64 t9y = t9x * t9x * t9x * t9x;
		energy_bt4 += GetModel()->prm.wtmp3 / t9y;
		
		f64 t9z = t9y * t9x;
		t9b -= 4.0 * GetModel()->prm.wtmp3 / t9z;
		
		f64 t3d = 1.0 - t3c * t3c;	// sin(kappa)^2
		f64 t3e = sqrt(t3d);		// sin(kappa)
		
// here we will calculate cos(lambda) with the derivatives. however, we also have terms which
// depend on sin(lambda), and there we will run into problems at 0 deg and 180 deg.

// this problem is mainly due to bad design, but the sine-terms are still quite important...

// easiest way out is to cheat a bit and use the LOWLIMIT-idea also here... this is certainly
// NOT an ideal fix, since it will affect also accuracy, but still better than dividing by zero.

		f64 t4a = 0.0;			// cos(lambda)
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			t4a += sb / t3e * bt1data[index1b].data2[1][n2] * t1a[n2] / t1c;
			t4a += cb / t3e * bt1data[index1b].data2[1][n2] * t2a[n2] / t2c;
		}
		
		if (t4a < -1.0) t4a = -1.0;	// domain check...
		if (t4a > +1.0) t4a = +1.0;	// domain check...
		
		f64 t4b[3];		// vector h
		t4b[0] = (bt1data[index1b].data2[1][0] - t3a[0] * t3c) / t3e;
		t4b[1] = (bt1data[index1b].data2[1][1] - t3a[1] * t3c) / t3e;
		t4b[2] = (bt1data[index1b].data2[1][2] - t3a[2] * t3c) / t3e;
		
		f64 t4c[3];		// vector k
		t4c[0] = (t2a[1] * t1a[2] - t2a[2] * t1a[1]) / (t2c * t1c);
		t4c[1] = (t2a[2] * t1a[0] - t2a[0] * t1a[2]) / (t2c * t1c);
		t4c[2] = (t2a[0] * t1a[1] - t2a[1] * t1a[0]) / (t2c * t1c);
		
		f64 t4d = t4b[0] * t4c[0] + t4b[1] * t4c[1] + t4b[2] * t4c[2];
		
		f64 t4e = sqrt(1.0 - t4a * t4a);	// sin(lambda)
		
		if (t4e > +1.0) t4e = +1.0;			// domain check...
		if (t4e < LOWLIMIT) t4e = LOWLIMIT;		// apply the LOWLIMIT cheat...
		if (t4d < 0.0) t4e = -t4e;			// sign check...
		
		f64 t9c = t4a * t4a;		// x*x
		f64 t9d = 4.0 * t9c - 1.0;	// 4*x*x-1
		
		// f1 = a*x + b*(2*x*x-1) + c*x*(4*x*x-3)
		// df1/dx = a + 4*b*x + 3*c*(4*x*x-1)
		
		f64 t9e = bt4_vector[n1].fscos[0] * t4a;
		t9e += bt4_vector[n1].fscos[1] * (2.0 * t9c - 1.0);
		t9e += bt4_vector[n1].fscos[2] * t4a * (4.0 * t9c - 3.0);
		
		f64 t9f = bt4_vector[n1].fscos[0];
		t9f += 4.0 * bt4_vector[n1].fscos[1] * t4a;
		t9f += 3.0 * bt4_vector[n1].fscos[2] * t9d;
		
		// f2 = d*Sy + e*2*Sy*x + f*Sy*(4*x*x-1)
		// df2/dx = -d*x/Sy + 2e*(-x*x/Sy+Sy) + f*(-x/Sy*(4*x*x-1)+8*x*Sy)
		
		t9e += bt4_vector[n1].fssin[0] * t4e;
		t9e += 2.0 * bt4_vector[n1].fssin[1] * t4e * t4a;
		t9e += bt4_vector[n1].fssin[2] * t4e * t9d;
		
		t9f -= bt4_vector[n1].fssin[0] * t4a / t4e;
		t9f += 2.0 * bt4_vector[n1].fssin[1] * (t4e - t9c / t4e);
		t9f += bt4_vector[n1].fssin[2] * (8.0 * t4a * t4e - t4a * t9d / t4e);
		
		energy_bt4 += t9e;
		
		if (p1 > 0)
		{
			for (i32s n2 = 0;n2 < 3;n2++)
			{
	f64 t6a[2];	// epsilon i
	t6a[0] = bt2data[index2].data2[0][n2] * bt2data[index2].data1 / t1b;
	t6a[1] = bt2data[index2].data2[2][n2] * bt2data[index2].data1 / t1b;
	
	f64 t6b[2];	// sigma ii / r2X
	t6b[0] = (1.0 - bt1data[index1a[0]].data2[dir1a[0]][n2] * bt1data[index1a[0]].data2[dir1a[0]][n2]) / bt1data[index1a[0]].data1;
	t6b[1] = (1.0 - bt1data[index1a[1]].data2[dir1a[1]][n2] * bt1data[index1a[1]].data2[dir1a[1]][n2]) / bt1data[index1a[1]].data1;
	
	i32s n3[2];
	n3[0] = (n2 + 1) % 3;		// index j
	n3[1] = (n2 + 2) % 3;		// index k
	
	f64 t6c[2];	// sigma ij / r2X
	t6c[0] = -bt1data[index1a[0]].data2[dir1a[0]][n2] * bt1data[index1a[0]].data2[dir1a[0]][n3[0]] / bt1data[index1a[0]].data1;
	t6c[1] = -bt1data[index1a[1]].data2[dir1a[1]][n2] * bt1data[index1a[1]].data2[dir1a[1]][n3[0]] / bt1data[index1a[1]].data1;
	
	f64 t6d[2];	// sigma ik / r2X
	t6d[0] = -bt1data[index1a[0]].data2[dir1a[0]][n2] * bt1data[index1a[0]].data2[dir1a[0]][n3[1]] / bt1data[index1a[0]].data1;
	t6d[1] = -bt1data[index1a[1]].data2[dir1a[1]][n2] * bt1data[index1a[1]].data2[dir1a[1]][n3[1]] / bt1data[index1a[1]].data1;
	
	f64 t5a[2][3];	// components of dc/di
	t5a[0][n2] = (t6c[0] * bt1data[index1a[1]].data2[dir1a[1]][n3[1]] -
		t6d[0] * bt1data[index1a[1]].data2[dir1a[1]][n3[0]] + t1a[n2] * t6a[0]) / t1c;
	t5a[0][n3[0]] = (t6d[0] * bt1data[index1a[1]].data2[dir1a[1]][n2] -
		t6b[0] * bt1data[index1a[1]].data2[dir1a[1]][n3[1]] + t1a[n3[0]] * t6a[0]) / t1c;
	t5a[0][n3[1]] = (t6b[0] * bt1data[index1a[1]].data2[dir1a[1]][n3[0]] -
		t6c[0] * bt1data[index1a[1]].data2[dir1a[1]][n2] + t1a[n3[1]] * t6a[0]) / t1c;
	t5a[1][n2] = (t6d[1] * bt1data[index1a[0]].data2[dir1a[0]][n3[0]] -
		t6c[1] * bt1data[index1a[0]].data2[dir1a[0]][n3[1]] + t1a[n2] * t6a[1]) / t1c;
	t5a[1][n3[0]] = (t6b[1] * bt1data[index1a[0]].data2[dir1a[0]][n3[1]] -
		t6d[1] * bt1data[index1a[0]].data2[dir1a[0]][n2] + t1a[n3[0]] * t6a[1]) / t1c;
	t5a[1][n3[1]] = (t6c[1] * bt1data[index1a[0]].data2[dir1a[0]][n2] -
		t6b[1] * bt1data[index1a[0]].data2[dir1a[0]][n3[0]] + t1a[n3[1]] * t6a[1]) / t1c;
	
	f64 t5b[2][3];	// components of dd/di
	f64 t6e = (bt1data[index1a[0]].data2[dir1a[0]][n2] + bt1data[index1a[1]].data2[dir1a[1]][n2]) / t2b;
	t5b[0][n2] = (t6b[0] - bt2data[index2].data2[0][n2] * t6e) / t2c;
	t5b[1][n2] = (t6b[1] - bt2data[index2].data2[2][n2] * t6e) / t2c;
	t6e = (bt1data[index1a[0]].data2[dir1a[0]][n3[0]] + bt1data[index1a[1]].data2[dir1a[1]][n3[0]]) / t2b;
	t5b[0][n3[0]] = (t6c[0] - bt2data[index2].data2[0][n2] * t6e) / t2c;
	t5b[1][n3[0]] = (t6c[1] - bt2data[index2].data2[2][n2] * t6e) / t2c;
	t6e = (bt1data[index1a[0]].data2[dir1a[0]][n3[1]] + bt1data[index1a[1]].data2[dir1a[1]][n3[1]]) / t2b;
	t5b[0][n3[1]] = (t6d[0] - bt2data[index2].data2[0][n2] * t6e) / t2c;
	t5b[1][n3[1]] = (t6d[1] - bt2data[index2].data2[2][n2] * t6e) / t2c;
				
				f64 t5c[4] = { 0.0, 0.0, 0.0, 0.0 };		// dcos(kappa)/di
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					t5c[0] += bt1data[index1b].data2[1][n3] * (cb * t5a[0][n3] - sb * t5b[0][n3]);
					t5c[2] += bt1data[index1b].data2[1][n3] * (cb * t5a[1][n3] - sb * t5b[1][n3]);
					
					if (n2 != n3) t6e = -bt1data[index1b].data2[1][n2] * bt1data[index1b].data2[1][n3];
					else t6e = 1.0 - bt1data[index1b].data2[1][n2] * bt1data[index1b].data2[1][n2];
					t5c[3] += t3a[n3] * t6e / bt1data[index1b].data1;
				}
				
				t5c[1] = -(t5c[0] + t5c[2] + t5c[3]);
				
				d1[atma[0]][n2] += t9b * t5c[0];
				d1[atma[1]][n2] += t9b * t5c[1];
				d1[atma[2]][n2] += t9b * t5c[2];
				d1[atmb][n2] += t9b * t5c[3];
				
				f64 t7e[3];
				t7e[0] = t3c * t5c[0] / t3d;
				t7e[1] = t3c * t5c[2] / t3d;
				t7e[2] = t3c * t5c[3] / t3d;
				
				f64 t5d[4] = { 0.0, 0.0, 0.0, 0.0 };		// dcos(lambda)/di
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					f64 t7a = t1a[n3] / t1c;	// f64 t7b = t7a * t7a;
					f64 t7c = t2a[n3] / t2c;	// f64 t7d = t7c * t7c;
					
					t5d[0] += sb * bt1data[index1b].data2[1][n3] * (t7e[0] * t7a + t5a[0][n3]) / t3e;
					t5d[0] += cb * bt1data[index1b].data2[1][n3] * (t7e[0] * t7c + t5b[0][n3]) / t3e;
					
					t5d[2] += sb * bt1data[index1b].data2[1][n3] * (t7e[1] * t7a + t5a[1][n3]) / t3e;
					t5d[2] += cb * bt1data[index1b].data2[1][n3] * (t7e[1] * t7c + t5b[1][n3]) / t3e;
					
					if (n2 != n3) t6e = -bt1data[index1b].data2[1][n2] * bt1data[index1b].data2[1][n3];
					else t6e = 1.0 - bt1data[index1b].data2[1][n2] * bt1data[index1b].data2[1][n2];
					
					f64 t7f = t7e[2] * bt1data[index1b].data2[1][n3] + t6e / bt1data[index1b].data1;
					t5d[3] += (sb * t7a + cb * t7c) * t7f / t3e;
				}
				
				t5d[1] = -(t5d[0] + t5d[2] + t5d[3]);
				
				d1[atma[0]][n2] += t9f * t5d[0];
				d1[atma[1]][n2] += t9f * t5d[1];
				d1[atma[2]][n2] += t9f * t5d[2];
				d1[atmb][n2] += t9f * t5d[3];
			}
		}
	}
}

void mm2_eng::ComputeNBT1(i32s p1)
{
	energy_nbt1a = 0.0;
	energy_nbt1b = 0.0;
	
	for (i32u n1 = 0;n1 < nbt1_vector.size();n1++)
	{
		i32s * atmi = nbt1_vector[n1].atmi;
		
		f64 t1a[3]; f64 t1b = 0.0;
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 t2a = crd[atmi[0]][n2];
			f64 t2b = crd[atmi[1]][n2];
			
			t1a[n2] = t2a - t2b;
			t1b += t1a[n2] * t1a[n2];
		}
		
		f64 t1c = sqrt(t1b);
		
		/*##############################################*/
		/*##############################################*/
		// this is for the surface area term...
		
		bool first = (atmi[0] > atmi[1]);
		dist2[dist1[atmi[first]] + (atmi[!first] - atmi[first]) - 1] = t1c;
		
		if (t1c < (vdwr1[atmi[0]] + vdwr1[atmi[1]]))
		{
			nbt3_nl[atmi[0]].index[nbt3_nl[atmi[0]].index_count++] = atmi[1];
			if (nbt3_nl[atmi[0]].index_count >= SIZE_NL) { cout << "NL overflow!!!" << endl; exit(EXIT_FAILURE); }
			
			nbt3_nl[atmi[1]].index[nbt3_nl[atmi[1]].index_count++] = atmi[0];
			if (nbt3_nl[atmi[1]].index_count >= SIZE_NL) { cout << "NL overflow!!!" << endl; exit(EXIT_FAILURE); }
		}
		
		// this is for the surface area term...
		/*##############################################*/
		/*##############################################*/
		
		// f = ((x+s)/a)^-9 - ((x+s)/b)^-6
		// df/dx = -9/a((x+s)/a)^-10 + 6/b((x+s)/b)^-7
		
		f64 t3a = t1c / nbt1_vector[n1].data[0];
		f64 t3b = t1c / nbt1_vector[n1].data[1];
		
		f64 t4a = t3a * t3a; f64 t4b = t4a * t3a; f64 t4c = t4a * t4b;
		f64 t5a = t3b * t3b; f64 t5b = t5a * t3b;
		
		energy_nbt1a += 1.0 / (t4b * t4b * t4b) - 1.0 / (t5b * t5b);
		
		// this complicated type checking stuff should be done earlier, outside this loop!!!
		// this complicated type checking stuff should be done earlier, outside this loop!!!
		// this complicated type checking stuff should be done earlier, outside this loop!!!
		
		bool flag1 = ((nbtp[atmi[0]] & NBTP_CHARGED) && (nbtp[atmi[1]] & NBTP_CHARGED));
		bool flag2 = ((nbtp[atmi[0]] & NBTP_CSIGN) ^ (nbtp[atmi[1]] & NBTP_CSIGN));
		
		bool flag3a = ((nbtp[atmi[0]] & NBTP_POLAR) && (nbtp[atmi[1]] & NBTP_CHARGED));
		bool flag3b = ((nbtp[atmi[0]] & NBTP_CHARGED) && (nbtp[atmi[1]] & NBTP_POLAR));
		bool flag4 = (flag3a || flag3b);
		
	// e = A + B/(1+k*exp(L*B*r))
	// de/dr = -(exp(L*B*r)*B^2*k*L)/(1+k*exp(L*B*r))^2
	
	f64 t7a = exp(GetModel()->prm.screen2 * GetModel()->prm.screen4 * t1c);
	f64 t7b = 1.0 + GetModel()->prm.screen3 * t7a; f64 t7c = t7b * t7b;
	f64 t7d = GetModel()->prm.screen1 + GetModel()->prm.screen2 / t7b;
	
	f64 t7e = t7d * t7d;
	
	f64 t7f = t7a * GetModel()->prm.screen2 * GetModel()->prm.screen2 * GetModel()->prm.screen3 * GetModel()->prm.screen4;
	
		f64 qq = 0.0;
		
		if (flag1)
		{
			qq = 138.9354518;
			if (flag2) qq = -qq;

			// f = Q/(e*r)
			// df/dr = -Q * ((1/e*r^2) + (de/dr)/(e^2*r))
			
			energy_nbt1b += qq / (t7d * t1c);
		}
		else if (flag4)
		{
			qq = -138.9354518 * GetModel()->prm.pbility1;
			
			// f = Q/(e*r^4)
			// df/dr = -Q * ((4/e*r^5) + (de/dr)/(e^2*r^4))
			
			energy_nbt1b += qq / (t7d * t1b * t1b);
		}
		
		if (p1 > 0)
		{
			f64 t6a = 9.0 / (nbt1_vector[n1].data[0] * t4c * t4c);
			f64 t6b = 6.0 / (nbt1_vector[n1].data[1] * t5a * t5a * t5b);
			f64 t2a = t6b - t6a;
			
			if (flag1) t2a += -qq * (1.0 / (t7d * t1b) - t7f / (t7c * t7e * t1c));
			else if (flag4) t2a += -qq * (4.0 / (t7d * t1b * t1b * t1c) - t7f / (t7c * t7e * t1b * t1b));
			
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				f64 t1e = (t1a[n2] / t1c) * t2a;
				
				d1[atmi[0]][n2] += t1e;
				d1[atmi[1]][n2] -= t1e;
			}
		}
	}
}

void mm2_eng::ComputeNBT2(i32s p1)
{
	energy_nbt2a = 0.0;
	energy_nbt2b = 0.0;
	
	// dipole moment for the peptide unit:
	// CRC handbook of Chem & Phys, 1st Student Edition, 1988, CRC Press Inc. (page E-53)
	// "acetyl methylamine" 3.73 debyes -> 3.73 * 3.338e-30 / 1.6021773e-19 * 1.0e+09 = 0.07771137
	
/*##############################################*/
/*##############################################*/

	for (i32u n1 = 0;n1 < bt3_vector.size();n1++)
	{
		if (bt3_vector[n1].skip) continue;	// skip all X-pro cases...
		i32s * atmi = bt3_vector[n1].atmi;
		
		for (i32u n2 = 0;n2 < index_vector.size();n2++)
		{
			if ((nbtp[n2] & NBTP_NEUTRAL)) continue;
			if (atmi[1] == (i32s) n2 || atmi[2] == (i32s) n2) continue;
			
			f64 t1a[3];
			t1a[0] = crd[n2][0] - 0.5 * (crd[atmi[1]][0] + crd[atmi[2]][0]);
			t1a[1] = crd[n2][1] - 0.5 * (crd[atmi[1]][1] + crd[atmi[2]][1]);
			t1a[2] = crd[n2][2] - 0.5 * (crd[atmi[1]][2] + crd[atmi[2]][2]);
			
			f64 t1b = t1a[0] * t1a[0] + t1a[1] * t1a[1] + t1a[2] * t1a[2];
			f64 t1c = sqrt(t1b); f64 t1d[2][3]; f64 t1e[2][3][3];
			
			// t1d = dr/dx, and therefore t1d[0] is a unit vector...
			// t1d = dr/dx, and therefore t1d[0] is a unit vector...
			// t1d = dr/dx, and therefore t1d[0] is a unit vector...
			
			for (i32s n3 = 0;n3 < 3;n3++)
			{
				t1d[0][n3] = t1a[n3] / t1c;
				t1d[1][n3] = -0.5 * t1d[0][n3];
			}
			
			f64 t1f[3]; f64 t1g[5][3];			// theta
			t1f[0] = t1d[0][0] * bt3_vector[n1].dv[0];
			t1f[1] = t1d[0][1] * bt3_vector[n1].dv[1];
			t1f[2] = t1d[0][2] * bt3_vector[n1].dv[2];
			f64 t1h = t1f[0] + t1f[1] + t1f[2];
			
////////// debug!!!
//energy_nbt2 += t1h;
////////// debug!!!

			if (p1 > 0)
			{
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					for (i32s n4 = 0;n4 < 3;n4++)
					{
						if (n3 != n4) t1e[0][n3][n4] = -t1d[0][n3] * t1d[0][n4] / t1c;
						else t1e[0][n3][n4] = (1.0 - t1d[0][n3] * t1d[0][n3]) / t1c;
						t1e[1][n3][n4] = -0.5 * t1e[0][n3][n4];
					}
				}
				
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					t1g[0][n3] = t1g[1][n3] = t1g[2][n3] = t1g[3][n3] = t1g[4][n3] = 0.0;
					
					for (i32s n4 = 0;n4 < 3;n4++)
					{
	t1g[0][n3] += t1d[0][n4] * bt3_vector[n1].ddv[0][n3][n4];
	t1g[1][n3] += t1d[0][n4] * bt3_vector[n1].ddv[1][n3][n4] + t1e[1][n3][n4] * bt3_vector[n1].dv[n4];
	t1g[2][n3] += t1d[0][n4] * bt3_vector[n1].ddv[2][n3][n4] + t1e[1][n3][n4] * bt3_vector[n1].dv[n4];
	t1g[3][n3] += t1d[0][n4] * bt3_vector[n1].ddv[3][n3][n4];
	t1g[4][n3] += t1e[0][n3][n4] * bt3_vector[n1].dv[n4];
					}
					
////////// debug!!!
//d1[atmi[0]][n3] += t1g[0][n3];
//d1[atmi[1]][n3] += t1g[1][n3];
//d1[atmi[2]][n3] += t1g[2][n3];
//d1[atmi[3]][n3] += t1g[3][n3];
//d1[n2][n3] += t1g[4][n3];
////////// debug!!!

				}
			}
			
			bool polar = ((nbtp[n2] & NBTP_POLAR));
			
	f64 t9x = t1b * t1b;	// this is r^4 !!!!!!!
	f64 t9y = t9x * t9x;	// this is r^8 !!!!!!!
	
	// e = A + B/(1+k*exp(L*B*r)) + Z/r^9
	// de/dr = -(exp(L*B*r)*B^2*k*L)/(1+k*exp(L*B*r))^2 - 9Z/r^10
	
	f64 t7a = exp(GetModel()->prm.screen2 * GetModel()->prm.screen4 * t1c);
	f64 t7b = 1.0 + GetModel()->prm.screen3 * t7a; f64 t7c = t7b * t7b;
	f64 t7d = GetModel()->prm.screen1 + GetModel()->prm.screen2 / t7b + GetModel()->prm.screen5 / (t9y * t1c);
	
	f64 t7e = t7d * t7d;
	
	f64 t7f = t7a * GetModel()->prm.screen2 * GetModel()->prm.screen2 * GetModel()->prm.screen3 * GetModel()->prm.screen4;
	f64 t7g = 9.0 * GetModel()->prm.screen5 / (t9y * t1b);
	
			f64 qd;
			
			if (!polar)
			{
				qd = 138.9354518 * 0.077711 * GetModel()->prm.dipole1;
				if (nbtp[n2] & NBTP_CSIGN) qd = -qd;
				
				// f = Q/(e*r^2)
				// df/dr = -Q * ((2/e*r^3) + (de/dr)/(e^2*r^2))
				
				energy_nbt2a += qd * t1h / (t7d * t1b);
			}
			else
			{
				qd = -138.9354518 * 0.077711 * GetModel()->prm.dipole1 * GetModel()->prm.pbility2;
				if (t1h < 0.0) qd = -qd;	// this term is always favourable!!!!!
				
				// f = Q/(e*r^6)
				// df/dr = -Q * ((6/e*r^7) + (de/dr)/(e^2*r^6))
				
				energy_nbt2a += qd * t1h / (t7d * t9x * t1b);
			}
			
			if (p1 > 0)
			{
				f64 t3a; f64 t3b;
				
				if (!polar)
				{
					t3a = -qd * t1h * (2.0 / (t7d * t1b * t1c) - (t7f / t7c + t7g) / (t7e * t1b));
					t3b = qd / (t7d * t1b);
				}
				else
				{
					t3a = -qd * t1h * (6.0 / (t7d * t9x * t1b * t1c) - (t7f / t7c + t7g) / (t7e * t9x * t1b));
					t3b = qd / (t7d * t9x * t1b);
				}
				
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					d1[atmi[1]][n3] += t3a * t1d[1][n3];
					d1[atmi[2]][n3] += t3a * t1d[1][n3];
					d1[n2][n3] += t3a * t1d[0][n3];
					
					d1[atmi[0]][n3] += t3b * t1g[0][n3];
					d1[atmi[1]][n3] += t3b * t1g[1][n3];
					d1[atmi[2]][n3] += t3b * t1g[2][n3];
					d1[atmi[3]][n3] += t3b * t1g[3][n3];
					d1[n2][n3] += t3b * t1g[4][n3];
				}
			}
		}
	}
	
/*##############################################*/
/*##############################################*/

	for (i32s n1 = 0;n1 < ((i32s) bt3_vector.size()) - 1;n1++)
	{
		if (bt3_vector[n1].skip) continue;		// skip all X-pro cases...
		i32s * atmi1 = bt3_vector[n1].atmi;
		
		for (i32s n2 = n1 + 1;n2 < (i32s) bt3_vector.size();n2++)
		{
			if (bt3_vector[n2].skip) continue;	// skip all X-pro cases...
			i32s * atmi2 = bt3_vector[n2].atmi;
			
			f64 t1a[3];
			t1a[0] = 0.5 * (crd[atmi2[1]][0] + crd[atmi2[2]][0] - crd[atmi1[1]][0] - crd[atmi1[2]][0]);
			t1a[1] = 0.5 * (crd[atmi2[1]][1] + crd[atmi2[2]][1] - crd[atmi1[1]][1] - crd[atmi1[2]][1]);
			t1a[2] = 0.5 * (crd[atmi2[1]][2] + crd[atmi2[2]][2] - crd[atmi1[1]][2] - crd[atmi1[2]][2]);
			
			f64 t1b = t1a[0] * t1a[0] + t1a[1] * t1a[1] + t1a[2] * t1a[2];
			f64 t1c = sqrt(t1b); f64 t1d[3]; f64 t1e[3][3];
			
			// t1d = dr/dx, and therefore is a vector of constant length 1/2...
			// t1d = dr/dx, and therefore is a vector of constant length 1/2...
			// t1d = dr/dx, and therefore is a vector of constant length 1/2...
			
			t1d[0] = 0.5 * t1a[0] / t1c;
			t1d[1] = 0.5 * t1a[1] / t1c;
			t1d[2] = 0.5 * t1a[2] / t1c;
			
			f64 t1f[3]; f64 t1g[5][3];			// theta1 * 1/2
			t1f[0] = t1d[0] * bt3_vector[n1].dv[0];
			t1f[1] = t1d[1] * bt3_vector[n1].dv[1];
			t1f[2] = t1d[2] * bt3_vector[n1].dv[2];
			f64 t1h = t1f[0] + t1f[1] + t1f[2];
			
			f64 t1i[3]; f64 t1j[5][3];			// theta2 * 1/2
			t1i[0] = t1d[0] * bt3_vector[n2].dv[0];
			t1i[1] = t1d[1] * bt3_vector[n2].dv[1];
			t1i[2] = t1d[2] * bt3_vector[n2].dv[2];
			f64 t1k = t1i[0] + t1i[1] + t1i[2];
			
			f64 t1l[3]; f64 t1m[8][3];
			t1l[0] = bt3_vector[n1].dv[0] * bt3_vector[n2].dv[0];
			t1l[1] = bt3_vector[n1].dv[1] * bt3_vector[n2].dv[1];
			t1l[2] = bt3_vector[n1].dv[2] * bt3_vector[n2].dv[2];
			f64 t1n = t1l[0] + t1l[1] + t1l[2];
			
////////// debug!!!
//energy_nbt2 += t1h;
//energy_nbt2 += t1k;
//energy_nbt2 += t1n;
////////// debug!!!

			if (p1 > 0)
			{
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					for (i32s n4 = 0;n4 < 3;n4++)
					{
						if (n3 != n4) t1e[n3][n4] = -t1d[n3] * t1d[n4] / t1c;
						else t1e[n3][n4] = (0.25 - t1d[n3] * t1d[n3]) / t1c;
					}
				}
				
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					t1g[0][n3] = t1g[1][n3] = t1g[2][n3] = t1g[3][n3] = t1g[4][n3] = 0.0;
					t1j[0][n3] = t1j[1][n3] = t1j[2][n3] = t1j[3][n3] = t1j[4][n3] = 0.0;
					
					t1m[0][n3] = t1m[1][n3] = t1m[2][n3] = t1m[3][n3] = 0.0;
					t1m[4][n3] = t1m[5][n3] = t1m[6][n3] = t1m[7][n3] = 0.0;
					
					for (i32s n4 = 0;n4 < 3;n4++)
					{
	t1g[0][n3] += t1d[n4] * bt3_vector[n1].ddv[0][n3][n4];
	t1g[1][n3] += t1d[n4] * bt3_vector[n1].ddv[1][n3][n4] - t1e[n3][n4] * bt3_vector[n1].dv[n4];
	t1g[2][n3] += t1d[n4] * bt3_vector[n1].ddv[2][n3][n4] - t1e[n3][n4] * bt3_vector[n1].dv[n4];
	t1g[3][n3] += t1d[n4] * bt3_vector[n1].ddv[3][n3][n4];
	t1g[4][n3] += t1e[n3][n4] * bt3_vector[n1].dv[n4];
	
	t1j[0][n3] += t1d[n4] * bt3_vector[n2].ddv[0][n3][n4];
	t1j[1][n3] += t1d[n4] * bt3_vector[n2].ddv[1][n3][n4] + t1e[n3][n4] * bt3_vector[n2].dv[n4];
	t1j[2][n3] += t1d[n4] * bt3_vector[n2].ddv[2][n3][n4] + t1e[n3][n4] * bt3_vector[n2].dv[n4];
	t1j[3][n3] += t1d[n4] * bt3_vector[n2].ddv[3][n3][n4];
	t1j[4][n3] -= t1e[n3][n4] * bt3_vector[n2].dv[n4];
	
	t1m[0][n3] += bt3_vector[n1].ddv[0][n3][n4] * bt3_vector[n2].dv[n4];
	t1m[1][n3] += bt3_vector[n1].ddv[1][n3][n4] * bt3_vector[n2].dv[n4];
	t1m[2][n3] += bt3_vector[n1].ddv[2][n3][n4] * bt3_vector[n2].dv[n4];
	t1m[3][n3] += bt3_vector[n1].ddv[3][n3][n4] * bt3_vector[n2].dv[n4];
	t1m[4][n3] += bt3_vector[n1].dv[n4] * bt3_vector[n2].ddv[0][n3][n4];
	t1m[5][n3] += bt3_vector[n1].dv[n4] * bt3_vector[n2].ddv[1][n3][n4];
	t1m[6][n3] += bt3_vector[n1].dv[n4] * bt3_vector[n2].ddv[2][n3][n4];
	t1m[7][n3] += bt3_vector[n1].dv[n4] * bt3_vector[n2].ddv[3][n3][n4];
					}
					
////////// debug!!!
//d1[atmi1[0]][n3] += t1g[0][n3];
//d1[atmi1[1]][n3] += t1g[1][n3];
//d1[atmi1[2]][n3] += t1g[2][n3];
//d1[atmi1[3]][n3] += t1g[3][n3];
//d1[atmi2[1]][n3] += t1g[4][n3];
//d1[atmi2[2]][n3] += t1g[4][n3];
//
//d1[atmi2[0]][n3] += t1j[0][n3];
//d1[atmi2[1]][n3] += t1j[1][n3];
//d1[atmi2[2]][n3] += t1j[2][n3];
//d1[atmi2[3]][n3] += t1j[3][n3];
//d1[atmi1[1]][n3] += t1j[4][n3];
//d1[atmi1[2]][n3] += t1j[4][n3];
//
//d1[atmi1[0]][n3] += t1m[0][n3];
//d1[atmi1[1]][n3] += t1m[1][n3];
//d1[atmi1[2]][n3] += t1m[2][n3];
//d1[atmi1[3]][n3] += t1m[3][n3];
//d1[atmi2[0]][n3] += t1m[4][n3];
//d1[atmi2[1]][n3] += t1m[5][n3];
//d1[atmi2[2]][n3] += t1m[6][n3];
//d1[atmi2[3]][n3] += t1m[7][n3];
////////// debug!!!

				}
			}
			
	f64 t9x = t1b * t1b;	// this is r^4 !!!!!!!
	f64 t9y = t9x * t9x;	// this is r^8 !!!!!!!
	
	// e = A + B/(1+k*exp(L*B*r)) + Z/r^9
	// de/dr = -(exp(L*B*r)*B^2*k*L)/(1+k*exp(L*B*r))^2 - 9Z/r^10
	
	f64 t7a = exp(GetModel()->prm.screen2 * GetModel()->prm.screen4 * t1c);
	f64 t7b = 1.0 + GetModel()->prm.screen3 * t7a; f64 t7c = t7b * t7b;
	f64 t7d = GetModel()->prm.screen1 + GetModel()->prm.screen2 / t7b + GetModel()->prm.screen5 / (t9y * t1c);
	
	f64 t7e = t7d * t7d;
	
	f64 t7f = t7a * GetModel()->prm.screen2 * GetModel()->prm.screen2 * GetModel()->prm.screen3 * GetModel()->prm.screen4;
	f64 t7g = 9.0 * GetModel()->prm.screen5 / (t9y * t1b);
	
			// f = Q/(e*r^3)
			// df/dr = -Q * ((3/e*r^4) + (de/dr)/(e^2*r^3))
			
			f64 dd = 138.9354518 * 0.077711 * 0.077711 * GetModel()->prm.dipole1 * GetModel()->prm.dipole1;
			
			// this complicated type checking stuff should be done earlier, outside this loop!!!
			// this complicated type checking stuff should be done earlier, outside this loop!!!
			// this complicated type checking stuff should be done earlier, outside this loop!!!
			
			// enhance alpha-helices...
			// enhance alpha-helices...
			// enhance alpha-helices...
			
			if ((n2 - n1) == 3)
			{
				bool at1 = (bt3_vector[n1 + 0].dip_type == TYPE_HELIX);
				bool at2 = (bt3_vector[n1 + 1].dip_type == TYPE_HELIX);
				bool at3 = (bt3_vector[n1 + 2].dip_type == TYPE_HELIX);
				bool at4 = (bt3_vector[n1 + 3].dip_type == TYPE_HELIX);
				
				bool same_chn = (bt3_vector[n1].atmi[3] == bt3_vector[n2].atmi[0]);
				
				if (at1 && at2 && at3 && at4 && same_chn) dd *= GetModel()->prm.dipole2;
			}
			
			// enhance beta-sheets...
			// enhance beta-sheets...
			// enhance beta-sheets...
			
			bool bt1 = (bt3_vector[n1].dip_type == TYPE_STRAND);
			bool bt2 = (bt3_vector[n2].dip_type == TYPE_STRAND);
			if (bt1 && bt2) dd *= GetModel()->prm.dipole2;
			
			f64 t1x = dd / (t7d * t1b * t1c);	// radial part...
			f64 t1y = t1n - 12.0 * t1h * t1k;	// angular part...
			energy_nbt2b += t1x * t1y;
			
			if (p1 > 0)
			{
				f64 t3a = -dd * t1y * (3.0 / (t7d * t9x) - (t7f / t7c + t7g) / (t7e * t1b * t1c));
				f64 t3b = -12.0 * t1k * t1x;
				f64 t3c = -12.0 * t1h * t1x;
				
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					d1[atmi1[1]][n3] -= t3a * t1d[n3];
					d1[atmi1[2]][n3] -= t3a * t1d[n3];
					d1[atmi2[1]][n3] += t3a * t1d[n3];
					d1[atmi2[2]][n3] += t3a * t1d[n3];
					
					d1[atmi1[0]][n3] += t3b * t1g[0][n3];
					d1[atmi1[1]][n3] += t3b * t1g[1][n3];
					d1[atmi1[2]][n3] += t3b * t1g[2][n3];
					d1[atmi1[3]][n3] += t3b * t1g[3][n3];
					d1[atmi2[1]][n3] += t3b * t1g[4][n3];
					d1[atmi2[2]][n3] += t3b * t1g[4][n3];
					
					d1[atmi2[0]][n3] += t3c * t1j[0][n3];
					d1[atmi2[1]][n3] += t3c * t1j[1][n3];
					d1[atmi2[2]][n3] += t3c * t1j[2][n3];
					d1[atmi2[3]][n3] += t3c * t1j[3][n3];
					d1[atmi1[1]][n3] += t3c * t1j[4][n3];
					d1[atmi1[2]][n3] += t3c * t1j[4][n3];
					
					d1[atmi1[0]][n3] += t1x * t1m[0][n3];
					d1[atmi1[1]][n3] += t1x * t1m[1][n3];
					d1[atmi1[2]][n3] += t1x * t1m[2][n3];
					d1[atmi1[3]][n3] += t1x * t1m[3][n3];
					d1[atmi2[0]][n3] += t1x * t1m[4][n3];
					d1[atmi2[1]][n3] += t1x * t1m[5][n3];
					d1[atmi2[2]][n3] += t1x * t1m[6][n3];
					d1[atmi2[3]][n3] += t1x * t1m[7][n3];
				}
			}
		}
	}
}

// Richmond TJ : "Solvent Accessible Surface Area and Extended Volume in Proteins"
// J. Mol. Biol 178, 63-89 (1984)

// Fraczkiewicz R, Braun W : "Exact and Efficient Analytical Calculation of the Accessible
// Surface Areas and Their Gradients for Macromolecules" J. Comp. Chem 19, 319-333, (1998)

// Weiser J, Weiser AA, Shenkin PS, Still WC : "Neigbor-List Reduction: Optimization for
// Computation of Molecular van der Waals and Solvent-Accessible Surface Areas"
// J. Comp. Chem. 19, 797-808, (1998)

// Do Carmo MP : "Differential Geometry of Curves and Surfaces" Prentice Hall Inc., 1976

void mm2_eng::ComputeNBT3(i32s p1)
{
	energy_nbt3 = 0.0;
	
	for (i32u n1 = 0;n1 < index_vector.size();n1++)
	{
		if (sasa[n1] == 0.0) continue;
		
		// if the coefficient is zero, no calculations are needed!!!
		// so, it's good to choose value zero whenever possible...
		
		mm2_nbt3_nd ndt[100];	// max 50
		for (i32s n2 = 0;n2 < nbt3_nl[n1].index_count;n2++)
		{
			ndt[n2].index = nbt3_nl[n1].index[n2];
			i32s atmi[2] = { n1, ndt[n2].index }; bool first = (atmi[0] > atmi[1]);
			ndt[n2].distance = dist2[dist1[atmi[first]] + (atmi[!first] - atmi[first]) - 1];
		}
		
		sort(ndt, ndt + nbt3_nl[n1].index_count);
		i32s n_count = 0; i32s nt[SIZE_NT];
		
		// neighbor-list reduction...
		// neighbor-list reduction...
		// neighbor-list reduction...
		
		for (i32s n2 = 0;n2 < nbt3_nl[n1].index_count;n2++)
		{
			i32s ind1 = ndt[n2].index;
			f64 dij = ndt[n2].distance;
			
			bool flag = true;
			
			for (i32s n3 = n2 + 1;n3 < nbt3_nl[n1].index_count;n3++)
			{
				i32s ind2 = ndt[n3].index;
				
				i32s atmi[2] = { ind1, ind2 }; bool first = (atmi[0] > atmi[1]);
				f64 djk = dist2[dist1[atmi[first]] + (atmi[!first] - atmi[first]) - 1];
				
				if (djk > dij) continue;
				
				f64 dij2 = dij * dij; f64 djk2 = djk * djk;
				f64 dik = ndt[n3].distance; f64 dik2 = dik * dik;
				
				// here dij and dik both represent distances which should never be
				// very close to zero (if LJ-terms work as they should) -> no checking
				
				f64 ca = (vdwr2[n1] + dij2 - vdwr2[ind1]) / (2.0 * vdwr1[n1] * dij);
				f64 cb = (vdwr2[n1] + dik2 - vdwr2[ind2]) / (2.0 * vdwr1[n1] * dik);
				f64 cg = (dij2 + dik2 - djk2) / (2.0 * dij * dik);
				
				f64 sa2 = 1.0 - ca * ca;
				f64 sg2 = 1.0 - cg * cg;
				
				f64 dc = sa2 * sg2;
				if (dc < 0.0) dc = 0.0;		// domain check...
				
				if (cb < ca * cg - sqrt(dc))
				{
					flag = false;
					break;
				}
			}
			
			if (flag)
			{
				nt[n_count++] = ind1;
				if (n_count >= SIZE_NT) { cout << "NT overflow!!!" << endl; exit(EXIT_FAILURE); }
			}
		}
		
		i32s coi_count = 0; mm2_nbt3_coi coit[SIZE_COI];
		
		// next we will create the coi-table...
		// next we will create the coi-table...
		// next we will create the coi-table...
		
		for (i32s n2 = 0;n2 < n_count;n2++)
		{
			coit[coi_count].index = nt[n2]; coit[coi_count].flag = false;
			
			f64 t1a[3]; f64 t1b = 0.0;
			for (i32s n3 = 0;n3 < 3;n3++)
			{
				f64 t9a = crd[n1][n3];
				f64 t9b = crd[coit[coi_count].index][n3];
				
				t1a[n3] = t9b - t9a;
				t1b += t1a[n3] * t1a[n3];
			}
			
			f64 t1c = sqrt(t1b);
			coit[coi_count].dist = t1c;
			
			// also t1c is a distance which should never be close to zero -> no checking
			
			f64 t2a[3];
			for (i32s n3 = 0;n3 < 3;n3++)
			{
				t2a[n3] = t1a[n3] / t1c;
				coit[coi_count].dv[n3] = t2a[n3];
			}
			
			coit[coi_count].g = (t1b + vdwr2[n1] - vdwr2[coit[coi_count].index]) / (2.0 * t1c);
			coit[coi_count].ct = coit[coi_count].g / vdwr1[n1];
			
			if (p1 > 0)
			{
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					for (i32s n4 = 0;n4 < 3;n4++)
					{
						f64 t9a = t2a[n3] * t2a[n4]; f64 t9b;
						if (n3 != n4) t9b = -t9a; else t9b = 1.0 - t9a;
						coit[coi_count].ddv[n3][n4] = t9b / t1c;
					}
				}
				
				f64 t3a = (t1c - coit[coi_count].g) / t1c;
				coit[coi_count].dg[0] = t3a * coit[coi_count].dv[0];
				coit[coi_count].dg[1] = t3a * coit[coi_count].dv[1];
				coit[coi_count].dg[2] = t3a * coit[coi_count].dv[2];
				
				coit[coi_count].dct[0] = coit[coi_count].dg[0] / vdwr1[n1];
				coit[coi_count].dct[1] = coit[coi_count].dg[1] / vdwr1[n1];
				coit[coi_count].dct[2] = coit[coi_count].dg[2] / vdwr1[n1];
			}
			
			coit[coi_count++].ipd_count = 0;
			if (coi_count >= SIZE_COI) { cout << "COI overflow!!!" << endl; exit(EXIT_FAILURE); }
		}
		
		i32s ips_total_count = 0;
		i32s ips_count = 0; mm2_nbt3_ips ipst[SIZE_IPS];
		
		// next we will create the ips-table...
		// next we will create the ips-table...
		// next we will create the ips-table...
		
		for (i32s n2 = 0;n2 < coi_count - 1;n2++)
		{
			for (i32s n3 = n2 + 1;n3 < coi_count;n3++)
			{
				f64 t1a[3];
				t1a[0] = coit[n2].dv[0] * coit[n3].dv[0];
				t1a[1] = coit[n2].dv[1] * coit[n3].dv[1];
				t1a[2] = coit[n2].dv[2] * coit[n3].dv[2];
				
				f64 t1b = t1a[0] + t1a[1] + t1a[2];	// cos phi
				
				if (t1b < -1.0) t1b = -1.0;	// domain check...
				if (t1b > +1.0) t1b = +1.0;	// domain check...
				
				f64 t1c = 1.0 - t1b * t1b;		// sin^2 phi
				if (t1c < LOWLIMIT) t1c = LOWLIMIT;
				
				f64 t2a = (coit[n2].g - coit[n3].g * t1b) / t1c;	// tau_kj
				f64 t2b = (coit[n3].g - coit[n2].g * t1b) / t1c;	// tau_jk
				
				f64 t2c = vdwr2[n1] - coit[n2].g * t2a - coit[n3].g * t2b;	// gamma^2
				if (t2c < LOWLIMIT) continue;		// these will not intercept...
				
				ips_total_count++;
				coit[n2].flag = true;
				coit[n3].flag = true;
				
				f64 t3a[3];	// eta
				t3a[0] = coit[n2].dv[0] * t2a + coit[n3].dv[0] * t2b;
				t3a[1] = coit[n2].dv[1] * t2a + coit[n3].dv[1] * t2b;
				t3a[2] = coit[n2].dv[2] * t2a + coit[n3].dv[2] * t2b;
				
				f64 t1d = sqrt(t1c);	// sin phi
				
				f64 t3b[3];	// omega
				t3b[0] = (coit[n2].dv[1] * coit[n3].dv[2] - coit[n2].dv[2] * coit[n3].dv[1]) / t1d;
				t3b[1] = (coit[n2].dv[2] * coit[n3].dv[0] - coit[n2].dv[0] * coit[n3].dv[2]) / t1d;
				t3b[2] = (coit[n2].dv[0] * coit[n3].dv[1] - coit[n2].dv[1] * coit[n3].dv[0]) / t1d;
				
				f64 t2d = sqrt(t2c);	// gamma
				
				for (i32s n4 = 0;n4 < 3;n4++)
				{
					f64 t9a = t3b[n4] * t2d;
					ipst[ips_count].ipv[0][n4] = t3a[n4] - t9a;
					ipst[ips_count].ipv[1][n4] = t3a[n4] + t9a;
				}
				
				bool skip_both = false;
				bool skip[2] = { false, false };
				for (i32s n4 = 0;n4 < n_count;n4++)
				{
					i32s n5 = nt[n4];
					if (n5 == coit[n2].index || n5 == coit[n3].index) continue;
					
					f64 t9a[3];
					t9a[0] = (crd[n1][0] + ipst[ips_count].ipv[0][0]) - crd[n5][0];
					t9a[1] = (crd[n1][1] + ipst[ips_count].ipv[0][1]) - crd[n5][1];
					t9a[2] = (crd[n1][2] + ipst[ips_count].ipv[0][2]) - crd[n5][2];
					f64 t9b = t9a[0] * t9a[0] + t9a[1] * t9a[1] + t9a[2] * t9a[2];
					if (t9b < vdwr2[n5]) skip[0] = true;
					
					f64 t9c[3];
					t9c[0] = (crd[n1][0] + ipst[ips_count].ipv[1][0]) - crd[n5][0];
					t9c[1] = (crd[n1][1] + ipst[ips_count].ipv[1][1]) - crd[n5][1];
					t9c[2] = (crd[n1][2] + ipst[ips_count].ipv[1][2]) - crd[n5][2];
					f64 t9d = t9c[0] * t9c[0] + t9c[1] * t9c[1] + t9c[2] * t9c[2];
					if (t9d < vdwr2[n5]) skip[1] = true;
					
					skip_both = (skip[0] && skip[1]);
					if (skip_both) break;
				}
				
				if (skip_both) continue;	// overwrite this one....
				
				ipst[ips_count].coi[0] = n2;
				ipst[ips_count].coi[1] = n3;
				
				if (!skip[0])
				{
					coit[n2].AddIPD(ipst[ips_count].ipv[0], ips_count);
					coit[n3].AddIPD(ipst[ips_count].ipv[0], ips_count | ORDER_FLAG);
				}
				
				if (!skip[1])
				{
					coit[n2].AddIPD(ipst[ips_count].ipv[1], ips_count | INDEX_FLAG | ORDER_FLAG);
					coit[n3].AddIPD(ipst[ips_count].ipv[1], ips_count | INDEX_FLAG);
				}
				
				if (p1 > 0)
				{
					f64 t1f[3];	// d(cos phi) / dXk
					t1f[0] = (coit[n3].dv[0] - t1b * coit[n2].dv[0]) / coit[n2].dist;
					t1f[1] = (coit[n3].dv[1] - t1b * coit[n2].dv[1]) / coit[n2].dist;
					t1f[2] = (coit[n3].dv[2] - t1b * coit[n2].dv[2]) / coit[n2].dist;
					
					f64 t1g[3];	// d(cos phi) / dXj
					t1g[0] = (coit[n2].dv[0] - t1b * coit[n3].dv[0]) / coit[n3].dist;
					t1g[1] = (coit[n2].dv[1] - t1b * coit[n3].dv[1]) / coit[n3].dist;
					t1g[2] = (coit[n2].dv[2] - t1b * coit[n3].dv[2]) / coit[n3].dist;
					
					f64 t2e[3];	// d(tau_kj) / dXk
					t2e[0] = (t1f[0] * (2.0 * t2a * t1b - coit[n3].g) + coit[n2].dg[0]) / t1c;
					t2e[1] = (t1f[1] * (2.0 * t2a * t1b - coit[n3].g) + coit[n2].dg[1]) / t1c;
					t2e[2] = (t1f[2] * (2.0 * t2a * t1b - coit[n3].g) + coit[n2].dg[2]) / t1c;
					
					f64 t2f[3];	// d(tau_kj) / dXj
					t2f[0] = (t1g[0] * (2.0 * t2a * t1b - coit[n3].g) - t1b * coit[n3].dg[0]) / t1c;
					t2f[1] = (t1g[1] * (2.0 * t2a * t1b - coit[n3].g) - t1b * coit[n3].dg[1]) / t1c;
					t2f[2] = (t1g[2] * (2.0 * t2a * t1b - coit[n3].g) - t1b * coit[n3].dg[2]) / t1c;
					
					f64 t2g[3];	// d(tau_jk) / dXk
					t2g[0] = (t1f[0] * (2.0 * t2b * t1b - coit[n2].g) - t1b * coit[n2].dg[0]) / t1c;
					t2g[1] = (t1f[1] * (2.0 * t2b * t1b - coit[n2].g) - t1b * coit[n2].dg[1]) / t1c;
					t2g[2] = (t1f[2] * (2.0 * t2b * t1b - coit[n2].g) - t1b * coit[n2].dg[2]) / t1c;
					
					f64 t2h[3];	// d(tau_jk) / dXj
					t2h[0] = (t1g[0] * (2.0 * t2b * t1b - coit[n2].g) + coit[n3].dg[0]) / t1c;
					t2h[1] = (t1g[1] * (2.0 * t2b * t1b - coit[n2].g) + coit[n3].dg[1]) / t1c;
					t2h[2] = (t1g[2] * (2.0 * t2b * t1b - coit[n2].g) + coit[n3].dg[2]) / t1c;
					
					f64 t3c[3][3];	// d(eta) / dXk
					f64 t3d[3][3];	// d(eta) / dXj
					
					for (i32s n4 = 0;n4 < 3;n4++)
					{
						for (i32s n5 = 0;n5 < 3;n5++)
						{
							f64 t9a = coit[n2].dv[n5]; f64 t9b = coit[n3].dv[n5];
							t3c[n4][n5] = t9a * t2e[n4] + t9b * t2g[n4] + t2a * coit[n2].ddv[n4][n5];
							t3d[n4][n5] = t9a * t2f[n4] + t9b * t2h[n4] + t2b * coit[n3].ddv[n4][n5];
						}
					}
					
					f64 t3e[3][3];	// d(omega) / dXk
					f64 t3f[3][3];	// d(omega) / dXj
					
					for (i32s n4 = 0;n4 < 3;n4++)
					{
						for (i32s n5 = 0;n5 < 3;n5++)
						{
							t3e[n4][n5] = t1b * t3b[n5] * t1f[n4] / t1c;
							t3f[n4][n5] = t1b * t3b[n5] * t1g[n4] / t1c;
						}
						
	t3e[n4][0] += (coit[n2].ddv[n4][1] * coit[n3].dv[2] - coit[n2].ddv[n4][2] * coit[n3].dv[1]) / t1d;
	t3e[n4][1] += (coit[n2].ddv[n4][2] * coit[n3].dv[0] - coit[n2].ddv[n4][0] * coit[n3].dv[2]) / t1d;
	t3e[n4][2] += (coit[n2].ddv[n4][0] * coit[n3].dv[1] - coit[n2].ddv[n4][1] * coit[n3].dv[0]) / t1d;
	
	t3f[n4][0] += (coit[n2].dv[1] * coit[n3].ddv[n4][2] - coit[n2].dv[2] * coit[n3].ddv[n4][1]) / t1d;
	t3f[n4][1] += (coit[n2].dv[2] * coit[n3].ddv[n4][0] - coit[n2].dv[0] * coit[n3].ddv[n4][2]) / t1d;
	t3f[n4][2] += (coit[n2].dv[0] * coit[n3].ddv[n4][1] - coit[n2].dv[1] * coit[n3].ddv[n4][0]) / t1d;
					}
					
					f64 t2i[3];	// d(gamma) / dXk
					t2i[0] = -(coit[n2].g * t2e[0] + t2a * coit[n2].dg[0] + coit[n3].g * t2g[0]) / (2.0 * t2d);
					t2i[1] = -(coit[n2].g * t2e[1] + t2a * coit[n2].dg[1] + coit[n3].g * t2g[1]) / (2.0 * t2d);
					t2i[2] = -(coit[n2].g * t2e[2] + t2a * coit[n2].dg[2] + coit[n3].g * t2g[2]) / (2.0 * t2d);
					
					f64 t2j[3];	// d(gamma) / dXj
					t2j[0] = -(coit[n2].g * t2f[0] + coit[n3].g * t2h[0] + t2b * coit[n3].dg[0]) / (2.0 * t2d);
					t2j[1] = -(coit[n2].g * t2f[1] + coit[n3].g * t2h[1] + t2b * coit[n3].dg[1]) / (2.0 * t2d);
					t2j[2] = -(coit[n2].g * t2f[2] + coit[n3].g * t2h[2] + t2b * coit[n3].dg[2]) / (2.0 * t2d);
					
					// the final result is derivatives for points dipv[2][2][3][3].
					// indexes are as follows: [point][atom][variable][xyz].
					
					for (i32s n4 = 0;n4 < 3;n4++)
					{
						for (i32s n5 = 0;n5 < 3;n5++)
						{
							ipst[ips_count].dipv[0][0][n4][n5] = t3c[n4][n5];
							ipst[ips_count].dipv[0][1][n4][n5] = t3d[n4][n5];
							ipst[ips_count].dipv[1][0][n4][n5] = t3c[n4][n5];
							ipst[ips_count].dipv[1][1][n4][n5] = t3d[n4][n5];
						}
						
						for (i32s n5 = 0;n5 < 3;n5++)
						{
							f64 t9a = t3b[n5] * t2i[n4] + t2d * t3e[n4][n5];
							f64 t9b = t3b[n5] * t2j[n4] + t2d * t3f[n4][n5];
							
							ipst[ips_count].dipv[0][0][n4][n5] -= t9a;
							ipst[ips_count].dipv[0][1][n4][n5] -= t9b;
							ipst[ips_count].dipv[1][0][n4][n5] += t9a;
							ipst[ips_count].dipv[1][1][n4][n5] += t9b;
						}
					}          
				}
				
				ips_count++;
				if (ips_count >= SIZE_IPS) { cout << "IPS overflow!!!" << endl; exit(EXIT_FAILURE); }
			}
		}
		
		i32s arc_count = 0; mm2_nbt3_arc arct[SIZE_ARC];
		
		// next we will create the arc-table...
		// next we will create the arc-table...
		// next we will create the arc-table...
		
		for (i32s n2 = 0;n2 < coi_count;n2++)
		{
			f64 t1z = vdwr2[n1] - coit[n2].g * coit[n2].g;
			if (t1z < 0.0) t1z = 0.0;		// domain check...
			
			f64 t1a = sqrt(t1z);
			if (t1a < LOWLIMIT) t1a = LOWLIMIT;
			
			sort(coit[n2].ipdt, coit[n2].ipdt + coit[n2].ipd_count);
			
			for (i32s n3 = 0;n3 < coit[n2].ipd_count;n3++)
			{
				if (coit[n2].ipdt[n3].ipdata & ORDER_FLAG) continue;
				i32s n4 = n3 + 1; if (n4 == coit[n2].ipd_count) n4 = 0;
				if (!(coit[n2].ipdt[n4].ipdata & ORDER_FLAG)) continue;
				
				arct[arc_count].coi = n2; arct[arc_count].flag = false;
				
				arct[arc_count].ipdata[0] = (coit[n2].ipdt[n3].ipdata & ~ORDER_FLAG);
				arct[arc_count].ipdata[1] = (coit[n2].ipdt[n4].ipdata & ~ORDER_FLAG);
				
				i32s i1a = (arct[arc_count].ipdata[0] & FLAG_MASK);
				bool i1b = (arct[arc_count].ipdata[0] & INDEX_FLAG ? 1 : 0);
				
				i32s i2a = (arct[arc_count].ipdata[1] & FLAG_MASK);
				bool i2b = (arct[arc_count].ipdata[1] & INDEX_FLAG ? 1 : 0);
				
				arct[arc_count].index[0][0] = coit[ipst[i1a].coi[i1b]].index;
				arct[arc_count].index[0][1] = coit[ipst[i1a].coi[!i1b]].index;
				
				arct[arc_count].index[1][0] = coit[ipst[i2a].coi[!i2b]].index;
				arct[arc_count].index[1][1] = coit[ipst[i2a].coi[i2b]].index;
				
				// let's compute the tangent vectors...
				
				f64 * ref1 = ipst[i1a].ipv[i1b];
				arct[arc_count].tv[0][0] = (ref1[1] * coit[n2].dv[2] - ref1[2] * coit[n2].dv[1]) / t1a;
				arct[arc_count].tv[0][1] = (ref1[2] * coit[n2].dv[0] - ref1[0] * coit[n2].dv[2]) / t1a;
				arct[arc_count].tv[0][2] = (ref1[0] * coit[n2].dv[1] - ref1[1] * coit[n2].dv[0]) / t1a;
				
				f64 * ref2 = ipst[i2a].ipv[i2b];
				arct[arc_count].tv[1][0] = (ref2[1] * coit[n2].dv[2] - ref2[2] * coit[n2].dv[1]) / t1a;
				arct[arc_count].tv[1][1] = (ref2[2] * coit[n2].dv[0] - ref2[0] * coit[n2].dv[2]) / t1a;
				arct[arc_count].tv[1][2] = (ref2[0] * coit[n2].dv[1] - ref2[1] * coit[n2].dv[0]) / t1a;
				
				if (p1 > 0)
				{
					for (i32s n4 = 0;n4 < 3;n4++)
					{
						f64 t9a = coit[n2].g * coit[n2].dg[n4] / t1a;
						for (i32s n5 = 0;n5 < 3;n5++)
						{
							arct[arc_count].dtv[0][0][n4][n5] = t9a * arct[arc_count].tv[0][n5];
							arct[arc_count].dtv[1][0][n4][n5] = t9a * arct[arc_count].tv[1][n5];
						}
						
	f64 * ref1a = ipst[i1a].dipv[i1b][i1b][n4];	// d(P1) / dXk
	arct[arc_count].dtv[0][0][n4][0] += ref1a[1] * coit[n2].dv[2] - ref1a[2] * coit[n2].dv[1];
	arct[arc_count].dtv[0][0][n4][1] += ref1a[2] * coit[n2].dv[0] - ref1a[0] * coit[n2].dv[2];
	arct[arc_count].dtv[0][0][n4][2] += ref1a[0] * coit[n2].dv[1] - ref1a[1] * coit[n2].dv[0];
	
	f64 * ref1b = ipst[i2a].dipv[i2b][!i2b][n4];	// d(P2) / dXk
	arct[arc_count].dtv[1][0][n4][0] += ref1b[1] * coit[n2].dv[2] - ref1b[2] * coit[n2].dv[1];
	arct[arc_count].dtv[1][0][n4][1] += ref1b[2] * coit[n2].dv[0] - ref1b[0] * coit[n2].dv[2];
	arct[arc_count].dtv[1][0][n4][2] += ref1b[0] * coit[n2].dv[1] - ref1b[1] * coit[n2].dv[0];
	
	f64 * ref2a = ipst[i1a].ipv[i1b];
	arct[arc_count].dtv[0][0][n4][0] += ref2a[1] * coit[n2].ddv[n4][2] - ref2a[2] * coit[n2].ddv[n4][1];
	arct[arc_count].dtv[0][0][n4][1] += ref2a[2] * coit[n2].ddv[n4][0] - ref2a[0] * coit[n2].ddv[n4][2];
	arct[arc_count].dtv[0][0][n4][2] += ref2a[0] * coit[n2].ddv[n4][1] - ref2a[1] * coit[n2].ddv[n4][0];
	
	f64 * ref2b = ipst[i2a].ipv[i2b];
	arct[arc_count].dtv[1][0][n4][0] += ref2b[1] * coit[n2].ddv[n4][2] - ref2b[2] * coit[n2].ddv[n4][1];
	arct[arc_count].dtv[1][0][n4][1] += ref2b[2] * coit[n2].ddv[n4][0] - ref2b[0] * coit[n2].ddv[n4][2];
	arct[arc_count].dtv[1][0][n4][2] += ref2b[0] * coit[n2].ddv[n4][1] - ref2b[1] * coit[n2].ddv[n4][0];
						
						for (i32s n5 = 0;n5 < 3;n5++)
						{
							arct[arc_count].dtv[0][0][n4][n5] /= t1a;
							arct[arc_count].dtv[1][0][n4][n5] /= t1a;
						}
						
	f64 * ref3a = ipst[i1a].dipv[i1b][!i1b][n4];	// d(P1) / dXj
	arct[arc_count].dtv[0][1][n4][0] = (ref3a[1] * coit[n2].dv[2] - ref3a[2] * coit[n2].dv[1]) / t1a;
	arct[arc_count].dtv[0][1][n4][1] = (ref3a[2] * coit[n2].dv[0] - ref3a[0] * coit[n2].dv[2]) / t1a;
	arct[arc_count].dtv[0][1][n4][2] = (ref3a[0] * coit[n2].dv[1] - ref3a[1] * coit[n2].dv[0]) / t1a;
	
	f64 * ref3b = ipst[i2a].dipv[i2b][i2b][n4];	// d(P2) / dXj
	arct[arc_count].dtv[1][1][n4][0] = (ref3b[1] * coit[n2].dv[2] - ref3b[2] * coit[n2].dv[1]) / t1a;
	arct[arc_count].dtv[1][1][n4][1] = (ref3b[2] * coit[n2].dv[0] - ref3b[0] * coit[n2].dv[2]) / t1a;
	arct[arc_count].dtv[1][1][n4][2] = (ref3b[0] * coit[n2].dv[1] - ref3b[1] * coit[n2].dv[0]) / t1a;
					}
				}
				
				arc_count++;
				if (arc_count >= SIZE_ARC) { cout << "ARC overflow!!!" << endl; exit(EXIT_FAILURE); }
			}
		}
		
		// all cases will pass through this point!!!
		// all cases will pass through this point!!!
		// all cases will pass through this point!!!
		
		f64 area;
		if (!arc_count)
		{

/*################################################################################################*/
// this must be used only if we wish to measure the distribution of areas (to derive parameters)...
//if (ips_total_count) { sasa[n1] = 0.0; continue; }	// MUST BE SET TO ZERO IF SASA IS MEASURED!!!
/*################################################################################################*/

			if (ips_total_count) continue;		// fully buried...
			else area = 4.0 * M_PI;
		}
		else
		{
			area = 0.0;
			i32s arc_counter = 0;
			
			do
			{
				i32s prev; i32s curr = 0;
				while (arct[curr].flag)
				{
					curr++;
					if (curr == arc_count)
					{
						cout << "area_panic: can't find the first arc!!!" << endl;
						goto area_panic;
					}
				}
				
				i32s first = curr;
				
				f64 sum1 = 0.0;
				f64 sum2 = 0.0;
				
				while (true)
				{
					i32s coi = arct[curr].coi;
					
					f64 t1a[3];
					t1a[0] = arct[curr].tv[1][1] * arct[curr].tv[0][2] - arct[curr].tv[1][2] * arct[curr].tv[0][1];
					t1a[1] = arct[curr].tv[1][2] * arct[curr].tv[0][0] - arct[curr].tv[1][0] * arct[curr].tv[0][2];
					t1a[2] = arct[curr].tv[1][0] * arct[curr].tv[0][1] - arct[curr].tv[1][1] * arct[curr].tv[0][0];
					
					f64 t1b[3];
					t1b[0] = coit[coi].dv[0] * t1a[0];
					t1b[1] = coit[coi].dv[1] * t1a[1];
					t1b[2] = coit[coi].dv[2] * t1a[2];
					
					f64 t1c = (t1b[0] + t1b[1] + t1b[2] < 0.0 ? -1.0 : +1.0);
					
					f64 t2a[3];
					t2a[0] = arct[curr].tv[0][0] * arct[curr].tv[1][0];
					t2a[1] = arct[curr].tv[0][1] * arct[curr].tv[1][1];
					t2a[2] = arct[curr].tv[0][2] * arct[curr].tv[1][2];
					
					f64 t2b = t2a[0] + t2a[1] + t2a[2];
					
					if (t2b < -1.0) t2b = -1.0;	// domain check...
					if (t2b > +1.0) t2b = +1.0;	// domain check...
					
					f64 t2c = (1.0 - t1c) * M_PI + t1c * acos(t2b);
					sum1 += t2c * coit[coi].ct;
					
					if (p1 > 0)
					{
						f64 t2x = fabs(sin(t2c));
						if (t2x < LOWLIMIT) t2x = LOWLIMIT;
						
						f64 t2y = -sasa[n1] * coit[coi].ct * t1c / t2x;
						
						// 1st are same points and 2nd are different ones...
						// 1st are same points and 2nd are different ones...
						// 1st are same points and 2nd are different ones...
						
						for (i32s n2 = 0;n2 < 3;n2++)
						{
							f64 t3a[3];
							t3a[0] = arct[curr].dtv[0][0][n2][0] * arct[curr].tv[1][0];
							t3a[1] = arct[curr].dtv[0][0][n2][1] * arct[curr].tv[1][1];
							t3a[2] = arct[curr].dtv[0][0][n2][2] * arct[curr].tv[1][2];
							f64 t3b = t3a[0] + t3a[1] + t3a[2];
							
							f64 t3c[3];
							t3c[0] = arct[curr].tv[0][0] * arct[curr].dtv[1][0][n2][0];
							t3c[1] = arct[curr].tv[0][1] * arct[curr].dtv[1][0][n2][1];
							t3c[2] = arct[curr].tv[0][2] * arct[curr].dtv[1][0][n2][2];
							f64 t3d = t3c[0] + t3c[1] + t3c[2];
							
							f64 t4a[3];
							t4a[0] = arct[curr].dtv[0][1][n2][0] * arct[curr].tv[1][0];
							t4a[1] = arct[curr].dtv[0][1][n2][1] * arct[curr].tv[1][1];
							t4a[2] = arct[curr].dtv[0][1][n2][2] * arct[curr].tv[1][2];
							f64 t4b = t4a[0] + t4a[1] + t4a[2];
							
							f64 t4c[3];
							t4c[0] = arct[curr].tv[0][0] * arct[curr].dtv[1][1][n2][0];
							t4c[1] = arct[curr].tv[0][1] * arct[curr].dtv[1][1][n2][1];
							t4c[2] = arct[curr].tv[0][2] * arct[curr].dtv[1][1][n2][2];
							f64 t4d = t4c[0] + t4c[1] + t4c[2];
							
							f64 t3e = t2y * (t3b + t3d) + sasa[n1] * t2c * coit[coi].dct[n2];
							f64 t5a = t2y * t4b; f64 t5b = t2y * t4d;
							
							d1[arct[curr].index[0][0]][n2] += t3e;
							d1[arct[curr].index[0][1]][n2] += t5a;
							d1[arct[curr].index[1][1]][n2] += t5b;
							d1[n1][n2] -= t3e + t5a + t5b;
						}
					}
					
					prev = curr; curr = 0;
					i32u ipd = arct[prev].ipdata[1];
					while (true)
					{
						if (arct[curr].ipdata[0] != ipd) curr++;
						else break;
						
						if (curr == arc_count)
						{
							cout << "area_panic: incomplete set of arcs!!!" << endl;
							goto area_panic;
						}
					}
					
					arc_counter++;
					arct[curr].flag = true;
					
					f64 t2d[3];
					t2d[0] = arct[prev].tv[1][0] * arct[curr].tv[0][0];
					t2d[1] = arct[prev].tv[1][1] * arct[curr].tv[0][1];
					t2d[2] = arct[prev].tv[1][2] * arct[curr].tv[0][2];
					
					f64 t2e = t2d[0] + t2d[1] + t2d[2];
					
					if (t2e < -1.0) t2e = -1.0;	// domain check...
					if (t2e > +1.0) t2e = +1.0;	// domain check...
					
					f64 t2f = -acos(t2e); sum2 += t2f;
					
					if (p1 > 0)
					{
						f64 t2x = fabs(sin(t2f));
						if (t2x < LOWLIMIT) t2x = LOWLIMIT;
						
						f64 t2y = sasa[n1] / t2x;
						
						// prev_k = curr_j and prev_j = curr_k !!!
						// prev_k = curr_j and prev_j = curr_k !!!
						// prev_k = curr_j and prev_j = curr_k !!!
						
						for (i32s n2 = 0;n2 < 3;n2++)
						{
							f64 t3a[3];
							t3a[0] = arct[prev].dtv[1][0][n2][0] * arct[curr].tv[0][0];
							t3a[1] = arct[prev].dtv[1][0][n2][1] * arct[curr].tv[0][1];
							t3a[2] = arct[prev].dtv[1][0][n2][2] * arct[curr].tv[0][2];
							f64 t3b = t3a[0] + t3a[1] + t3a[2];
							
							f64 t3c[3];
							t3c[0] = arct[prev].tv[1][0] * arct[curr].dtv[0][1][n2][0];
							t3c[1] = arct[prev].tv[1][1] * arct[curr].dtv[0][1][n2][1];
							t3c[2] = arct[prev].tv[1][2] * arct[curr].dtv[0][1][n2][2];
							f64 t3d = t3c[0] + t3c[1] + t3c[2];
							
							f64 t4a[3];
							t4a[0] = arct[prev].dtv[1][1][n2][0] * arct[curr].tv[0][0];
							t4a[1] = arct[prev].dtv[1][1][n2][1] * arct[curr].tv[0][1];
							t4a[2] = arct[prev].dtv[1][1][n2][2] * arct[curr].tv[0][2];
							f64 t4b = t4a[0] + t4a[1] + t4a[2];
							
							f64 t4c[3];
							t4c[0] = arct[prev].tv[1][0] * arct[curr].dtv[0][0][n2][0];
							t4c[1] = arct[prev].tv[1][1] * arct[curr].dtv[0][0][n2][1];
							t4c[2] = arct[prev].tv[1][2] * arct[curr].dtv[0][0][n2][2];
							f64 t4d = t4c[0] + t4c[1] + t4c[2];
							
							f64 t3e = t2y * (t3b + t3d);
							f64 t4e = t2y * (t4b + t4d);
							
							d1[arct[prev].index[1][0]][n2] += t3e;
							d1[arct[prev].index[1][1]][n2] += t4e;
							d1[n1][n2] -= t3e + t4e;
						}    
					}
					
					if (curr == first) break;
				}
				
				area += 2.0 * M_PI + sum1 + sum2;
			} while (arc_counter < arc_count);
			
			// when we have some problems somewhere above (for example, if we have
			// an incomplete set of arcs or no arcs at all; these things are possible
			// in rare special cases; for example we might have to reject some arcs
			// if they contained some singular intermediate values) we will truncate
			// the sum and jump right here.
			
			// in this case we will calculate incorrect value for the area, but the
			// good news is that the value and the gradient will still be consistent.
			
			// since these cases are very rare, this probably won't make big problems
			// in any applications...
			
			area_panic:	// we will jump here in all problematic cases...
			
			while (area > 4.0 * M_PI) area -= 4.0 * M_PI;
		}
		
		// finally here we will handle the single separate patches...
		// finally here we will handle the single separate patches...
		// finally here we will handle the single separate patches...
		
		for (i32s n2 = 0;n2 < coi_count;n2++)
		{
			if (coit[n2].flag) continue;
			
			f64 t1a = 2.0 * M_PI / vdwr1[n1];
			area -= t1a * (vdwr1[n1] - coit[n2].g);
			
			if (p1 > 0)
			{
				for (i32s n3 = 0;n3 < 3;n3++)
				{
					f64 t1b = sasa[n1] * t1a * coit[n2].dg[n3];
					d1[coit[n2].index][n3] += t1b;
					d1[n1][n3] -= t1b;
				}
			}
		}
		
		energy_nbt3 += sasa[n1] * area;
		
/*################################################################################################*/
// this must be used only if we wish to measure the distribution of areas (to derive parameters)...
//sasa[n1] = area;	// HERE WE STORE THE AREAS AND LATER USE THEM TO MEASURE THE DISTRIBUTION!!!
/*################################################################################################*/

	}
}

void mm2_eng::Compute(i32s p1)
{
	if (p1 > 0)
	{
		for (i32u n1 = 0;n1 < index_vector.size();n1++)
		{
			d1[n1][0] = 0.0;
			d1[n1][1] = 0.0;
			d1[n1][2] = 0.0;
		}
	}
	
	for (i32u n1 = 0;n1 < index_vector.size();n1++)		// this is for the surface area term...
	{
		nbt3_nl[n1].index_count = 0;
	}
	
	ComputeBT1(p1);		// we need this also for the surface area term...
	ComputeBT2(p1);
	ComputeBT3(p1);
	ComputeBT4(p1);
//energy_bt1 = 0.0;
//energy_bt2 = 0.0;
//energy_bt3 = 0.0;
//energy_bt4 = 0.0;

	ComputeNBT1(p1);		// we need this also for the surface area term...
//energy_nbt1a = 0.0;
//energy_nbt1b = 0.0;

	ComputeNBT2(p1);
//energy_nbt2a = 0.0;
//energy_nbt2b = 0.0;

	ComputeNBT3(p1);
//energy_nbt3 = 0.0;

	energy = constraints_OLD_FIXME;
	energy += energy_bt1 + energy_bt2 + energy_bt3 + energy_bt4;
	energy += energy_nbt1a + energy_nbt1b + energy_nbt2a + energy_nbt2b + energy_nbt3;
}

#define DELTA 0.000001		// the finite difference step...

void mm2_eng::Check(i32s)
{
	Compute(1); f64 tmp1 = energy;
	
	for (i32u n1 = 0;n1 < index_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			f64 old = crd[n1][n2];
			crd[n1][n2] = old + DELTA; Compute(0);
			f64 tmp2 = (energy - tmp1) / DELTA;
			crd[n1][n2] = old;
			
			cout << n1 << ": a = " << d1[n1][n2] << " n = " << tmp2 << endl;
		}
		
		if ((n1 % 5) == 4)
		{
			int delay;
			cin >> delay;
		}
	}
}

f64 mm2_eng::GetGradientVectorLength(void)
{
	f64 sum = 0.0;
	
	for (i32u n1 = 0;n1 < index_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			f64 tmp1 = d1[n1][n2];
			sum += tmp1 * tmp1;
		}
	}
	
	return sqrt(sum);
}

/*################################################################################################*/

// eof
