/*
 * $Id: gt_http_server.c,v 1.63 2003/12/22 02:50:42 hipnod Exp $
 *
 * Copyright (C) 2001-2003 giFT project (gift.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include "gt_gnutella.h"

#include "gt_xfer_obj.h"
#include "gt_xfer.h"

#include "gt_http_server.h"
#include "gt_http_client.h"

#include "gt_accept.h"
#include "gt_ban.h"
#include "gt_version.h"

/*****************************************************************************/

static void server_handle_get    (GtTransfer *xfer);

/*****************************************************************************/
/* SERVER HELPERS */

static char *lookup_http_code (int code, char **desc)
{
	char *err;
	char *txt;

	switch (code)
	{
	 case 200: err = "OK";
		txt = "Success";
		break;
	 case 206: err = "Partial Content";
		txt = "Resume successful";
		break;
	 case 403: err = "Forbidden";
		txt = "You do not have access to this file";
		break;
	 case 404: err = "Not Found";
		txt = "File is not available";
		break;
	 case 500: err = "Internal Server Error";
		txt = "Stale file entry, retry later";
		break;
	 case 501: err = "Not Implemented";
		txt = "???";
		break;
	 case 503: err = "Service Unavailable";
		txt = "Upload queue is currently full, please try again later";
		break;
	 default:  err = NULL;
		txt = NULL;
		break;
	}

	if (desc)
		*desc = txt;

	return err;
}

/*
 * Construct and send a server reply
 *
 * NOTE:
 * A similar function is available in http_client.c
 */
static BOOL gt_http_server_send (TCPC *c, int code, ...)
{
	char       *key;
	char       *value;
	char       *code_text;
	String     *s;
	int         ret;
	size_t      len;
	va_list     args;

	/* so that we can communicate both the numerical code and the human
	 * readable string */
	if (!(code_text = lookup_http_code (code, NULL)))
		return FALSE;

	if (!(s = string_new (NULL, 0, 0, TRUE)))
		return FALSE;

	string_appendf (s, "HTTP/1.1 %i %s\r\n", code, code_text);

	/* Add "Server: " header */
	string_appendf (s, "Server: %s\r\n", gt_version ());

	va_start (args, code);

	for (;;)
	{
		if (!(key = va_arg (args, char *)))
			break;

		if (!(value = va_arg (args, char *)))
			continue;

		string_appendf (s, "%s: %s\r\n", key, value);
	}

	va_end (args);

	/* append final message terminator */
	string_append (s, "\r\n");

	if (HTTP_DEBUG)
		GT->DBGSOCK (GT, c, "sending reply to client =\n%s", s->str);

	len = s->len;
	ret = tcp_send (c, s->str, s->len);

	string_free (s);

	return (ret == len);
}

static char *get_error_page (TCPC *c, int code)
{
	char *page;
	char *err;
	char *errtxt = NULL;

	if (!(err = lookup_http_code (code, &errtxt)))
		return 0;

	page = stringf ("<h1>%i %s</h1><br>%s.", code, err, errtxt);

	return page;
}

static void get_content_urns (GtTransfer *xfer, char **key, char **val)
{
	*key = NULL;
	*val = NULL;

	if (xfer && xfer->content_urns)
	{
		*key = "X-Gnutella-Content-URN";
		*val = xfer->content_urns;
	}
}

static BOOL supports_queue (GtTransfer *xfer)
{
	if (!(dataset_lookupstr (xfer->header, "x-queue")))
		return FALSE;

	return TRUE;
}

static char *get_queue_line (GtTransfer *xfer)
{
	String *s;

	/* do nothing if not queued */
	if (xfer->queue_pos == 0)
		return NULL;

	if (!(s = string_new (NULL, 0, 0, TRUE)))
		return NULL;

	string_appendf (s, "position=%d,length=%d,pollMin=%d,pollMax=%d", 
	                xfer->queue_pos, xfer->queue_ttl, 45, 120);

	return string_free_keep (s);
}

static int gt_http_server_send_error (TCPC *c, int code)
{
	char        *error_page;
	int          len;
	int          sent_len;
	char         content_len[256];
	GtTransfer  *xfer            = NULL;
	char        *queue_line      = NULL;
	char        *content_urn_key = NULL;
	char        *content_urn_val = NULL;

	gt_transfer_unref (&c, NULL, &xfer);

	error_page = get_error_page (c, code);
	assert (error_page != NULL);

	len = strlen (error_page);
	snprintf (content_len, sizeof (content_len) - 1, "%u", len);

	/* Append any content urns */
	get_content_urns (xfer, &content_urn_key, &content_urn_val);

	if (code == 503 && supports_queue (xfer))
		queue_line = get_queue_line (xfer);

	gt_http_server_send (c, code,
	                     "Content-Type",    "text/html",
	                     "Content-Length",   content_len,
	                     content_urn_key,    content_urn_val,
	                     "X-Queue",          queue_line,
	                     NULL);

	sent_len = tcp_send (c, error_page, len);

	/* if the whole page was sent, then its safe to use persistent
	 * http by setting the headers to transmitted and the remaining len to
	 * zero */
	if (sent_len == len)
	{
		xfer->transmitted_hdrs = TRUE;
		xfer->remaining_len    = 0;
	}

	free (queue_line);

	return sent_len;
}

/* parse the Range: bytes=0-10000 format */
static void parse_client_request_range (Dataset *dataset,
                                        off_t *r_start, off_t *r_stop)
{
	char         *range;
	off_t         start;
	off_t         stop;

	if (!r_start && !r_stop)
		return;

	if (r_start)
		*r_start = 0;
	if (r_stop)
		*r_stop = 0;

	/* leave stop as 0 if we can't figure anything out yet.  This is expected
	 * to be handled separately by GET and PUSH */
	if (!(range = dataset_lookupstr (dataset, "range")))
		return;

	/* WARNING: this butchers the data in the dataset! */
	string_sep     (&range, "bytes");
	string_sep_set (&range, " =");

	if (!range)
	{
		if (HTTP_DEBUG)
			GT->DBGFN (GT, "error parsing Range: header");

		return;
	}

	start = (off_t) ATOI (string_sep (&range, "-"));
	stop  = (off_t) ATOI (string_sep (&range, " "));

	/*
	 * The end of the range is optional (e.g. "Range: 0-"),
	 * and in that case stop == 0. In the case of a single
	 * byte file, stop == 1.
	 *
	 * TODO: this is broken for one-byte requests.
	 */
	if (stop > 0)
		stop = stop + 1;

	if (r_start)
		*r_start = start;
	if (r_stop)
		*r_stop = stop;
}

/* 
 * Parse requests of the forms: 
 *  
 *  /get/47/File maybe with spaces HTTP
 *  /get/47/files_%20_url_encoded HTTP/1.1
 *  /uri-res/N2R?urn:sha1:ABCD HTTP/1.0
 *
 * and combinations thereof. The "HTTP" trailer is mandatory.
 */
static void get_request_and_version (char *data, char **request, 
                                     char **version)
{
	size_t  len;
	char   *next;
	char   *dup;
	char   *http_version = NULL;

	*request = NULL;
	*version = NULL;

	if (!(dup = STRDUP (data)))
		return;

	string_trim (dup);
	string_upper (dup);

	next = dup;

	/* find the last instance of "HTTP" in the string */
	while ((next = strstr (next, "HTTP")))
	{
		http_version = next;
		next += sizeof ("HTTP") - 1;
	}

	/* the rest of the string must be the request */
	if (http_version != NULL && http_version != dup)
	{
		len = http_version - dup;
		data[len - 1] = 0;

		*request = data;
		*version = data + len;
	}

	free (dup);
}

/*
 * Break down the clients HTTP request
 */
static int parse_client_request (Dataset **r_dataset, char **r_command,
								 char **r_request, char **r_version,
								 off_t *r_start, off_t *r_stop, char *hdr)
{
	Dataset *dataset = NULL;
	char    *command; /* GET */
	char    *request; /* /file.tar.gz */
	char    *version; /* HTTP/1.1 */
	char    *req_line;

	if (!hdr)
		return FALSE;

	/*
	 * Get the first line of the request
	 */
	req_line = string_sep_set (&hdr, "\r\n");

	/* get the command (GET, HEAD, etc.) */
	command = string_sep (&req_line, " ");

	/*
	 * Handle non-url-encoded requests as well as encoded
	 * ones and get the request and version from this HTTP line.
	 */
	get_request_and_version (req_line, &request, &version);

	GT->DBGFN (GT, "command=%s version=%s request=%s",
	           command, version, request);

	if (!request || string_isempty (request))
		return FALSE;

	if (r_command)
		*r_command = command;
	if (r_request)
		*r_request = request;
	if (r_version)
		*r_version = version;

	http_headers_parse (hdr, &dataset);

	if (r_dataset)
		*r_dataset = dataset;

	/* handle Range: header */
	parse_client_request_range (dataset, r_start, r_stop);

	if (r_start && r_stop)
	{
		if (HTTP_DEBUG)
			GT->dbg (GT, "range: [%i, %i)", *r_start, *r_stop);
	}

	return TRUE;
}

/*****************************************************************************/

/*
 * Send the request reply back to the client
 *
 * NOTE:
 * This is used by both GET / and PUSH /
 */
static void reply_to_client_request (GtTransfer *xfer)
{
	TCPC       *c     = NULL;
	Chunk      *chunk = NULL;
	off_t       entity_size;
	char        range[128];
	char        length[32];
	char       *content_urn_key;
	char       *content_urn_val;
	BOOL        ret;

	if (!xfer)
		return;

	gt_transfer_unref (&c, &chunk, &xfer);

	/*
	 * Determine the "total" entity body that we have locally, not necessarily
	 * the data that we are uploading.  HTTP demands this, but OpenFT really
	 * doesn't give a shit.
	 *
	 * NOTE:
	 * This only works to standard when operating on a GET / request, PUSH's
	 * merely use the range!
	 */
	if (xfer->open_path_size)
		entity_size = xfer->open_path_size;
	else
		entity_size = xfer->stop - xfer->start;

	/* NOTE: we are "working" around the fact that HTTP's Content-Range
	 * reply is inclusive for the last byte, whereas giFT's is not. */
	snprintf (range, sizeof (range) - 1, "bytes %i-%i/%i",
	          (int) xfer->start, (int) (xfer->stop - 1), (int) entity_size);

	snprintf (length, sizeof (length) - 1, "%i",
	          (int) (xfer->stop - xfer->start));

	get_content_urns (xfer, &content_urn_key, &content_urn_val);

	ret = gt_http_server_send (c, xfer->code,
	                           "Content-Range",  range,
	                           "Content-Length", length,
	                           "Content-Type",   xfer->content_type,
	                           content_urn_key,  content_urn_val,
	                           NULL);

	/* if we transmitted all headers successfully, set transmitted_hdrs
	 * to keep the connection alive, possibly */
	if (ret)
		xfer->transmitted_hdrs = TRUE;
}

/*****************************************************************************/

/*
 * Handle the client's GET or PUSH commands
 */
void gt_get_client_request (int fd, input_id id, TCPC *c)
{
	GtTransfer    *xfer;
	Dataset       *dataset = NULL;
	char          *command = NULL;
	char          *request = NULL;
	char          *version = NULL;
	off_t          start   = 0;
	off_t          stop    = 0;
	FDBuf         *buf;
	unsigned char *data;
	size_t         data_len = 0;
	int            n;

	buf = tcp_readbuf (c);

	if ((n = fdbuf_delim (buf, "\n")) < 0)
	{
		gt_http_connection_close (c, TRUE, GT_TRANSFER_UPLOAD);
		return;
	}

	if (n > 0)
		return;

	data = fdbuf_data (buf, &data_len);

	/* TODO: this should be timed out */
	if (!http_headers_terminated (data, data_len))
		return;

	fdbuf_release (buf);

	if (HTTP_DEBUG)
		GT->DBGSOCK (GT, c, "client request:\n%s", data);

	/* parse the client's reply and determine how we should proceed */
	if (!parse_client_request (&dataset, &command, &request, &version,
	                           &start, &stop, data))
	{
		GT->DBGSOCK (GT, c, "invalid http header");
		gt_http_connection_close (c, TRUE, GT_TRANSFER_UPLOAD);
		return;
	}

	/*
	 * We have enough information now to actually allocate the transfer
	 * structure and pass it along to all logic that follows this
	 *
	 * NOTE:
	 * Each individual handler can determine when it wants to let giFT
	 * in on this
	 */
	xfer = gt_transfer_new (GT_TRANSFER_UPLOAD, NULL,
	                        net_peer (c->fd), 0, start, stop);

	gt_transfer_ref (c, NULL, xfer);

	/* assign all our own memory */
	xfer->command = STRDUP (command);
	xfer->header  = dataset;
	xfer->version = STRDUP (version);

	if (!gt_transfer_set_request (xfer, request))
	{
		if (HTTP_DEBUG)
			GT->DBGSOCK (GT, c, "invalid request '%s'", request);

		gt_transfer_close (xfer, TRUE);
		return;
	}

	/* check if the client is using a newer version than we are */
	gt_version_warn_if_newer (xfer->ip,
	                          dataset_lookupstr (xfer->header, "User-Agent"));

	/* no need for this function again */
	input_remove (id);

	/* figure out how to handle this request */
	if (!strcasecmp (xfer->command, "GET") ||
	    !strcasecmp (xfer->command, "HEAD"))
	{
		server_handle_get (xfer);
		return;
	}

	gt_http_server_send_error (c, 501);
	gt_transfer_close (xfer, FALSE);
}

static Transfer *start_upload (GtTransfer *xfer, Chunk **chunk)
{
	Transfer *transfer;
	char     *user;

	user = net_ip_str (xfer->ip);

	transfer = GT->upload_start (GT, chunk, user, xfer->share_authd,
	                             xfer->start, xfer->stop);

	assert (transfer != NULL);

	return transfer;
}

/* setup the structure for uploading.  this will be called from within
 * client space for PUSH requests as well */
int gt_server_setup_upload (GtTransfer *xfer)
{
	Transfer   *transfer;              /* giFT structure */
	Chunk      *chunk;
	TCPC       *c = NULL;

	if (!xfer)
		return FALSE;

	gt_transfer_unref (&c, NULL, &xfer);

	/*
	 * Ban the host if they don't have access -- this gives no information
	 * about whether we have the file or not, and i think this is in violation
	 * of the HTTP spec (supposed to return "404 not found" before 403, but
	 * i'm not sure.
	 */
	if (gt_ban_ipv4_is_banned (c->host))
	{
		xfer->code = 403;
		return FALSE;
	}

	/* open the file that was requested before we go bringing giFT into
	 * this */
	if (!(xfer->f = gt_transfer_open_request (xfer, &xfer->code)))
		return FALSE;

	/* assign stop a value before we proceed */
	if (xfer->stop == 0)
	{
		struct stat st;

		if (!file_stat (xfer->open_path, &st) || st.st_size == 0)
		{
			/* stupid bastards have a 0 length file */
			GT->DBGSOCK (GT, c, "cannot satisfy %s: invalid share",
			             xfer->open_path);
			return FALSE;
		}

		xfer->stop = st.st_size;
		xfer->remaining_len = xfer->stop - xfer->start;
	}

	/* we can now be certain that we are handling a download request from
	 * the client.  allocate the appropriate structures to hook into giFT */
	if (!(transfer = start_upload (xfer, &chunk)))
	{
		GT->DBGFN (GT, "unable to register upload with the daemon");
		return FALSE;
	}

	/* override 200 w/ 206 if the request is not the whole file size */
	if (xfer->remaining_len != xfer->share_authd->size)
		xfer->code = 206;

	/* assign the circular references for passing this data along */
	gt_transfer_ref (c, chunk, xfer);

	/* finally, seek the file descriptor where it needs to be */
	fseek (xfer->f, xfer->start, SEEK_SET);

	return TRUE;
}

/*
 * Serve client GET / requests
 */
static void server_handle_get (GtTransfer *xfer)
{
	TCPC *c = NULL;

	if (!xfer)
		return;

	gt_transfer_unref (&c, NULL, &xfer);

	/* WARNING: this block is duplicated in http_client:client_push_request */
	if (!gt_server_setup_upload (xfer))
	{
		/* gracefully reject */
		switch (xfer->code)
		{
		 case 503:
			gt_http_server_send_error (c, xfer->code);
			break;
		 case 200:                     /* general failure */
			xfer->code = 404;
			/* fall through */
		 default:
			gt_http_server_send_error (c, xfer->code);
			break;
		}

		/* perform an orderly close: keep the connection alive if possible */
		gt_transfer_close (xfer, FALSE);
		return;
	}

	/* ok, give the client the accepted header */
	reply_to_client_request (xfer);

	if (!strcasecmp (xfer->command, "HEAD"))
	{
		gt_transfer_close (xfer, TRUE);
		return;
	}

	input_add (c->fd, c, INPUT_WRITE,
			   (InputCallback) gt_server_upload_file, 0);
}

/*
 * Uploads the file requests
 */
void gt_server_upload_file (int fd, input_id id, TCPC *c)
{
	Chunk       *chunk = NULL;
	GtTransfer  *xfer  = NULL;
	char         buf[RW_BUFFER];
	size_t       read_len;
	size_t       size;
	int          sent_len = 0;
	off_t        remainder;

	gt_transfer_unref (&c, &chunk, &xfer);

	assert (xfer->f != NULL);

	/* number of bytes left to be uploaded by this chunk */
	if ((remainder = xfer->remaining_len) <= 0)
	{
		/* for whatever reason this function may have been called when we have
		 * already overrun the transfer...in that case we will simply fall
		 * through to the end-of-transfer condition */
		gt_transfer_write (xfer, chunk, NULL, 0);
		return;
	}

	size = sizeof (buf);

	if (size > remainder)
		size = remainder;

	/*
	 * Ask giFT for the size we should send.  If this returns 0, the upload
	 * was suspended.
	 */
	if ((size = upload_throttle (chunk, size)) == 0)
		return;

	/* read as much as we can from the local file */
	if (!(read_len = fread (buf, sizeof (char), size, xfer->f)))
	{
		GT->DBGFN (GT, "unable to read from %s: %s", xfer->open_path,
		           GIFT_STRERROR ());
		gt_transfer_status (xfer, SOURCE_CANCELLED, "Local read error");
		gt_transfer_close (xfer, TRUE);
		return;
	}

	if ((sent_len = tcp_send (c, buf, MIN (read_len, remainder))) <= 0)
	{
		gt_transfer_status (xfer, SOURCE_CANCELLED,
		                    "Unable to send data block");
		gt_transfer_close (xfer, TRUE);
		return;
	}

	/* check if the file was too small for the transfer TODO: this isn't
	 * checked earlier, but should be */
	if (read_len != size)
	{
		gt_transfer_status (xfer, SOURCE_CANCELLED, "Unexpected end of file");
		gt_transfer_close (xfer, TRUE);
		return;
	}

	/* 
	 * Check for short send(). This could use fseek(), but I have this
	 * growing feeling that using stdio everywhere is a bad idea.
	 */
	if (read_len != sent_len)
	{
		gt_transfer_status (xfer, SOURCE_CANCELLED, "Short send()");
		gt_transfer_close (xfer, TRUE);
		return;
	}

	/* 
	 * Call gt_upload to report back to giFT. This will also cancel
	 * the transfer if the upload has completed.
	 */
	gt_transfer_write (xfer, chunk, buf, sent_len);

}

/*****************************************************************************/

void gt_http_server_reset (TCPC *c)
{
	/* This can happen because the GtTransfer and TCPC can be decoupled.
	 * as in the case of a push request sent. */
	if (!c)
		return;

	/* finish all queued writes before we reset */
	tcp_flush (c, TRUE);

	/* reset the input state */
	input_remove_all (c->fd);
	input_add (c->fd, c, INPUT_READ,
			   (InputCallback) gt_get_client_request, 2 * MINUTES);
}
