/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gimageview.h"

#include "dnd.h"
#include "fileload.h"
#include "fileutil.h"
#include "prefs.h"
#include "thumbnail.h"
#include "thumbnail_view.h"
#include "thumbnail_window.h"
#include "thumbview_table.h"


typedef struct Thumbata_Tag
{
   GtkWidget *button;
   GtkWidget *button_vbox;
   GtkTooltips *tooltips;
} ThumbData;


typedef struct ThumbTableData_Tag
{
   GtkWidget   *table;
   GtkWidget   *hbox;
   GtkWidget   *event_box;
   gint         colnum;

   Thumbnail   *focused;   /* focused thumbnail */
   gfloat       page_pos_x;
   gfloat       page_pos_y;
} ThumbTableData;


/* callback functions */
static void       cb_thumbbutton_enter          (GtkWidget      *button,
						 gpointer        data);
static void       cb_thumbbutton_toggle         (GtkWidget      *button,
						 gpointer        data);
static void       cb_thumbbutton_toggle         (GtkWidget      *button,
						 gpointer        data);
static void       cb_button_focus_in            (GtkWidget      *button,
						 GdkEventFocus  *event,
						 Thumbnail      *thumb);
static gint       cb_expose                     (GtkWidget      *widget,
						 GdkEventExpose *event,
						 ThumbView      *tv);

/* private functions */
#if 0
static void       set_style                     (GtkWidget *widget,
						 GtkStyle  *style);
static void       hilight_thumbnail             (GtkWidget *widget);
#endif

static guint      calc_thumbtable_col_row_num   (ThumbView *tv,
						 gint       num);
static gboolean   calc_thumbbutton_pos          (Thumbnail *thumb,
						 gint      *col,
						 gint      *row);
static GtkWidget *create_thumbnail_button       (Thumbnail *thumb,
						 gint       thumb_size);
static GtkWidget *thumbtable_append_thumb_frame (ThumbView *tv,
						 Thumbnail *thumb,
						 gint       dest_mode);
static gint       idle_thumbtable_redraw        (gpointer   data);


/* static GtkStyle *normal_style = NULL; */


/******************************************************************************
 *
 *   Callback functions.
 *
 ******************************************************************************/
static void
cb_thumb_button_press (GtkWidget *icon, GdkEventButton *event, gpointer data)
{
   Thumbnail *thumb = data;
   ThumbView *tv;
   gboolean found = FALSE;

   g_return_if_fail (icon && thumb);

   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   /* set selection */
   if (event->type == GDK_BUTTON_PRESS && (event->button == 1)) {
      if (event->state & GDK_SHIFT_MASK) {
	 found = thumbview_set_selection_multiple (thumb, TRUE, FALSE);
	 if (!found)
	    thumbview_set_selection_multiple (thumb, FALSE, FALSE);
	 thumbview_set_selection (thumb, FALSE);
      }
   }

   /* relay management to reference callback function */
   thumbview_thumb_button_press_cb (icon, event, thumb);
}


/*
 *  cb_button_selected:
 *     @ Callback function for thumbnail button selected.
 *     @ Currently, do nothing.
 *
 *  widget :
 *  data   :
 */
static void
cb_thumbbutton_enter (GtkWidget *button, gpointer data)
{
   /*
   g_print ("%s\n", filename);
   */
}


static void
cb_thumbbutton_toggle (GtkWidget *button, gpointer data)
{
   Thumbnail *thumb = data;

   g_return_if_fail (thumb);

   thumb->selected = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
   /*
   if (thumb->selected) {
      hilight_thumbnail (button);
   } else {
      if (!normal_style)
	 normal_style = gtk_style_copy (gtk_widget_get_style (button));
      set_style(button, normal_style);
   }
   */
}


static void
cb_button_focus_in (GtkWidget *button, GdkEventFocus *event, Thumbnail *thumb)
{
   ThumbView *tv;
   ThumbTableData *tt;
   ThumbData *thumb_data;

   g_return_if_fail (thumb);

   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_if_fail (tt && thumb_data);

   if (tt->focused == thumb) return;
   tt->focused = thumb;

   thumbtable_adjust (tv, thumb);
}


static gint
cb_expose (GtkWidget *widget, GdkEventExpose *event, ThumbView *tv)
{
   thumbview_resize (tv);

   return TRUE;
}


static void
cb_thumbview_drag_begin (GtkWidget *widget,
			 GdkDragContext *context,
			 gpointer data)
{
   Thumbnail *thumb = data;
   ThumbView *tv;

   g_return_if_fail (thumb);
   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   if (!thumb->selected) {
      thumbview_set_selection_all (tv, FALSE);
      thumbview_set_selection (thumb, TRUE);
   }

   thumbview_drag_begin_cb (widget, context, thumb->thumb_view);
}


/******************************************************************************
 *
 *   private functions.
 *
 ******************************************************************************/
#if 0
static void
set_style (GtkWidget *widget, GtkStyle *style)
{
   gtk_widget_set_style (widget, style);

   if (GTK_IS_CONTAINER(widget))
      gtk_container_foreach (GTK_CONTAINER (widget),
			     GTK_SIGNAL_FUNC(set_style), style);
}


static void
hilight_thumbnail (GtkWidget *widget)
{
   GtkStyle *style = gtk_style_copy (gtk_widget_get_style (widget));

   if (!normal_style)
      normal_style = gtk_style_copy (gtk_widget_get_style (widget));

   style->bg[GTK_STATE_NORMAL]   = style->bg[GTK_STATE_SELECTED];
   style->bg[GTK_STATE_ACTIVE]   = style->bg[GTK_STATE_SELECTED];
   style->bg[GTK_STATE_PRELIGHT] = style->bg[GTK_STATE_SELECTED];
   style->fg[GTK_STATE_NORMAL]   = style->fg[GTK_STATE_SELECTED];
   style->fg[GTK_STATE_ACTIVE]   = style->fg[GTK_STATE_SELECTED];
   style->fg[GTK_STATE_PRELIGHT] = style->fg[GTK_STATE_SELECTED];

   set_style (widget, style);
}
#endif


/*
 *  calc_thumbtable_col_row_num:
 *     @ Calculate thumbnail table's columns and rows num. Calculated col num will be
 *       stored to ThumbView structure, and calcurated rows num is return value.
 *
 *  tv     : Pointer to ThumbView structure.
 *  num    : Number of thumbnails.
 *  Return : Rows number.
 */
static guint
calc_thumbtable_col_row_num (ThumbView *tv, gint num)
{
   ThumbWindow *tw;
   ThumbTableData *tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   GtkAdjustment *hadj;
   gint table_rows = 0, container_width;
   gint celwidth;

   tw = tv->thumb_window;
   hadj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (tv->container));

   container_width = hadj->page_size;
   if (container_width < 10)
      container_width = tw->window->allocation.width;

   celwidth = tv->ThumbnailSize + 6
      + conf.thumbbutton_border_width *2 + conf.thumbtable_col_space;
   tt->colnum = (container_width /*+ celwidth * 2*/) / (celwidth);

   if (tt->colnum < conf.thumbtable_colnum_min)
      tt->colnum = conf.thumbtable_colnum_min;
   if (tt->colnum > conf.thumbtable_colnum_max)
      tt->colnum = conf.thumbtable_colnum_max;

   if (num)
      table_rows = num / tt->colnum + 1;

   return table_rows;
}


/*
 *  calc_thumbbutton_pos:
 *     @ calculate thumbnail position in the table (column and row).
 *
 *  thumb  : Pointer to the Thumbnail widget.
 *  col    : column num for return. 
 *  row    : row num for return.
 *  Return : TRUE if success.
 */
static gboolean
calc_thumbbutton_pos (Thumbnail *thumb, gint *col, gint *row)
{
   ThumbView *tv;
   ThumbTableData *tt;
   GList *node;
   gint pos;

   if (!thumb) return FALSE;

   tv = thumb->thumb_view;
   if (!tv) return FALSE;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return FALSE;

   node = g_list_find (tv->thumblist, thumb);

   if (!node)
      return FALSE;

   pos = g_list_position (tv->thumblist, node);

   *col = pos % tt->colnum;
   *row = pos / tt->colnum;

   return TRUE;
}


/*
 *  create_thumbnail_button:
 *     @ Create a thumbnail button. 
 *
 *  thumb      : Pointer to Thumbnail struture.
 *  thumb_size : Size of thumbnail.
 *  type       : Thumbnail load type
 *               (from cache or original image (or other method...?)).
 *  Return     : Pointer to GtkButton widget.
 */
static GtkWidget *
create_thumbnail_button (Thumbnail *thumb, gint thumb_size)
{
   GtkWidget *button;
   GtkWidget *label;
   GtkTooltips *tooltips;
   gchar *str;
   gchar buf[BUF_SIZE];
   ThumbData *thumb_data;

   if (!thumb) return NULL;

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   if (!thumb_data) return NULL;

   thumb_data->button_vbox = gtk_vbox_new (FALSE, 0);
   button = gtk_toggle_button_new ();
   gtk_container_add (GTK_CONTAINER (button), thumb_data->button_vbox);
   gtk_container_set_border_width(GTK_CONTAINER(thumb_data->button_vbox),
				  conf.thumbbutton_border_width);

   /* get filename */
   str = g_basename (thumb->info->filename);
   label = gtk_label_new (str);
   gtk_widget_set_usize(label, thumb_size + conf.thumbbutton_border_width * 2, -1);
   gtk_box_pack_end (GTK_BOX(thumb_data->button_vbox), label, TRUE, TRUE, 0);

   gtk_widget_show (thumb_data->button_vbox);
   gtk_widget_show (label);

   /* set signal and tooltip */
   gtk_signal_connect (GTK_OBJECT(button), "enter",
		       GTK_SIGNAL_FUNC(cb_thumbbutton_enter), thumb);
   gtk_signal_connect (GTK_OBJECT(button), "toggled",
		       GTK_SIGNAL_FUNC(cb_thumbbutton_toggle), thumb);
   gtk_signal_connect_after (GTK_OBJECT(button),"button_press_event",
			     GTK_SIGNAL_FUNC(cb_thumb_button_press), thumb);
   gtk_signal_connect (GTK_OBJECT(button),"button_release_event",
		       GTK_SIGNAL_FUNC(thumbview_thumb_button_release_cb), thumb);
   gtk_signal_connect (GTK_OBJECT(button),"motion_notify_event",
		       GTK_SIGNAL_FUNC(thumbview_motion_notify_cb), thumb);
   gtk_signal_connect (GTK_OBJECT(button),"focus_in_event",
		       GTK_SIGNAL_FUNC(cb_button_focus_in), thumb);

   /* for drag file list */
   dnd_src_set (button, dnd_types, dnd_types_num);
   gtk_signal_connect (GTK_OBJECT (button), "drag_begin",
		       GTK_SIGNAL_FUNC (cb_thumbview_drag_begin),
		       thumb);
   gtk_signal_connect (GTK_OBJECT (button), "drag_data_get",
		       GTK_SIGNAL_FUNC (thumbview_drag_data_get_cb),
		       thumb->thumb_view);
   gtk_signal_connect (GTK_OBJECT (button), "drag-data-delete",
		       GTK_SIGNAL_FUNC (thumbview_drag_data_delete_cb),
		       thumb->thumb_view);
   gtk_signal_connect (GTK_OBJECT (button), "drag_end",
		       GTK_SIGNAL_FUNC (thumbview_drag_end_cb),
		       thumb->thumb_view);

   /* set tooltip */
   g_snprintf(buf, BUF_SIZE, "%s (%dkB)",
	      (gchar *) thumb->info->filename,
	      (gint) (thumb->info->st.st_size) / 1024);
   tooltips = gtk_tooltips_new();
   gtk_tooltips_set_tip(GTK_TOOLTIPS (tooltips), button,
   			buf, NULL);
   thumb_data->tooltips = tooltips;

   /* initialize button state */
   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), thumb->selected);

   thumb_data->button = button;

   return button;
}


/*
 *  thumbtable_append_thumb_frame:
 *     @ append a thumbnail button to the table.
 *
 *  tv     : Pointer to the ThumbView struct.
 *  thumb  : Pointer to the Thumbnail struct to create frame.
 *  Return : Added thumbnail button widget.
 */
static GtkWidget *
thumbtable_append_thumb_frame (ThumbView *tv, Thumbnail *thumb, gint dest_mode)
{
   GtkWidget *button;
   gint col, row;
   ThumbTableData *tt;
   ThumbData *thumb_data;

   if (!tv) return NULL;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return NULL;

   thumb_data = g_new0 (ThumbData, 1);
   g_hash_table_insert (thumb->mode_data, THUMB_TABLE_LABEL, thumb_data);

   button = create_thumbnail_button (thumb, tv->ThumbnailSize);
   if (thumb->pixmap)
      thumbtable_add_thumbnail (thumb, dest_mode, LOAD_CACHE);

   calc_thumbbutton_pos (thumb, &col, &row);
   gtk_table_attach (GTK_TABLE (tt->table), button,
		     col, col + 1, row, row + 1,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
   gtk_widget_show (button);	 

   return button;
}


static gint
idle_thumbtable_redraw (gpointer data)
{
   ThumbView *tv = data;
   ThumbTableData *tt;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *hadj, *vadj;
   GList *node;

   g_return_val_if_fail (tv, FALSE);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return FALSE;

   if (tv->disp_mode != thumbview_label_to_num (THUMB_TABLE_LABEL))
     return FALSE;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return FALSE;

   gtk_widget_queue_draw (tt->table);

   /* reset page position */
   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   hadj = gtk_scrolled_window_get_hadjustment (scrollwin);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);

   if (tt->page_pos_x < 0 - 0.1)
     tt->page_pos_x = 0;
   if (tt->page_pos_x > hadj->upper - hadj->page_size + 0.1)
     tt->page_pos_x = hadj->upper - hadj->page_size;

   if (tt->page_pos_y < 0 - 0.1)
     tt->page_pos_y = 0;
   if (tt->page_pos_y > vadj->upper - vadj->page_size + 0.1)
     tt->page_pos_y = vadj->upper - vadj->page_size;

   gtk_adjustment_set_value (hadj, 0.0);
   gtk_adjustment_set_value (vadj, 0.0);
   gtk_adjustment_set_value (hadj, tt->page_pos_x);
   gtk_adjustment_set_value (vadj, tt->page_pos_y);

   gtk_widget_queue_draw (tt->table);

   return FALSE;
}



/******************************************************************************
 *
 *   public functions.
 *
 ******************************************************************************/
/*
 *  thumbtable_destroy_table:
 *     @ destroy thumbnail table.
 *
 *  tv : Pointer to the ThumbView struct.
 */
void
thumbtable_destroy_table (ThumbView *tv)
{
   ThumbTableData *tt;

   if (!tv) return;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return;

   if (tt->table)
      gtk_widget_destroy (tt->table);

   tt->event_box = NULL;
   tt->table = NULL;
}


/*
 *  thumbtable_remove_thumbview_data:
 *
 *  tv : Pointer to the ThumbView struct.
 */
void
thumbtable_remove_thumbview_data (ThumbView *tv)
{
   ThumbTableData *tt;

   if (!tv) return;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return;

   g_hash_table_remove (tv->disp_mode_data, THUMB_TABLE_LABEL);
   g_free (tt);
}


/*
 *  thumbtable_remove_thumbnail_data:
 *
 *  thumb : Pointer to the Thumbnail struct.
 */
void
thumbtable_remove_thumbnail_data (Thumbnail *thumb)
{
   ThumbData *thumb_data;

   if (!thumb) return;

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   if (!thumb_data) return;

   g_hash_table_remove (thumb->mode_data, THUMB_TABLE_LABEL);
   gtk_object_unref (GTK_OBJECT (thumb_data->tooltips));
   g_free (thumb_data);
}


/*
 *  thumbtable_append_thumb_frames:
 *     @ add thumbnail buttons to thumbnail table
 *       (and add thumbnail to button if exist).
 *
 *  tv     : Pointer to the ThumbView struct.
 *  start  : Pointer to Start of the Thumbnail list.
 *  Return : Return FALSE if canceled by user.
 */
gboolean
thumbtable_append_thumb_frames (ThumbView *tv, GList *start, gint dest_mode)
{
   Thumbnail  *thumb;
   GList *node;

   g_return_val_if_fail (tv, FALSE);
   if (!start) return FALSE;

   node = start;
   while (node) {
      thumb = (Thumbnail *) node->data;

      /* add button */
      thumbtable_append_thumb_frame (tv, thumb, dest_mode);

      node = g_list_next (node);
   }

   return TRUE;
}


/*
 *  thumbtable_add_thumbnail:
 *     @ append thumbnail to thumbnail frame.
 *
 *  thumb     : Pointer to the Thumbnail struct.
 *  dest_mode : New thumbnail view display mode.
 *  Return    : Added thumbnail pixmap widget.
 */
GtkWidget *
thumbtable_add_thumbnail (Thumbnail *thumb, gint dest_mode,
			  ThumbLoadType type)
{
   ThumbView *tv = thumb->thumb_view;
   ThumbData *thumb_data;

   if (!thumb) return NULL;

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   if (!thumb_data) return NULL;

   if (!thumb->pixmap)
      thumb->pixmap = create_thumbnail_auto (thumb, tv->ThumbnailSize, type);

   if (thumb->pixmap) {
      gtk_widget_ref (thumb->pixmap);
      gtk_box_pack_start (GTK_BOX (thumb_data->button_vbox), thumb->pixmap, TRUE, TRUE, 0);
      gtk_widget_show (thumb->pixmap);
   }

   return thumb->pixmap;
}


/*
 *  thumbtable_redraw:
 *     @ For resize table or sort files list, once tear off thumbnail buttons
 *       from thumbnail table, and attach to new resized table.
 *
 *  tv        : Pointer to ThumbView struct for redraw.
 *  dest_mode : New thumbnail view display mode.
 *  Return    : Pointer to the new GtkTable widget.
 */
GtkWidget *
thumbtable_redraw (ThumbView *tv, gint dest_mode, GtkWidget *scroll_win)
{
   ThumbTableData *tt;
   Thumbnail *thumb;
   ThumbData *thumb_data;
   GtkWidget *table, *event_box, *button;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *vadj;
   GList *node;
   gint pos, colnum;
   gint col, row;
   GtkWidget *hbox;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (ThumbViewList, tv);
   g_return_val_if_fail (node, NULL);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);

   if (!tt) {
      thumbtable_create (tv, dest_mode);
      tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   }

   if (tv->container
       && tv->disp_mode == thumbview_label_to_num (THUMB_TABLE_LABEL))
   {
     scrollwin = GTK_SCROLLED_WINDOW (tv->container);
     vadj = gtk_scrolled_window_get_vadjustment (scrollwin);
     tt->page_pos_y = vadj->value;
   }

   colnum = tt->colnum;
   calc_thumbtable_col_row_num (tv, 0);

   if (tt->table) {
      event_box = gtk_event_box_new ();
      hbox = gtk_hbox_new (FALSE, 0);
      table = gtk_table_new((g_list_length (tv->thumblist) - 1) / tt->colnum + 1,
			    tt->colnum, TRUE);
      gtk_container_set_border_width (GTK_CONTAINER (table), conf.thumbtable_col_space);
      gtk_container_add (GTK_CONTAINER (event_box), hbox);
      gtk_box_pack_start (GTK_BOX (hbox), table, TRUE, FALSE, 0);

      gtk_table_set_row_spacings (GTK_TABLE (table), conf.thumbtable_row_space);
      gtk_table_set_col_spacings (GTK_TABLE (table), conf.thumbtable_col_space);

      node = tv->thumblist;
      pos = 0;

      while (node) {
	 thumb = node->data;

	 thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);

	 /* If thumbnail was appended at other disp-mode,
	    append it to original table. */
	 if (!thumb_data) {
	    thumbtable_append_thumb_frame (tv, thumb, dest_mode);
	    thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
	 }

	 col = pos % tt->colnum;
	 row = pos / tt->colnum;

	 button = thumb_data->button;

	 if (button) {
	    gtk_widget_ref (button);
	    gtk_container_remove (GTK_CONTAINER (tt->table), button);
	    gtk_table_attach (GTK_TABLE (table), button,
			      col, col + 1,
			      row, row + 1,
			      GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
	    gtk_widget_unref (button);
	 }

	 /* reset button state */
	 gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), thumb->selected);

	 node = g_list_next (node);
	 pos++;
      }

      tt->event_box = event_box;
      tt->hbox = hbox;
      tt->table = table;

      gtk_widget_show (tt->event_box);
      gtk_widget_show (tt->hbox);
      gtk_widget_show (tt->table);

      /* set callback */
      gtk_signal_connect (GTK_OBJECT (tt->event_box), "expose_event",
			  GTK_SIGNAL_FUNC (cb_expose), tv);

      /* for drag and drop */
      dnd_dest_set (tt->event_box, dnd_types, dnd_types_num);
      gtk_signal_connect(GTK_OBJECT (tt->event_box), "drag_data_received",
			 GTK_SIGNAL_FUNC (thumbview_drag_data_received_cb), tv);
      gtk_signal_connect(GTK_OBJECT (tt->event_box), "drag_end",
			 GTK_SIGNAL_FUNC (thumbview_drag_end_cb), tv);
   } else {
      thumbtable_create (tv, dest_mode);
   }

   /* redraw */
   if (scroll_win) {
      scrollwin = GTK_SCROLLED_WINDOW (scroll_win);
      if (GTK_BIN (tv->container)->child)
	 gtk_widget_destroy(GTK_BIN (tv->container)->child);   
      gtk_scrolled_window_add_with_viewport (scrollwin, tt->event_box);
   }

   if (dest_mode == thumbview_label_to_num (THUMB_TABLE_LABEL)) {
     gtk_idle_add (idle_thumbtable_redraw, tv);
   }

   return tt->event_box;
}


/*
 *  thumbtable_refresh_thumbnail:
 *     @ Create thumbnail from original image, and recreate thumbnail button.
 *
 *  thumb  : Pointer to the Thumbnail struct.
 *  Return : TRUE if success.
 */
gboolean
thumbtable_refresh_thumbnail (Thumbnail *thumb)
{
   ThumbView *tv;
   ThumbTableData *tt;
   GtkWidget *button;
   gint col, row;
   ThumbData *thumb_data;

   if (!thumb) return FALSE;

   tv = thumb->thumb_view;
   if (!tv) return FALSE;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return FALSE;

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   if (!thumb_data) return FALSE;

   gtk_widget_destroy (thumb_data->button);

   button = create_thumbnail_button (thumb, tv->ThumbnailSize);

   if (!button)
      return FALSE;

   thumbtable_add_thumbnail (thumb, tv->ThumbnailSize, CREATE_THUMB);
   calc_thumbbutton_pos (thumb, &col, &row);
   gtk_table_attach (GTK_TABLE (tt->table), button,
		     col, col + 1, row, row + 1,
		     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
   gtk_widget_show (button);

   thumb_data->button = button;

   if (button)
      return TRUE;
   else
      return FALSE;
}


/*
 *  thumbtable_resize:
 *     @ Resize thumbnail table (only call thumbtable_redraw ())
 *
 *  tv     : Pointer to the ThumbView struct.
 *  Return : Pointer to the New thumbnail table widget.
 */
GtkWidget *
thumbtable_resize (ThumbView *tv)
{
   ThumbTableData *tt;
   GList *node;
   gint colnum;

   if (!tv) return NULL;

   node = g_list_find (ThumbViewList, tv);
   if (!node) return NULL;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return NULL;

   colnum = tt->colnum;
   calc_thumbtable_col_row_num (tv, 0);
   if (colnum == tt->colnum)
      return tt->event_box;
   else
      return thumbtable_redraw (tv, tv->disp_mode, tv->container);
}


void
thumbtable_adjust (ThumbView *tv, Thumbnail *thumb)
{
   ThumbTableData *tt;
   ThumbData *thumb_data;   
   GList *node;

   GtkWidget *button;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *hadj, *vadj;
   gint top, buttom, left, right;
   gint frame_top, frame_buttom, frame_left, frame_right;
   gfloat pos;

   g_return_if_fail (tv);
   g_return_if_fail (thumb);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   g_return_if_fail (tt);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_if_fail (thumb_data);

   button = thumb_data->button;

   /*
   tt->focused = NULL;
   gtk_widget_grab_focus (thumb_data->button);
   */

   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   hadj = gtk_scrolled_window_get_hadjustment (scrollwin);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);

   left = button->allocation.x;
   right = left + button->allocation.width;
   top = button->allocation.y;
   buttom = top + button->allocation.height;

   frame_left = hadj->value;
   frame_right = frame_left + hadj->page_size;
   frame_top = vadj->value;
   frame_buttom = frame_top + vadj->page_size;

   if (right > frame_right) {
      pos = right - (int) hadj->page_size;
      gtk_adjustment_set_value (hadj, pos);
   } else if (left < frame_left) {
      gtk_adjustment_set_value (hadj, left);
   }

   if (buttom > frame_buttom) {
      pos = buttom - (int) vadj->page_size;
      gtk_adjustment_set_value (vadj, pos);
   } else if (top < frame_top) {
      gtk_adjustment_set_value (vadj, top);
   }
}


gboolean
thumbtable_set_selection (Thumbnail *thumb, gboolean select)
{
   ThumbData *thumb_data;
   GtkWidget *button;

   g_return_val_if_fail (thumb, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (thumb_data, FALSE);

   thumb->selected = select;
   button = thumb_data->button;
   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), thumb->selected);

   /*
   if (thumb->selected) {
      hilight_thumbnail (button);
   } else {
      if (!normal_style)
	 normal_style = gtk_style_copy (gtk_widget_get_style (button));
      set_style(button, normal_style);
   }
   */

   return TRUE;
}


/*
 *  thumbtable_create:
 *     @ Create thumbnail table widget.
 *
 *  tv        : Pointer to the ThumbView struct.
 *  dest_mode : New display mode.
 *  Return    : GtkTable widget.
 */
GtkWidget *
thumbtable_create (ThumbView *tv, gint dest_mode)
{
   ThumbTableData *tt;
   gint table_rows, num;

   g_return_val_if_fail (tv, NULL);

   /*
   if (!normal_style)
      normal_style = gtk_style_copy (gtk_widget_get_style(tv->thumb_window->window));
   */

   tt = g_new0 (ThumbTableData, 1);
   tt->focused = NULL;
   tt->page_pos_x = 0.0;
   tt->page_pos_y = 0.0;
   g_hash_table_insert (tv->disp_mode_data, THUMB_TABLE_LABEL, tt);

   num = g_list_length (g_list_first(tv->thumblist));
   table_rows = calc_thumbtable_col_row_num (tv, num);

   tt->event_box = gtk_event_box_new ();
   tt->hbox = gtk_hbox_new (FALSE, 0);
   tt->table = gtk_table_new (table_rows, tt->colnum, FALSE);
   gtk_container_add (GTK_CONTAINER (tt->event_box), tt->hbox);
   gtk_box_pack_start (GTK_BOX (tt->hbox), tt->table, TRUE, TRUE, 0);

   gtk_table_set_row_spacings (GTK_TABLE (tt->table), conf.thumbtable_row_space);
   gtk_table_set_col_spacings (GTK_TABLE (tt->table), conf.thumbtable_col_space);

   gtk_widget_show (tt->event_box);
   gtk_widget_show (tt->hbox);
   gtk_widget_show (tt->table);

   /* set callback */
   gtk_signal_connect (GTK_OBJECT (tt->event_box), "expose_event",
		       GTK_SIGNAL_FUNC (cb_expose), tv);

   /* for drag and drop */
   dnd_dest_set (tt->event_box, dnd_types, dnd_types_num);
   gtk_signal_connect(GTK_OBJECT (tt->event_box), "drag_data_received",
		      GTK_SIGNAL_FUNC (thumbview_drag_data_received_cb), tv);
   gtk_signal_connect(GTK_OBJECT (tt->event_box), "drag_end",
		      GTK_SIGNAL_FUNC (thumbview_drag_end_cb), tv);

   /* create buttons */
   thumbtable_append_thumb_frames (tv, tv->thumblist, dest_mode);

   return tt->event_box;
}
