/*  GKrellMMS: GKrellM XMMS Plugin
 |  Copyright (C) 2000 Sander Klein Lebbink
 |
 |  Author:  Sander Klein Lebbink <sander@cerberus.demon.nl>
 |  Latest versions might be found at:  http://gkrellm.luon.net/
 |
 |  This program is free software which I release under the GNU General Public
 |  License. You may redistribute and/or modify this program under the terms
 |  of that license as published by the Free Software Foundation; either
 |  version 2 of the License, or (at your option) any later version.
 |
 |  This program is distributed in the hope that it will be useful,
 |  but WITHOUT ANY WARRANTY; without even the implied warranty of
 |  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 |  GNU General Public License for more details.
 |
 |  To get a copy of the GNU General Public License,  write to the
 |  Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gkrellmms.h"

#include "pixmaps/prev.xpm"
#include "pixmaps/play_pause.xpm"
#include "pixmaps/stop.xpm"
#include "pixmaps/next.xpm" 
#include "pixmaps/eject.xpm"

#include "pixmaps/prev_clicked.xpm"
#include "pixmaps/play_pause_clicked.xpm"
#include "pixmaps/stop_clicked.xpm"
#include "pixmaps/next_clicked.xpm" 
#include "pixmaps/eject_clicked.xpm"

/* Define xmms-vars */
static gint cur_len;
static gint where_to_jump;
static gint t;

static gboolean xmms_playing = FALSE;
static gboolean xmms_draw_green = FALSE;

/* Evil var! */
static GtkWidget *gkrellm_vbox;

/* Define gkrellm-vars */
static TextStyle *ts, *ts_alt;
static Style *style;
static Style *scroll_style;

/* for scrolling text */
static Panel *panel = NULL;
static Decal *scroll_text;
static Decal *scroll_in_motion;
static gint scroll_motion_x, x_scroll;
static GtkTooltips *scrolling_tooltip;
static gchar *scrolling_tooltip_text;

/* for time-bar */
static Krell *time_krell, *slider_in_motion;
static Decal *xmms_label,
             *xmms_indicator;
static gboolean got_motion;

/* for button-bar */
static Panel *button_bar = NULL;

#if ((VERSION_MAJOR>0)||(VERSION_MINOR>9))
static gint style_id;
#endif

void do_xmms_command(gint i)
{
  if (xmms_running)
  {
    switch (i)
    {
      case prev: xmms_remote_playlist_prev(xmms_session); break;
      case play:
        if (xmms_remote_is_playing(xmms_session) &&
            !xmms_remote_is_paused(xmms_session))
          xmms_remote_pause(xmms_session);
        else
          xmms_remote_play(xmms_session);
        break;
      case paus: xmms_remote_pause(xmms_session); break;
      case stop:
        xmms_remote_stop(xmms_session);
        time_krell->previous =
        t = 0;
        break;
      case next: xmms_remote_playlist_next(xmms_session); break;
      case eject:
        if (eject_opens_playlist)
          pl_open_func(NULL, NULL);
        else
          xmms_remote_eject(xmms_session);
        break;
      default:
        gkrellm_message_window("GKrellMMS Error", "You shouldn't get this window!", NULL);
    }
  }
}

static void
cb_button(DecalButton *button)
{
  do_xmms_command(GPOINTER_TO_INT(button->data));
}

static void
update_gkrellmms()
{
  static gint output_time, len, w, separator_len;
  static gint prev_where;
  static gboolean previous_state;
  static gboolean blink;
  gchar *scrolling_title_text;
  gchar *prev_scrolling_tooltip_text;
  gchar *more_scrolled;
  gint pos_time;
  gchar *filename;
  gchar *time_bar_text;
  playlist_def *templ;
  gint rate;
  gint freq;
  gint nch;

  if (xmms_remote_is_running(xmms_session))
  {
    xmms_running = TRUE;
    previous_state = xmms_draw_green;
    xmms_playing = xmms_remote_is_playing(xmms_session);
    prev_where = xmms_pos;
    xmms_pos = xmms_remote_get_playlist_pos(xmms_session);
    cur_len = xmms_remote_get_playlist_time(xmms_session, xmms_pos);
    xmms_draw_green = (xmms_playing && !xmms_remote_is_paused(xmms_session)) ? TRUE : FALSE;

    /* Update gkrellmms-panels when xmms is going on */
    if ((scroll_enable && panel == NULL)
       || (buttons_enable && button_bar == NULL))
      update_gkrellmms_config(1);

    /*
     | Check if there is a local playlist set; if not, load it!
     */
     if ((plist == NULL) ||
       (g_list_length(plist) != xmms_remote_get_playlist_length(xmms_session)))
       update_playlist();

    if (g_list_length(plist) > 0)
      templ = (playlist_def *) g_list_nth(plist, xmms_pos)->data;
    else
    {
      templ = g_malloc(sizeof(playlist_def *));
      templ->time = 0;
      templ->title = "";
      templ->filename = "";
    }
    pos_time = templ->time / 1000;

    if (panel) {
      if ((prev_where != xmms_pos) || (pos_time != cur_len / 1000))
      {
        update_plist_statusbar(xmms_pos);
        update_plist_window_row(prev_where, xmms_pos);
      }

      /* Scrollbar */
      if (scroll_enable && !scroll_in_motion)
      {
        w = gkrellm_chart_width() - (2 * style->margin) - 2;

        if (templ->time > 0)
          scrolling_title_text = g_strdup_printf("%d. %s (%d:%02d)",
            xmms_pos + 1, templ->title, pos_time / 60, pos_time % 60);
        else
          scrolling_title_text = g_strdup_printf("%d. %s",
            xmms_pos + 1, templ->title);

        if (scrolling_tooltip != NULL)
        {
          xmms_remote_get_info(xmms_session, &rate, &freq, &nch);
          prev_scrolling_tooltip_text = g_strdup(scrolling_tooltip_text);

          g_free(scrolling_tooltip_text);

          scrolling_tooltip_text = g_strdup_printf("%s\n%d%s - %dHz - %s",
            scrolling_title_text, rate / 1000,
            (templ->time  == -1) ? "bpm" : "kb/s", freq,
            (nch == 1) ? "mono" : "stereo");

          if (strcmp(prev_scrolling_tooltip_text, scrolling_tooltip_text) != 0)
            gtk_tooltips_set_tip(scrolling_tooltip, panel->drawing_area,
              scrolling_tooltip_text, NULL);

          g_free(prev_scrolling_tooltip_text);
        }

        len = gdk_string_width(panel->textstyle->font, scrolling_title_text);
        separator_len = gdk_string_width(panel->textstyle->font, SCROLL_SEPARATOR);

        if (len > w)
        {
          more_scrolled = g_strdup_printf("%s%s%s",
            scrolling_title_text, SCROLL_SEPARATOR, scrolling_title_text);
          x_scroll = (x_scroll + 1) % (len + separator_len);
          scroll_text->x_off = w - x_scroll - len;
        }
        else
        {
          more_scrolled = g_strdup("");
          x_scroll = scroll_text->x_off = 0;
        }

        gkrellm_draw_decal_text(panel, scroll_text,
          (len > w) ? more_scrolled : scrolling_title_text,
          (gulong) (len > w) ? (w - x_scroll - len) : -1);
        gkrellm_draw_layers(panel);

        g_free(scrolling_title_text);
        g_free(more_scrolled);
      }
    }

    /* Title-scroll is done, check for time-bar update */
    if (GK.second_tick)
    {
      /*
       | Always draw a green led on second and third tick.
       |
       | Third tick isn't fetchable, so there will also be set
       | a var 'blink', so that the led doesn't blink to fast
       | when stopped. :)
       */
      xmms_draw_green = TRUE;

      if (g_list_length(plist))
      {
        /* Little after-check */
        filename = xmms_remote_get_playlist_file(xmms_session, xmms_pos);
        if (strcmp(templ->filename, filename) != 0)
          update_playlist();
        g_free(filename);
      }
    }
  
    /* The time bar has a max.-scale of 100, so count the output-time
     | to a max. of 100
     */
    output_time = xmms_remote_get_output_time(xmms_session);
    if (!slider_in_motion /* && (GK.second_tick || (prev_where != xmms_pos)) */)
    {
      t = cur_len ?
        ((output_time * 100) / cur_len)
        : 0;
      if (t < 0)
        t = 0;
      else if (t > 100)
        t = 100;
    }
  } else {
    xmms_running = FALSE;
    xmms_draw_green = FALSE;
    xmms_pos = -1;
    t = 0;

    /* update gkrellmms-panels when xmms is going off */
    if ((scroll_enable && panel != NULL)
       || (buttons_enable && button_bar != NULL))
      update_gkrellmms_config(1);

    if (plist != NULL)
      update_playlist();

    if (panel != NULL)
    {
      /* If all goes well, this piece of code should never
       | be executed.
       */
      gkrellm_draw_decal_text(panel, scroll_text, (gchar *) "", (gulong) t);
      gkrellm_draw_layers(panel);
    }
  }

  /* Also draw xmms-status when xmms isn't running,
   |  but don't while seeking
   */
  if (slider_in_motion == NULL)
  {
    if (xmms_running && (xmms_playing || (prev_where != xmms_pos)) && draw_time)
    {
      /* Have to re-define templ, or else gcc cries. :( */
      if (g_list_length(plist) > 0)
        templ = (playlist_def *) g_list_nth(plist, xmms_pos)->data;
      else
      {
        templ = g_malloc(sizeof(playlist_def *));
        templ->time = 0;
        templ->title = "";
        templ->filename = "";
      }

      time_bar_text = (time_format || templ->time <= 0)
                    ? g_strdup_printf("%02d:%02d",
                      output_time / 60000, (output_time / 1000) % 60)
                    : g_strdup_printf(draw_minus ? "-%02d:%02d" : "%02d:%02d",
                      (cur_len - output_time) / 60000,
                      ((cur_len - output_time) / 1000) % 60);
    } else
      time_bar_text = g_strdup(gkrellmms_label);

    if (xmms_remote_is_paused(xmms_session))
      xmms_draw_green = xmms_draw_green ? FALSE : TRUE;
    gkrellm_draw_decal_text(time_bar, xmms_label, time_bar_text, -1);
    gkrellm_draw_decal_pixmap(time_bar, xmms_indicator,
      (xmms_draw_green) ? D_MISC_LED1 : D_MISC_LED0);
    time_krell->previous = 0;
    gkrellm_update_krell(time_bar, time_krell, (gulong) t);
    gkrellm_draw_layers(time_bar);
    g_free(time_bar_text);
  }

  if (blink)
    blink = FALSE;
  if (xmms_draw_green && !blink)
    blink = TRUE;
  else
    blink = FALSE;
}

static gint
panel_expose_event(GtkWidget *widget, GdkEventExpose *ev, int i)
{
  gdk_draw_pixmap(widget->window,
                  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                  (i == 1) ? panel->pixmap :
                  ((i == 2) ? time_bar->pixmap : button_bar->pixmap),
                  ev->area.x, ev->area.y, ev->area.x,
                  ev->area.y, ev->area.width, ev->area.height);
  return FALSE;
}

static void
slider_motion(GtkWidget *widget, GdkEventMotion *ev, gpointer data)
{
  gint   z;
  GdkModifierType state;

  if ((slider_in_motion != NULL) && g_list_length(plist))
  {
    /* Check if button is still pressed, in case missed button_release */
    state = ev->state;
    if (!(state & GDK_BUTTON1_MASK))
    {
      slider_in_motion = NULL;
      return;
    }
    z = ev->x * time_krell->full_scale / (gkrellm_chart_width() - 1);
    if (z < 0)
      z = 0;
    else if (z > 100)
      z = 100;
    t = z;
    where_to_jump = (z * cur_len) / 100;

    got_motion = TRUE;
    time_krell->previous = 0;
    gkrellm_update_krell(time_bar, time_krell, (gulong) t);
    gkrellm_draw_layers(time_bar);
  }
}

static void
panel_button_release(GtkWidget *widget, GdkEventButton *ev, gpointer data)
{
  gint z;
  gint timer;
  time_t lt;
  playlist_def *templ;

  if (slider_in_motion && g_list_length(plist))
  {
    if (!got_motion)
    {
      /* Also jump to time if you've clicked once on the Krell */
      z = ev->x * time_krell->full_scale / (gkrellm_chart_width() - 1);
      if (z < 0)
        z = 0;
      else if (z > 100)
        z = 100;
      t = z;
      where_to_jump = (z * cur_len) / 100;
    }

    templ = (playlist_def *) g_list_nth(plist, xmms_pos)->data;
    if (where_to_jump > templ->time)
      return;

    time_krell->previous = 0;
    gkrellm_update_krell(time_bar, time_krell, (gulong) t);
    gkrellm_draw_layers(time_bar);
    /*
     | Let gkrellm sleep for about 0 seconds when sending this commands
     | when xmms isn't playing, so that xmms received the functions for
     | sure, and gkrellm doesn't do weird stuff on it.
     | Maybe it's not neccasary if you've got a fast computer, but on my
     | P200 I really notice some bugs without it.
     */
    if (!xmms_playing)
      xmms_remote_play(xmms_session);
    timer = localtime(&lt)->tm_sec;

    /* Do nothing, wait until xmms really plays;
     | stop waiting after 10 seconds.
     */
    while (!xmms_remote_is_playing(xmms_session) &&
      ((localtime(&lt)->tm_sec - timer) < 10))
    {
      usleep(0);
    }

    xmms_remote_jump_to_time(xmms_session, where_to_jump);

    timer = localtime(&lt)->tm_sec;

    /* Wait till really jumped before we continue. */
    while ((xmms_remote_get_output_time(xmms_session) / 1000) != (where_to_jump / 1000)
           && ((localtime(&lt)->tm_sec - timer) < 10))
      usleep(0);
  }
  else if ((slider_in_motion != NULL) && (!g_list_length(plist) || !xmms_playing))
    xmms_remote_play(xmms_session);

  slider_in_motion = NULL;
  got_motion = FALSE;
}

static void
panel_button_press(GtkWidget *widget, GdkEventButton *ev, gpointer data)
{
  switch (ev->button)
  {
    case 1:
      slider_in_motion = xmms_running ? time_krell : NULL;
      if ((ev->type == GDK_2BUTTON_PRESS) && !xmms_running)
        xmms_start_func();
      break;
    case 2:
      if (xmms_running && xmms_remote_is_playing(xmms_session))
      {
        if (krell_mmb_pause)
          xmms_remote_pause(xmms_session);
        else
          xmms_remote_stop(xmms_session);
      }
      else if (xmms_running)
        xmms_remote_play(xmms_session);
      else
        xmms_start_func();
      break;
    case 3:
      options_menu(ev);
      break;
  }
}

static void
scroll_bar_press(GtkWidget *widget, GdkEventButton *ev, gpointer data)
{
  if (g_list_length(plist) && (ev->button == 1))
  {
    scroll_in_motion = scroll_text;
    scroll_motion_x = ev->x;
  }
}

static void
scroll_bar_release(GtkWidget *widget, GdkEventButton *ev, gpointer data)
{
  scroll_in_motion = NULL;
}

static void
scroll_bar_motion(GtkWidget *widget, GdkEventButton *ev, gpointer data)
{
  gint w, len, change, separator_len;
  GdkModifierType state;
  playlist_def *templ;
  gchar *scrolling_title_text;
  gchar *more_scrolled;

  if (scroll_in_motion != NULL)
  {
    /* Check if button is still pressed, in case missed button_release */
    state = ev->state;
    if (!(state & GDK_BUTTON1_MASK))
    {
      scroll_in_motion = NULL;
      return;
    }

    w = gkrellm_chart_width() - (2 * style->margin) - 2;
    templ = (playlist_def *) g_list_nth(plist, xmms_pos)->data;
    change = scroll_motion_x - ev->x;
    scroll_motion_x = ev->x;

    if (templ->time > 0)
      scrolling_title_text = g_strdup_printf("%d. %s (%d:%02d)",
        xmms_pos + 1, templ->title, (templ->time / 1000) / 60, (templ->time / 1000) % 60);
    else
      scrolling_title_text = g_strdup_printf("%d. %s",
        xmms_pos + 1, templ->title);

    len = gdk_string_width(ts_alt->font, scrolling_title_text);
    separator_len = gdk_string_width(ts_alt->font, SCROLL_SEPARATOR);

    if (len <= w)
    {
      scroll_in_motion = NULL;
      g_free(scrolling_title_text);
      return;
    }

    x_scroll = (x_scroll + change) % (len + separator_len);
    if (x_scroll < 0)
      x_scroll = len + separator_len;

    more_scrolled = g_strdup_printf("%s%s%s",
      scrolling_title_text, SCROLL_SEPARATOR, scrolling_title_text);

    scroll_text->x_off = w - x_scroll - len;
    gkrellm_draw_decal_text(panel, scroll_text, more_scrolled, (gulong) w - x_scroll - len);

    gkrellm_draw_layers(panel);
    g_free(scrolling_title_text);
    g_free(more_scrolled);

  }
}

#if ((VERSION_MAJOR<1)&&(VERSION_MINOR<10))

static gchar *image_type[] =
{
  "png", "jpg", "xpm", "gif"
};

static gchar *
button_file_exists(char *directory)
{
  gint i;
  static gchar *button_file;
  struct stat  st;

  if (*(GK.theme_path) == '\0')
    return NULL;
  if (button_file)
    g_free(button_file);
  
  for (i = 0; i < sizeof(image_type) / sizeof(char *); ++i)
  {
    button_file = g_strdup_printf("%s.%s", directory, image_type[i]);

    if (stat(button_file, &st) == 0 &&
       (S_ISREG(st.st_mode) || S_ISLNK(st.st_mode)))
    {
      return button_file;
    }
    g_free(button_file);
    button_file = NULL;
  }
  return NULL;
}

#endif

static DecalButton *prev_but;
static DecalButton *play_but;
static DecalButton *stop_but;
static DecalButton *next_but;
static DecalButton *eject_but;

static void
create_gkrellmms(GtkWidget *vbox, gint first_create)
{
  static gboolean first_time = TRUE;
  GdkBitmap *mask, *button_mask;
  GdkPixmap *pixmap;
  GdkImlibImage *time_button;

  /* for button-bar */
  gint x, y;
  Decal *prev_decal,
        *play_decal,
        *stop_decal,
        *next_decal,
        *eject_decal;
  GdkImlibImage *prev_image = NULL, *prev_image_clicked = NULL,
                *play_image = NULL, *play_image_clicked = NULL,
                *stop_image = NULL, *stop_image_clicked = NULL,
                *next_image = NULL, *next_image_clicked = NULL,
                *eject_image = NULL, *eject_image_clicked = NULL;
  static GdkImlibImage *bg_scroll_image;
  static GdkPixmap *prev_pix = NULL,
                   *play_pix = NULL,
                   *stop_pix = NULL,
                   *next_pix = NULL,
                   *eject_pix = NULL;
  static gchar *theme_dir;

#if ((VERSION_MAJOR<1)&&(VERSION_MINOR<10))
  static gchar *button_file;
  static gchar *button_file_fixed;
#endif

  gkrellm_vbox = vbox;

  if (first_time && xmms_autostart && !xmms_remote_is_running(xmms_session))
    xmms_start_func();

  xmms_running = xmms_remote_is_running(xmms_session);

  if (auto_main_close && xmms_running && first_create && !config_update)
    xmms_remote_main_win_toggle(xmms_session, (gboolean) 0);

  if (!first_time && xmms_running && first_create && !config_update)
  {
    /* I hope this should always work! */
    if (!xmms_remote_is_playing(xmms_session))
    {
/* Obsolete!
      open_playlist(playlist_file, 0);
*/
      set_playlist_position(0);
      set_time_position(0);
    }
  }

  if (plist == NULL)
    update_playlist();

  first_time = FALSE;

  if (first_create || (time_bar == NULL))
  {
    time_bar = gkrellm_panel_new0();
  } else {
    gkrellm_destroy_krell_list(time_bar);
    gkrellm_destroy_decal_list(time_bar);
  }
  style = gkrellm_meter_style(DEFAULT_STYLE);
  if (scroll_style)
    g_free(scroll_style);
#if ((VERSION_MAJOR>1)||(VERSION_MAJOR==1 && VERSION_MINOR==0 && VERSION_REV>2)||(VERSION_MAJOR==1 && VERSION_MINOR>0))
    scroll_style = gkrellm_copy_style(style);
#else
    scroll_style = gkrellm_style_new0();
    *scroll_style = *style;
#endif

  ts = gkrellm_meter_textstyle(GKRELLMMS_STYLE);
  ts_alt = gkrellm_meter_alt_textstyle(GKRELLMMS_STYLE);
  time_bar->textstyle = ts;

  /* Scroll Bar */
  if (xmms_running && scroll_enable)
  {

    if (first_create)
      panel = gkrellm_panel_new0();
    else
      gkrellm_destroy_decal_list(panel);

    panel->textstyle = ts_alt;

#if ((VERSION_MAJOR>0)||(VERSION_MINOR>9))
    /* For versions greater than 0.9.x, check for a custom themed background
     | for the scrolling text.
     */
    if (bg_scroll_image)
      { /* Avoid memory leaks at theme changes */
      gdk_imlib_kill_image(bg_scroll_image);
      bg_scroll_image = NULL;
    }
    gkrellm_load_image("bg_scroll", NULL, &bg_scroll_image, STYLE_NAME);
    if (bg_scroll_image)
       		gkrellm_set_image_border("gkrellmms_bg_scroll",
       		bg_scroll_image, scroll_style);
#if ((VERSION_MAJOR>1)||(VERSION_MAJOR==1 && VERSION_MINOR==0 && VERSION_REV>2)||(VERSION_MAJOR==1 && VERSION_MINOR>0))
	gkrellm_get_gkrellmrc_integer("gkrellmms_scroll_margin",
       		&scroll_style->margin);
	gkrellm_get_gkrellmrc_integer("gkrellmms_scroll_top_margin",
		&scroll_style->top_margin);
	gkrellm_get_gkrellmrc_integer("gkrellmms_scroll_bottom_margin",
		&scroll_style->bottom_margin);
#endif
#endif

    /* Text decal for scrolling title */
    scroll_text = gkrellm_create_decal_text(panel, (gchar *) "Apif0", ts_alt, scroll_style, -1, -1, -1);

    gkrellm_configure_panel(panel, NULL, scroll_style);
    gkrellm_create_panel(gkrellm_vbox, panel, bg_scroll_image ? bg_scroll_image : gkrellm_bg_meter_image(DEFAULT_STYLE));
    gkrellm_monitor_height_adjust(panel->h);

    if (scrolling_tooltip == NULL)
    {
      scrolling_tooltip = gtk_tooltips_new();
      scrolling_tooltip_text = g_strdup("xmms");
      gtk_tooltips_set_tip(scrolling_tooltip, panel->drawing_area,
                           scrolling_tooltip_text, NULL);
      gtk_tooltips_set_delay(scrolling_tooltip, 750);
    }

    if (first_create)
    {
      gtk_signal_connect(GTK_OBJECT (panel->drawing_area), "expose_event",
                         (GtkSignalFunc) panel_expose_event, (int *) 1);
      gtk_signal_connect(GTK_OBJECT(panel->drawing_area), "button_press_event",
                         (GtkSignalFunc) scroll_bar_press, NULL);
      gtk_signal_connect(GTK_OBJECT(panel->drawing_area), "button_release_event",
                         (GtkSignalFunc) scroll_bar_release, NULL);
      gtk_signal_connect(GTK_OBJECT(panel->drawing_area), "motion_notify_event",
                         (GtkSignalFunc) scroll_bar_motion, NULL);
    }
  }

  /* Stuff for time bar */
  time_button = gkrellm_krell_meter_image(DEFAULT_STYLE);
  time_krell = gkrellm_create_krell(time_bar, time_button, style);
  time_krell->full_scale = 100;

  xmms_label = gkrellm_create_decal_text(time_bar, (gchar *) "Ap0", ts, style, -1, -1,
  gkrellm_chart_width() - ((GdkWindowPrivate *) gkrellm_decal_misc_pixmap())->width - 2 * style->margin);
  xmms_label->x_off = style->margin;

  pixmap = (GdkPixmap *) gkrellm_decal_misc_pixmap();
  mask = (GdkBitmap *) gkrellm_decal_misc_mask();
  xmms_indicator = gkrellm_create_decal_pixmap(time_bar, gkrellm_decal_misc_pixmap(),
                   gkrellm_decal_misc_mask(), N_MISC_DECALS,
                   style, gkrellm_chart_width() -
                   ((GdkWindowPrivate *) gkrellm_decal_misc_pixmap())->width
                   - style->margin, -1);

  gkrellm_configure_panel(time_bar, NULL, style);
  gkrellm_create_panel(vbox, time_bar, gkrellm_bg_meter_image(DEFAULT_STYLE));
  gkrellm_monitor_height_adjust(time_bar->h);
  gkrellm_draw_decal_text(time_bar, xmms_label, gkrellmms_label, -1);
  gkrellm_draw_decal_pixmap(time_bar, xmms_indicator, D_MISC_LED0);
  gkrellm_update_krell(time_bar, time_krell, (gulong) 0);
  gkrellm_draw_layers(time_bar);

  if (xmms_running && buttons_enable)
  {
    if (first_create)
      button_bar = gkrellm_panel_new0();
    else
      gkrellm_destroy_decal_list(button_bar);

    button_bar->textstyle = ts;

    g_free(theme_dir);
#if ((VERSION_MAJOR>0)||(VERSION_MINOR>9))
    theme_dir = g_strdup_printf("%s/gkrellmms", gkrellm_get_theme_path());
#else
    theme_dir = g_strdup_printf("%s/gkrellmms", GK.theme_path);
#endif


    /* Load images */

#if ((VERSION_MAJOR>0)||(VERSION_MINOR>9))
    gkrellm_load_image("prev", prev_xpm, &prev_image, STYLE_NAME);
    gkrellm_load_image("play_pause", play_pause_xpm, &play_image, STYLE_NAME);
    gkrellm_load_image("stop", stop_xpm, &stop_image, STYLE_NAME);
    gkrellm_load_image("next", next_xpm, &next_image, STYLE_NAME);
    gkrellm_load_image("eject", eject_xpm, &eject_image, STYLE_NAME);

    gkrellm_load_image("prev_clicked", prev_clicked_xpm, &prev_image_clicked, STYLE_NAME);
    gkrellm_load_image("play_pause_clicked", play_pause_clicked_xpm, &play_image_clicked, STYLE_NAME);
    gkrellm_load_image("stop_clicked", stop_clicked_xpm, &stop_image_clicked, STYLE_NAME);
    gkrellm_load_image("next_clicked", next_clicked_xpm, &next_image_clicked, STYLE_NAME);
    gkrellm_load_image("eject_clicked", eject_clicked_xpm, &eject_image_clicked, STYLE_NAME);
#else

    g_free(button_file);
    button_file = g_strdup_printf("%s/prev", theme_dir);
    prev_image = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(prev_xpm);

    g_free(button_file);
    button_file = g_strdup_printf("%s/play_pause", theme_dir);
    play_image = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(play_pause_xpm);

    g_free(button_file);
    button_file = g_strdup_printf("%s/stop", theme_dir);
    stop_image = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(stop_xpm);

    g_free(button_file);
    button_file = g_strdup_printf("%s/next", theme_dir);
    next_image = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(next_xpm);

    g_free(button_file);
    button_file = g_strdup_printf("%s/eject", theme_dir);
    eject_image = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(eject_xpm);

    /* Load clicked buttons */
    g_free(button_file);
    button_file = g_strdup_printf("%s/prev_clicked", theme_dir);
    prev_image_clicked = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(prev_clicked_xpm);

    g_free(button_file);
    button_file = g_strdup_printf("%s/play_pause_clicked", theme_dir);
    play_image_clicked = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(play_pause_clicked_xpm);

    g_free(button_file);
    button_file = g_strdup_printf("%s/stop_clicked", theme_dir);
    stop_image_clicked = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(stop_clicked_xpm);

    g_free(button_file);
    button_file = g_strdup_printf("%s/next_clicked", theme_dir);
    next_image_clicked = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(next_clicked_xpm);

    g_free(button_file);
    button_file = g_strdup_printf("%s/eject_clicked", theme_dir);
    eject_image_clicked = ((button_file_fixed = button_file_exists(button_file)) != NULL) ?
      gdk_imlib_load_image(button_file_fixed) :
      gdk_imlib_create_image_from_xpm_data(eject_clicked_xpm);

#endif /* Load images */

    /* Render and other stuff */

    /* Janitor:                                                         */
    /* Made changes so that the width and height are pulled out of the  */
    /* image structure instead of hard coded.                           */
    gkrellm_render_to_pixmap(prev_image, &prev_pix, &button_mask,
      prev_image->rgb_width, prev_image->rgb_height);
    gkrellm_render_to_pixmap(play_image, &play_pix, NULL,
      play_image->rgb_width, play_image->rgb_height);
    gkrellm_render_to_pixmap(stop_image, &stop_pix, NULL,
      stop_image->rgb_width, stop_image->rgb_height);
    gkrellm_render_to_pixmap(next_image, &next_pix, NULL,
      next_image->rgb_width, next_image->rgb_height);
    gkrellm_render_to_pixmap(eject_image, &eject_pix, NULL,
      eject_image->rgb_width, eject_image->rgb_height);

    /* Janitor:                                                         */
    /* This code centers the button-bar horizontally in the chart.      */
    x = (gkrellm_chart_width() - (prev_image->rgb_width + play_image->rgb_width +
                                  stop_image->rgb_width + next_image->rgb_width +
                                  eject_image->rgb_width)) / 2;
    if (x < 0)
    {
      /* The chart is too narrow to fit the entire button-bar */
      x = 0;
    }
#if ((VERSION_MAJOR>1)||(VERSION_MAJOR==1 && VERSION_MINOR==0 && VERSION_REV>2)||(VERSION_MAJOR==1 && VERSION_MINOR>0))
    gkrellm_get_top_bottom_margins(style, &y, NULL);
#else
    y = style->margin;
#endif

    prev_decal = gkrellm_create_decal_pixmap(button_bar, prev_pix,
                 button_mask, 1, style, x, y);
    x += prev_decal->w;
    play_decal = gkrellm_create_decal_pixmap(button_bar, play_pix,
                 button_mask, 1, style, x, y);
    x += play_decal->w;
    stop_decal = gkrellm_create_decal_pixmap(button_bar, stop_pix,
                 button_mask, 1, style, x, y);
    x += stop_decal->w;
    next_decal = gkrellm_create_decal_pixmap(button_bar, next_pix,
                 button_mask, 1, style, x, y);
    x += next_decal->w;
    eject_decal = gkrellm_create_decal_pixmap(button_bar, eject_pix,
                  button_mask, 1, style, x, y);
    
    gkrellm_configure_panel(button_bar, NULL, style);
    gkrellm_create_panel(vbox, button_bar, gkrellm_bg_meter_image(DEFAULT_STYLE));
    gkrellm_monitor_height_adjust(button_bar->h);

    /* Janitor:                                                         */
    /* Made changes so that the width and height are pulled out of the  */
    /* image structure instead of hard coded.                           */
    prev_but = gkrellm_make_overlay_button(button_bar, cb_button,
               (int *) prev, prev_decal->x, y,
               prev_image->rgb_width, prev_image->rgb_height,
               prev_image, prev_image_clicked);
    play_but = gkrellm_make_overlay_button(button_bar, cb_button,
               (int *) play, play_decal->x, y,
               play_image->rgb_width, play_image->rgb_height,
               play_image, play_image_clicked);
    stop_but = gkrellm_make_overlay_button(button_bar, cb_button,
               (int *) stop, stop_decal->x, y,
               stop_image->rgb_width, stop_image->rgb_height,
               stop_image, stop_image_clicked);
    next_but = gkrellm_make_overlay_button(button_bar, cb_button,
               (int *) next, next_decal->x, y,
               next_image->rgb_width, next_image->rgb_height,
               next_image, next_image_clicked);
    eject_but = gkrellm_make_overlay_button(button_bar, cb_button,
                (int *) eject, eject_decal->x, y,
                eject_image->rgb_width, eject_image->rgb_height,
                eject_image, eject_image_clicked);

    gkrellm_draw_decal_pixmap(button_bar, prev_decal, 0);
    gkrellm_draw_decal_pixmap(button_bar, play_decal, 0);
    gkrellm_draw_decal_pixmap(button_bar, stop_decal, 0);
    gkrellm_draw_decal_pixmap(button_bar, next_decal, 0);
    gkrellm_draw_decal_pixmap(button_bar, eject_decal, 0);

    gkrellm_draw_layers(button_bar);

    gdk_imlib_kill_image(prev_image);
    gdk_imlib_kill_image(play_image);
    gdk_imlib_kill_image(stop_image);
    gdk_imlib_kill_image(next_image);
    gdk_imlib_kill_image(eject_image);

    gdk_imlib_kill_image(prev_image_clicked);
    gdk_imlib_kill_image(play_image_clicked);
    gdk_imlib_kill_image(stop_image_clicked);
    gdk_imlib_kill_image(next_image_clicked);
    gdk_imlib_kill_image(eject_image_clicked);

    if (first_create)
      gtk_signal_connect(GTK_OBJECT (button_bar->drawing_area), "expose_event",
                         (GtkSignalFunc) panel_expose_event, (int *) 3);
  }

  if (first_create)
  {

    gtk_signal_connect(GTK_OBJECT (time_bar->drawing_area), "expose_event",
                       (GtkSignalFunc) panel_expose_event, (int *) 2);
    gtk_signal_connect(GTK_OBJECT (time_bar->drawing_area), "button_press_event",
                       (GtkSignalFunc) panel_button_press, NULL);
    gtk_signal_connect(GTK_OBJECT (time_bar->drawing_area), "button_release_event",
                       (GtkSignalFunc) panel_button_release, NULL);
    gtk_signal_connect(GTK_OBJECT (time_bar->drawing_area), "motion_notify_event",
                       (GtkSignalFunc) slider_motion, NULL);

  }
}

void update_gkrellmms_config(gint pack)
{
  if (panel != NULL)
  {
    gkrellm_monitor_height_adjust(- panel->h);
    gkrellm_destroy_decal_list(panel);
    gkrellm_destroy_panel(panel);
    g_free(panel);
  }
  panel = NULL;

  if (time_bar != NULL)
  {
    gkrellm_monitor_height_adjust(- time_bar->h);
    gkrellm_destroy_decal_list(time_bar);
    gkrellm_destroy_panel(time_bar);
    g_free(time_bar);
  }
  time_bar = NULL;

  if (button_bar != NULL)
  {
    gkrellm_destroy_button(prev_but);
    gkrellm_destroy_button(play_but);
    gkrellm_destroy_button(stop_but);
    gkrellm_destroy_button(next_but);
    gkrellm_destroy_button(eject_but);
    
    gkrellm_monitor_height_adjust(- button_bar->h);
    gkrellm_destroy_decal_list(button_bar);
    gkrellm_destroy_panel(button_bar);
    g_free(button_bar);
  }
  button_bar = NULL;

  create_gkrellmms(gkrellm_vbox, 1);
  if (pack)
    gkrellm_pack_side_frames();
}

void mainwin_back_func()
{
  if (auto_main_close && xmms_running)
    xmms_remote_main_win_toggle(xmms_session, TRUE);
}

static Monitor  plugin_mon  =
{
  "GKrellMMS",             /* Name, for config tab.                   */
  0,                       /* Id, 0 if a plugin                       */
  create_gkrellmms,        /* The create_plugin() function            */
  update_gkrellmms,        /* The update_plugin() function            */

  create_gkrellmms_config, /* Create_plugin_tab() config func */
  apply_gkrellmms_config,  /* The apply_plugin_config() function      */

  save_gkrellmms_config,   /* The save_plugin_config() function */
  load_gkrellmms_config,   /* The load_plugin_config() function */
  CONFIG_KEYWORD,          /* Config Keyword                    */

  NULL,                    /* Undefined 2 */
  NULL,                    /* Undefined 1 */
  NULL,                    /* Undefined 0 */

  GKRELLMMS_PLACE,         /* Insert plugin before this monitor     */
  NULL,                    /* Handle if a plugin, filled by GKrellM */
  NULL                     /* Path if a plugin, filled by GKrellM   */
};

Monitor *
init_plugin(void)
{
  gchar *tmp;

  /* Popup-menu set */
  running_factory = options_menu_factory(1);
  not_running_factory = options_menu_factory(0);
  pl_factory = playlist_menu_factory();

  /* Default settings */
  config_update = FALSE;
  time_bar = NULL;
  xmms_running = FALSE;
  xmms_pos = 0;
  x_scroll = 0;
  total_plist_time = 0;
  plist = NULL;
  scrolling_tooltip = NULL;

#if ((VERSION_MAJOR>1)||(VERSION_MAJOR==1 && VERSION_MINOR==0 && VERSION_REV>3)||(VERSION_MAJOR==1 && VERSION_MINOR>0))
  tmp = g_strdup_printf("%s/%s", gkrellm_homedir(), GKRELLM_DATA_DIR);
#else
  tmp = g_strdup_printf("%s/%s", homedir(), GKRELLM_DATA_DIR);
#endif

  playlist_file = g_strdup_printf("%s/gkrellmms_playlist", tmp);
  position_file = g_strdup_printf("%s/gkrellmms_position", tmp);
  time_file = g_strdup_printf("%s/gkrellmms_time", tmp);

  g_free(tmp);

  playlist_dir = g_strdup("/");
  files_directory = g_strdup("/");
  gkrellmms_label = g_strdup("xmms");
  xmms_session = 0;
  scroll_enable = TRUE;
  buttons_enable = TRUE;
  draw_time = 1;
  xmms_exec_command = g_strdup("xmms");
  xmms_autostart = 0;
  auto_main_close = 0;
  eject_opens_playlist = 1;
  krell_mmb_pause = 1;
  time_format = 1;
  auto_play_start = 0;
  auto_seek = 0;
  always_load_info = 0;
  draw_minus = 1;

  g_atexit(mainwin_back_func);

#if ((VERSION_MAJOR>0)||(VERSION_MINOR>9))
    style_id = gkrellm_add_meter_style(&plugin_mon, STYLE_NAME);
#endif

  /* Done */
  return &plugin_mon;
}
