/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/****************
 * The menubar. *
 ****************/

#include "gliv.h"

static gchar **array;
static guint length;

static void add_menu_item(GtkMenuShell * menu, gchar * name, gchar * load)
{
    GtkMenuItem *item;

    if (load != NULL)
        name = add_mnemonic(name);

#ifdef GTK2
    name = g_filename_to_utf8(name, -1, NULL, NULL, NULL);
#endif

    item = GTK_MENU_ITEM(gtk_menu_item_new_with_mnemonic(name));
    gtk_menu_shell_append(menu, GTK_WIDGET(item));

    if (load == NULL)
        /* A dirname */
        gtk_widget_set_sensitive(GTK_WIDGET(item), FALSE);
    else
        /* A filename */
        g_signal_connect_swapped(item, "activate", G_CALLBACK(menu_load), load);

#ifdef GTK2
    g_free(name);
#endif
}

static GtkMenuShell *add_menu(GtkMenuShell * parent, gchar * name)
{
    GtkMenuShell *menu;
    GtkMenuItem *item;

    name = locale_to_utf8(add_mnemonic(name));
    item = GTK_MENU_ITEM(gtk_menu_item_new_with_mnemonic(name));
    gtk_menu_shell_append(parent, GTK_WIDGET(item));

    menu = GTK_MENU_SHELL(gtk_menu_new());
    gtk_menu_shell_append(menu, gtk_tearoff_menu_item_new());
    gtk_menu_item_set_submenu(item, GTK_WIDGET(menu));

    return menu;
}

typedef enum {
    DIR_PARENT,
    DIR_EQUAL,
    DIR_DIFFERENT
} dir_type;

/* Wether dir1 is parent, equal or different from the dir where file_id2 is. */
static dir_type file1_wrt_file2(gchar * dir1, guint file_id2)
{
    gchar *ptr1, *dir2, *ptr2;
    dir_type res;

    ptr1 = dir1;
    ptr2 = dir2 = g_path_get_dirname(array[file_id2]);

    /* Skip identical characters in the paths. */
    while (*ptr1 != '\0' && *ptr2 != '\0' && *ptr1 == *ptr2) {
        ptr1++;
        ptr2++;
    }

    if (*ptr1 == *ptr2)
        res = DIR_EQUAL;

    else if (*ptr1 == '\0' && *ptr2 == '/')
        res = DIR_PARENT;

    else
        res = DIR_DIFFERENT;

    g_free(dir2);

    return res;
}

G_GNUC_PURE static gchar *dir_minus_parent(gchar * dir, gchar * parent)
{
    while (*dir == *parent) {
        dir++;
        parent++;
    }

    /* + 1 : we don't want the leading separator. */
    return dir + 1;
}

/*
 * Underscores in filenames are replaced
 * with mnemonics, so we duplicate them.
 */
#ifdef GTK2
static gchar *duplicate_underscores(gchar * orig)
{
    guint length = 1;
    guint nb_underscores = 0;
    gchar *new;
    gchar *ptr_orig, *ptr_new;

    /* How many underscores ? the first char isn't a problem. */
    for (ptr_orig = orig + 1; *ptr_orig != '\0'; ptr_orig++) {
        length++;
        if (*ptr_orig == '_')
            nb_underscores++;
    }

    new = g_new(gchar, length + nb_underscores + 1);

    /* The first char isn't checked. */
    new[0] = orig[0];
    ptr_new = new + 1;

    for (ptr_orig = orig + 1; *ptr_orig != '\0'; ptr_orig++) {
        *ptr_new = *ptr_orig;
        if (*ptr_orig == '_') {
            ptr_new++;
            *ptr_new = '_';
        }
        ptr_new++;
    }

    *ptr_new = '\0';

    return new;
}
#else
#define duplicate_underscores(str) (str)
#endif

static void add_file_item(GtkMenuShell * menu, guint file_id)
{
    gchar *filename, *menu_name;

    filename = array[file_id];
    menu_name = duplicate_underscores(g_path_get_basename(filename));

    if (file_id == 0 || g_str_equal(filename, array[file_id - 1]) == FALSE)
        add_menu_item(menu, menu_name, filename);

#ifdef GTK2
    g_free(menu_name);
#endif
}

static void make_directories_menu_rec(GtkMenuShell * parent,
                                      guint * file_id, gchar * current_dir)
{
    gchar *dir;
    GtkMenuShell *menu;

    while (*file_id < length) {
        switch (file1_wrt_file2(current_dir, *file_id)) {
        case DIR_PARENT:
            dir = g_path_get_dirname(array[*file_id]);
            menu = add_menu(parent, dir_minus_parent(dir, current_dir));
            add_file_item(menu, *file_id);
            (*file_id)++;
            make_directories_menu_rec(menu, file_id, dir);
            g_free(dir);
            break;

        case DIR_EQUAL:
            add_file_item(parent, *file_id);
            (*file_id)++;
            break;

        case DIR_DIFFERENT:
            return;
        }
    }
}

static GtkMenuShell *init_rebuild(GtkMenuItem * root)
{
    GtkMenuShell *menu;

    menu = GTK_MENU_SHELL(gtk_menu_new());
    gtk_menu_shell_append(menu, gtk_tearoff_menu_item_new());
    gtk_menu_item_set_submenu(root, GTK_WIDGET(menu));

    return menu;
}

void rebuild_directories(GtkMenuItem * root)
{
    static GtkMenuShell *menu;
    static GtkMenuItem *item;
    GtkMenuShell *submenu;
    gchar *dir;
    guint file_id = 0;

    if (root != NULL) {
        item = root;
        return;
    }

    menu = init_rebuild(item);

    while (file_id < length) {
        dir = g_path_get_dirname(array[file_id]);
        submenu = add_menu(menu, dir);
        add_file_item(submenu, file_id);
        file_id++;
        make_directories_menu_rec(submenu, &file_id, dir);
        g_free(dir);
    }

    gtk_widget_show_all(GTK_WIDGET(item));
}

void rebuild_images(GtkMenuItem * root)
{
    static GtkMenuShell *menu;
    static GtkMenuItem *item;
    guint file_id;
    gchar *old_dir, *new_dir;

    if (root != NULL) {
        item = root;
        return;
    }

    menu = init_rebuild(item);

    /* The file_id = 0 case. */
    old_dir = g_path_get_dirname(array[0]);
    add_menu_item(menu, old_dir, NULL);
    add_file_item(menu, 0);

    for (file_id = 1; file_id < length; file_id++) {
        new_dir = g_path_get_dirname(array[file_id]);

        if (g_str_equal(old_dir, new_dir) == FALSE)
            add_menu_item(menu, new_dir, NULL);

        add_file_item(menu, file_id);

        g_free(old_dir);
        old_dir = new_dir;
    }

    gtk_widget_show_all(GTK_WIDGET(item));
}

void rebuild_images_menu(void)
{
    length = array_length();
    if (length == 0)
        return;

    array = get_file_list_sorted(0, length);

    rebuild_directories(NULL);
    rebuild_images(NULL);

    g_free(array);
}
