/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/****************
 * The menubar. *
 ****************/

#include "gliv.h"

#include <gdk/gdkkeysyms.h>     /* GDK_? */

extern options_struct *options;
extern GtkMenuBar *menu_bar;

#ifndef GTK2
/* Connection id for the handler of all events that grabs the keyboard. */
static guint handler_id;

/*
 * In fullscreen mode, when the menu bar is used,
 * we give the grab back to the window.
 */
void intercept_events(gboolean intercepted)
{
    if (intercepted == TRUE)
        gtk_signal_handler_unblock(GTK_OBJECT(menu_bar), handler_id);
    else
        gtk_signal_handler_block(GTK_OBJECT(menu_bar), handler_id);
}

static void event(void)
{
    static GdkWindow *parent = NULL;

    if (parent == NULL)
        /* First time. */
        parent = gtk_widget_get_parent_window(GTK_WIDGET(menu_bar));

    gdk_keyboard_grab(parent, TRUE, GDK_CURRENT_TIME);
}
#endif

static GtkMenuShell *add_menu(gchar * name)
{
    GtkMenuShell *menu;
    GtkMenuItem *item;

    name = locale_to_utf8(add_mnemonic(name));
    item = GTK_MENU_ITEM(gtk_menu_item_new_with_mnemonic(name));

    gtk_menu_shell_append(GTK_MENU_SHELL(menu_bar), GTK_WIDGET(item));
    menu = GTK_MENU_SHELL(gtk_menu_new());
    gtk_menu_shell_append(menu, gtk_tearoff_menu_item_new());
    gtk_menu_item_set_submenu(item, GTK_WIDGET(menu));

    return menu;
}

static void add_menu_item(GtkMenuShell * menu, GtkAccelGroup * accel_group,
                          gchar * name, GCallback func, guint key,
                          gpointer data)
{
    GtkMenuItem *menu_item;
    GdkModifierType mod;

    name = locale_to_utf8(add_mnemonic(name));
    menu_item = GTK_MENU_ITEM(gtk_menu_item_new_with_mnemonic(name));

    gtk_menu_shell_append(menu, GTK_WIDGET(menu_item));
    g_signal_connect_swapped(menu_item, "activate", func, data);

    if (key == 0)
        return;

    mod = (key >= GDK_A && key <= GDK_Z) ? GDK_SHIFT_MASK : 0;
    gtk_widget_add_accelerator(GTK_WIDGET(menu_item), "activate", accel_group,
                               key, mod, GTK_ACCEL_VISIBLE);
}

static void toggle_alpha_checks(void)
{
    options->alpha_checks ^= TRUE;
    refresh(REFRESH_IMAGE);
}

static void menu_fullscreen(void)
{
    toggle_fullscreen(!options->fullscreen);
}

#define MAXIMIZE   (1 << 0)
#define SCALE_DOWN (1 << 1)

static void menu_zoom(guchar what)
{
    gboolean old_maximize, old_scale_down;

    old_maximize = options->maximize;
    old_scale_down = options->scaledown;

    options->maximize = ((what & MAXIMIZE) != 0);
    options->scaledown = ((what & SCALE_DOWN) != 0);

    matrix_set_max_zoom(-1, -1, TRUE);

    options->maximize = old_maximize;
    options->scaledown = old_scale_down;

    refresh(REFRESH_IMAGE | REFRESH_STATUS | APPEND_HISTORY);
}

static void menu_flip_h(void)
{
    matrix_flip_h();
    refresh(REFRESH_IMAGE | REFRESH_STATUS | APPEND_HISTORY);
}

static void menu_flip_v(void)
{
    matrix_flip_v();
    refresh(REFRESH_IMAGE | REFRESH_STATUS | APPEND_HISTORY);
}

#ifdef GTK2
#define SEPARATOR gtk_menu_shell_append(submenu, gtk_separator_menu_item_new())
#else
#define SEPARATOR gtk_menu_shell_append(submenu, gtk_menu_item_new())
#endif

#define ADD_MENU_ITEM(str, signal_func, key, val) \
    add_menu_item(submenu, accel_group, str, \
                  G_CALLBACK(signal_func), key, GINT_TO_POINTER(val))

#define ADD_MENU(str) submenu = add_menu(str)

static GtkMenuItem *add_special_menu(GtkAccelGroup * accel_group, gchar * name)
{
    GList *children;
    GtkMenuShell *submenu;
    GtkMenuItem *item;

    submenu = add_menu(name);

    children = gtk_container_get_children(GTK_CONTAINER(submenu));
    item = GTK_MENU_ITEM(gtk_menu_get_attach_widget(GTK_MENU(submenu)));
    g_list_free(children);

    if (options->build_menus == FALSE)
        ADD_MENU_ITEM(_("Rebuild images menus"), rebuild_images_menu, 0, 0);

    return item;
}

#define ADD_SPECIAL_MENU(str) add_special_menu(accel_group, str)

GtkAccelGroup *create_menus(void)
{
    GtkMenuShell *submenu;
    GtkAccelGroup *accel_group;

    menu_bar = GTK_MENU_BAR(gtk_menu_bar_new());
    accel_group = gtk_accel_group_new();

    /* File menu. */
    ADD_MENU(_("File"));
    ADD_MENU_ITEM(_("Open..."), menu_open, GDK_o, 0);
    ADD_MENU_ITEM(_("Quit"), gtk_main_quit, GDK_q, 0);

    /* Commands menu. */
    ADD_MENU(_("Commands"));
    ADD_MENU_ITEM(_("Maximize small image"), menu_zoom, GDK_M, MAXIMIZE);
    ADD_MENU_ITEM(_("Scale down large image"), menu_zoom, GDK_l, SCALE_DOWN);
    ADD_MENU_ITEM(_("Image fit window"), menu_zoom, GDK_m,
                  MAXIMIZE | SCALE_DOWN);
    SEPARATOR;
    ADD_MENU_ITEM(_("Horizontal flip"), menu_flip_h, GDK_z, 0);
    ADD_MENU_ITEM(_("Vertical flip"), menu_flip_v, GDK_e, 0);
    SEPARATOR;
    ADD_MENU_ITEM(_("Load previous image"), load_direction, GDK_p, -1);
    ADD_MENU_ITEM(_("Load next image"), load_direction, GDK_n, 1);
    SEPARATOR;
    ADD_MENU_ITEM(_("Undo"), undo, GDK_u, 0);
    ADD_MENU_ITEM(_("Redo"), redo, GDK_y, 0);
    ADD_MENU_ITEM(_("Clear history"), clear_history, GDK_c, 0);
    SEPARATOR;
    ADD_MENU_ITEM(_("Rebuild images menus"), rebuild_images_menu, 0, 0);

    /* Options menu. */
    ADD_MENU(_("Options"));
    ADD_MENU_ITEM(_("Toggle Fullscreen mode"), menu_fullscreen, GDK_f, 0);
    ADD_MENU_ITEM(_("Toggle Menu Bar"), toggle_menu_bar, GDK_b, 0);
    ADD_MENU_ITEM(_("Toggle Status Bar"), toggle_status_bar, GDK_i, 0);
    ADD_MENU_ITEM(_("Toggle Alpha checks"), toggle_alpha_checks, GDK_a, 0);
    SEPARATOR;
    ADD_MENU_ITEM(_("Hide the cursor"), toggle_cursor, GDK_d, FALSE);
    SEPARATOR;
    ADD_MENU_ITEM(_("Options..."), show_options, GDK_t, 0);

    /* Directories menu. */
    rebuild_directories(ADD_SPECIAL_MENU(_("Directories")));

    /* Images menu. */
    rebuild_images(ADD_SPECIAL_MENU(_("Images")));

    /* Help menu. */
    ADD_MENU(_("Help"));
    ADD_MENU_ITEM(_("About..."), show_about_box, 0, 0);
    ADD_MENU_ITEM(_("Controls..."), toggle_help, GDK_h, 0);

    if (options->build_menus == TRUE)
        rebuild_images_menu();

#ifndef GTK2
    handler_id = gtk_signal_connect_after(GTK_OBJECT(menu_bar), "event",
                                          G_CALLBACK(event), NULL);
    intercept_events(FALSE);
#endif

    return accel_group;
}
