/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Operation to perform on lfn sent to RLI server (update timestamp, or
 * remove it).  Following defines prepended to lfns sent to rli during
 * update to indicate if lfn should be added or deleted from rli.  They
 * correspond to RLIOP enum.
 */
#define RLI_ADD		'A'
#define RLI_DELETE	'D'

typedef enum { rliop_add, rliop_delete } RLIOP;

/*
 * Following are constant UPDATE values that have special meaning to
 * the RLI update processing code.  EXIT means rli update thread should
 * flush pending updates and exit (because rlilist has changed), SOFTSTATEEND
 * means it should log the time between when an rli update started (recorded
 * in rli->softstatestart) and the time the SOFTSTATEEND record is processed.
 */
#define RU_EXIT		((UPDATE *) 0)
#define RU_SOFTSTATEEND	((UPDATE *) -1)
				
typedef struct update_ {
  globus_mutex_t	mtx;		/* Sync access to refcnt	*/
  char			*lfn;
  int			lfnlen;		/* Includes cmd and trailing null*/
  char			*lrc;
  int			refcnt;
  struct update_	*nxt;		/* Used by freelist only	*/
} UPDATE;

typedef struct update_list_ {
  UPDATE		*u;
  struct update_list_	*nxt;
} UPDATE_LIST;

typedef struct updatepat_ {
  char			*pattern;
  struct updatepat_	*nxt;
} UPDATEPAT;

typedef struct {
  globus_mutex_t	mtx;		/* Sync access to this struct	*/
  globus_cond_t		cond;
  UPDATE_LIST		*first;		/* first,last point to begin and*/
  UPDATE_LIST		*last;		/* end of updates to be proc'ed	*/
  UPDATE_LIST		*pending;	/* Updates waiting to be written*/
} UPDATEQUEUE;

#define RLIIOV	100
typedef struct rli_ {
  globus_mutex_t	mtx;		/* Sync access to this struct	*/
  globus_cond_t		cond;		/* Wait for updater to exit	*/
  char			*url;		/* URL of RLI server to update	*/
  globus_rls_handle_t	*h;		/* Handle to send updates over	*/
  UPDATEPAT		*patternlist;	/* Partition for this RLI	*/
  UPDATEQUEUE		queue;		/* Queue for updates to RLI	*/
  struct iovec		iov[RLIIOV];	/* Write queue			*/
  int			idx;		/* Index into iov		*/
  time_t		bufferage;	/* Time of last buffer flush	*/
  time_t		softstatestart;	/* Start of softstate update	*/
  time_t		retrytime;	/* Time to attempt reconnect	*/
  time_t		lastupdate;	/* Last successful update	*/
  void			*lastlrc;	/* Last lrc sent in update	*/
  int			flags;
#define FR_OPEN			0x1	/* RLI connection is open	*/
#define FR_UPDATETHREAD		0x2	/* Update thread running	*/
#define FR_SYNCED		0x4	/* LRC, RLI in sync		*/
#define FR_EXITING		0x8	/* RU_EXIT update queued	*/
#define FR_DOUPDATE		0x10	/* Do softstate update		*/
#define FR_BLOOMFILTER		0x20	/* Update w/ bloom filter	*/
#define FR_FLUSHPENDING		0x40	/* Flush pending updates	*/
  struct rli_		*nxt;
} RLI;

extern void		update_freelist(RLI **rlilistp);
extern void		update_init();
extern UPDATE		*update_new(char *lfn, char *lrc, RLIOP op);
extern globus_bool_t	update_patternmatch(RLI *rli, char *lfn);
extern void		update_queue(RLI *rli, UPDATE *u);
extern void		update_flushqueue(RLI *rli);
extern void		update_readrli(void *dbh, LOCK *listlock,
				       RLI **rlilist);
extern void		update_sendbf(RLI *rli, bloomfilter_t *bloomfilter,
				      char *lrcurl);
