/* parse_vorbis.c - Processing of Ogg Vorbis header data */

#include "gnapster.h"

/* #ifdef HAVE_VORBIS */

#include "parse_vorbis.h"
#include "parse_ext.h"
#include "ogg.h"

#define CHUNKSIZE 4096

void vorbis_register() {
   REGISTER_EXT("ogg", "audio", parse_vorbis, vorbis_ext);
}

int vorbis_ext(char *ext) {
   /* 1 on match, 0 on failure */
   return (strcasecmp(ext, "ogg") == 0);
}

static int read32bits(ogg_packet *op, int off)
{
	unsigned char *buf = op->packet;
	return (buf[off])|(buf[off+1]<<8)|(buf[off+2]<<16)|(buf[off+3]<<24);
}

/* Try to find the final page. Can be slightly off (but not enough to cause
 * problems) for non-standard page lengths */
static int get_final_page(int fd, ogg_sync_state *oy, int len, ogg_page *og)
{
	int size = 0;
	unsigned char *buffer;
	int bytes;
	int found=0;
	int seekp;
	int tlen=len;
		
	while(1)
	{
		len -= CHUNKSIZE;
		size += CHUNKSIZE;
		if(lseek(fd, len, SEEK_SET) == -1)
			return -1;

		ogg_sync_reset(oy);
		buffer = ogg_sync_buffer(oy, size);
		bytes = read(fd, buffer, size);
		if(bytes<=0)
			return -1;

		ogg_sync_wrote(oy, bytes);

		found = ogg_sync_pageseek(oy, og);

		seekp = len;
		while(found && seekp <= len+CHUNKSIZE && seekp-found<tlen)
		{
			seekp -= found;
			lseek(fd, seekp, SEEK_SET); /* Seek to where we were told it is. */

			ogg_sync_reset(oy);
			buffer = ogg_sync_buffer(oy, size);
			bytes = read(fd, buffer, size);
			if(bytes<=0)
				return -1;
			ogg_sync_wrote(oy,bytes);
	
			found = ogg_sync_pageseek(oy, og);
			if(found>0)
				return 0;
		}
	}

	return -1;
}

/* Don't call this until everything non-vorbis specific has been set in shr.
 * It uses this info for some stuff */
static int vorbis_get_info(int fd, char *filename, ShareData *shr)
{
	ogg_sync_state oy;
	ogg_stream_state os;
	ogg_page og;
	ogg_packet op;

	unsigned char *buffer;
	int bytes;
	int serial;
	int streaminit=0;
	int length;

	ogg_sync_init(&oy);
	
	buffer = ogg_sync_buffer(&oy, CHUNKSIZE);
	bytes = read(fd, buffer, CHUNKSIZE);
	ogg_sync_wrote(&oy, bytes);

	if(ogg_sync_pageout(&oy,&og)!=1)
		goto failure;

	serial = ogg_page_serialno(&og);
	ogg_stream_init(&os, serial);
	streaminit=1;
	if(ogg_stream_pagein(&os, &og)<0)
		goto failure;
	if(ogg_stream_packetout(&os, &op)!=1)
		goto failure;
	
	if(read32bits(&op,7) != 0) 
		goto failure;

	shr->frequency = read32bits(&op, 12);

	ogg_stream_clear(&os); /* We're done with that, now. */
	streaminit=0;

	if(get_final_page(fd, &oy, shr->filesize, &og) != 0)
		goto failure;
	if(serial != ogg_page_serialno(&og))
		fprintf(stderr, 
			"WARNING: %s is a chained stream. The napster protocol doesn't\n"
			"have any provision for such a concept. As a result, the output\n"
			"file length and bitrate will be inaccurate.\n", filename);

	length = ogg_page_granulepos(&og);

	shr->time = (time_t)(length/shr->frequency);
	shr->bitrate = (int)(shr->filesize/shr->time*8/1000);

	ogg_sync_clear(&oy);
	return 0;

failure:
	fprintf(stderr, "WARNING: %s wasn't a valid vorbis file\n", filename);
	if(streaminit) 
		ogg_stream_clear(&os);
	ogg_sync_clear(&oy);
	return -1;
}

ShareData *parse_vorbis(char *file, int fd) {
   ShareData *shr;
   struct stat st;

   shr = d_new(SHARE);

   fstat(fd, &st);

   shr->checksum = d_strdup("00000000000000000000000000000000");
   shr->filename = d_strdup(file);

   shr->mtime = st.st_mtime;
   shr->filesize = st.st_size;

   if(vorbis_get_info(fd, file, shr)==0)
   		return shr;
   else
   {
	   d_free(shr);
	   return NULL;
   }
}

/* #endif *//* HAVE_VORBIS */
