with Rbtrees;
with Ada.Text_IO; use Ada.Text_IO;
with Ada.Numerics.Discrete_Random;
with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;

procedure Test_Rbtree is
   type Item is new Natural;

   procedure Free (It : in out Item);

   package Item_Random is new Ada.Numerics.Discrete_Random (Item);
   use Item_Random;

   package Item_Rbtrees is new Rbtrees (Item, Free);
   use Item_Rbtrees;

   function Print (Tree : Rbtree; Iter : Rbtree_Iterator) return String;
   procedure Print_Recursive
     (Tree        : in out Rbtree;
      Iter        : Rbtree_Iterator;
      With_Remove : Boolean := False);
   procedure Assert (Item1, Item2 : String; Comment : String := "");
   procedure Assert (Item1, Item2 : Natural; Comment : String := "");
   function Image (J : Integer) return String;
   function Image (J : Item) return String;

   Str : Unbounded_String;
   --  Output for the current test

   ----------
   -- Free --
   ----------

   procedure Free (It : in out Item) is
   begin
      Str := Str & " (Freeing " & Image (It) & ')';
   end Free;

   -----------
   -- Print --
   -----------

   function Print (Tree : Rbtree; Iter : Rbtree_Iterator) return String is
   begin
      if Iter = Null_Iterator then
         return "<null>";
      else
         return Image (Get (Iter))
            & " (index=" & Image (Index (Iter)) & "/"
            & Image (Length (Tree)) & ")";
      end if;
   end Print;

   -----------
   -- Image --
   -----------

   function Image (J : Integer) return String is
      Js : constant String := Integer'Image (J);
   begin
      if Js (Js'First) = ' ' then
         return Js (Js'First + 1 .. Js'Last);
      else
         return Js;
      end if;
   end Image;

   -----------
   -- Image --
   -----------

   function Image (J : Item) return String is
   begin
      return Image (Integer (J));
   end Image;

   ------------
   -- Assert --
   ------------

   procedure Assert (Item1, Item2 : String; Comment : String := "") is
   begin
      if Item1 /= Item2 then
         Put_Line ("--Failed: " & Comment);
         Put_Line ("  Item1=" & Item1);
         Put_Line ("  Item2=" & Item2);
      end if;
   end Assert;

   procedure Assert (Item1, Item2 : Natural; Comment : String := "") is
   begin
      if Item1 /= Item2 then
         Put_Line ("--Failed: " & Comment);
         Put_Line ("  Item1=" & Image (Item1));
         Put_Line ("  Item2=" & Image (Item2));
      end if;
   end Assert;

   ---------------------
   -- Print_Recursive --
   ---------------------

   procedure Print_Recursive
     (Tree        : in out Rbtree;
      Iter        : Rbtree_Iterator;
      With_Remove : Boolean := False)
   is
      Prev : Item := 0;
      It : Rbtree_Iterator := Iter;
   begin
      Str := To_Unbounded_String ("");
      while It /= Null_Iterator loop
         if Get (It) < Prev then
            Put_Line ("!!! Error : not sorted");
            raise Program_Error;
         end if;

         Prev := Get (It);

         Str := Str & ' ' & Print (Tree, It);

         if With_Remove then
            Remove (Tree, It);
            It := Minimum (Tree);
         else
            It := Next (It);
         end if;
      end loop;
   end Print_Recursive;

   Tree : Rbtree;
   Iter : Rbtree_Iterator;
   Gen  : Generator;

begin
   Reset (Gen);

   Insert (Tree, 5);
   Insert (Tree, 1);
   Insert (Tree, 9);
   Insert (Tree, 2);
   Insert (Tree, 12);
   Insert (Tree, 7);
   Insert (Tree, 8);
   Insert (Tree, 2);

   --  for J in 1 .. 1_000_000 loop
   --     Insert (Tree, Random (Gen));
   --  end loop;

   Assert (Length (Tree), 8, "Incorrect length");

   Print_Recursive (Tree, Minimum (Tree));
   Assert (To_String (Str),
           " 1 (index=1/8) 2 (index=2/8) 2 (index=3/8) 5 (index=4/8) 7"
           & " (index=5/8) 8 (index=6/8) 9 (index=7/8) 12 (index=8/8)",
           "Simple traversing");

   Assert (Print (Tree, Get_Nth (Tree, 3)),
           "2 (index=3/8)",
           "3rd entry");
   Assert (Print (Tree, Get_Nth (Tree, 4)),
           "5 (index=4/8)",
           "4th entry");
   Assert (Print (Tree, Get_Nth (Tree, 7)),
           "9 (index=7/8)",
           "7th entry");
   Assert (Print (Tree, Get_Nth (Tree, 17)),
           "<null>",
           "17th entry");

   Print_Recursive (Tree, Find (Tree, 8));
   Assert (To_String (Str), " 8 (index=6/8) 9 (index=7/8) 12 (index=8/8)",
           "Simple traversing starting at 8");

   Str := To_Unbounded_String ("");
   Iter := Maximum (Tree);
   while Iter /= Null_Iterator loop
      Str := Str & ' ' & Print (Tree, Iter);
      Iter := Previous (Iter);
   end loop;

   Assert (To_String (Str),
           " 12 (index=8/8) 9 (index=7/8) 8 (index=6/8) 7 (index=5/8) 5 "
           & "(index=4/8) 2 (index=3/8) 2 (index=2/8) 1 (index=1/8)",
           "Simple reverse traversing");

   --  Clear (Tree);

   Print_Recursive (Tree, Minimum (Tree), True);
   Assert (To_String (Str),
           " 1 (index=1/8) (Freeing 1) 2 (index=1/7) (Freeing 2) 2 (index=1/6)"
           & " (Freeing 2) 5 (index=1/5) (Freeing 5) 7 (index=1/4) (Freeing 7)"
           & " 8 (index=1/3) (Freeing 8) 9 (index=1/2) (Freeing 9) 12 "
           & "(index=1/1) (Freeing 12)",
           "Simple traversing with removal");
   Assert (Length (Tree), 0, "Tree should be empty now");

end Test_Rbtree;
