/******************************************************************************\
 gnofin/clipboard.c   $Revision: 1.9 $
 Copyright (C) 2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gtk/gtksignal.h>
#include <gtk/gtkselection.h>
#include <gtk/gtkmain.h>
#include "clipboard.h"
#include "xml-io.h"
#include "data-if.h"
#include "merging.h"


#define GNOFIN_CLIP_ATOM  "text/gnofin-clip"
#define TARGETS_ATOM      "TARGETS"


/******************************************************************************
 * Clipboard structures */

typedef struct {
  GtkWidget          *widget;
  guint               selection_recv_id;
  ClipboardNotifyFunc notify_func;
  ClipboardPasteFunc  paste_func;
  gpointer            user_data;
} ClipboardSink;

static struct {
  GHashTable *sinks;     /* can_paste may only be TRUE if sinks is non-empty */
  GtkWidget  *source;    /* might be NULL even though we can paste (ie. source is in another process) */
  gchar      *clip;      /* this is the data available on the clipboard */
  guint       clip_size;
  guint       selection_get_id;
  guint       selection_clear_id;
  guint       monitor_id;
  guint       can_paste : 1;
} clipboard = {0};

/*****************************************************************************/

static Bankbook *
build_bankbook (const GSList *records)
{
  Bankbook *book;
  AccountInfo acc = {0};
  Account *account;
  Record *record;

  trace ("");

  book = if_bankbook_new ();

  acc.name = "clip";
  account = if_bankbook_insert_account (book, &acc);
  
  for (; records; records=records->next)
  {
    record = LIST_DEREF (Record, records);
    if (!merge_record_into_account (account, record, TRUE))
      g_print ("Warning: error copying record into clipboard!\n");
  }

  return book;
}

static inline GdkAtom
get_atom (const gchar *name)
{
  return gdk_atom_intern (name, FALSE);
}

static void
get_widget (GtkWidget *widget, ClipboardSink *sink, GtkWidget **out)
{
  if (*out == NULL)
    *out = widget;
}

static inline GtkWidget *
get_a_widget (void)
{
  GtkWidget *widget = NULL;
  g_hash_table_foreach (clipboard.sinks, (GHFunc) get_widget, &widget);
  return widget;
}

static inline ClipboardSink *
lookup_sink (GtkWidget *widget)
{
  return (ClipboardSink *) g_hash_table_lookup (clipboard.sinks, widget);
}

static void
notify_sink (GtkWidget *widget, ClipboardSink *sink, gboolean have_selection)
{
  if (sink->notify_func)
    sink->notify_func (widget, have_selection, sink->user_data);
}

static void
scan_targets (GtkWidget *widget, GtkSelectionData *selection_data)
{
  GdkAtom *atoms;
  gboolean have_selection = FALSE;
  gint i;

  if (selection_data->type != GDK_SELECTION_TYPE_ATOM)
    trace ("Selection \"TARGETS\" was not returned as atoms!");
  else
  {
    atoms = (GdkAtom *) selection_data->data;
    for (i=0; i < selection_data->length / sizeof(GdkAtom); ++i)
    {
      gchar *name = gdk_atom_name (atoms[i]);
      if (name != NULL && (strcmp (name, GNOFIN_CLIP_ATOM) == 0))
      {
	have_selection = TRUE;
	break;
      }
    }
  }
  clipboard.can_paste = have_selection;

  g_hash_table_foreach (clipboard.sinks, (GHFunc) notify_sink,
			GINT_TO_POINTER (have_selection));
}

/*****************************************************************************/

static void
on_selection_get (GtkWidget *widget, GtkSelectionData *selection_data,
		  guint info, guint time_stamp, gpointer data)
{
  trace ("");
  gtk_selection_data_set (selection_data, get_atom (GNOFIN_CLIP_ATOM),
  			  8, clipboard.clip, clipboard.clip_size);
}

static void
on_selection_clear (GtkWidget *widget, GdkEventSelection *event, gpointer data)
{
  trace ("");
  clipboard_clear ();
}

static void
on_selection_received (GtkWidget *widget, GtkSelectionData *selection_data, gpointer data)
{
  trace ("");

  if (selection_data->length < 0)
  {
    trace ("Selection retrieval failed");
    return;
  }

  if (selection_data->target == get_atom (TARGETS_ATOM))
    scan_targets (widget, selection_data);
  else if (selection_data->target == get_atom (GNOFIN_CLIP_ATOM))
  {
    Bankbook *book;
    ClipboardSink *sink = lookup_sink (widget);

    trace ("GNOFIN_CLIP: size=%d", selection_data->length);

    book = if_bankbook_new ();

    if (!xml_io_clip_read (selection_data->data, selection_data->length, book))
      trace ("xml_io_clip_read() failed");

    if (sink->paste_func);
      sink->paste_func (book, sink->user_data);

    if_bankbook_destroy (book);
  }
}

/******************************************************************************
 * Timeout procedure for monitoring available atoms */

static void
on_clipboard_monitor (void)
{
  trace ("");
  g_assert (clipboard.sinks != NULL);

  gtk_selection_convert (get_a_widget (),
  			 GDK_SELECTION_PRIMARY,
			 get_atom (TARGETS_ATOM),
			 GDK_CURRENT_TIME);
}

/******************************************************************************
 * Interface
 */

void
clipboard_attach_sink (GtkWidget *widget, ClipboardSignals *sigs, gpointer data)
{
  ClipboardSink *sink;

  trace ("");
  g_return_if_fail (widget);
  g_return_if_fail (sigs);

  sink = g_new0 (ClipboardSink, 1);
  sink->widget = widget;
  sink->notify_func = sigs->notify_func;
  sink->paste_func = sigs->paste_func;
  sink->user_data = data;

  if (clipboard.sinks == NULL)
  {
    clipboard.sinks = g_hash_table_new (g_direct_hash, g_direct_equal);
    clipboard.monitor_id =
      gtk_timeout_add (1000, (GtkFunction) on_clipboard_monitor, NULL);
  }

  g_hash_table_insert (clipboard.sinks, widget, sink);

  sink->selection_recv_id =
    gtk_signal_connect (GTK_OBJECT (widget), "selection_received",
			GTK_SIGNAL_FUNC (on_selection_received), sink);

  /* Update immediately */
  if (g_hash_table_size (clipboard.sinks) > 0)
    on_clipboard_monitor ();
}

void
clipboard_remove_sink (GtkWidget *widget)
{
  ClipboardSink *sink;

  trace ("");

  sink = lookup_sink (widget);
  if (sink)
  {
    gtk_signal_disconnect (GTK_OBJECT (widget), sink->selection_recv_id);
    gtk_widget_set_sensitive (widget, FALSE);

    g_hash_table_remove (clipboard.sinks, widget);
    g_free (sink);
  }

  if (g_hash_table_size (clipboard.sinks) == 0)
  {
    gtk_timeout_remove (clipboard.monitor_id);

    g_hash_table_destroy (clipboard.sinks);
    clipboard.sinks = NULL;
  }

  /* Update immediately */
  if (g_hash_table_size (clipboard.sinks) > 0)
    on_clipboard_monitor ();
}

gboolean
clipboard_can_paste (void)
{
  return clipboard.can_paste;
}

void
clipboard_paste (GtkWidget *widget)
{
  trace ("");
  g_return_if_fail (lookup_sink (widget));

  gtk_selection_convert (widget,
			 GDK_SELECTION_PRIMARY,
			 get_atom (GNOFIN_CLIP_ATOM),
			 GDK_CURRENT_TIME);
}

void
clipboard_set_source (GtkWidget *source, const GSList *records)
{
  Bankbook *book;

  trace ("");
  g_return_if_fail (source);
  g_return_if_fail (records);

  if (clipboard.source)
    clipboard_clear ();

  clipboard.source = source;

  /* Generate clipboard data */
  book = build_bankbook (records);
  if (!xml_io_clip_write (&clipboard.clip, &clipboard.clip_size, book))
  {
    trace ("Error writing XML clip\n");
    goto fail1;
  }
  if_bankbook_destroy (book);

  gtk_selection_add_target (source,
			    GDK_SELECTION_PRIMARY,
			    get_atom (GNOFIN_CLIP_ATOM),
			    1);
  clipboard.selection_clear_id =
    gtk_signal_connect (GTK_OBJECT (source), "selection_clear_event",
    			GTK_SIGNAL_FUNC (on_selection_clear), NULL);
  clipboard.selection_get_id =
    gtk_signal_connect (GTK_OBJECT (source), "selection_get",
			GTK_SIGNAL_FUNC (on_selection_get), NULL);
  
  if (!gtk_selection_owner_set (source, GDK_SELECTION_PRIMARY, GDK_CURRENT_TIME))
  {
    g_print ("Failed to acquire selection\n");
    goto fail2;
  }

  /* Update immediately */
  if (g_hash_table_size (clipboard.sinks) > 0)
    on_clipboard_monitor ();

  return;

fail1:
  if_bankbook_destroy (book);
fail2:
  clipboard_clear ();
}

void
clipboard_clear (void)
{
  trace ("");

  if (clipboard.source)
  {
    gtk_signal_disconnect (GTK_OBJECT (clipboard.source), clipboard.selection_get_id);
    gtk_signal_disconnect (GTK_OBJECT (clipboard.source), clipboard.selection_clear_id);
    clipboard.source = NULL;

    g_free (clipboard.clip);
    clipboard.clip = NULL;
    clipboard.clip_size = 0;
  }
}
